#include <glut.h>

#include "flow.h"
#include "../../materials/plastic.h"

extern const Scene *scene;

#define FALL_MAXT 1000.0f
#define BOUNCE_MAXT 0.5f
#define FLOW_MAXT 2.5f
#define ATTACH_MAXT (BOUNCE_MAXT * 2 + FLOW_MAXT * 2)

#define GRAVITY_DELTA_THETA 5
#define SET_GRAVITY_INTERVAL 16

#define MEAN_DISTANCE_FACTOR 100.0f

#define MAX_GROUND_COUNTER 20

#define MAX_HOP_COUNTER 500

#define GROUND_WIDTH 700
#define GROUND_HEIGHT 1000
#define GROUND_ALTITUDE -100
#define RAIN_ALTITUDE 400
#define RAIN_DELTA 30			// rain angle perturbation
#define FALL_DELTA 15			// fall angle perturbation

//static Vector gravity(0, 0, -1);

// get perpendicular (to b) component in a
static Vector perpendicular(const Vector& a, const Vector& b)
{
	return  a - b * Dot(a, b) / Dot(b,b) ;
}

static bool tossCoin(float p)
{
	return RandomFloat() < p;
}

static Vector randomDisk()
{
	Vector ret;
	UniformSampleDisk(RandomFloat(), RandomFloat(), &ret.x, &ret.y);
	return ret;
}

static Vector randomCircle()
{
	float phi = 2 * M_PI * RandomFloat();
	return Vector(cos(phi), sin(phi), 0);
}

static float randomFloat2() // from -1 to 1
{
	return RandomFloat() * 2 - 1;
}

void
Particle::bounce()
{
	// count, book keeping
	if (tri->mesh->particleId[tri->faceIndex] != id) {
		tri->mesh->particleId[tri->faceIndex] = id;
		tri->mesh->count[tri->faceIndex]++;
	}


	Ray r(pos, (Vector)tri->mesh->faceN[tri->faceIndex], RAY_EPSILON, BOUNCE_MAXT);
	Intersection inter;

	if (scene->aggregate->Intersect(r, &inter)) {
		Triangle *fallTri = dynamic_cast<Triangle*>((Shape *)inter.dg.shape);
		assert(fallTri);

		state = BOUNCE;
		pos = inter.dg.p;
		tri = fallTri;
	}
	else {
		state = FLOW;
		pos = r(BOUNCE_MAXT);
	}
}

void
Particle::flow()
{
	Vector &normal = (Vector)tri->mesh->faceN[tri->faceIndex];
	float cosN = Dot(normal, gravity);

	if (cosN > 0 && cosN < 1) {
		/* random free fall */
		float meanDistance = MEAN_DISTANCE_FACTOR * sqrt(1 - cosN * cosN) / cosN; // tanN
		float probability = FLOW_MAXT / meanDistance;
		if (tossCoin(probability)) {
			state = FREEFALL;
			return;
		}
	}

	Vector direction;
	if (triPrev) {
		/* process concave triangles */
		direction = Cross(normal, (Vector)triPrev->mesh->faceN[triPrev->faceIndex]);
		if (direction.z > 0) {
			direction = - direction;
		}
	}
	else {
		direction = perpendicular(gravity, normal);
	}

	if (direction.Length() == 0) {
		setGravity();
		triPrev = NULL;
		return;
	}

	direction = Normalize(direction);

	Ray r(pos, direction, RAY_EPSILON, FLOW_MAXT);
	Intersection inter;

	triPrev = NULL;

	if (scene->aggregate->Intersect(r, &inter)) {

		Triangle *fallTri = dynamic_cast<Triangle*>((Shape *)inter.dg.shape);
		assert(fallTri);

		Vector direction2 = perpendicular(gravity, (Vector)fallTri->mesh->faceN[fallTri->faceIndex]);

		/* detected concave triangles */
		if (Dot(direction, direction2) <= 0 ||
			Dot((Vector)tri->mesh->faceN[tri->faceIndex], (Vector)fallTri->mesh->faceN[fallTri->faceIndex]) <= 0){
			triPrev = fallTri;
			state = FLOW;

//			printf("e ");
			edgeCounter++;
			if (edgeCounter % SET_GRAVITY_INTERVAL == 0) {
				setGravity();
				triPrev = NULL;
			}
		}
		else {
			state = BOUNCE;
			pos = inter.dg.p;
			tri = fallTri;
		}
	}
	else {
		flowCounter++;
		state = ATTACH;
		pos = r(FLOW_MAXT);
	}
}

void
Particle::attach()
{
	Ray r(pos, (Vector) - tri->mesh->faceN[tri->faceIndex], RAY_EPSILON, ATTACH_MAXT);
	Intersection inter;

	if (scene->aggregate->Intersect(r, &inter)) {
		Triangle *fallTri = dynamic_cast<Triangle*>((Shape *)inter.dg.shape);
		assert(fallTri);

		// on a same triangle
		if (tri == fallTri) {
			state = FLOW;
		}
		else {
			state = BOUNCE;
			pos = inter.dg.p;
			tri = fallTri;
		}
	}
	else {
		state = FREEFALL;
		pos = r(ATTACH_MAXT);
	}
}

void
Particle::freefall()
{
	Vector pert = randomCircle() * tan(Radians(FALL_DELTA)) * pow(RandomFloat(), 1);
	Vector direction = gravity + pert;

	Ray r(pos, direction, RAY_EPSILON, FALL_MAXT);
	Intersection inter;

	if (scene->aggregate->Intersect(r, &inter)) {
		Triangle *fallTri = dynamic_cast<Triangle*>((Shape *)inter.dg.shape);
		assert(fallTri);
		
		state = BOUNCE;
		pos = inter.dg.p;
		tri = fallTri;
	}
	else {
		float hitt = (-GROUND_ALTITUDE + r.o.z);
		pos = r(hitt);
		state = STUCK_FALL;
	}
}

void
Flow::renderGL()
{
	// rain area
	glColor3f(1, 0, 0);
	glPointSize(3);
	glBegin(GL_LINE_STRIP);
		glVertex3f(GROUND_WIDTH, GROUND_HEIGHT, GROUND_ALTITUDE);
		glVertex3f(-GROUND_WIDTH, GROUND_HEIGHT, GROUND_ALTITUDE);
		glVertex3f(-GROUND_WIDTH, -GROUND_HEIGHT, GROUND_ALTITUDE);
		glVertex3f(GROUND_WIDTH, -GROUND_HEIGHT, GROUND_ALTITUDE);
		glVertex3f(GROUND_WIDTH, GROUND_HEIGHT, GROUND_ALTITUDE);
	glEnd();

	// rain line
	glColor3f(0, 0, 1);
	glPointSize(3);
	glBegin(GL_LINES);
		glVertex3fv((float *)&rainBegin);
		glVertex3fv((float *)&rainEnd);
	glEnd();

	// trace
	glColor3f(0, 1, 0);
	for (u_int i = 0; i < trace.size(); i++) {
		trace[i].renderGL();
	}

	// stuck
	glColor3f(1, 1, 0);
	for (u_int i = 0; i < stuck.size(); i++) {
		stuck[i].renderGL();
	}

	// current particle
	glColor3f(1, 0, 1);
	glPointSize(4);
	glBegin(GL_POINTS);
		glVertex3fv((float *)&particle.pos);
	glEnd();
}

void
Flow::init() 
{
	particleId = 0;
	outputGround.open ("ground.dat", ios::out);
	outputSurface.open ("surface.dat", ios::out);
	trace.empty();
	stuck.empty();
	drop();
}

void
Flow::simulate()
{
	if (particle.simulCounter++ > MAX_HOP_COUNTER) {
		stuck.push_back(Deposit(particle));
		printf("%d ", particle.simulCounter);
		drop();
	}

	switch (particle.state) {
		case BOUNCE:
			particle.bounce();
			break;

		case FLOW:
			particle.flow();
			break;

		case ATTACH:
			particle.attach();
			break;

		case FREEFALL:
			particle.freefall();
			break;

		case STUCK_FALL:
		case STUCK_GROUND:
			trace.push_back(Deposit(particle));
			write(particle);
			printf("%d ", particle.simulCounter);
			drop();
			break;
	}
}

void
Flow::writeSurfacePoints()
{
	ofstream output;
	output.open ("surface.dat", ios::out);

	for (u_int i = 0; i < scene->shapes.size(); i++) {
		TriangleMesh *mesh = dynamic_cast<TriangleMesh *>(scene->shapes[i].ptr);

		if (!mesh) { continue;}

		for (int j = 0; j < mesh->ntris; j++) {
			int v1 = mesh->vertexIndex[3*j];
			int v2 = mesh->vertexIndex[3*j+1];
			int v3 = mesh->vertexIndex[3*j+2];

			Point &pos = (mesh->p[v1] + mesh->p[v2] + mesh->p[v3]) / 3.0f;

			output << pos.x << " " << pos.y << " " << pos.z << " " 
				<< mesh->count[j] << endl;
		}
	}
	output.close();
}

void
Flow::write(const Particle &particle)
{
	if (particle.state == STUCK_FALL) {
		const Point &pos = particle.pos;
		outputGround << pos.x << " " << pos.y << " " << pos.z << " " 
			<< particle.flowCounter << " " << particle.simulCounter << endl;
	}
}
	
void
Flow::drop()
{
	Intersection inter;

	while(true) {
		rainEnd = Point(GROUND_WIDTH * randomFloat2(), GROUND_HEIGHT * randomFloat2(), GROUND_ALTITUDE);
		rainBegin = rainEnd + randomDisk() * (RAIN_ALTITUDE - GROUND_ALTITUDE)* tan(Radians(RAIN_DELTA));
		rainBegin.z = RAIN_ALTITUDE;

		Ray r(rainBegin, rainEnd - rainBegin, RAY_EPSILON, FALL_MAXT);
		/* rain hit a object */
		if (scene->aggregate->Intersect(r, &inter)) {
			const GeometricPrimitive *geo = dynamic_cast<const GeometricPrimitive *>(inter.primitive);
			if (geo) {
			
				const Plastic *plastic = dynamic_cast<const Plastic *>(geo->material.ptr);

				/* rain hit plastic object, which is a tire */
				if (plastic) {
					continue;
				}
			}

			rainEnd = inter.dg.p;
			break;
		}
	};

	Triangle *fallTri = dynamic_cast<Triangle*>((Shape *)inter.dg.shape);
	assert(fallTri);
	
	particle.id = particleId++;
	particle.state = BOUNCE;
	particle.pos = inter.dg.p;
	particle.tri = fallTri;
	particle.triPrev = NULL;
	particle.edgeCounter = 0;
	particle.simulCounter = 0;
	particle.flowCounter = 0;

	particle.setGravity();

	if (particleId % 5000 == 0) {
		writeSurfacePoints();
	}
}

void
Particle::setGravity()
{
	float theta = Radians(GRAVITY_DELTA_THETA);
	float phi = 2 * M_PI * RandomFloat();

	gravity = Vector(sin(theta) * cosf(phi), sin(theta) * sinf(phi), - cos(theta));
}

Deposit::Deposit(const Particle &particle) : pos(particle.pos)
{
	state = particle.state;
}

void
Deposit::renderGL()
{
	glPointSize(2);
	glBegin(GL_POINTS);
		glVertex3fv((float *)&pos);
	glEnd();
}
