#include "lrt.h"
#include "color.h"
#include "reflection.h"
#include "shapes.h"
#include <stdarg.h>
BSDF::BSDF(const DifferentialGeometry &dg, BxDF *r, ...)
	: N(dg.N), S(dg.S), T(dg.T)
{
	va_list args;
	va_start(args, r);
	while (r != NULL) {
		Float wt = va_arg(args, double);
		if (wt > 0) {
			bxdfs.push_back(r);
			weights.push_back(wt);
		}
		r = va_arg(args, BxDF *);
	}
}
BSDF::~BSDF() {
	for (u_int i = 0; i < bxdfs.size(); ++i)
		delete bxdfs[i];
}
int BSDF::NumSpecular() const {
	int n = 0;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		if (bxdfs[i]->IsSpecular()) ++n;
	return n;
}
Spectrum BSDF::f(const Vector &wiW, const Vector &woW) const {
	Vector wi = WorldToLocal(wiW), wo = WorldToLocal(woW);
	Spectrum f = 0.;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		f += weights[i] * bxdfs[i]->f(wi, wo);
	return f;
}
Spectrum BSDF::f_delta(int component, const Vector &w,
		Vector *wi) const {
	BxDF *spec = NULL;
	Float weight = 0.;
	for (u_int i = 0; i < bxdfs.size(); ++i) {
		if (bxdfs[i]->IsSpecular()) {
			if (component-- == 0) {
				spec = bxdfs[i];
				weight = weights[i];
				break;
			}
		}
	}
	if (spec) {
		Vector wo = WorldToLocal(w);
		Spectrum f = weight * spec->f_delta(wo, wi);
		*wi = LocalToWorld(*wi);
		return f;
	}
	return 0.;
}
Spectrum BSDF::rho(int component)
{
    BxDF *spec = NULL;
	Float weight = 0.;
	for (u_int i = 0; i < bxdfs.size(); ++i) {
		if (bxdfs[i]->IsSpecular()) {
			if (component-- == 0) {
				spec = bxdfs[i];
				weight = weights[i];
				break;
			}
		}
	}
  	if (spec) {		
		Spectrum f = weight * spec->rho();
		return f;
	}
	return 0.;
}

Spectrum LambertianReflection::f(const Vector &wi,
		const Vector &wo) const {
	if (sameHemisphere(wi, wo))
		return R / M_PI;
	else
		return 0.;
}
Spectrum LambertianTransmission::f(const Vector &wi,
		const Vector &wo) const {
	if (!sameHemisphere(wi, wo))
		return T / M_PI;
	else
		return 0.;
}
Microfacet::Microfacet(const Spectrum &reflectance,
	MicrofacetDistribution *d) {
	R = reflectance;
	distribution = d;
}
Microfacet::~Microfacet() {
	delete distribution;
}
Spectrum Microfacet::f(const Vector &wi, const Vector &wo) const {
	if (sameHemisphere(wi, wo)) {
		Float cosThetaI = fabs(wi.z);
		Float cosThetaO = fabs(wo.z);
		return R * distribution->D(wi, wo) * G(wi, wo) *
			F(wi, wo) / (4. * cosThetaI * cosThetaO);
	}
	else return 0.;
}
SpecularReflection::SpecularReflection(const Spectrum &r) {
	R = r;
}
Spectrum SpecularReflection::f_delta(const Vector &wo,
		Vector *wi) const {
	*wi = Vector(-wo.x, -wo.y, wo.z);
	return R;
}
SpecularTransmission::SpecularTransmission(const Spectrum &r,
		Float indexi, Float indext) {
	R = r;
	eta = indexi / indext;
}
Spectrum SpecularTransmission::f_delta(const Vector &wo,
		Vector *wi) const {
	Float cosi = wo.z;
	Vector N(0,0,1);
	Float e = eta;
	if (cosi < 0.) {
		N = -N;
		cosi = -cosi;
		e = 1./eta;
	}

	Float cost = 1.0 - e * e * (1.0 - cosi * cosi);
	if (cost < 0.)
		return 0.; // *wi = Vector(-wo.x, -wo.y, wo.z);
	else {
		Float k = e * cosi - sqrtf(cost);
		*wi = k * N + e * -wo;
	}
	return R;
}
void BxDF::sample_f(const Vector &wi, Vector *wo) const {
	Float u1 = RandomFloat(), u2 = RandomFloat();
	Float r = sqrtf(u2);
	Float theta = 2. * M_PI * u1;
	Float x = r * cos(theta);
	Float y = r * sin(theta);
	Float z = sqrtf(1. - x*x - y*y);
	*wo = Vector(x, y, z);
	if (wi.z * wo->z < 0.) *wo = -*wo;
}
Float BxDF::weight(const Vector &wi, const Vector &wo) const {
	return M_PI / fabs(wo.Hat().z);
}
void LambertianTransmission::sample_f(const Vector &wi,
		Vector *wo) const {
	BxDF::sample_f(wi, wo);
	wo->z *= -1;
}
Spectrum BxDF::rho(const Vector &w) const {
	Spectrum r = 0.;
	int nSamples = 10;
	for (int i = 0; i < nSamples; ++i) {
		Vector wi;
		sample_f(w, &wi);
		r += f(w, wi) * weight(w, wi);
	}
	return r / nSamples;
}
Spectrum BxDF::rho() const {
	Spectrum r = 0.;
	int nSamples = 10;
	for (int i = 0; i < nSamples; ++i) {
		Vector wo, wi;
		while (1) {
			wo = Vector(RandomFloat(-1, 1), RandomFloat(-1, 1),
				RandomFloat(-1, 1));
			if (wo.LengthSquared() < 1.) break;
		}
		wo = wo.Hat();
		if (wo.z < 0.) wo.z *= -1;
		Float weight_o = 2.*M_PI;
		sample_f(wo, &wi);
		Float weight_i = weight(wo, wi);
		r += f(wo, wi) * weight_o * weight_i / M_PI;
	}
	return r / nSamples;
}
void Microfacet::sample_f(const Vector &wo, Vector *wi) const {
	distribution->sample_f(wo, wi);
}
Float Microfacet::weight(const Vector &wo, const Vector &wi) const {
	return distribution->weight(wo, wi);
}
void Blinn::sample_f(const Vector &wo, Vector *wi) const {
	float costheta = pow(RandomFloat(), 1. / (exponent+1));
	float sintheta = sqrtf(1. - costheta*costheta);
	float phi = RandomFloat() * 2. * M_PI;
	Vector H(sintheta * sin(phi), sintheta * cos(phi), costheta);
	*wi = -wo + 2. * Dot(wo, H) * H;
}
Float Blinn::weight(const Vector &wo, const Vector &wi) const {
	if (wi.z * wo.z < 0.) return 0;
	Vector H = (wi + wo).Hat();
	float costheta = H.z;
	return (4. * Dot(wi, H)) /
		((exponent + 1) * pow(costheta, exponent));

}
Spectrum BSDF::sample_f(const Vector &wiW, Vector *wo) const {
	Vector wi = WorldToLocal(wiW);
	int which = int(RandomFloat() * bxdfs.size());
	bxdfs[which]->sample_f(wi, wo);
	*wo = LocalToWorld(*wo);
	return f(wiW, *wo);
}
Float BSDF::weight(const Vector &wiW, const Vector &woW) const {
	Vector wi = WorldToLocal(wiW), wo = WorldToLocal(woW);
	Float wt = 0;
	for (u_int i = 0; i < bxdfs.size(); ++i)
		wt += bxdfs[i]->weight(wi, wo);
	return wt / bxdfs.size();
}
Spectrum BxDF::rho() const;
Spectrum BxDF::rho(const Vector &w) const;
