#ifndef PHOTONMAP_H
#define PHOTONMAP_H

#include "lrt.h"
#include "color.h"
#include "geometry.h"
#include "light.h"
#include "reflection.h"
#include "scene.h"
#include <vector>


struct Photon {
    Photon() {}
    Point pos; // photon position
    Normal norm; // normal where photon strikes
    short plane; // splitting plane for kd-tree
    Vector pdir;
    Spectrum power; // photon power
};

struct NearestPhotons {
    NearestPhotons() {}
    int max;
    int found;
    int got_heap;
    Point pos;
    std::vector<float> dist2;
    std::vector<Photon*> index;
};


class Photon_map {
public:
    Photon_map(Scene* scene, int max_phot);
    virtual ~Photon_map();

    void store(const Spectrum& power, const Point& pos, const Vector& dir, 
        const Normal& norm);
    void scale_photon_power(const float scale);
    void balance(void);
    void irradiance_estimate(Spectrum& irrad, const Point& pos, 
        const Normal& normal, const float max_dist, const int nphotons) const;
    void radiance_estimate(Spectrum& rad, BSDF* bsdf, 
	const Vector& wi, const Point& pos, const Normal& normal, 
			   const float max_dist, const int nphotons) const;
    void locate_photons(NearestPhotons * const np, const int index) const;
    // photon direction
    void photon_dir(Vector& dir, const Photon* p) const; 

    virtual void make_photon_map();

protected:
    void balance_segment(Photon** pbal, Photon** porg, const int index,
        const int start, const int end);
    void median_split(Photon** p, const int start, const int end,
        const int median, const int axis);

    // data members
    Scene* pScene;
    Photon* photons;

    int stored_photons;
    int half_stored_photons;
    int max_photons;
    int prev_scale;

    float costheta[256];
    float sintheta[256];
    float cosphi[256];
    float sinphi[256];

    BBox bbox;    
};

class GlobalPhotonMap : public Photon_map {
 public:
    GlobalPhotonMap(Scene* scene, int max_phot) : Photon_map(scene, max_phot) {}
    virtual void make_photon_map();
    virtual void make_photon_map_terrain();
    void ComputeBSSRDFRadiance(Spectrum &rad, Surf& surf, const Vector& wi);
    void ComputeBSSRDFRadianceTerrain(Spectrum &rad, Surf& surf, const Vector& wi);
//    void radiance_estimate(Spectrum& rad, BSDF* bsdf, 
//	const Vector& wi, const Point& pos, const Normal& normal, 
//			   const float max_dist, const int nphotons) const;
};


#endif // PHOTONMAP_H
