#include "photonmap.h"
#include "primitives.h"


Photon_map::Photon_map(Scene* scene, int max_phot) : 
    pScene(scene), photons(NULL), stored_photons(0), 
    max_photons(max_phot), prev_scale(1), bbox(Point(1e8, 1e8, 1e8), Point(-1e8, -1e8, -1e8))
{
    photons =  new Photon[max_photons+1]; // allocate memory
    if(photons == NULL) {
        std::cerr << "Out of memory" << std::endl;
        exit(-1);
    }

    double val = (1.0/256.0)*M_PI;
    // initialize direction conversion tables
    for(int i=0; i < 256; i++) {
        double angle = double(i) * val;
        costheta[i] = cos(angle);
        sintheta[i] = sin(angle);
        cosphi[i] = cos(2.0 * angle);
        sinphi[i] = sin(2.0 * angle);
    }        
}
Photon_map::~Photon_map()
{
    delete [] photons;
}

// store puts a photon into the flat array that will form the final kd-tree
void Photon_map::store(const Spectrum& power, const Point& pos, 
                       const Vector& dir, const Normal& norm)
{
    if(stored_photons > max_photons)
        return;

    stored_photons++;
    Photon *const node = &photons[stored_photons];

    node->pos.x = pos.x; node->pos.y = pos.y; node->pos.z  = pos.z;
    if(node->pos.x < bbox.pMin.x)  bbox.pMin.x = node->pos.x;
    if(node->pos.y < bbox.pMin.y)  bbox.pMin.y = node->pos.y;
    if(node->pos.z < bbox.pMin.z)  bbox.pMin.z = node->pos.z;

    if(node->pos.x > bbox.pMax.x)  bbox.pMax.x = node->pos.x;
    if(node->pos.y > bbox.pMax.y)  bbox.pMax.y = node->pos.y;
    if(node->pos.z > bbox.pMax.z)  bbox.pMax.z = node->pos.z;

    node->norm = norm.Hat();    
    node->power.Set(power);   
    node->pdir = dir.Hat();
}

// scale_photon_power is used to scale the power of all 
// photons, once they have been emitted from source
// scale = 1/(#emitted photons)
// call this function after each light source is processed
void Photon_map::scale_photon_power(const float scale)
{
    for(int i = prev_scale; i <= stored_photons; i++) {
        photons[i].power *= scale;
    }
    prev_scale = stored_photons;
}
void Photon_map::balance(void)
{
    if(stored_photons > 1) {
        // allocate two temporary arrays for the balancing procedure
        Photon **pa1 = new Photon*[stored_photons+1];
        Photon **pa2 = new Photon*[stored_photons+1];

        int i;
        for(i = 0; i <= stored_photons; i++) 
            pa2[i] = &photons[i];

        balance_segment(pa1, pa2, 1, 1, stored_photons);

        delete [] pa2;

        // reorganize balanced kd-tree (make a heap)
        int d, j=1, foo=1;
        Photon foo_photon = photons[j];

        for(i = 1; i <= stored_photons; i++) {
            d = pa1[j] - photons;
            pa1[j] = NULL;
            if(d!= foo)
                photons[j] = photons[d];
            else {
                photons[j] = foo_photon;

                if(i < stored_photons) {
                    for(; foo <= stored_photons; foo++)
                        if(pa1[foo] != NULL)
                            break;
                    foo_photon = photons[foo];
                    j = foo;
                }
                continue;
            }
            j = d;
        }
        delete [] pa1;
    }
    half_stored_photons = stored_photons/2 - 1;
}

// computes irradiance estimate at a given surface position
void Photon_map::irradiance_estimate(Spectrum& irrad, const Point& pos, 
    const Normal& normal, const float max_dist, const int nphotons) const
{
    NearestPhotons np;
    np.dist2.resize(nphotons + 1);
    np.index.resize(nphotons + 1);
    np.found = 0;
    np.got_heap = 0;
    np.dist2[0] = max_dist*max_dist;

    // locate nearest photons
    locate_photons(&np, 1);

    // if less than 8 photons return
    if(np.found < 8)
        return;

    // sum irradiance from all photons
    for(int i = 1; i <= np.found; ++i) {
        const Photon* p = np.index[i];
        // the photon_dir call and following if can be omitted
        // if the scene does not have any thin surfaces        
        if( Dot(p->pdir, normal) > 0.0f) {
            irrad += p->power;
        }
    }
    // estimate of density    
    // may need to multiply by some factor
    irrad *= ((1.0f/M_PI)/(np.dist2[0])); 
}

void Photon_map::radiance_estimate(Spectrum& rad, BSDF* bsdf, 
	const Vector& wi, const Point& pos, const Normal& normal, 
	const float max_dist, const int nphotons) const
{
    NearestPhotons np;    
    np.dist2.resize(nphotons + 1);
    np.index.resize(nphotons + 1);
    np.max = nphotons;
    np.pos = pos;
    np.found = 0;
    np.got_heap = 0;
    np.dist2[0] = max_dist*max_dist;

    // locate nearest photons
    locate_photons(&np, 1);


    // if less than 8 photons return
    if(np.found < 8) {
        return;
    }    

    float k = 1.1; // filter constant to make edges look sharper
    float filter = 1. / (k*np.dist2[0]);
    float wpc;

    // sum irradiance from all photons
    for(int i = 1; i <= np.found; ++i) {
        const Photon* p = np.index[i];
        // the photon_dir call and following if can be omitted
        // if the scene does not have any thin surfaces
        // apply cone filter
	    wpc = 1 - np.dist2[i] * filter;
	    rad += bsdf->f(wi, p->pdir) * p->power * wpc;
    }

    // estimate of density    
    // may need to multiply by some factor
      rad *= M_PI*((1.0f/M_PI)/(np.dist2[0])); 
}
    

void Photon_map::locate_photons(NearestPhotons * const np, const int index) const
{
    Photon* p = &photons[index];
    float dist1;

    if( index < half_stored_photons) {
        dist1 = np->pos[p->plane] - p->pos[p->plane];

        if(dist1 > 0.0) { // if dist1 is positive search right plane
            locate_photons(np, 2*index+1);
            if(dist1*dist1 < np->dist2[0] )
                locate_photons(np, 2*index);
        }
        else {
            locate_photons(np, 2*index);
            if(dist1*dist1 < np->dist2[0])
                locate_photons(np, 2*index+1);
        }
    }
    // compute squared distance between current photon and np->pos

    Vector dist(p->pos - np->pos);
    float dist2 = dist.LengthSquared();    
    
    if(dist2 < np->dist2[0]) {
        // we found a photon ...... Insert it in the candidate list                
        if(np->found < np->max) {
            // heap is not full; use array                               
            np->found++;
            np->dist2[np->found] = dist2;
            np->index[np->found] = p;
        }
        else {
            int j, parent;
            
            if(np->got_heap == 0) {// do we need to build the heap?
                // Build heap
                float dst2;
                Photon* phot;
                int half_found = np->found >> 1;
                for(int k = half_found; k >= 1; k--) {
                    parent = k;
                    phot = np->index[k];
                    dst2 = np->dist2[k];
                    while(parent <= half_found) {
                        j = parent+parent;
                        if(j < np->found && np->dist2[j] < np->dist2[j+1])
                            j++;
                        if(dst2 >= np->dist2[j])
                            break;

                        np->dist2[parent] = np->dist2[j];
                        np->index[parent] = np->index[j];
                        parent = j;
                    }
                    np->dist2[parent] = dst2;
                    np->index[parent] = phot;
                }
                np->got_heap = 1;
            }
            // insert new photon into max heap
            // delete largest element, insert new and reorder the heap

            parent = 1;
            j = 2;
            while( j <= np->found) {
                if(j < np->found && np->dist2[j] < np->dist2[j+1])
                    j++;
                if( dist2 > np->dist2[j])
                    break;
                np->dist2[parent] = np->dist2[j];
                np->index[parent] = np->index[j];
                parent = j;
                j += j;
            }
            np->index[parent] = p;
            np->dist2[parent] = dist2;

            np->dist2[0] = np->dist2[1];
        }
    }
}
// return photon direction
void Photon_map::photon_dir(Vector& dir, const Photon* p) const
{
//    dir.x = sintheta[p->theta]*cosphi[p->phi];
  //  dir.y = sintheta[p->theta]*sinphi[p->phi];
   // dir.z = costheta[p->theta];
}

void Photon_map::balance_segment(Photon** pbal, Photon** porg, const int index,
    const int start, const int end)
{
    // compute new median
    int median = 1;
    while((4*median) <= (end-start+1))
        median+=median;

    if((3*median) <= (end-start+1)) {
        median += median;
        median += start - 1;
    }
    else 
        median = end - median +1;

    // find axis to split along

    int axis = 2;
    if( (bbox.pMax[0] - bbox.pMin[0]) > (bbox.pMax[1] - bbox.pMin[1]) &&
        (bbox.pMax[0] - bbox.pMin[0]) > (bbox.pMax[2] - bbox.pMin[2]) )
        axis = 0;
    else if( (bbox.pMax[1] - bbox.pMin[1]) > (bbox.pMax[2] - bbox.pMin[2]) )
        axis = 1;

    // partition photon block around the median
    median_split(porg, start, end, median, axis);
    pbal[index] = porg[median];
    pbal[index]->plane = axis;

    // recursively balance left and right block

    if(median > start) {
        // balance left segment
        if( start < median - 1) {
            const float tmp = bbox.pMax[axis];
            bbox.pMax[axis] = pbal[index]->pos[axis];
            balance_segment(pbal, porg, 2*index, start, median-1);
            bbox.pMax[axis] = tmp;
        }
        else {
            pbal[2*index] = porg[start];
        }
    }

    if(median < end) {
        // balance right segment
        if(median + 1 < end) {
            const float tmp = bbox.pMin[axis];
            bbox.pMin[axis] = pbal[index]->pos[axis];
            balance_segment(pbal, porg, 2*index + 1, median+1, end);
            bbox.pMin[axis] = tmp;
        }
        else {
            pbal[2*index+1] = porg[end];
        }
    }
}

#define swap(ph, a,  b) {Photon *ph2 = ph[a]; ph[a] = ph[b]; ph[b] = ph2;}
// splits the photon array into two separated pieces around the median, with
// all photons below the mdean in the lower half and all photons above
// the median in upper half. The comparison criteria
void Photon_map::median_split(Photon** p, const int start, const int end,
    const int median, const int axis)
{
    int left = start;
    int right = end;

    while( right > left) {
        const float v = p[right]->pos[axis];
        int i = left-1;
        int j = right;
        for(;;) {
            while(p[++i]->pos[axis] < v)
                ;
            while(p[--j]->pos[axis] > v && j > left)
                ;
            if( i >= j)
                break;
            swap(p, i, j);
        }

        swap(p, i, right);
        if(i >= median)
            right = i - 1;
        if(i <= median)
            left = i + 1;
    }
}

void Photon_map::make_photon_map()
{
    int numPoints = 0;
    std::cerr << "Making Photon Map " << std::endl;

    int photonsPerLight = max_photons / (pScene->lights.size()-1);

    cerr << "num photons per light " << photonsPerLight << endl;

    // for all lights calculate the photon map
    for(int numLight = 0; numLight < (int)pScene->lights.size()-1; ++numLight) 
    {
        // get current light
        Vector ww;
	PointLight *pl = (PointLight*)pScene->lights[numLight];
        Spectrum pr = 4*M_PI*pl->I(ww);
        if(pr.Intensity() < 0.1)
            continue;

        const Light *light = pScene->lights[numLight];        
        for (int i = 0; i < photonsPerLight; i++)
	    {	
		Surf surf;
	        PointLight* ipLight = (PointLight*) light;            
	        Ray photonRay;
            
	        // Generate the initial photon ray
		    Float x = 1, y = 1, z = 1;            
		    while (x * x + y * y + z * z > 1)
		    {                                
		        x = RandomFloat(-1, 1);                
		        y = RandomFloat(-1, 1);
		        z = RandomFloat(-1, 1);                                
		    }
                        
		    Vector photonDirection(x, y, z);
		    photonDirection.Normalize();
		    Float maxt = INFINITY;
		    photonRay = Ray(ipLight->lightPos, photonDirection, 1e-4, maxt);
		
		    //cerr << i << " : " << photonRay.O << " :  " << photonRay.D << endl;
        	// Generate the photon, making sure that it always first hits
	        // a surface capable of generating caustics if in caustic map
	        // mode
	        if (pScene->Intersect(photonRay, &surf))
	        {                			
		        bool addPhotonToMap = false;
		        bool isIndirectPhoton = false;		        
                Spectrum powerMod(1.);                
                while (true)
		        {		            
		            BSDF* bsdf = surf.getBSDF();		           
		            // Normalize the Russian roulette probabilities in
		            // preparation for generating the global map
		            Float total;

                    Normal nm = surf.dgShading.N;
			        Vector wi = photonRay.D - 2.0 * Dot(nm, photonRay.D) * Vector(nm);        
                    Spectrum diffFactor = bsdf->f(wi, -photonRay.D);
                    Float pDiffuse = diffFactor.Intensity();

                    Spectrum refFactor(0.), transFactor(0.);
                    if(bsdf->NumSpecular() > 0)
                        refFactor = bsdf->rho(0);
                    if(bsdf->NumSpecular() > 1)
                        transFactor = bsdf->rho(1);

                    Float pRef = refFactor.Intensity();
                    Float pTrans = transFactor.Intensity();

                    // normalize probabilities
                    Float pAbsorption = pDiffuse;
			        total = pAbsorption + pDiffuse;/* + pRef + pTrans*/;			        pAbsorption /= total;
			        pDiffuse /= total;
//                    pRef /= total;
				//                  pTrans /= total;
			        
                    maxt = INFINITY;
                    photonRay.D = -photonRay.D;

           			// For the global map, randomly select absorption,
			        // diffuse reflection, specular reflection or
			        // transmission for the photon, and handle each case
			        // separately
			        Float epsilon = RandomFloat(0, 1);
			        if (epsilon < pAbsorption)
			        {
			            // absorption
			            if (isIndirectPhoton == false) {
				            addPhotonToMap = false;
				            break;
			            }

			            addPhotonToMap = true;
			            break;
			        }
			        // it can only be an Indirect photon by now...
			        else if (epsilon < pAbsorption + pDiffuse) {
			            // diffuse reflection
			            Normal nm = surf.dgShading.N;
			            Vector wo = photonRay.D;
                        photonRay.D = -photonRay.D - 2.0 * Dot(nm,
			            -photonRay.D) * Vector(nm);               
                        Spectrum f = bsdf->f(photonRay.D, wo);
                        if(f.Intensity() < 1e-4) {
                            // intensity is very low absorb it 
			                if (isIndirectPhoton == false) {
				                addPhotonToMap = false;
				                break;
			                }

			                addPhotonToMap = true;
			                break;
                        }
                        //f /= f.Intensity(); // normalize the intensity
                        Float mulFactor = 1.0f / (1.0f - pAbsorption);
			            powerMod *= mulFactor*f;
			        }
/*                    else if(epsilon < pAbsorption + pDiffuse + pRef) {     
                        Vector wo;                        
                        Spectrum f = bsdf->f_delta(0, photonRay.D, &wo);
                        if(f.Intensity() < 1e-4) {
                            // intensity is very low absorb it 
			                if (isIndirectPhoton == false) {
				                addPhotonToMap = false;
				                break;
			                }

			                addPhotonToMap = true;
			                break;
                        }
                        //f /= f.Intensity(); // normalize the intensity

                        Float mulFactor = 1.0f / (1.0f - pAbsorption - pDiffuse);
                        powerMod *= f*mulFactor;
                        photonRay.D = wo;
                    }
                    else if(epsilon < pAbsorption + pDiffuse + pRef + pTrans) {
                        Vector wo;                        
                        Spectrum f = bsdf->f_delta(0, photonRay.D, &wo);
                        if(f.Intensity() < 1e-4) {
                            // intensity is very low absorb it 
			                if (isIndirectPhoton == false) {
				                addPhotonToMap = false;
				                break;
			                }

			                addPhotonToMap = true;
			                break;
                        }
                      //f /= f.Intensity(); // normalize the intensity

                        Float mulFactor = 1.0f / (1.0f - pAbsorption - pDiffuse - pRef);
                        powerMod *= f*mulFactor;
                        photonRay.D = wo;
                    }*/
                    isIndirectPhoton = true;

                    photonRay.O = surf.dgGeom.P;
                    photonRay.mint = 1e-4;
                    photonRay.maxt = INFINITY;

		            if (!pScene->Intersect(photonRay, &surf))
		            {			         
			            addPhotonToMap = false;
			            break;
		            }
                }

         		if (addPhotonToMap)
		        {		            
        		    Spectrum power = 4*M_PI*ipLight->I(photonRay.D) /
		        photonsPerLight; 

			    //	    cerr << "Intensity " << ipLight->I(photonRay.D) << " power    " <<  power << endl;
//                    powerMod /= powerMod.Intensity();
		    power *= powerMod;            
//		    cerr << powerMod << endl;
//		    cerr << "Effective power entered " << power << endl;
                    Vector direction = Vector(photonRay.O - surf.dgGeom.P);
                    store(power, surf.dgGeom.P, direction, surf.dgShading.N);
		            
		            numPoints++;

                    if (numPoints % 10000 == 0) cerr << '#';
		            else if (numPoints % 1000 == 0) cerr << '!';
                }
            }
        }
    }       

    std::cerr << stored_photons << " size map generated " << std::endl;
    balance();
}

void GlobalPhotonMap::make_photon_map_terrain()
{
    int numPoints = 0;
    std::cerr << "Making Global Photon Map for Transluscent Terrain..." << std::endl;

    int photonsPerLight = max_photons / (pScene->lights.size() - 1);

    cerr << "num photons per light " << photonsPerLight << endl;

    // for all lights calculate the photon map
    for(int numLight = 0; numLight < (int)pScene->lights.size() - 1; ++numLight) 
    {
        // get current light
        Vector ww;
	PointLight *pl = (PointLight*)pScene->lights[numLight];
        Spectrum power = 4*M_PI*pl->I(ww);
        if(power.Intensity() < 0.1)
            continue;
        power /= photonsPerLight;

        const Light *light = pScene->lights[numLight];        
        for (int i = 0; i < photonsPerLight; i++)
	    {	
    		Surf surf;
	        PointLight* ipLight = (PointLight*) light;            
	        Ray photonRay;
            
	        // Generate the initial photon ray
		    Float x = 1, y = 1, z = 1;            
		    while (x * x + y * y + z * z > 1)
		    {                                
		        x = RandomFloat(-1, 1);                
		        y = RandomFloat(-1, 1);
		        z = RandomFloat(-1, 1);                                
		    }
                        
		    Vector photonDirection(x, y, z);
		    photonDirection.Normalize();
		    Float maxt = INFINITY;
		    photonRay = Ray(ipLight->lightPos, photonDirection, 1e-4, maxt);
		
		    //cerr << i << " : " << photonRay.O << " :  " << photonRay.D << endl;
        	// Generate the photon, making sure that it always first hits
	        // a surface capable of generating caustics if in caustic map
	        // mode
	        if (pScene->Intersect(photonRay, &surf))
	        {     
		 // only add intersections with transluscent materials                
//			cerr << "Surface Intersection occured " << endl;
                if(surf.primitive->material->getType() != TRANSLUSCENT_TERRAIN)
			        continue;
        		Vector direction = Vector(photonRay.O - surf.dgGeom.P);
                store(power, surf.dgGeom.P, direction.Hat(), surf.dgShading.N);
		        
		        numPoints++;

                if (numPoints % 10000 == 0) cerr << '#';
		            else if (numPoints % 1000 == 0) cerr << '!';
            }
        }
    }       

    std::cerr << stored_photons << " size map generated " << std::endl;
    balance();

}

void GlobalPhotonMap::make_photon_map()
{
    int numPoints = 0;
    std::cerr << "Making Global Photon Map for Transluscent materials..." << std::endl;

    int photonsPerLight = max_photons / (pScene->lights.size() - 1);

    cerr << "num photons per light " << photonsPerLight << endl;

    // for all lights calculate the photon map
    for(int numLight = 0; numLight < (int)pScene->lights.size() - 1; ++numLight) 
    {
        // get current light
        Vector ww;
	PointLight *pl = (PointLight*)pScene->lights[numLight];
        Spectrum power = 4*M_PI*pl->I(ww);
        if(power.Intensity() < 0.1)
            continue;
        power /= photonsPerLight;

        const Light *light = pScene->lights[numLight];        
        for (int i = 0; i < photonsPerLight; i++)
	    {	
    		Surf surf;
	        PointLight* ipLight = (PointLight*) light;            
	        Ray photonRay;
            
	        // Generate the initial photon ray
		    Float x = 1, y = 1, z = 1;            
		    while (x * x + y * y + z * z > 1)
		    {                                
		        x = RandomFloat(-1, 1);                
		        y = RandomFloat(-1, 1);
		        z = RandomFloat(-1, 1);                                
		    }
                        
		    Vector photonDirection(x, y, z);
		    photonDirection.Normalize();
		    Float maxt = INFINITY;
		    photonRay = Ray(ipLight->lightPos, photonDirection, 1e-4, maxt);
		
		    //cerr << i << " : " << photonRay.O << " :  " << photonRay.D << endl;
        	// Generate the photon, making sure that it always first hits
	        // a surface capable of generating caustics if in caustic map
	        // mode
	        if (pScene->Intersect(photonRay, &surf))
	        {     
		 // only add intersections with transluscent materials                
//			cerr << "Surface Intersection occured " << endl;
                if(surf.primitive->material->getType() != TRANSLUSCENT)
			        continue;
        		Vector direction = Vector(photonRay.O - surf.dgGeom.P);
                store(power, surf.dgGeom.P, direction.Hat(), surf.dgShading.N);
		        
		        numPoints++;

                if (numPoints % 10000 == 0) cerr << '#';
		            else if (numPoints % 1000 == 0) cerr << '!';
            }
        }
    }       

    std::cerr << stored_photons << " size map generated " << std::endl;
    balance();
}

static Float Ft(Float index, const Vector& w, const Normal& N) 
{    
    Float cosTheta = Dot(w, N);
    // using Schlick approximation
    Float Fo = 0.04;
    Float OneMinusCosThetaPower5 = (1 - cosTheta) * (1 - cosTheta);
    OneMinusCosThetaPower5 *= OneMinusCosThetaPower5;
    OneMinusCosThetaPower5 *= (1 - cosTheta);
    Float Fr =  Fo + (1-Fo)*OneMinusCosThetaPower5;
    return fabs((1 - Fr));
}
void GlobalPhotonMap::ComputeBSSRDFRadianceTerrain(Spectrum &rad, Surf& surf, const Vector& wi)
{
    TransluscentTerrain* mat = (TransluscentTerrain*)surf.primitive->material;
    Spectrum sigmaS_dash = mat->sigS*(1 - mat->g);
    // compute extinction coefficent
    Spectrum sigmaT_dash = sigmaS_dash + mat->sigA; 
    // compute diffusion constant
//    Spectrum D = Spectrum(1.) / (3.*sigmaT_dash);
    // compute the effective transport coefficient
    Spectrum sigmaTR = 3.*mat->sigA*sigmaT_dash;
    sigmaTR.Pow(Spectrum(0.5)); // take square root

    Float Fdr = -1.440/(mat->index*mat->index) + 0.710/mat->index + 0.668
        + 0.0636*mat->index;

    Float A = (1 + Fdr) / (1 - Fdr);

    ///////////////////// Diffusion Approximation ////////////////////////
    // mean free path length 
    Float lu = 1. / sigmaT_dash.Intensity();
    // now get the photons in this range

    int nphotons = 500;
    NearestPhotons np;
    np.dist2.resize(nphotons + 1);
    np.index.resize(nphotons + 1);
    np.max = nphotons;
    np.pos = surf.dgShading.P; // point where light leaves
    np.found = 0;
    np.got_heap = 0;
//    np.dist2[0] = lu*lu;
   // np.dist2[0] = lu*lu;
    //cerr << 'lu =' << lu << endl;
    np.dist2[0] = 1.5*1.5;

    // locate nearest photons
    locate_photons(&np, 1);

    Spectrum albedo_dash = sigmaS_dash / sigmaT_dash;


    //float k = 1.1; // filter constant to make edges look sharper
    //float filter = 1. / (k*np.dist2[0]);
    //float wpc;
    // dipole offsets
    Float zr = lu;
    Float zv = lu * (1. + 4./(3*A));
    Float sigTR = sigmaTR.Intensity();
    Float sigT = sigmaT_dash.Intensity();    

    Float fto = Ft(mat->index, wi, surf.dgShading.N.Hat());
    // Compute Lo
    for(int i = 1; i <= np.found; ++i) {
        const Photon* p = np.index[i];
        // the photon_dir call and following if can be omitted
        // if the scene does not have any thin surfaces
        // apply cone filter
	  //  wpc = 1 - np.dist2[i] * filter;
	    Spectrum dphi =  p->power;// * wpc;
        Float r = Distance(surf.dgShading.P, p->pos);
        Float dr = sqrt(r*r + zr*zr);
        Float dv = sqrt(r*r + zv*zv);

        Float C1 = zr * (sigTR*dr + 1);
        Float C2 = zv * (sigTR*dv + 1);

        Spectrum Rd = (albedo_dash /(4*M_PI))*( 
            (C1 * expf(-sigTR*dr) / (sigT * dr * dr * dr)) +
            (C2 * expf(-sigTR*dv) / (sigT * dv * dv * dv)) );

        Spectrum Sd = (1./M_PI)*Ft(mat->index, p->pdir, p->norm)*Rd*fto;

        rad += Sd * dphi;	
    }
    // may need to scale it by some factor ?
    rad*=10;
}
void GlobalPhotonMap::ComputeBSSRDFRadiance(Spectrum &rad, Surf& surf, const Vector& wi)
{
    Transluscent* mat = (Transluscent*)surf.primitive->material;
    Spectrum sigmaS_dash = mat->sigS*(1 - mat->g);
    // compute extinction coefficent
    Spectrum sigmaT_dash = sigmaS_dash + mat->sigA; 
    // compute diffusion constant
//    Spectrum D = Spectrum(1.) / (3.*sigmaT_dash);
    // compute the effective transport coefficient
    Spectrum sigmaTR = 3.*mat->sigA*sigmaT_dash;
    sigmaTR.Pow(Spectrum(0.5)); // take square root

    Float Fdr = -1.440/(mat->index*mat->index) + 0.710/mat->index + 0.668
        + 0.0636*mat->index;

    Float A = (1 + Fdr) / (1 - Fdr);

    ///////////////////// Diffusion Approximation ////////////////////////
    // mean free path length 
    Float lu = 1. / sigmaT_dash.Intensity();
    // now get the photons in this range

    int nphotons = 500;
    NearestPhotons np;
    np.dist2.resize(nphotons + 1);
    np.index.resize(nphotons + 1);
    np.max = nphotons;
    np.pos = surf.dgShading.P; // point where light leaves
    np.found = 0;
    np.got_heap = 0;
//    np.dist2[0] = lu*lu;
   // np.dist2[0] = lu*lu;
    //cerr << 'lu =' << lu << endl;
    np.dist2[0] = 0.5*0.5;

    // locate nearest photons
    locate_photons(&np, 1);

    Spectrum albedo_dash = sigmaS_dash / sigmaT_dash;


    //float k = 1.1; // filter constant to make edges look sharper
    //float filter = 1. / (k*np.dist2[0]);
    //float wpc;
    // dipole offsets
    Float zr = lu;
    Float zv = lu * (1. + 4./(3*A));
    Float sigTR = sigmaTR.Intensity();
    Float sigT = sigmaT_dash.Intensity();    

    Float fto = Ft(mat->index, wi, surf.dgShading.N.Hat());
    // Compute Lo
    for(int i = 1; i <= np.found; ++i) {
        const Photon* p = np.index[i];
        // the photon_dir call and following if can be omitted
        // if the scene does not have any thin surfaces
        // apply cone filter
	  //  wpc = 1 - np.dist2[i] * filter;
	    Spectrum dphi =  p->power;// * wpc;
        Float r = Distance(surf.dgShading.P, p->pos);
        Float dr = sqrt(r*r + zr*zr);
        Float dv = sqrt(r*r + zv*zv);

        Float C1 = zr * (sigTR*dr + 1);
        Float C2 = zv * (sigTR*dv + 1);

        Spectrum Rd = (albedo_dash /(4*M_PI))*( 
            (C1 * expf(-sigTR*dr) / (sigT * dr * dr * dr)) +
            (C2 * expf(-sigTR*dv) / (sigT * dv * dv * dv)) );

        Spectrum Sd = (1./M_PI)*Ft(mat->index, p->pdir, p->norm)*Rd*fto;

        rad += Sd * dphi;	
    }
    // may need to scale it by some factor ?
    //rad*=10;
}
