#include "heightfield.h"
#define VERTEX_VALUE(x,y) ((x)+(y)*nx)

Heightfield::Heightfield(const Transform &o2w, int x, int y, float *zs)
	: Shape(o2w), nx(x), ny(y), _bbExists(false), _bb(Point(0, 0, 0))
{
    _gWidthX = 1.0f/((float)(nx-1));
    _gWidthY = 1.0f/((float)(ny-1));
	z = new float[nx*ny]; // no. of heights required
	memcpy(z, zs, nx*ny*sizeof(float));// copy data in z
	_bb = Bound();
	_bbExists = true;
	CalculatePhongNormals();
}

Heightfield::~Heightfield() {
	delete[] z; // deallocate memory for heights
}

void Heightfield::CalculatePhongNormals()
{
	_phongNormal.reserve(nx*ny);
	Assert(_phongNormal.capacity() >= (unsigned int)(nx*ny));
	Normal v(1e-6, 1e-6, 1e-6);
	// add nx*ny copies of v
	_phongNormal.insert(_phongNormal.begin(), nx*ny, v);

	int i, j, index0, index1, index2, index3;
	for (j = 0; j < ny-1; ++j) {
		for (i = 0; i < nx-1; ++i) {
			// compute points in object space from current index
			index0 = VERTEX_VALUE(i, j);
			Point p0(i*_gWidthX, j*_gWidthY, z[index0]);
			p0 = ObjectToWorld(p0);
			index1 = VERTEX_VALUE(i+1, j);
			Point p1((i+1)*_gWidthX, j*_gWidthY, z[index1]);
			p1 = ObjectToWorld(p1);
			index2 = VERTEX_VALUE(i+1, j+1);
			Point p2((i+1)*_gWidthX, (j+1)*_gWidthY, z[index2]);
			p2 = ObjectToWorld(p2);
			index3 = VERTEX_VALUE(i, j+1);
			Point p3(i*_gWidthX, (j+1)*_gWidthY, z[index3]);
			p3 = ObjectToWorld(p3);
			
			Vector edge1 = p1 - p0;
			Vector edge2 = p2 - p0;
   
	// begin calculating determinant - also used to calculate U parameter 
			Normal n = Normal(Cross(edge2, edge1));
			n = n.Hat()/6.;
			
			_phongNormal[index0] += n;
			_phongNormal[index1] += n;
			_phongNormal[index2] += n;

			Vector edge3 = p3 - p0;
			n = Normal(Cross(edge3, edge2));
			n = n.Hat()/6;
			
			_phongNormal[index0] += n;
			_phongNormal[index2] += n;
			_phongNormal[index3] += n;
		}
	}	
}


// calculate bounding box for the heightfield
BBox Heightfield::Bound() const {
	if(_bbExists)
		return _bb;
	float minz = z[0], maxz = z[0];
	for (int i = 1; i < nx*ny; ++i) {
		if (z[i] < minz) minz = z[i];
		if (z[i] > maxz) maxz = z[i];
	}

	return BBox(Point(0,0,minz), Point(1,1,maxz));
}
bool Heightfield::CanIntersect() const {
	return true;
}
void Heightfield::Refine(vector<Shape *> &refined) const {
	// compute no. of triangles in the heightfield
	int ntris = 2*(nx-1)*(ny-1);
	// we need 3 vertex indexes for each triangle
	int *verts = new int[3*ntris];
	Point *P = new Point[nx*ny];
	int x, y;
	P = new Point[nx*ny];// allocate memory for total no.  points
	int pos = 0;
	for (y = 0; y < ny; ++y) {
		for (x = 0; x < nx; ++x) {
		        // calculate x pos in world coordinates
			P[pos].x = (float)x / (float)(nx-1);
			// calculate y pos in world coordinates
			P[pos].y = (float)y / (float)(ny-1);
			P[pos].z = z[pos];
			++pos;
		}
	}
	int* vp = verts;
	for (y = 0; y < ny-1; ++y) {
		for (x = 0; x < nx-1; ++x) {
	#define VERT(x,y) ((x)+(y)*nx)
			*vp++ = VERT(x, y);
			*vp++ = VERT(x+1, y);
			*vp++ = VERT(x+1, y+1);
	
			*vp++ = VERT(x, y);
			*vp++ = VERT(x+1, y+1);
			*vp++ = VERT(x, y+1);
		}
	#undef VERT
	}
	refined.push_back(new TriangleMesh(ObjectToWorld, ntris,
		nx*ny, verts, P));
	delete[] P;
	delete[] verts;
}


/*
*****************************************************************
Fast, Minimum Storage Ray-Triangle Intersection
Tomas Mller
*****************************************************************
*/
 bool Heightfield::intersect_triangle(const Ray& ray, const Point& pp0, 
			       const Point& pp1, const Point& pp2, 
			       DifferentialGeometry *dg,
			       int i0, int i1, int i2) const
{   
   float u, v;
   // find vectors for two edges sharing vert0 
   Point p0 = ObjectToWorld(pp0);
   Point p1 = ObjectToWorld(pp1);
   Point p2 = ObjectToWorld(pp2);

// get interpolated normals of the three vertices and convert them to 
// world space

   Normal n0 = _phongNormal[i0].Hat();
   Normal n1 = _phongNormal[i1].Hat();
   Normal n2 = _phongNormal[i2].Hat();   
   
   Vector edge1 = p1 - p0;
   Vector edge2 = p2 - p0;
   
   // begin calculating determinant - also used to calculate U parameter 
   Vector pvec = Cross(ray.D, edge2);

   // if determinant is near zero, ray lies in plane of triangle 
   float det = Dot(edge1, pvec);

// the non-culling branch 
   if (det == 0. )
     return false;
   float inv_det = 1. / det;

  // calculate distance from vert0 to ray origin 
   Vector tvec = ray.O - p0;   

   // calculate U parameter and test bounds 
   u = Dot(tvec, pvec) * inv_det;
   if (u < 0. || u > 1.0)
     return false;

   // prepare to test V parameter 
   Vector qvec = Cross(tvec, edge1);   

   // calculate V parameter and test bounds 
   v = Dot(ray.D, qvec) * inv_det;
   if (v < 0.0 || u + v > 1.0)
     return false;

   // calculate t, ray intersects triangle 
   float t = Dot(edge2, qvec) * inv_det;

   if(t < ray.mint || t > ray.maxt)
	   return false;

   // calculate parameters for differential geometry structure


   float tex[3][2];
   tex[0][0] = 0;  tex[0][1] = 0;
   tex[1][0] = 1;  tex[1][1] = 0;
   tex[2][0] = 1;  tex[2][1] = 1;

   
// get interpolated normal here from u & v barycentric coordinates

   Normal N = Normal((1-u-v)*n0 + u*n1 + v*n2);   

   Vector dPdu, dPdv;
   dPdv = Cross(Vector(N), Vector(N.y, N.z, N.x));
   dPdu = Cross(Vector(N), dPdv);

   Float uu = (1 - u - v) * tex[0][0] + u * tex[1][0] + v * tex[2][0];
   Float vv = (1 - u - v) * tex[0][1] + u * tex[1][1] + v * tex[2][1];
   *dg = DifferentialGeometry(ray(t), N.Hat(), dPdu.Hat(),
			      Cross(Vector(N), dPdu).Hat(), uu, vv);

   ray.maxt = t;
   return true;
}


bool Heightfield::TriIntersectTest(const Ray& r, int i, int j, 
				   DifferentialGeometry *dg) const
{
    int index0 = VERTEX_VALUE(i, j);
    Point p0(i*_gWidthX, j*_gWidthY, z[index0]);
    int index1 = VERTEX_VALUE(i+1, j);
    Point p1((i+1)*_gWidthX, j*_gWidthY, z[index1]);
    int index2 = VERTEX_VALUE(i+1, j+1);
    Point p2((i+1)*_gWidthX, (j+1)*_gWidthY, z[index2]);
    int index3 = VERTEX_VALUE(i, j+1);
    Point p3(i*_gWidthX, (j+1)*_gWidthY, z[index3]);

    bool found = false;

    if(intersect_triangle(r, p0, p1, p2, dg, index0, index1, index2)) {      
	    found = true;
    }
    if(intersect_triangle(r, p0, p2, p3, dg, index0, index2, index3)) {	
        found =  true;
    }
    return found;
}


/*
*****************************************************************
// Algorithm Cleary and Wyvill ray walking algorithm
*****************************************************************
*/
bool Heightfield::IntersectTest(const Ray &r, const Ray& ray,
				DifferentialGeometry *dg) const
{
//    Point p1, p2; 
    Float rayT = r.maxt;
     // p1 is near intersection of ray with bounding box in object space
    if(_bb.Inside(r(r.mint))) { // ray is already inside
	    rayT = r.mint;
    }
    else {
	    Ray rb = r;	    
	    if(!_bb.IntersectP(rb)) {     
		    return false;  
	    }  
	    rayT = rb.maxt;
    }

    Point gIntersect = r(rayT);

    // setup x stepping
    int xx = (int)(gIntersect.x * (nx-1));
    if( xx == nx) xx--;

    Assert( xx >= 0 && xx < nx);
    float NextXCrossing, DeltaX;
    int StepX, OutX;

    if(fabs(r.D.x) < 1e-6) {
	    // handle ray perpendicular to x  
	    NextXCrossing = INFINITY;
	    DeltaX = 0;
	    OutX = -1;
    }
    else if(r.D.x > 0) {
	    // handle ray with +ve x direction
	    NextXCrossing = 
		    rayT + ( (xx+1)*_gWidthX - gIntersect.x) / r.D.x;
	    DeltaX =  _gWidthX / r.D.x;
	    StepX = 1;
	    OutX = nx;
    }
    else {
	    // handle ray with -ve x direction
	    NextXCrossing = 
		    rayT + (xx*_gWidthX - gIntersect.x) / r.D.x;
	    DeltaX = - _gWidthX / r.D.x;
	    StepX = -1;
	    OutX = -1;
    }


    // setup y stepping
    int yy = (int)(gIntersect.y * (ny-1));
    if( yy >= ny) yy--;

    Assert( yy >= 0 && yy < ny);
    float NextYCrossing, DeltaY;
    int StepY, OutY;

    if(fabs(r.D.y) < 1e-6) {
	    // handle ray perpendicular to y
	    NextYCrossing = INFINITY;
	    DeltaY = 0;
	    OutY = -1;
    }
    else if(r.D.y > 0) {
	    // handle ray with +ve x direction
	    NextYCrossing = 
		    rayT + ( (yy+1)*_gWidthY - gIntersect.y) / r.D.y;
	    DeltaY =  _gWidthY / r.D.y;
	    StepY = 1;
	    OutY = ny;
    }
    else {
	    // handle ray with -ve x direction
	    NextYCrossing = 
		    rayT + (yy*_gWidthY - gIntersect.y) / r.D.y;
	    DeltaY = - _gWidthY / r.D.y;
	    StepY = -1;
	    OutY = -1;
    }

    for(;;) {
	    // check for triangle intersection           
	    if(VERTEX_VALUE(xx+1, yy+1) >= nx*ny)
		    break;
	    if(TriIntersectTest(ray, xx, yy, dg)) {
		    return true;         		    
	    }
	    if(NextXCrossing < NextYCrossing) {
		    // Step in X
		    if(r.maxt < NextXCrossing) break;
		    xx += StepX;
		    if(xx == OutX) break;
		    NextXCrossing += DeltaX;
	    }
	    else {
		    // Step in Y
		    if(r.maxt < NextYCrossing) break;
		    yy += StepY;
		    if(yy == OutY) break;
		    NextYCrossing+= DeltaY;		    
	    }	    
    }

    return false;

}

bool Heightfield::IntersectP(const Ray &ray) const
{    
    // keep a copy of our ray in object space
    return true;
}

bool Heightfield::Intersect(const Ray &r, DifferentialGeometry *dg) const
{
   // keep a copy of our ray in object space
    Ray ray = WorldToObject(r);
    if(IntersectTest(ray, r, dg)) 
        return true;

    return false;
}
