#ifndef TEXTURE_H
#define TEXTURE_H
#include "lrt.h"
#include "color.h"
#include "geometry.h"
#include "transform.h"
#include "shapes.h"
#include "primitives.h"
#include <string>
using std::string;
template <class T> class Texture {
public:
	virtual T evaluate(const Surf *) const = 0;
	virtual ~Texture() { }
};
template <class T>
class ConstantTexture: public Texture<T> {
public:
	ConstantTexture(const T &d) {
		datum = d;
	}
	T evaluate(const Surf *) const;
private:
	T datum;
};
template <class T1, class T2>
class ScaleTexture: public Texture<T2> {
public:
	ScaleTexture(Texture<T1> *s, Texture<T2> *m) {
		scale = s;
		map = m;
	}
	~ScaleTexture() {
		delete scale;
		delete map;
	}
	T2 evaluate(const Surf *) const;
private:
	Texture<T1> *scale;
	Texture<T2> *map;
};
template <class T>
class MixTexture : public Texture<T> {
public:
	MixTexture(Texture<T> *t1, Texture<T> *t2,
		Texture<Float> *amt);
	~MixTexture();
	T evaluate(const Surf *) const;
private:
	Texture<T> *tex1, *tex2;
	Texture<Float> *amount;
};
template <class T>
class BilerpTexture : public Texture<T> {
public:
	BilerpTexture(const Transform &xform, Texture<T> *t00, Texture<T> *t01, Texture<T> *t10,
		Texture<T> *t11);
	~BilerpTexture();
	T evaluate(const Surf *) const;
private:
	Texture<T> *tex00, *tex01, *tex10, *tex11;
	Transform xform;
};
class UVTexture : public Texture<Spectrum> {
public:
	UVTexture(const Transform &x): xform(x) { }
	Spectrum evaluate(const Surf *) const;
private:
	Transform xform;
};
class TextureMap {
public:
	TextureMap(const char *filename);
	~TextureMap();
	Spectrum Lookup(Float u, Float v) const;
	Spectrum Lookup(Float u, Float v, Float width) const;
	Spectrum Lookup(Float u, Float v, Float dudx, Float dudy,
	                Float dvdx, Float dvdy) const;
private:
	Spectrum *the_map;
	int width, height;
};
template <class T>
class ImageTexture: public Texture<T> {
public:
	ImageTexture(const Transform &x, const string &filename);
	T evaluate(const Surf *) const;
private:
	TextureMap *texmap;
	Transform xform;
};
template <class T> class UVCheckerboard : public Texture<T> {
public:
	UVCheckerboard(const Transform &x, Texture<T> *c1, Texture<T> *c2)
		: xform(x) {
		map1 = c1;
		map2 = c2;
	}
	~UVCheckerboard();
	T evaluate(const Surf *) const;
private:
	Texture<T> *map1, *map2;
	Transform xform;
};
template <class T> class SolidCheckerboard : public Texture<T> {
public:
	SolidCheckerboard(const Transform &x, Texture<T> *c1,
		Texture<T> *c2) : xform(x) {
		map1 = c1;
		map2 = c2;
	}
	~SolidCheckerboard();
	
	T evaluate(const Surf *surf) const;
private:
	Texture<T> *map1, *map2;
	Transform xform;
};
template <class T>
T ConstantTexture<T>::evaluate(const Surf *) const {
	return datum;
}
template <class T1, class T2>
T2 ScaleTexture<T1, T2>::evaluate(const Surf *surf) const {
	return scale->evaluate(surf) * map->evaluate(surf);
}
template <class T>
MixTexture<T>::MixTexture(Texture<T> *t1, Texture<T> *t2,
		Texture<Float> *amt) {
	tex1 = t1;
	tex2 = t2;
	amount = amt;
}
template <class T>
MixTexture<T>::~MixTexture() {
	delete tex1;
	delete tex2;
	delete amount;
}
template <class T>
T MixTexture<T>::evaluate(const Surf *surf) const {
	T t1 = tex1->evaluate(surf), t2 = tex2->evaluate(surf);
	Float amt = amount->evaluate(surf);
	return (1. - amt) * t1 + amt * t2;
}
template <class T>
BilerpTexture<T>::BilerpTexture(const Transform &x,
		Texture<T> *t00, Texture<T> *t01, Texture<T> *t10,
		Texture<T> *t11) : xform(x) {
	tex00 = t00;
	tex01 = t01;
	tex10 = t10;
	tex11 = t11;
}
template <class T>
BilerpTexture<T>::~BilerpTexture() {
	delete tex00;
	delete tex01;
	delete tex10;
	delete tex11;
}
template <class T>
T BilerpTexture<T>::evaluate(const Surf *surf) const {
	T t00 = tex00->evaluate(surf);
	T t01 = tex01->evaluate(surf);
	T t10 = tex10->evaluate(surf);
	T t11 = tex11->evaluate(surf);
	Point pt = xform(Point(surf->dgShading.u, surf->dgShading.v, 0.));
	Float u = pt.u, v = pt.v;
	return (1-u)*(1-v) * t00 + (1-u)*v * t01 + u*(1-v) * t10 +
		u*v * t11;
}
template <class T>
ImageTexture<T>::ImageTexture(const Transform &x,
		const string &filename)
	: xform(x) {
	texmap = GetTexture(filename.c_str());
}
template <class T>
T ImageTexture<T>::evaluate(const Surf *surf) const {
	Float u = surf->dgShading.u, v = surf->dgShading.v;
	Point ptrans = xform(Point(u, v, 0.));
	return texmap->Lookup(ptrans.x, ptrans.y);
}
template <class T> UVCheckerboard<T>::~UVCheckerboard() {
	delete map1;
	delete map2;
}
template <class T>
T UVCheckerboard<T>::evaluate(const Surf *surf) const {
	Float u = surf->dgShading.u, v = surf->dgShading.v;
	Point ptrans = xform(Point(u, v, 0.));
	if ((Round(ptrans.x) + Round(ptrans.y)) % 2 == 0)
		return map1->evaluate(surf);
	return map2->evaluate(surf);
}
template <class T>
SolidCheckerboard<T>::~SolidCheckerboard() {
	delete map1;
	delete map2;
}
template <class T>
T SolidCheckerboard<T>::evaluate(const Surf *surf) const {
	Point ptrans = xform(surf->dgShading.P);
	if ((Round(ptrans.x) + Round(ptrans.y) + Round(ptrans.z)) % 2 == 0)
		return map1->evaluate(surf);
	return map2->evaluate(surf);
}
#endif // TEXTURE_H
