#ifndef SHAPES_H
#define SHAPES_H
#include "lrt.h"
#include "geometry.h"
#include "transform.h"
class Shape {
public:
	Shape(const Transform &o2w)
		: ObjectToWorld(o2w) {
		WorldToObject = Transform(ObjectToWorld.GetInverse());
	}
	virtual BBox Bound() const = 0;
	virtual BBox BoundWorldSpace() const;
	virtual bool CanIntersect() const { return true; }
	virtual void Refine(vector<Shape *> &refined) const;
	virtual bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	virtual bool IntersectP(const Ray &ray) const;
	virtual Float area() const;
	virtual void sample(const DifferentialGeometry &dgShading,
		Vector *dir) const;
	virtual Float weight(const DifferentialGeometry &dgShading,
			const Vector &dir) const;
	virtual ~Shape() { }
protected:
	Transform ObjectToWorld, WorldToObject;
};
struct DifferentialGeometry 
{
  DifferentialGeometry() { u = v = 0.; }
  DifferentialGeometry(const Point &P, const Normal &N, const Vector &S, const Vector &T, Float u, Float v) 
  {
    this->P = P;
    this->N = N.Hat();
    this->S = S.Hat();
    this->T = T.Hat();
    this->u = u;
    this->v = v;
  }
	
  Point P;
  Normal N;
  Vector S, T;
  Float u, v;

  // RRG: Added to help with simple bump-mapping. 
  Vector X;
  Vector Y;
  Vector Z;

};
class Sphere: public Shape {
public:
	Sphere(const Transform &o2w, Float rad, Float zmin, Float zmax, Float thetaMax);
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Sphere::area() const;
	virtual void sample(const DifferentialGeometry &dgShading,
		Vector *dir) const;
protected:
	Float radius;
	Float zmin, zmax;
	Float thetaMax;
};
class Cylinder: public Shape {
public:
	Cylinder( const Transform &o2w, Float rad, Float zmin, Float zmax, Float thetaMax );
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Cylinder::area() const;
	virtual void sample(const DifferentialGeometry &dgShading, Vector *dir) const;
protected:
	Float radius;
	Float zmin, zmax;
	Float thetaMax;
};
class Cone: public Shape {
public:
	Cone(const Transform &o2w, Float height, Float rad, Float tm );
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Cone::area() const;
protected:
	Float radius, height, thetaMax;
};
class Paraboloid: public Shape {
public:
	Paraboloid(const Transform &o2w, Float rad, Float z0, Float z1, Float tm );
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Paraboloid::area() const;
protected:
	Float radius;
	Float zmin, zmax;
	Float thetaMax;
};
class Hyperboloid: public Shape {
public:
	Hyperboloid(const Transform &o2w, const Point &point1, const Point &point2,
		Float tm );
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Hyperboloid::area() const;
protected:
	Point p1, p2;
	Float zmin, zmax;
	Float thetaMax;
	Float rmax;
};
class Disk : public Shape {
public:
	Disk(const Transform &o2w, Float height, Float radius, Float thetaMax);
	BBox Bound() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Disk::area() const;
	virtual void sample(const DifferentialGeometry &dgShading,
		Vector *dir) const;
private:
	Float height, radius, thetaMax;
};
class TriangleMesh: public Shape {
public:
	TriangleMesh(const Transform &o2w, int ntris, int nverts, int *vptr, Point *P);
	~TriangleMesh();
	BBox Bound() const;
	BBox BoundWorldSpace() const;
	bool CanIntersect() const { return false; }
	void Refine(vector<Shape *> &refined) const;
	friend class Triangle;
public:
	int ntris;
	int nverts;
	int *vertexIndex;
	Point *p;
};
class Triangle : public Shape {
public:
	Triangle(const Transform &o2w, const TriangleMesh *m, int n, Float *uv = NULL)
		: Shape(o2w) {
		mesh = m;
		triNum = n;
		if (uv)
			memcpy(tex, uv, 6*sizeof(Float));
		else {
			tex[0][0] = 0;  tex[0][1] = 0;
			tex[1][0] = 1;  tex[1][1] = 0;
			tex[2][0] = 1;  tex[2][1] = 1;
		}
	}
	BBox Bound() const;
	BBox BoundWorldSpace() const;
	bool Intersect(const Ray &ray, DifferentialGeometry *dg) const;
	bool IntersectP(const Ray &ray) const;
	Float Triangle::area() const;
private:
	const TriangleMesh *mesh;
	int triNum;
	Float tex[3][2];
};
#endif // SHAPES_H
