#include "materials.h"
#include "color.h"
#include "reflection.h"
#include "texture.h"


DifferentialGeometry Material::bump(
	const DifferentialGeometry &dgg) const {
	return dgg;
}
Matte::~Matte() {
	delete Kd;
}
BSDF *Matte::getBSDF(const Surf *surf) const {
	Spectrum r = Kd->evaluate(surf);
	return new BSDF(surf->dgShading,
		new LambertianReflection(r), 1., NULL);
}
Plastic::~Plastic() {
	delete Kd;
	delete Ks;
	delete roughness;
}
BSDF *Plastic::getBSDF(const Surf *surf) const {
	BxDF *diff = new LambertianReflection(Kd->evaluate(surf));
	BxDF *spec = new Microfacet(Ks->evaluate(surf),
		new Blinn(1. / roughness->evaluate(surf)));
	return new BSDF(surf->dgShading, diff, 1., spec, 1., NULL);
}


Glass::~Glass() {
	delete Kr;
	delete Kt;
	delete index;
}
BSDF *Glass::getBSDF(const Surf *surf) const {
	Spectrum R = Kr->evaluate(surf), T = Kt->evaluate(surf);
	Float ior = index->evaluate(surf);
	BxDF *brdf = NULL, *btdf = NULL;
	if (R.Intensity() > 0.)
		brdf = new SpecularReflection(R);
	if (T.Intensity() > 0.)
		btdf = new SpecularTransmission(T, 1., ior);
	if (brdf && btdf) return new BSDF(surf->dgShading, brdf, 1.,
						btdf, 1., NULL);
	else if (brdf)    return new BSDF(surf->dgShading, brdf, 1., NULL);
	else if (btdf)    return new BSDF(surf->dgShading, btdf, 1., NULL);
	else              return new BSDF(surf->dgShading, NULL);
}

BSDF *
FresnelGlass::getBSDF( const Surf * surf ) const {
    Float ior = index->evaluate( surf );

    return new BSDF( surf->dgShading, new FresnelScattering( color, 1., ior ), 
        1., NULL );
}

ShinyMetal::~ShinyMetal() {
	delete Ks;
	delete roughness;
	delete Kr;
}
BSDF *ShinyMetal::getBSDF(const Surf *surf) const {
	Spectrum spec = Ks->evaluate(surf);
	Float rough = roughness->evaluate(surf);
	Spectrum R = Kr->evaluate(surf);

	MicrofacetDistribution *md = new Blinn(1. / rough);
	return new BSDF(surf->dgShading, new Microfacet(spec, md), 1.,
		new SpecularReflection(R), 1., NULL, NULL);
}

BSDF *
LiquidInterface::getBSDF( const Surf * surf ) const {
    return new BSDF( surf->dgShading, new FresnelScattering( _color, indexi, indext ),
        1., NULL );
}

void 
LiquidInterface::SamplePhaseScatter( const Vector &wi, Vector *wo ) const {
    Float u1 = RandomFloat();
    Float u2 = RandomFloat();
    Float cosTheta = (2*u1 + k - 1) / (2*u1*k - k + 1);
    Float cosAlpha = cos( u2 * 2 * M_PI );
    Float sinTheta = sqrt( 1. - cosTheta * cosTheta );
    Float sinAlpha = sin( u2 * 2 * M_PI );
 
    // Construct a rotation matrix that brings the world wi vector to a local
    // coordinate system so that it becomes the z-axis of the new local system.

    Vector N = wi;
    Vector tangent1;

    if ( N.x == 0 ) {
        tangent1.x = 1;
        tangent1.y = 0;
        tangent1.z = 0;
    }
    else if ( N.z == 0 ) {
        tangent1.y = sqrt( 1 / ((N.y/N.x)*(N.y/N.x) + 1) );
        tangent1.x = (N.y/N.x) * tangent1.y;
        tangent1.z = 0;
    }
    else {
        float nySquared = N.y*N.y;
        float nzSquared = N.z*N.z;
        float term = (nySquared + nzSquared) / (N.x*N.z);
        tangent1.z = sqrt( 1 / ( (term*term) + (nySquared/nzSquared) + 1 ) );
        tangent1.x = -tangent1.z * (nySquared + nzSquared) / (N.x*N.z);
        tangent1.y = (N.y/N.z) * tangent1.z;
    }

    Vector tangent2 = Cross( tangent1, N );
    Transform LocalToWorld( tangent1.x, tangent2.x, N.x, 0,
        tangent1.y, tangent2.y, N.y, 0,
        tangent1.z, tangent2.z, N.z, 0,
        0, 0, 0, 1 );
    
    Vector scatteredVector( sinTheta * cosAlpha, sinTheta * sinAlpha, cosTheta );

    *wo = LocalToWorld( scatteredVector ); 
}

 
Float
LiquidInterface::PhaseScatter( const Vector & wi, const Vector & wo ) const {
    Float cosTheta = Dot( wi, wo );  // Assumes unit vectors
    Float term = 1. + k * cosTheta;
    return ( 1. - k*k ) / ( 4 * M_PI * term * term ); 
}
