#include "lrt.h"
#include "shapes.h"
BBox Shape::BoundWorldSpace() const {
	return ObjectToWorld(Bound());
}
void Shape::Refine(vector<Shape *> &refined) const {
	Severe("Unimplemented Shape::Refine() method called");
}
bool Shape::Intersect(const Ray &ray,
		DifferentialGeometry *dg) const {
	Severe("Unimplemented Shape::Intersect() method called");
	return false;
}

bool Shape::IntersectP(const Ray &ray) const {
	Severe("Unimplemented Shape::IntersectP() method called");
	return false;
}
Float Shape::area() const {
	Severe("Unimplemented Shape::area() method called");
	return 0.;
}
Sphere::Sphere(const Transform &o2w, Float rad, Float z0,
		Float z1, Float tm )
	: Shape(o2w) {
	radius = rad;
	zmin = Clamp(min(z0, z1), -radius, radius);
	zmax = Clamp(max(z0, z1), -radius, radius);
	thetaMax = Radians( Clamp( tm, 0.0f, 360.0f ) );
}
BBox Sphere::Bound() const {
	return BBox( Point( -radius, -radius, zmin ),
				 Point(  radius,  radius,  zmax ) );
}
bool Sphere::Intersect(const Ray &r,
		DifferentialGeometry *dg) const {
	Float u, v;
	Float theta, phi;
	Ray ray = WorldToObject(r);
	Float A = ray.D.x*ray.D.x + ray.D.y*ray.D.y + ray.D.z*ray.D.z;
	Float B = 2 * (ray.D.x*ray.O.x + ray.D.y*ray.O.y +
				   ray.D.z*ray.O.z);
	Float C = ray.O.x*ray.O.x + ray.O.y*ray.O.y +
			  ray.O.z*ray.O.z - radius*radius;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	phi = acos(-Phit.z / radius);
	v = phi / M_PI;
	if (v == 0 || v == 1) {
		u = 0;
	} else {
		Float val = Clamp( Phit.x / (radius*sin(phi)), -1.f, 1.f );
		theta = acos(val) / (2*M_PI);
		if (Phit.y > 0)
			u = theta;
		else
			u = 1 - theta;
		u *= (2*M_PI)/thetaMax;
	}
	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1) return false;
		if (t1 > ray.maxt) return false;
		thit = t1;
		Phit = ray(thit);
		Point Phit = ray(thit);
		phi = acos(-Phit.z / radius);
		v = phi / M_PI;
		if (v == 0 || v == 1) {
			u = 0;
		} else {
			Float val = Clamp( Phit.x / (radius*sin(phi)), -1.f, 1.f );
			theta = acos(val) / (2*M_PI);
			if (Phit.y > 0)
				u = theta;
			else
				u = 1 - theta;
			u *= (2*M_PI)/thetaMax;
		}
		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}
	Float vmin = acos(-zmin/radius);
	Float vmax = acos(-zmax/radius);
	v = (v*M_PI - vmin)/(vmax-vmin);
	Normal N = Normal( Phit - Point(0,0,0) );
	Vector S, T;
	Float denom = sqrt(Phit.x*Phit.x + Phit.y*Phit.y);
	if (denom == 0) {
		S = Vector( 1,0,0 );
	} else  {
		S = Vector( - Phit.y / denom, Phit.x / denom );
	}
	T = Cross( Vector(N), S );
	*dg = DifferentialGeometry(Phit, N, S, T, u, v);
	*dg = ObjectToWorld(*dg);
	r.maxt = thit;
	return true;
}
bool Sphere::IntersectP(const Ray &r) const {
	Float u, v;
	Float theta, phi;
	Ray ray = WorldToObject(r);
	Float A = ray.D.x*ray.D.x + ray.D.y*ray.D.y + ray.D.z*ray.D.z;
	Float B = 2 * (ray.D.x*ray.O.x + ray.D.y*ray.O.y +
				   ray.D.z*ray.O.z);
	Float C = ray.O.x*ray.O.x + ray.O.y*ray.O.y +
			  ray.O.z*ray.O.z - radius*radius;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	phi = acos(-Phit.z / radius);
	v = phi / M_PI;
	if (v == 0 || v == 1) {
		u = 0;
	} else {
		Float val = Clamp( Phit.x / (radius*sin(phi)), -1.f, 1.f );
		theta = acos(val) / (2*M_PI);
		if (Phit.y > 0)
			u = theta;
		else
			u = 1 - theta;
		u *= (2*M_PI)/thetaMax;
	}
	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1) return false;
		if (t1 > ray.maxt) return false;
		thit = t1;
		Phit = ray(thit);
		Point Phit = ray(thit);
		phi = acos(-Phit.z / radius);
		v = phi / M_PI;
		if (v == 0 || v == 1) {
			u = 0;
		} else {
			Float val = Clamp( Phit.x / (radius*sin(phi)), -1.f, 1.f );
			theta = acos(val) / (2*M_PI);
			if (Phit.y > 0)
				u = theta;
			else
				u = 1 - theta;
			u *= (2*M_PI)/thetaMax;
		}
		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}
	return true;
}
Float Sphere::area() const {
	return thetaMax * radius * (zmax-zmin);
}
Cylinder::Cylinder( const Transform &o2w, Float rad, Float z0,
		Float z1, Float tm )
	: Shape(o2w) {
	radius = rad;
	zmin = min(z0, z1);
	zmax = max(z0, z1);
	thetaMax = Radians( Clamp( tm, 0.0f, 360.0f ) );
}
BBox Cylinder::Bound() const {
	Point p1 = Point( -radius, -radius, zmin );
	Point p2 = Point(  radius,  radius, zmax );
	return BBox( p1, p2 );
}
bool Cylinder::Intersect(const Ray &r,
		DifferentialGeometry *dg) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float A = ray.D.x*ray.D.x + ray.D.y*ray.D.y;
	Float B = 2 * (ray.D.x*ray.O.x + ray.D.y*ray.O.y);
	Float C = ray.O.x*ray.O.x + ray.O.y*ray.O.y - radius*radius;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = (Phit.z - zmin) / (zmax - zmin);
	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = (Phit.z - zmin) / (zmax - zmin);
		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}
	Normal N = Normal(Phit - Point(0,0,Phit.z));
	Vector S, T;
	Float denom = sqrt(Phit.x*Phit.x + Phit.y*Phit.y);
	if (denom == 0) {
		S = Vector( 1,0,0 );
	} else  {
		S = Vector( - Phit.y / denom, Phit.x / denom );
	}
	T = Cross( Vector(N), S );
	*dg = DifferentialGeometry(Phit, N, S, T, u, v);
	*dg = ObjectToWorld(*dg);
	r.maxt = thit;
	return true;
}
bool Cylinder::IntersectP(const Ray &r) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float A = ray.D.x*ray.D.x + ray.D.y*ray.D.y;
	Float B = 2 * (ray.D.x*ray.O.x + ray.D.y*ray.O.y);
	Float C = ray.O.x*ray.O.x + ray.O.y*ray.O.y - radius*radius;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = (Phit.z - zmin) / (zmax - zmin);
	if (Phit.z < zmin || Phit.z > zmax || u > 1.) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = (Phit.z - zmin) / (zmax - zmin);
		if (Phit.z < zmin || Phit.z > zmax || u > 1.)
			return false;
	}
	return true;
}
Float Cylinder::area() const {
	return (zmin-zmax)*thetaMax*radius;
}
Cone::Cone(const Transform &o2w, Float ht, Float rad, Float tm )
	: Shape(o2w) {
	radius = rad;
	height = ht;
	thetaMax = Radians( Clamp( tm, 0.0f, 360.0f ) );
}
BBox Cone::Bound() const {
	Point p1 = Point( -radius, -radius, 0 );
	Point p2 = Point(  radius,  radius, height );
	return BBox( p1, p2 );
}
bool Cone::Intersect(const Ray &r, DifferentialGeometry *dg) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float k = radius / height;
	k = k*k;
	Float A = ray.D.x * ray.D.x +
			  ray.D.y * ray.D.y -
		  k * ray.D.z * ray.D.z;
	Float B = 2 * (ray.D.x * ray.O.x +
				   ray.D.y * ray.O.y -
			   k * ray.D.z * (ray.O.z-height) );
	Float C = ray.O.x * ray.O.x +
			  ray.O.y * ray.O.y -
		  k * (ray.O.z -height) * (ray.O.z-height);
	
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = Phit.z / height;
	if (Phit.z < 0 || Phit.z > height || u > 1.) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = Phit.z / height;
		if (Phit.z < 0 || Phit.z > height || u > 1.)
			return false;
	}
	Normal N(height * Phit.x, -height * Phit.y, radius*radius*(1.-v));
	Vector S, T;
	Float denom = sqrt(Phit.x*Phit.x + Phit.y*Phit.y);
	if (denom == 0) {
		S = Vector( 1,0,0 );
	} else  {
		S = Vector( - Phit.y / denom, Phit.x / denom );
	}
	T = Cross( Vector(N), S );
	*dg = DifferentialGeometry(Phit, N, S, T, u, v);
	*dg = ObjectToWorld(*dg);
	r.maxt = thit;
	return true;
}

bool Cone::IntersectP(const Ray &r) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float k = radius / height;
	k = k*k;
	Float A = ray.D.x * ray.D.x +
			  ray.D.y * ray.D.y -
		  k * ray.D.z * ray.D.z;
	Float B = 2 * (ray.D.x * ray.O.x +
				   ray.D.y * ray.O.y -
			   k * ray.D.z * (ray.O.z-height) );
	Float C = ray.O.x * ray.O.x +
			  ray.O.y * ray.O.y -
		  k * (ray.O.z -height) * (ray.O.z-height);
	
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
	v = Phit.z / height;
	if (Phit.z < 0 || Phit.z > height || u > 1.) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / thetaMax;
		v = Phit.z / height;
		if (Phit.z < 0 || Phit.z > height || u > 1.)
			return false;
	}
	return true;
}
Float Cone::area() const {
	return thetaMax*height*height*
		sqrt((height*height)+(radius*radius))/(2.0*radius);
}
Paraboloid::Paraboloid(const Transform &o2w, Float rad, Float z0,
		Float z1, Float tm)
	: Shape(o2w) {
	radius = rad;
	zmin = min(z0,z1);
	zmax = max(z0,z1);
	thetaMax = Radians( Clamp( tm, 0.0f, 360.0f ) );
}
BBox Paraboloid::Bound() const {
	Point p1 = Point( -radius, -radius, zmin );
	Point p2 = Point(  radius,  radius, zmax );
	return BBox( p1, p2 );
}
bool Paraboloid::Intersect(const Ray &r, DifferentialGeometry *dg) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float k = zmax/(radius*radius);
	Float A =   k*(ray.D.x * ray.D.x + ray.D.y * ray.D.y );
	Float B = 2*k*(ray.D.x * ray.O.x + ray.D.y * ray.O.y ) - ray.D.z;
	Float C =   k*(ray.O.x * ray.O.x + ray.O.y * ray.O.y ) - ray.O.z;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	v = (Phit.z-zmin) / (zmax-zmin);
	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
		v = (Phit.z-zmin) / (zmax-zmin);
		if (Phit.z < zmin || Phit.z > zmax ||
			u * 2 * M_PI > thetaMax)
			return false;
	}
	u *= (2*M_PI)/thetaMax;
	
	Float crossu = atan2(Phit.y, -Phit.x) + M_PI;
	Float crossv = Phit.z / zmax;
	Vector dPdv( (radius*cos(crossu))/(2*sqrt(crossv)*sqrt(zmax) ),
		   (radius*sin(crossu))/(2*sqrt(crossv)*sqrt(zmax) ),
		   1);
	Vector dPdu( (-radius*sqrt(crossv)*sin(crossu))/sqrt(zmax),
		   (radius*sqrt(crossv)*cos(crossu))/sqrt(zmax),
		   0);
	Normal N = Normal(Cross(dPdu, dPdv));
	Vector S, T;
	Float denom = sqrt(Phit.x*Phit.x + Phit.y*Phit.y);
	if (denom == 0) {
		S = Vector( 1,0,0 );
	} else  {
		S = Vector( - Phit.y / denom, Phit.x / denom );
	}
	T = Cross( Vector(N), S );
	*dg = DifferentialGeometry(Phit, N, S, T, u, v);
	*dg = ObjectToWorld(*dg);
	r.maxt = thit;
	return true;
}

bool Paraboloid::IntersectP(const Ray &r) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float k = zmax/(radius*radius);
	Float A =   k*(ray.D.x * ray.D.x + ray.D.y * ray.D.y );
	Float B = 2*k*(ray.D.x * ray.O.x + ray.D.y * ray.O.y ) - ray.D.z;
	Float C =   k*(ray.O.x * ray.O.x + ray.O.y * ray.O.y ) - ray.O.z;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	Point Phit = ray(thit);
	u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	v = (Phit.z-zmin) / (zmax-zmin);
	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		Point Phit = ray(thit);
		u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
		v = (Phit.z-zmin) / (zmax-zmin);
		if (Phit.z < zmin || Phit.z > zmax ||
			u * 2 * M_PI > thetaMax)
			return false;
	}
	return true;
}
Float Paraboloid::area() const {
	return thetaMax/12.0 *
		(pow(1+4*zmin,3.0/2.0) - pow(1+4*zmax,3.0/2.0));
}
Hyperboloid::Hyperboloid(const Transform &o2w, const Point &point1,
	const Point &point2, Float tm )
	: Shape(o2w) {
	p1 = point1;
	p2 = point2;
	thetaMax = Radians( Clamp( tm, 0.0f, 360.0f ) );
	Float rad1 = sqrt( p1.x*p1.x + p1.y*p1.y );
	Float rad2 = sqrt( p2.x*p2.x + p2.y*p2.y );
	rmax = max(rad1,rad2);
	zmin = min(p1.z,p2.z);
	zmax = max(p1.z,p2.z);
}
BBox Hyperboloid::Bound() const {
	Point p1 = Point( -rmax, -rmax, zmin );
	Point p2 = Point(  rmax,  rmax, zmax );
	return BBox( p1, p2 );
}
bool Hyperboloid::Intersect(const Ray &r, DifferentialGeometry *dg) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float t, a2, c2;
	Float a, c;
	Point pr;
	
	if(p1.z == p2.z)
	return false;
	t = -p2.z/(p1.z-p2.z);
	pr = Point(t*p1.x + (1-t)*p2.x,
		 t*p1.y + (1-t)*p2.y,
		 t*p1.z + (1-t)*p2.z);
	a2 = pr.x*pr.x + pr.y*pr.y;
	if(p1.x*p1.x + p1.y*p1.y == a2)
	return false;
	c2 = (a2*p1.z*p1.z)/(p1.x*p1.x + p1.y*p1.y - a2);
	
	Float A = ray.D.x*ray.D.x/a2 +
			  ray.D.y*ray.D.y/a2 -
			  ray.D.z*ray.D.z/c2;
	Float B = 2 * (ray.D.x*ray.O.x/a2 +
				   ray.D.y*ray.O.y/a2 -
				   ray.D.z*ray.O.z/c2);
	Float C = ray.O.x*ray.O.x/a2 +
			  ray.O.y*ray.O.y/a2 -
			  ray.O.z*ray.O.z/c2 - 1;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	c = sqrt(c2);
	a = sqrt(a2);
	
	Point Phit = ray(thit);
	v = (Phit.z - p1.z)/(p2.z - p1.z);
	
	pr = Point((1-v)*p1.x + v*p2.x,
		 (1-v)*p1.y + v*p2.y,
		 (1-v)*p1.z + v*p2.z);
	
	u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
	if(u < 0)
	u += 2*M_PI;
	u /= 2*M_PI;
	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		c = sqrt(c2);
		a = sqrt(a2);
		
		Point Phit = ray(thit);
		v = (Phit.z - p1.z)/(p2.z - p1.z);
		
		pr = Point((1-v)*p1.x + v*p2.x,
			 (1-v)*p1.y + v*p2.y,
			 (1-v)*p1.z + v*p2.z);
		
		u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
		if(u < 0)
		u += 2*M_PI;
		u /= 2*M_PI;
		if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax)
			return false;
	}
	Float crossu = atan2(Phit.y, Phit.x) + M_PI;
	Float crossv = Phit.z/c;
	Vector dPdu(-a*cosh(crossv)*sin(crossu),
		a*cosh(crossv)*cos(crossu), 0);
	Vector dPdv(a*sinh(crossv)*cos(crossu), a*sinh(crossv)*sin(crossu),
		c*cosh(crossv));
	Normal N = Normal(Cross(dPdu, dPdv));
	*dg = DifferentialGeometry(Phit, N, dPdu, Cross(Vector(N), dPdu),
		u, v);
	*dg = ObjectToWorld(*dg);
	r.maxt = thit;
	return true;
}
bool Hyperboloid::IntersectP(const Ray &r) const {
	Float u, v;
	Ray ray = WorldToObject(r);
	Float t, a2, c2;
	Float a, c;
	Point pr;
	
	if(p1.z == p2.z)
	return false;
	t = -p2.z/(p1.z-p2.z);
	pr = Point(t*p1.x + (1-t)*p2.x,
		 t*p1.y + (1-t)*p2.y,
		 t*p1.z + (1-t)*p2.z);
	a2 = pr.x*pr.x + pr.y*pr.y;
	if(p1.x*p1.x + p1.y*p1.y == a2)
	return false;
	c2 = (a2*p1.z*p1.z)/(p1.x*p1.x + p1.y*p1.y - a2);
	
	Float A = ray.D.x*ray.D.x/a2 +
			  ray.D.y*ray.D.y/a2 -
			  ray.D.z*ray.D.z/c2;
	Float B = 2 * (ray.D.x*ray.O.x/a2 +
				   ray.D.y*ray.O.y/a2 -
				   ray.D.z*ray.O.z/c2);
	Float C = ray.O.x*ray.O.x/a2 +
			  ray.O.y*ray.O.y/a2 -
			  ray.O.z*ray.O.z/c2 - 1;
	Float discrim = B * B - 4. * A * C;
	if (discrim < 0.) return false;
	Float rootDiscrim = sqrt(discrim);
	Float q;
	if (B < 0) q = -.5 * (B - rootDiscrim);
	else       q = -.5 * (B + rootDiscrim);
	Float t0 = q / A, t1 = C / q;
	if (t0 > t1) swap(t0, t1);
	if (t0 > ray.maxt || t1 < ray.mint)
		return false;
	Float thit = t0;
	if (t0 < ray.mint) thit = t1;
	if (thit > ray.maxt) return false;
	c = sqrt(c2);
	a = sqrt(a2);
	
	Point Phit = ray(thit);
	v = (Phit.z - p1.z)/(p2.z - p1.z);
	
	pr = Point((1-v)*p1.x + v*p2.x,
		 (1-v)*p1.y + v*p2.y,
		 (1-v)*p1.z + v*p2.z);
	
	u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
	if(u < 0)
	u += 2*M_PI;
	u /= 2*M_PI;
	if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax) {
		if (thit == t1) return false;
		thit = t1;
		if (t1 > ray.maxt) return false;
		Phit = ray(thit);
		c = sqrt(c2);
		a = sqrt(a2);
		
		Point Phit = ray(thit);
		v = (Phit.z - p1.z)/(p2.z - p1.z);
		
		pr = Point((1-v)*p1.x + v*p2.x,
			 (1-v)*p1.y + v*p2.y,
			 (1-v)*p1.z + v*p2.z);
		
		u = atan2(Phit.y, Phit.x) - atan2(pr.y, pr.x);
		if(u < 0)
		u += 2*M_PI;
		u /= 2*M_PI;
		if (Phit.z < zmin || Phit.z > zmax || u * 2 * M_PI > thetaMax)
			return false;
	}
	return true;
}
Float Hyperboloid::area() const {
	Float x_1 = p1.x, x_2 = p2.x;
	Float y_1 = p1.y, y_2 = p2.y;
	Float z_1 = p1.z, z_2 = p2.z;
	return thetaMax/6.0 * (2*pow(x_1,4) - 2*pow(x_1,3)*x_2 +
		2*pow(x_2,4) + 2*(pow(y_1,2) + y_1*y_2 + pow(y_2,2))*
			(pow(y_1 - y_2,2) + pow(z_1 - z_2,2)) +
		   pow(x_2,2)*(5*pow(y_1,2) + 2*y_1*y_2 - 4*pow(y_2,2) +
			  2*pow(z_1 - z_2,2)) +
		   pow(x_1,2)*(-4*pow(y_1,2) + 2*y_1*y_2 + 5*pow(y_2,2) +
			  2*pow(z_1 - z_2,2)) -
		   2*x_1*x_2*(pow(x_2,2) - pow(y_1,2) + 5*y_1*y_2 -
			  pow(y_2,2) - pow(z_1,2) + 2*z_1*z_2 - pow(z_2,2)));
}
Disk::Disk(const Transform &o2w, Float ht, Float r, Float tmax)
	: Shape(o2w) {
	height = ht;
	radius = r;
	thetaMax = Radians( Clamp( tmax, 0.0f, 360.0f ) );
}
BBox Disk::Bound() const {
	return BBox(Point(-radius,-radius,height),
				Point( radius, radius,height) );
}
bool Disk::Intersect(const Ray &r, DifferentialGeometry *dg) const {
	Ray ray = WorldToObject(r);
	Float thit = (height - ray.O.z) / ray.D.z;
	if (thit < ray.mint || thit > ray.maxt)
		return false;
	Point Phit = ray(thit);
	Float Dist2 = Phit.x * Phit.x + Phit.y * Phit.y;
	if (Dist2 > radius * radius)
		return false;
	Float u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	Float v = 1. - (sqrt(Phit.x * Phit.x + Phit.y * Phit.y) / radius);
	if (u * 2. * M_PI > thetaMax)
		return false;
	u *= (2*M_PI)/thetaMax;
	Normal N = Normal(0,0,1);
	Vector S, T;
	Float denom = sqrt(Phit.x*Phit.x + Phit.y*Phit.y);
	if (denom == 0) {
		S = Vector( 1,0,0 );
	} else  {
		S = Vector( - Phit.y / denom, Phit.x / denom );
	}
	T = Cross( Vector(N), S );
	*dg = DifferentialGeometry(Phit, N, S, T, u, v );
	r.maxt = thit;
	return true;
}
bool Disk::IntersectP(const Ray &r) const {
	Ray ray = WorldToObject(r);
	Float thit = (height - ray.O.z) / ray.D.z;
	if (thit < ray.mint || thit > ray.maxt)
		return false;
	Point Phit = ray(thit);
	Float Dist2 = Phit.x * Phit.x + Phit.y * Phit.y;
	if (Dist2 > radius * radius)
		return false;
	Float u = 1. - (atan2(Phit.y, -Phit.x) + M_PI) / (2. * M_PI);
	Float v = 1. - (sqrt(Phit.x * Phit.x + Phit.y * Phit.y) / radius);
	if (u * 2. * M_PI > thetaMax)
		return false;
	(void) v;
	return true;
}
Float Disk::area() const {
	return thetaMax * radius * radius;
}
TriangleMesh::TriangleMesh(const Transform &o2w, int nt, int nv,
		int *vi, Point *P)
	: Shape(o2w) {
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	p = new Point[nverts];
	for (int i  = 0; i < nverts; ++i)
		p[i] = ObjectToWorld(P[i]);
}
TriangleMesh::~TriangleMesh() {
	delete [] vertexIndex;
	delete [] p;
}
BBox TriangleMesh::Bound() const {
	BBox bobj(WorldToObject(p[0]), WorldToObject(p[1]));
	for (int i = 2; i < nverts ; i++)
		bobj = Union(bobj, WorldToObject(p[i]));
	return bobj;
}
BBox TriangleMesh::BoundWorldSpace() const {
	BBox worldBounds(p[0], p[1]);
	for (int i = 2; i < nverts ; i++)
		worldBounds = Union(worldBounds, p[i]);
	return worldBounds;
}
void TriangleMesh::Refine(vector<Shape *> &refined) const {
	for (int i = 0; i < ntris; ++i)
		refined.push_back(new Triangle(ObjectToWorld, this, i));
}
BBox Triangle::Bound() const {
	Point p0 = WorldToObject(mesh->p[mesh->vertexIndex[triNum*3]]);
	Point p1 = WorldToObject(mesh->p[mesh->vertexIndex[triNum*3 + 1]]);
	Point p2 = WorldToObject(mesh->p[mesh->vertexIndex[triNum*3 + 2]]);
	BBox objectBounds(p0, p1);
	return Union(objectBounds, p2);
}

BBox Triangle::BoundWorldSpace() const {
	Point p0 = mesh->p[mesh->vertexIndex[triNum*3]];
	Point p1 = mesh->p[mesh->vertexIndex[triNum*3 + 1]];
	Point p2 = mesh->p[mesh->vertexIndex[triNum*3 + 2]];
	BBox objectBounds(p0, p1);
	return Union(objectBounds, p2);
}
bool Triangle::Intersect(const Ray &ray,
		DifferentialGeometry *dg) const {
	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry", "Triangle Ray Intersections",
			&triangleHits, &triangleTests);
	++triangleTests;
	int *vptr = &(mesh->vertexIndex[3*triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;
	
	const Point *p = mesh->p;
	Vector E1 = p[vertexIndex1] - p[vertexIndex0];
	Vector E2 = p[vertexIndex2] - p[vertexIndex0];
	Vector S_1 = Cross( ray.D, E2 );
	Float divisor = Dot( S_1, E1 );
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;
	Vector T = ray.O - mesh->p[vertexIndex0];
	Float u = Dot( T, S_1 ) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;
	Vector S_2 = Cross( T, E1 );
	Float v = Dot( ray.D, S_2 ) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;
	Float t = Dot( E2, S_2 ) * invDivisor;
	if (t < ray.mint || t > ray.maxt)
		return false;
	++triangleHits;
	Normal N = Normal(Cross(E2, E1));
	Vector dPdu, dPdv;
	Float du1 = tex[1][0] - tex[0][0];
	Float du2 = tex[2][0] - tex[0][0];
	Float dv1 = tex[1][1] - tex[0][1];
	Float dv2 = tex[2][1] - tex[0][1];
	const Point &p1 = mesh->p[vertexIndex0];
	const Point &p2 = mesh->p[vertexIndex1];
	const Point &p3 = mesh->p[vertexIndex2];
	Float dx1 = p2.x - p1.x;
	Float dx2 = p3.x - p1.x;
	Float dy1 = p2.y - p1.y;
	Float dy2 = p3.y - p1.y;
	Float dz1 = p2.z - p1.z;
	Float dz2 = p3.z - p1.z;
	Float determinant = du1 * dv2 - dv1 * du2;
	if (determinant == 0) {
		dPdv = Cross(Vector(N), Vector(N.y, N.z, N.x));
		dPdu = Cross(Vector(N), dPdv);
	}
	else {
		Float invdet = 1. / determinant;
		dPdu = Vector((dx1 * dv2 - dv1 * dx2) * invdet,
			(dy1 * dv2 - dv1 * dy2) * invdet,
			(dz1 * dv2 - dv1 * dz2) * invdet);
		dPdv = Vector((du1 * dx2 - dx1 * du2) * invdet,
			(du1 * dy2 - dy1 * du2) * invdet,
			(du1 * dz2 - dz1 * du2) * invdet);
	}
	Float uu = (1 - u - v) * tex[0][0] + u * tex[1][0] + v * tex[2][0];
	Float vv = (1 - u - v) * tex[0][1] + u * tex[1][1] + v * tex[2][1];
	*dg = DifferentialGeometry(ray(t), N.Hat(), dPdu.Hat(),
		Cross(Vector(N), dPdu).Hat(), uu, vv);
	ray.maxt = t;
	return true;
}
bool Triangle::IntersectP(const Ray &ray) const {
	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry", "Triangle Ray Intersections",
			&triangleHits, &triangleTests);
	++triangleTests;
	int *vptr = &(mesh->vertexIndex[3*triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;
	
	const Point *p = mesh->p;
	Vector E1 = p[vertexIndex1] - p[vertexIndex0];
	Vector E2 = p[vertexIndex2] - p[vertexIndex0];
	Vector S_1 = Cross( ray.D, E2 );
	Float divisor = Dot( S_1, E1 );
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;
	Vector T = ray.O - mesh->p[vertexIndex0];
	Float u = Dot( T, S_1 ) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;
	Vector S_2 = Cross( T, E1 );
	Float v = Dot( ray.D, S_2 ) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;
	Float t = Dot( E2, S_2 ) * invDivisor;
	if (t < ray.mint || t > ray.maxt)
		return false;
	++triangleHits;
	return true;
}
Float Triangle::area() const {
	int *vptr = &(mesh->vertexIndex[3*triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;

	const Point &p1 = mesh->p[vertexIndex0];
	const Point &p2 = mesh->p[vertexIndex1];
	const Point &p3 = mesh->p[vertexIndex2];
	Float x_1 = p1.x, x_2 = p2.x, x_3 = p3.x;
	Float y_1 = p1.y, y_2 = p2.y, y_3 = p3.y;
	Float z_1 = p1.z, z_2 = p2.z, z_3 = p3.z;

#define DET_HELPER(a,b) \
	a##_2*b##_3-a##_3*b##_2 - \
	(a##_1*b##_3-a##_3*b##_1) + \
	a##_1*b##_2-a##_2*b##_1

	Float discrim = pow( DET_HELPER(y,z), 2 );
	discrim += pow( DET_HELPER(z,x), 2 );
	discrim += pow( DET_HELPER(x,y), 2 );

	return sqrt(discrim)/2.0;
}
Float Sphere::Sample(Float u[2], DifferentialGeometry *dg) const {
	// YOUR CODE HERE: sample a point P, and compute
	// the inverse mapping to find uu and vv.  Then
	// pass this and the Disk's Normal, (0,0,1) into
	// the (misnamed) RecordHit() method of DifferentialGeometry.

	// dg->RecordHit(P, N, uu,
	//                    vv, attributes, surfaceFunction);
	// return Pdf;
	return 0.;
}

Float Cylinder::Sample(Float u[2], DifferentialGeometry *dg) const {
	// YOUR CODE HERE: sample a point P, and compute
	// the inverse mapping to find uu and vv.  Then
	// pass this and the Disk's Normal, (0,0,1) into
	// the (misnamed) RecordHit() method of DifferentialGeometry.

	// dg->RecordHit(P, N, uu,
	//                    vv, attributes, surfaceFunction);
	// return Pdf;
	return 0.;
}

Float Cone::Sample(Float u[2], DifferentialGeometry *dg) const {
	// YOUR CODE HERE: sample a point P, and compute
	// the inverse mapping to find uu and vv.  Then
	// pass this and the Disk's Normal, (0,0,1) into
	// the (misnamed) RecordHit() method of DifferentialGeometry.

	// dg->RecordHit(P, N, uu,
	//                    vv, attributes, surfaceFunction);
	// return Pdf;
	return 0.;
}
Float Sphere::Pdf(const Ray &ray,
				  Float *costhetao,
				  Point *Plight) const {
	// YOUR CODE HERE
	return 0.;
}
Float Cylinder::Pdf(const Ray &ray,
					Float *costhetao,
					Point *Plight) const {
	// YOUR CODE HERE
	return 0.;
}
Float Cone::Pdf(const Ray &ray,
				Float *costhetao,
				Point *Plight) const {
	// YOUR CODE HERE
	return 0.;
}
