#include "light.h"
#include "primitives.h"
#include "color.h"
#include "scene.h"

Light::~Light() {
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	const Vector &w) const {
	return Spectrum(0.);
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	Vector *w) const {
	*w = Vector(0,0,0);
	return Spectrum(0.);
}
bool Light::visible(const Scene *scene, const Point &x1,
	const Point &x2) const {
	if (!CastsShadows) return 1.;
	Ray r(x1, x2-x1, 1e-3, .999);
	return scene->IntersectP(r) ? false : true;
}
float Light::unoccluded(const Scene *scene, const Point &x,
		const Vector &w) const {
	if (!CastsShadows) return 1.;
	Ray r(x, w, 1e-3);
	return scene->IntersectP(r) ? 0. : 1.;
}
Spectrum Light::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) const {
	*weight = 1;
	return dE(scene, dg, wo);
}
PointLight::PointLight(bool shadows, const Transform &world2light,
	const Spectrum &power, const Point &Plight)
	: Light(shadows, world2light, power) {
	lightPos = LightToWorld(Plight);
}
Spectrum PointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = lightPos - dg.P;
	w->Normalize();
	if (visible(scene, dg.P, lightPos))
	  return I(*w) * fabs(Dot(*w, dg.N));
	  /*
	  return I(*w) * fabs(Dot(*w, dg.N)) /
	    DistanceSquared(lightPos, dg.P);
	  */
	return Spectrum(0.);
}
InfinitePointLight::InfinitePointLight(bool shadows,
	const Transform &world2light, const Spectrum &power,
	const Vector &dir)
	: Light(shadows, world2light, power) {
	lightDir = LightToWorld(dir);
}
Spectrum InfinitePointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = - lightDir;
	w->Normalize();
	return unoccluded(scene, dg.P, *w) *
		Power * fabs(Dot(*w, dg.N));
}
AreaLight::AreaLight(bool shadows, const Transform &world2light,
	const Spectrum &power, Shape *s)
	: Light(shadows, world2light, power) {
	shape = s;
	Area = shape->area();
}
Spectrum AreaLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	const Vector &w) const {
	DifferentialGeometry hit;
	Ray ray(dg.P, w);
	if (shape->Intersect(ray, &hit) &&
		visible( scene, dg.P, hit.P ))
	    return L(hit.P, -w);
	return 0.;
}

Spectrum AreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *wt) const {
	shape->sample(dg, wo);
	*wt = shape->weight(dg, *wo);
	return dE(scene, dg, *wo);
}

// * * * * * * * * *
//  SUN PARAMETERS *
// * * * * * * * * *

// 6:30am
float sunrise = 6.5;

// 8:30pm
float sunset = 20.5;

// theta greater than half circle on either horizon = PI/32
static float theta_additional = 0.098175;

// highest point sun reaches = PI/2 - PI/16
static float phi_max = 1.3744;

//static float max_sun_intensity = 200.0; // sun is white, rest of sky changes color

//static float sky_intensity_factor = 0.25; // from max_sun_intensity

float max_sky_intensity = 15.0;

float sun_intensity_factor = 3.0;

// M_PI / 8.0
static float sun_falloff_distance = 0.3927;

// angle; only used for sunrise/sunset
static float sun_radius = 2.0*0.176;

static float sun_falloff_power = 12;

static Vector AngleVector(float theta, float phi)
{
  float y = sin(phi);
  float cosPhi = cos(phi);
  float x = sin(theta)*cosPhi;
  float z = cos(theta)*cosPhi;
  return Vector(x, y, z);
}

EnvironmentLight::EnvironmentLight(bool shadows, const Transform &world2light,
				   const Spectrum &power, float time)
  : Light(shadows, world2light, power) {

  float timeNorm = (time-sunrise) / (sunset-sunrise);

  cerr << "EnvironmentLight: time = " << time << endl;

  // [((time-sunrise) / (sunset-sunrise))*(pi+2*theta_additional)]
  // + pi - theta_additional
  sunTheta = timeNorm * (M_PI+(2*theta_additional));
  sunTheta += M_PI - theta_additional;

  // sin[((time-sunrise) / (sunset-sunrise)) * pi] + pi/2
  sunPhi = timeNorm * M_PI;
  sunPhi = sin(sunPhi);
  sunPhi *= M_PI/2.0;
  sunVector = AngleVector(sunTheta, sunPhi);

  skyBase = SkyColor(sunPhi);

#if 0
  float theta, phi;
  int i, j, k;
  //build degree array, by 1 degree in theta and phi
  Spectrum *env = new Spectrum[360*90];
  for(i=0; i<360; i++){
    for(j=0; j<90; j++){
      float r = sqrt(j/90.);
      phi = acos(r);
      phi *= M_PI/180.;
      theta = (i/180.)*M_PI;
      env[(i*90)+j] = ComputeE(theta, phi);
    }
  }

  //place into probablility array
  int array_size = 0;
  int array_max = 0;
  int array_min = INT_MAX;
  for(i=0; i<360*90; i++){
    int intensity = Round(env[i].Intensity());
    array_size += intensity;
    if(intensity < array_min){
      array_min = intensity;
    }
    if(intensity > array_max){
      array_max = intensity;
    }
  }

  // TK: round?
  if (array_min > 0) array_size /= array_min;
  light_array = new EnvMap[array_size];
  light_array_size = array_size;

  //copy values into array
  int index = 0;
  for(i=0; i<360; i++){
    theta = i/180.*M_PI;
    for(j=0; j<90; j++){
      float r = sqrt(j/90.);
      phi = acos(r);
      phi *= M_PI/180.;
      int kMax = Round(env[(i*90)+j].Intensity());
      if (array_min > 0) kMax /= array_min;
      for(k=0; k<kMax; k++){
	light_array[index].theta = theta;
	light_array[index].phi = phi;
	light_array[index].value = env[(i*90)+j];
	index++;
      }
    }
  }
  cerr << "Constructed " << index << " sky sample buckets." << endl;
  if (index != light_array_size) {
    cerr << "Bucket total mismatch: " << index << "buckets filled != ";
    cerr << light_array_size << " buckets" << endl;
    exit(1);
  }
  delete env;
#endif // 0
}
/*
Spectrum EnvironmentLight::E(float theta, float phi)
{
  if (theta > 360 || theta < 0 || phi > 90 || phi < 0)
    return Spectrum(0.);

  int th = Round(Clamp(theta*180.0/M_PI, 0.f, 360.f));
  int ph = Round(Clamp(phi*180.0/M_PI, 0.f, 90.f));
  return env[(th*90)+ph];
}

Spectrum EnvironmentLight::E(Vector v)
{
  float theta = 0.;
  float phi = 0.;
  VectorAngle(v, theta, phi);
  return E(theta, phi);
}
*/

static void VectorAngle(Vector v, float &theta, float &phi)
{
  phi = asin(v.y);
  theta = acos(v.z / cos(phi));
}

/* Should be fine now... //DO NOT CALL THIS DIRECTLY!!!!! */

Spectrum EnvironmentLight::dE(const Scene *scene,
			      const DifferentialGeometry &dg,
			      const Vector &w) {
  //cerr << "NEVER, EVER call dE() directly on Environment lights!!!" << endl;
  //return Spectrum(1.);

  if (CastsShadows) {
    Ray r(dg.P, w);
    if (scene->IntersectP(r)) {
      cerr << "Shadow" << endl;
      return Spectrum(0.);
    }
  }

  return ComputeE(w);
}

// num must be of the form (2n)^2
void gen_stratified_samples(Vector w[], int num)
{
  int nPhi = sqrt(num)/2;
  int nTheta = 4*nPhi;
  if ((2*nPhi)*(2*nPhi) != num) {
    cerr << "gen_stratified_samples num samples of the form (2n)^2." << endl;
    exit(1);
  }
  for (int i = 0; i < nTheta; i++) {
    for (int j = 0; j < nPhi; j++) {
      float u1 = RandomFloat(j, j+1);
      float r = sqrt(u1/nPhi);
      float phi = acos(r);
      float u2 = RandomFloat(i, i+1);
      float theta = 2.*M_PI*u2/nTheta;
      w[(i*nPhi)+j] = AngleVector(theta, phi);
    }
  }
}

Spectrum EnvironmentLight::sample_dE(const Scene *scene,
				     const DifferentialGeometry &dg,
				     Vector *wo, Float *wt) const {
  //cerr << "EnvLight::sample_dE" << endl;
  //shape->sample(dg, wo);
  //*wt = shape->weight(dg, *wo);
  float rand = RandomFloat(0,1);

  //create a vector 
  float R, r, x, y, z;
  R = 10000.;
  r = R * cos(light_array[(int)(rand*light_array_size)].phi);
  x = r * cos(light_array[(int)(rand*light_array_size)].theta);
  z = r * sin(light_array[(int)(rand*light_array_size)].theta);
  y = R * sin(light_array[(int)(rand*light_array_size)].phi);

  *wo = Vector(Point(x,y,z)-dg.P);
  *wt = 1;
  if (CastsShadows) {
    Ray r(dg.P, *wo);
    if (scene->IntersectP(r)) {
      cerr << "Shadow" << endl;
      return Spectrum(0.);
    }
  }
  return light_array[(int)(rand*light_array_size)].value;
}

Spectrum EnvironmentLight::SkyColor(float sunPhi)
{
  static const float color_scale[256*3] = {  0,   0,   0, 
					     35,   0,   0, 
					     52,   0,   0, 
					     60,   0,   0, 
					     63,   1,   0, 
					     64,   2,   0, 
					     68,   5,   0, 
					     69,   6,   0, 
					     72,   8,   0, 
					     74,  10,   0, 
					     77,  12,   0, 
					     78,  14,   0, 
					     81,  16,   0, 
					     83,  17,   0, 
					     85,  19,   0, 
					     86,  20,   0, 
					     89,  22,   0, 
					     91,  24,   0, 
					     92,  25,   0, 
					     94,  26,   0, 
					     95,  28,   0, 
					     98,  30,   0, 
					     100,  31,   0, 
					     102,  33,   0, 
					     103,  34,   0, 
					     105,  35,   0, 
					     106,  36,   0, 
					     108,  38,   0, 
					     109,  39,   0, 
					     111,  40,   0, 
					     112,  42,   0, 
					     114,  43,   0, 
					     115,  44,   0, 
					     117,  45,   0, 
					     119,  47,   0, 
					     119,  47,   0, 
					     120,  48,   0, 
					     122,  49,   0, 
					     123,  51,   0, 
					     125,  52,   0, 
					     125,  52,   0, 
					     126,  53,   0, 
					     128,  54,   0, 
					     129,  56,   0, 
					     129,  56,   0, 
					     131,  57,   0, 
					     132,  58,   0, 
					     134,  59,   0, 
					     134,  59,   0, 
					     136,  61,   0, 
					     137,  62,   0, 
					     137,  62,   0, 
					     139,  63,   0, 
					     139,  63,   0, 
					     140,  65,   0, 
					     142,  66,   0, 
					     142,  66,   0, 
					     143,  67,   0, 
					     143,  67,   0, 
					     145,  68,   0, 
					     145,  68,   0, 
					     146,  70,   0, 
					     146,  70,   0, 
					     148,  71,   0, 
					     148,  71,   0, 
					     149,  72,   0, 
					     149,  72,   0, 
					     151,  73,   0, 
					     151,  73,   0, 
					     153,  75,   0, 
					     153,  75,   0, 
					     154,  76,   0, 
					     154,  76,   0, 
					     154,  76,   0, 
					     156,  77,   0, 
					     156,  77,   0, 
					     157,  79,   0, 
					     157,  79,   0, 
					     159,  80,   0, 
					     159,  80,   0, 
					     159,  80,   0, 
					     160,  81,   0, 
					     160,  81,   0, 
					     162,  82,   0, 
					     162,  82,   0, 
					     163,  84,   0, 
					     163,  84,   0, 
					     165,  85,   0, 
					     165,  85,   0, 
					     166,  86,   0, 
					     166,  86,   0, 
					     166,  86,   0, 
					     168,  87,   0, 
					     168,  87,   0, 
					     170,  89,   0, 
					     170,  89,   0,
					     171,  90,   0, 
					     171,  90,   0, 
					     173,  91,   0, 
					     173,  91,   0, 
					     174,  93,   0, 
					     174,  93,   0, 
					     176,  94,   0, 
					     176,  94,   0, 
					     177,  95,   0, 
					     177,  95,   0, 
					     179,  96,   0, 
					     179,  96,   0, 
					     180,  98,   0, 
					     182,  99,   0, 
					     182,  99,   0, 
					     183, 100,   0, 
					     183, 100,   0, 
					     185, 102,   0, 
					     185, 102,   0, 
					     187, 103,   0, 
					     187, 103,   0, 
					     188, 104,   0, 
					     188, 104,   0, 
					     190, 105,   0, 
					     191, 107,   0, 
					     191, 107,   0, 
					     193, 108,   0, 
					     193, 108,   0, 
					     194, 109,   0, 
					     196, 110,   0, 
					     196, 110,   0, 
					     197, 112,   0, 
					     197, 112,   0, 
					     199, 113,   0, 
					     200, 114,   0, 
					     200, 114,   0, 
					     202, 116,   0, 
					     202, 116,   0, 
					     204, 117,   0, 
					     205, 118,   0, 
					     205, 118,   0, 
					     207, 119,   0, 
					     208, 121,   0, 
					     208, 121,   0, 
					     210, 122,   0, 
					     211, 123,   0, 
					     211, 123,   0, 
					     213, 124,   0, 
					     214, 126,   0, 
					     214, 126,   0, 
					     216, 127,   0, 
					     217, 128,   0, 
					     217, 128,   0, 
					     219, 130,   0, 
					     221, 131,   0, 
					     221, 131,   0, 
					     222, 132,   0, 
					     224, 133,   0, 
					     224, 133,   0, 
					     225, 135,   0, 
					     227, 136,   0, 
					     227, 136,   0, 
					     228, 137,   0, 
					     230, 138,   0, 
					     230, 138,   0, 
					     231, 140,   0, 
					     233, 141,   0, 
					     233, 141,   0, 
					     234, 142,   0, 
					     236, 144,   0, 
					     236, 144,   0, 
					     238, 145,   0, 
					     239, 146,   0, 
					     241, 147,   0,
					     241, 147,   0, 
					     242, 149,   0, 
					     244, 150,   0, 
					     244, 150,   0, 
					     245, 151,   0, 
					     247, 153,   0, 
					     247, 153,   0, 
					     248, 154,   0, 
					     250, 155,   0, 
					     251, 156,   0, 
					     251, 156,   0, 
					     253, 158,   0, 
					     255, 159,   0, 
					     255, 159,   0, 
					     255, 160,   0, 
					     255, 161,   0, 
					     255, 163,   0, 
					     255, 163,   0, 
					     255, 164,   0, 
					     255, 165,   0, 
					     255, 167,   0, 
					     255, 167,   0, 
					     255, 168,   0, 
					     255, 169,   0, 
					     255, 169,   0, 
					     255, 170,   0, 
					     255, 172,   0, 
					     255, 173,   0, 
					     255, 173,   0, 
					     255, 174,   0, 
					     255, 175,   0, 
					     255, 177,   0, 
					     255, 178,   0, 
					     255, 179,   0, 
					     255, 181,   0, 
					     255, 181,   0, 
					     255, 182,   0, 
					     255, 183,   0, 
					     255, 184,   0, 
					     255, 187,   7, 
					     255, 188,  10, 
					     255, 189,  14, 
					     255, 191,  18, 
					     255, 192,  21, 
					     255, 193,  25, 
					     255, 195,  29, 
					     255, 197,  36,
					     255, 198,  40, 
					     255, 200,  43, 
					     255, 202,  51, 
					     255, 204,  54, 
					     255, 206,  61, 
					     255, 207,  65, 
					     255, 210,  72, 
					     255, 211,  76, 
					     255, 214,  83, 
					     255, 216,  91, 
					     255, 219,  98, 
					     255, 221, 105, 
					     255, 223, 109, 
					     255, 225, 116, 
					     255, 228, 123, 
					     255, 232, 134, 
					     255, 234, 142, 
					     255, 237, 149, 
					     255, 239, 156, 
					     255, 240, 160, 
					     255, 243, 167, 
					     255, 246, 174, 
					     255, 248, 182, 
					     255, 249, 185, 
					     255, 252, 193, 
					     255, 253, 196, 
					     255, 255, 204, 
					     255, 255, 207, 
					     255, 255, 211, 
					     255, 255, 218, 
					     255, 255, 222, 
					     255, 255, 225, 
					     255, 255, 229, 
					     255, 255, 233, 
					     255, 255, 236, 
					     255, 255, 240, 
					     255, 255, 244, 
					     255, 255, 247, 
					     255, 255, 255 };
  // TK: add color, currently just modulating intensity
  float intensity = max_sky_intensity;

  // if in sunrise/sunset
  if (fabs(sunPhi) < sun_radius) {
    // [0,1] on range [begin sunset, end sunset]
    //intensity *= (sunPhi+sun_radius)/(2.0*sun_radius);
    
    int i = (int)(ceil((fabs(sunPhi)/sun_radius*255)*3));   
    int j = (int)(floor((fabs(sunPhi)/sun_radius*255)*3));
    //lookup in map
    // TK: WTF!?! r,g,b are fucked!
    /*    return (Spectrum(color_scale[i+1]*intensity/255.,
		     color_scale[i+2]*intensity/255.,
		     color_scale[i+0]*intensity/255. )); */
    return (Spectrum(intensity));
  }
  // if sun is set
  else if (sunPhi < -sun_radius) {
    return Spectrum(0.);
  }
  // sun fully visible
  return Spectrum(intensity);
}

// angles in radians!
Spectrum EnvironmentLight::ComputeE(float theta, float phi) {
  Vector v = AngleVector(theta, phi);
  return ComputeE(v);
}

Spectrum EnvironmentLight::ComputeE(Vector v) {
  float sunIntensity = Clamp(Dot(sunVector, v), 0., 1.);
  sunIntensity = pow(sunIntensity, sun_falloff_power);
  return skyBase + (sunIntensity*sun_intensity_factor*skyBase);
}

Spectrum InfiniteAreaLight::dE(const Scene *scene,
	 	const DifferentialGeometry &dg,	const Vector &w) const {
	return unoccluded(scene, dg.P, w) * Power *
		max(0.f, Dot(w, dg.N));
}
Spectrum InfiniteAreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) {
	Float u1 = RandomFloat(), u2 = RandomFloat();
	Float r = sqrtf(u2);
	Float theta = 2. * M_PI * u1;
	Float x = r * cos(theta);
	Float y = r * sin(theta);
	Float z = sqrtf(1. - x*x - y*y);
	if (RandomFloat() < .5) z *= -1;
	*wo = Vector(x, y, z);
	*wo = Vector(dg.S.x * wo->x + dg.T.x * wo->y + dg.N.x * wo->z,
		     dg.S.y * wo->x + dg.T.y * wo->y + dg.N.y * wo->z,
		     dg.S.z * wo->x + dg.T.z * wo->y + dg.N.z * wo->z);
	*weight = 2.*M_PI / fabs(wo->z);
	return dE(scene, dg, *wo);
}
AmbientLight::AmbientLight(const Spectrum &p)
	: Light(false, Transform(), p) {
}
Spectrum AmbientLight::dE(const Scene *,
		const DifferentialGeometry &, const Vector &) const {
	return Power;
}
