#ifndef ACCEL_H
#define ACCEL_H
#include "lrt.h"
class GridAccelerator : public PrimitiveSet {
	struct MailboxPrim;
public:
	GridAccelerator(const vector<Primitive *> &p)
		: PrimitiveSet(p) {
		float delta = max(fabsf(bounds.pMin.x),
			max(fabsf(bounds.pMin.y), fabsf(bounds.pMin.z)));
		delta = max(delta, max(fabsf(bounds.pMax.x),
				max(fabsf(bounds.pMax.y), fabsf(bounds.pMax.z))));
		bounds.pMin -= Vector(delta, delta, delta);
		bounds.pMax += Vector(delta, delta, delta);
		int cubeRoot = (int)pow(prims.size(), .333333333);
		Float dx = 1.00001 * (bounds.pMax.x - bounds.pMin.x);
		Float dy = 1.00001 * (bounds.pMax.y - bounds.pMin.y);
		Float dz = 1.00001 * (bounds.pMax.z - bounds.pMin.z);
		Float invmaxWidth = 1.0/max(dx, max(dy, dz));
		Assert(invmaxWidth > 0.);
		XVoxels = Clamp(3 * Round(cubeRoot * dx * invmaxWidth), 1, 100);
		YVoxels = Clamp(3 * Round(cubeRoot * dy * invmaxWidth), 1, 100);
		ZVoxels = Clamp(3 * Round(cubeRoot * dz * invmaxWidth), 1, 100);
		XWidth = dx / XVoxels;
		YWidth = dy / YVoxels;
		ZWidth = dz / ZVoxels;
		InvXWidth = (XWidth == 0.) ? 0. : 1. / XWidth;
		InvYWidth = (YWidth == 0.) ? 0. : 1. / YWidth;
		InvZWidth = (ZWidth == 0.) ? 0. : 1. / ZWidth;
		cells = new list<MailboxPrim *>[XVoxels * YVoxels * ZVoxels];
		for (u_int i = 0; i < prims.size(); ++i) {
			BBox primBounds = prims[i]->BoundWorldSpace();
			int x0 = max(x2v(primBounds.pMin.x), 0);
			int x1 = min(x2v(primBounds.pMax.x), XVoxels-1);
			int y0 = max(y2v(primBounds.pMin.y), 0);
			int y1 = min(y2v(primBounds.pMax.y), YVoxels-1);
			int z0 = max(z2v(primBounds.pMin.z), 0);
			int z1 = min(z2v(primBounds.pMax.z), ZVoxels-1);
			MailboxPrim *mp = new MailboxPrim(prims[i]);
			for (int x = x0; x <= x1; ++x)
				for (int y = y0; y <= y1; ++y)
					for (int z = z0; z <= z1; ++z) {
						int offset = z*XVoxels*YVoxels + y*XVoxels + x;
						cells[offset].push_back(mp);
					}
		}
		curMailboxId = 0;
	}
	int x2v( Float x ) const { return int(( x - bounds.pMin.x ) *
						InvXWidth); }
	int y2v( Float y ) const { return int(( y - bounds.pMin.y ) *
						InvYWidth); }
	int z2v( Float z ) const { return int(( z - bounds.pMin.z ) *
						InvZWidth); }
	Float v2x( int x ) const { return x * XWidth + bounds.pMin.x; }
	Float v2y( int y ) const { return y * YWidth + bounds.pMin.y; }
	Float v2z( int z ) const { return z * ZWidth + bounds.pMin.z; }
	~GridAccelerator();
	bool Intersect(const Ray &ray, Surf *surf) const;
	bool IntersectP(const Ray &ray) const;
private:
	int XVoxels, YVoxels, ZVoxels;
	Float XWidth, YWidth, ZWidth;
	Float InvXWidth, InvYWidth, InvZWidth;
	list<MailboxPrim *> *cells;
	struct MailboxPrim {
		MailboxPrim(Primitive *p) { primitive = p; lastMailboxId = -1; }
		Primitive *primitive;
		int lastMailboxId;
	};
	mutable int curMailboxId;
};
struct HBVNode {
	HBVNode(const pair<Primitive *, BBox *> &prim);
	HBVNode(HBVNode *child1, HBVNode *child2);
	~HBVNode();
	BBox bounds;
	bool isLeaf;
	union {
		HBVNode *children[2];
		const Primitive *primitive;
	};
};

class HBVAccelerator : public PrimitiveSet {
public:
	HBVAccelerator(const vector<Primitive *> &primitives);
	~HBVAccelerator();
	bool Intersect(const Ray &ray, Surf *surf) const;
private:
	static HBVNode *BuildHBV(const vector<Primitive *> &primitives);
	static HBVNode *RecursiveBuild(const vector<pair<Primitive *, BBox *> > &primitives);
	static int XPartition(vector<pair<Primitive *, Point> > &prims, int start, int end);
	static bool RecursiveIntersect(HBVNode *&node, const Ray &ray, Surf *surf);
	HBVNode *root;
};
#endif // ACCEL_H
