#include "lrt.h"
#include <stdarg.h>
#define ERROR_IGNORE 0
#define ERROR_CONTINUE 1
#define ERROR_ABORT 2
static void processError(const char *format, va_list args,
                         const char *message, int disposition) {
	#define ERR_BUF_SZ 1024
	static char errorBuf[ERR_BUF_SZ];
	vsprintf(errorBuf, format, args);
	switch (disposition) {
	case ERROR_IGNORE:
		return;
	case ERROR_CONTINUE:
		fprintf(stderr, "%s: %s\n", message, errorBuf);
		break;
	case ERROR_ABORT:
		fprintf(stderr, "%s: %s\n", message, errorBuf);
		abort();
	}
}
void Info(const char *format, ...) {
	va_list args;
	va_start(args, format);
	processError(format, args, "Notice", ERROR_CONTINUE);
	va_end(args);
}
void Warning(const char *format, ...) {
	va_list args;
	va_start(args, format);
	processError(format, args, "Warning", ERROR_CONTINUE);
	va_end(args);
}
void Error(const char *format, ...) {
	va_list args;
	va_start(args, format);
	processError(format, args, "Error", ERROR_CONTINUE);
	va_end(args);
}
void Severe(const char *format, ...) {
	va_list args;
	va_start(args, format);
	processError(format, args, "Fatal Error", ERROR_ABORT);
	va_end(args);
}
void MatrixInvert( Float m[4][4], Float minv[4][4] ) {
	int indxc[4], indxr[4];
	int ipiv[4] = { 0, 0, 0, 0 };

	memcpy( minv, m, 4*4*sizeof(Float) );

	for (int i = 0; i < 4; i++) {
		int irow, icol;
		Float big = 0.;
		for (int j = 0; j < 4; j++) {
			if (ipiv[j] != 1) {
				for (int k = 0; k < 4; k++) {
					if (ipiv[k] == 0) {
						if (fabs(minv[j][k]) >= big) {
							big = Float(fabs(minv[j][k]));
							irow = j;
							icol = k;
						}
					}
					else if (ipiv[k] > 1)
						Error("Singular matrix in MatrixInvert");
				}
			}
		}
		++ipiv[icol];
		if (irow != icol) {
			for (int k = 0; k < 4; ++k)
				swap(minv[irow][k], minv[icol][k]);
		}
		indxr[i] = irow;
		indxc[i] = icol;
		if (minv[icol][icol] == 0.)
			Error("Singular matrix in MatrixInvert");
		Float pivinv = 1. / minv[icol][icol];
		minv[icol][icol] = 1.;
		for (int j = 0; j < 4; j++)
			minv[icol][j] *= pivinv;
		for (int j = 0; j < 4; j++) {
			if (j != icol) {
				Float save = minv[j][icol];
				minv[j][icol] = 0;
				for (int k = 0; k < 4; k++)
					minv[j][k] -= minv[icol][k]*save;
			}
		}
	}
	for (int j = 3; j >= 0; j--) {
		if (indxr[j] != indxc[j]) {
			for (int k = 0; k < 4; k++)
				swap(minv[k][indxr[j]], minv[k][indxc[j]]);
		}
	}
}
struct StatTracker {
	StatTracker(const char *cat, const char *n, int l,
	            int *pa, int *pb = NULL);
	~StatTracker();

	char *category;
	char *name;
	int level;
	int *ptra, *ptrb;
};
static vector<StatTracker *> trackers;
static void addTracker(StatTracker *newTracker) {
	for (u_int i = 0; i < trackers.size(); ++i) {
		if (strcmp(newTracker->category, trackers[i]->category) == 0 &&
		    strcmp(newTracker->name, trackers[i]->name) == 0) {
				Error("Statistic %s/%s multiply reported. Discarding.",
					newTracker->category, newTracker->name);
				return;
		}
	}
	trackers.push_back(newTracker);
}
StatTracker::StatTracker(const char *cat, const char *n, int l,
                         int *pa, int *pb) {
	category = Strdup(cat);
	name = Strdup(n);
	level = l;
	ptra = pa;
	ptrb = pb;
}
StatTracker::~StatTracker() {
	delete[] category;
	delete[] name;
}
void StatsRegisterCounter(int level, const char *category,
                             const char *name, int *ptr) {
	addTracker(new StatTracker(category, name, level, ptr));
}
void StatsRegisterRatio(int level, const char *category,
                        const char *name, int *pa, int *pb) {
	addTracker(new StatTracker(category, name, level, pa, pb));
}
void StatsPrint(int level, FILE *dest) {
	if (level == STATS_NONE) return;
	fprintf(dest, "Statistics (%s)\n",
		level == STATS_BASIC ? "Basic" : "Detailed");
	for (u_int i = 0; i < trackers.size(); ++i) {
		if (trackers[i]->level <= level) {
			fprintf(dest, "%s/%s", trackers[i]->category, trackers[i]->name);
			int textLength = strlen(trackers[i]->category) + 1 + strlen(trackers[i]->name);
			int resultsColumn = 52;
			int paddingSpaces = resultsColumn - textLength;
			while (paddingSpaces--)
				putc(' ', dest);
			if (trackers[i]->ptrb == NULL)
				fprintf(dest, "%d\n", *trackers[i]->ptra);
			else {
				if (*trackers[i]->ptrb > 0) {
					Float ratio = (Float)*trackers[i]->ptra / (Float)*trackers[i]->ptrb;
					fprintf(dest, "%d:%d (%f%%)\n", *trackers[i]->ptra, *trackers[i]->ptrb,
						    100. * ratio);
				}
				else
					fprintf(dest, "%d:%d\n", *trackers[i]->ptra, *trackers[i]->ptrb);
			}
		}
	}
}
void StatsCleanup() {
	for (u_int i = 0; i < trackers.size(); ++i) {
		trackers[i]->ptra = 0;
		if (trackers[i]->ptrb)
			trackers[i]->ptrb = 0;
	}
	for (u_int i = 0; i < trackers.size(); ++i)
		delete trackers[i];
	trackers.erase(trackers.begin(), trackers.end());
}
/* Period parameters */
#define N 624
#define M 397
#define MATRIX_A 0x9908b0df   /* constant vector a */
#define UPPER_MASK 0x80000000 /* most significant w-r bits */
#define LOWER_MASK 0x7fffffff /* least significant r bits */

/* Tempering parameters */
#define TEMPERING_MASK_B 0x9d2c5680
#define TEMPERING_MASK_C 0xefc60000
#define TEMPERING_SHIFT_U(y)  (y >> 11)
#define TEMPERING_SHIFT_S(y)  (y << 7)
#define TEMPERING_SHIFT_T(y)  (y << 15)
#define TEMPERING_SHIFT_L(y)  (y >> 18)

static unsigned long mt[N]; /* the array for the state vector  */
static int mti=N+1; /* mti==N+1 means mt[N] is not initialized */
static void sgenrand(u_long seed) {
	/* setting initial seeds to mt[N] using the generator Line 25 of Table 1
	   in [KNUTH 1981, The Art of Computer Programming Vol. 2 (2nd Ed.),
	   pp102
	*/
	mt[0]= seed & 0xffffffff;
	for (mti=1; mti<N; mti++)
		mt[mti] = (69069 * mt[mti-1]) & 0xffffffff;
}
static double genrand() {
	unsigned long y;
	static unsigned long mag01[2]={0x0, MATRIX_A};
	/* mag01[x] = x * MATRIX_A  for x=0,1 */

	if (mti >= N) { /* generate N words at one time */
		int kk;
		if (mti == N+1)   /* if sgenrand() has not been called, */
			sgenrand(4357); /* a default initial seed is used   */

		for (kk=0;kk<N-M;kk++) {
			y = (mt[kk]&UPPER_MASK)|(mt[kk+1]&LOWER_MASK);
			mt[kk] = mt[kk+M] ^ (y >> 1) ^ mag01[y & 0x1];
		}
		for (;kk<N-1;kk++) {
			y = (mt[kk]&UPPER_MASK)|(mt[kk+1]&LOWER_MASK);
			mt[kk] = mt[kk+(M-N)] ^ (y >> 1) ^ mag01[y & 0x1];
		}
		y = (mt[N-1]&UPPER_MASK)|(mt[0]&LOWER_MASK);
		mt[N-1] = mt[M-1] ^ (y >> 1) ^ mag01[y & 0x1];
		mti = 0;
	}

	y = mt[mti++];
	y ^= TEMPERING_SHIFT_U(y);
	y ^= TEMPERING_SHIFT_S(y) & TEMPERING_MASK_B;
	y ^= TEMPERING_SHIFT_T(y) & TEMPERING_MASK_C;
	y ^= TEMPERING_SHIFT_L(y);

	return ( (double)y / (unsigned long)0xffffffff ); /* reals */
	/* return y; */ /* for integer generation */
}
Float RandomFloat(Float min, Float max) {
	return Lerp(genrand(), min, max);
}
u_int StringHashTable::Hash(const char *str) const {
	u_int hashValue = 0;
	while (*str) {
		hashValue <<= 1;
		++str;
	}
	return hashValue % NUM_BUCKETS;
}

void *StringHashTable::Search(const char *key) const {
	u_int index = Hash(key);
	ItemType::const_iterator iter;
	for (iter = buckets[index].begin(); iter != buckets[index].end(); iter++) {
		if(strcmp(key, iter->first) == 0)
			return iter->second;
	}
	return NULL;
}
void StringHashTable::Add(const char *key, void *data) {
	u_int index = Hash(key);
	ItemType::iterator iter = buckets[index].begin();
	while (iter != buckets[index].end()) {
		if (strcmp(key, iter->first) == 0)
			buckets[index].erase(iter);
		++iter;
	}
	buckets[index].push_front(make_pair(strdup(key), data));
}
