#ifndef REFLECTION_H
#define REFLECTION_H
#include "lrt.h"
#include "geometry.h"
class BSDF {
public:
	BSDF(const DifferentialGeometry &dg, BxDF *r, ...);
	~BSDF();
	int NumSpecular() const;
	Vector WorldToLocal(const Vector &v) const {
		return Vector(Dot(v, S), Dot(v, T), Dot(v, N)).Hat();
	}
	Vector LocalToWorld(const Vector &v) const {
		return Vector(S.x * v.x + T.x * v.y + N.x * v.z,
			      S.y * v.x + T.y * v.y + N.y * v.z,
			      S.z * v.x + T.z * v.y + N.z * v.z);
	}

	//pass in scene light spectrum for bsdfs

	Spectrum f(const Spectrum &de, const Vector &wiW, const Vector &woW) const;
	Spectrum f_delta(int component, const Vector &wi,
		Vector *wo) const;
	Spectrum sample_f(const Spectrum &de, const Vector &wi, Vector *wo) const;
	Float weight(const Vector &wi, const Vector &wo) const;
private:
	Vector N, S, T;
	vector<BxDF *> bxdfs;
	vector<Float> weights;
};
class BxDF {
public:
	virtual ~BxDF() { }
	virtual Spectrum f(const Spectrum &de, const Vector &wi, const Vector &wo) const = 0;
	virtual bool IsSpecular() const { return false; }
	virtual Spectrum f_delta(const Vector &wi, Vector *wo) const {
		return 0.;
	}
	static bool sameHemisphere(const Vector &wi, const Vector &wo) {
		return wi.z * wo.z > 0.;
	}
	virtual void sample_f(const Vector &wi, Vector *wo) const;
	virtual Float weight(const Vector &wi, const Vector &wo) const;
	virtual Spectrum rho() const;
	virtual Spectrum rho(const Vector &w) const;
};
class LambertianReflection : public BxDF {
public:
	LambertianReflection(const Spectrum &reflectance) {
		R = reflectance;
	}
	Spectrum f(const Spectrum &de, const Vector &wi, const Vector &wo) const;
	Spectrum rho() const { return R; }
	Spectrum rho(const Vector &w) const { return R; }
private:
	Spectrum R;
};
class LambertianTransmission : public BxDF {
public:
	LambertianTransmission(const Spectrum &tr) {
		T = tr;
	}
	Spectrum f(const Spectrum &de, const Vector &wi, const Vector &wo) const;
	virtual void sample_f(const Vector &wi, Vector *wo) const;
	Spectrum rho() const { return T; }
	Spectrum rho(const Vector &w) const { return T; }
private:
	Spectrum T;
};
class MicrofacetDistribution {
public:
	virtual ~MicrofacetDistribution() { }
	virtual Float D(const Vector &wi, const Vector &wo) const = 0;
	virtual void sample_f(const Vector &wi, Vector *wo) const = 0;
	virtual Float weight(const Vector &wi, const Vector &wo) const = 0;
};

class Microfacet : public BxDF {
public:
	Microfacet(const Spectrum &reflectance, MicrofacetDistribution *d);
	~Microfacet();
	Spectrum f(const Spectrum &de, const Vector &wi, const Vector &wo) const;
	Float G(const Vector &wi, const Vector &wo) const {
		return 1.f;
	}
	Float F(const Vector &wi, const Vector &wo) const {
		return 1.f;
	}
	virtual void sample_f(const Vector &wi, Vector *wo) const;
	virtual Float weight(const Vector &wi, const Vector &wo) const;
private:
	Spectrum R;
	MicrofacetDistribution *distribution;
};
class Blinn : public MicrofacetDistribution {
public:
	Blinn(Float e) { exponent = e; }
	Float D(const Vector &wi, const Vector &wo) const {
		Vector H = (wi + wo).Hat();
		Float costhetah = fabs(H.z);
		return pow(max(0.f, costhetah), exponent);
	}
	virtual void sample_f(const Vector &wi, Vector *wo) const;
	virtual Float weight(const Vector &wi, const Vector &wo) const;
private:
	Float exponent;
};
class SpecularReflection : public BxDF {
public:
	SpecularReflection(const Spectrum &r);
	Spectrum f(const Spectrum &, const Vector &, const Vector &) const {
		return Spectrum(0.);
	}
	bool IsSpecular() const { return true; }
	Spectrum f_delta(const Vector &wi, Vector *wo) const;
private:
	Spectrum R;
};
class SpecularTransmission : public BxDF {
public:
	SpecularTransmission(const Spectrum &r, Float indexi, Float indext);
	Spectrum f(const Spectrum &, const Vector &, const Vector &) const {
		return Spectrum(0.);
	}
	bool IsSpecular() const { return true; }
	Spectrum f_delta(const Vector &wi, Vector *wo) const;
private:
	Spectrum R;
	Float eta;
};
class ThinFilmReflection : public BxDF {
public:
        ThinFilmReflection(const Spectrum &d, const Normal &n, const Float
			   &thick, const Float &transmission, const Float
			   &modulation, const Float &Tfcn) {
	  D = d;
	  N = n;
	  th = thick;
	  trans = transmission;
	  tfcn = Tfcn;
	  mod = modulation;
	}
	Spectrum f(const Spectrum &de, const Vector &wi, const Vector &wo) const; 
	
private:
	Spectrum D;
	Normal N;
	Float th, n, trans, tfcn, mod;
	static const Float xyz[81][3];
	static const Float xyz2rgb[3][3];
};

#endif // REFLECTION_H
