#include "nurbs.h"

#define DICEU 50
#define DICEV 50

static int KnotOffset(const float *knot, int order, int np, float t) {
    int firstKnot = order - 1;
    int lastKnot = np;

    int knotOffset = firstKnot;
    while (t > knot[knotOffset+1])
	++knotOffset;
    assert(knotOffset < lastKnot);
    assert(t >= knot[knotOffset] && t <= knot[knotOffset + 1]);
    return knotOffset;
}

// doesn't handle flat out discontinuities in the curve...

struct Homogeneous3 {
Homogeneous3() { x = y = z = w = 0.; }
Homogeneous3(Float xx, Float yy, Float zz, Float ww) {
    x = xx; y = yy; z = zz; w = ww;
}
Float x, y, z, w;
};
static Homogeneous3
NURBSEvaluate(int order, const float *knot, const Homogeneous3 *cp, int np,
	      int cpStride, float t, Vector *deriv = NULL) {
//    int nKnots = np + order;
    float alpha;

    int knotOffset = KnotOffset(knot, order, np, t);
    knot += knotOffset;

    int cpOffset = knotOffset - order + 1;
    assert(cpOffset >= 0 && cpOffset < np);

    Homogeneous3 *cpWork =
	(Homogeneous3 *)alloca(order * sizeof(Homogeneous3));
    for (int i = 0; i < order; ++i)
	cpWork[i] = cp[(cpOffset+i) * cpStride];

    for (int i = 0; i < order - 2; ++i)
	for (int j = 0; j < order - 1 - i; ++j) {
	    alpha = (knot[1 + j] - t) /
		(knot[1 + j] - knot[j + 2 - order + i]);
	    assert(alpha >= 0. && alpha <= 1.);

	    cpWork[j].x = cpWork[j].x * alpha + cpWork[j+1].x * (1. - alpha);
	    cpWork[j].y = cpWork[j].y * alpha + cpWork[j+1].y * (1. - alpha);
	    cpWork[j].z = cpWork[j].z * alpha + cpWork[j+1].z * (1. - alpha);
	    cpWork[j].w = cpWork[j].w * alpha + cpWork[j+1].w * (1. - alpha);
	}

    alpha = (knot[1] - t) / (knot[1] - knot[0]);
    assert(alpha >= 0. && alpha <= 1.);

    Homogeneous3 val(cpWork[0].x * alpha + cpWork[1].x * (1. - alpha),
		     cpWork[0].y * alpha + cpWork[1].y * (1. - alpha),
		     cpWork[0].z * alpha + cpWork[1].z * (1. - alpha),
		     cpWork[0].w * alpha + cpWork[1].w * (1. - alpha));

    if (deriv) {
		float factor = (order - 1) / (knot[1] - knot[0]);
		Homogeneous3 delta((cpWork[1].x - cpWork[0].x) * factor,
			   (cpWork[1].y - cpWork[0].y) * factor,
			   (cpWork[1].z - cpWork[0].z) * factor,
			   (cpWork[1].w - cpWork[0].w) * factor);

		deriv->x = delta.x / val.w - (val.x * delta.w / (val.w * val.w));
		deriv->y = delta.y / val.w - (val.y * delta.w / (val.w * val.w));
		deriv->z = delta.z / val.w - (val.z * delta.w / (val.w * val.w));
    }

    return val;
}
static Point
NURBSEvaluateSurface(int uOrder, const float *uKnot, int ucp, float u,
		     int vOrder, const float *vKnot, int vcp, float v,
		     const Homogeneous3 *cp, Vector *dPdu, Vector *dPdv) {
    Homogeneous3 *iso = (Homogeneous3 *)alloca(max(uOrder, vOrder) *
					       sizeof(Homogeneous3));

    int uOffset = KnotOffset(uKnot, uOrder, ucp, u);
    int uFirstCp = uOffset - uOrder + 1;
    assert(uFirstCp >= 0 && uFirstCp + uOrder - 1 < ucp);

    for (int i = 0; i < uOrder; ++i)
		iso[i] = NURBSEvaluate(vOrder, vKnot, &cp[uFirstCp + i], vcp,
			       ucp, v);

    int vOffset = KnotOffset(vKnot, vOrder, vcp, v);
    int vFirstCp = vOffset - vOrder + 1;
    assert(vFirstCp >= 0 && vFirstCp + vOrder - 1 < vcp);

    Homogeneous3 P = NURBSEvaluate(uOrder, uKnot, iso - uFirstCp, ucp,
				   1, u, dPdu);

    if (dPdv) {
		for (int i = 0; i < vOrder; ++i)
		    iso[i] = NURBSEvaluate(uOrder, uKnot, &cp[(vFirstCp+i)*ucp], ucp,
				   1, u);
		(void)NURBSEvaluate(vOrder, vKnot, iso - vFirstCp, vcp, 1, v, dPdv);
    }
    return Point(P.x/P.w, P.y/P.w, P.z/P.w);;
}
NURBS::NURBS(const Transform &o2w, int numu, int uo, Float *uk, Float u0, Float u1,
	int numv, int vo, Float *vk, Float v0, Float v1,
	Float *p, bool homogeneous, bool film, bool bump, bool noise)
		: Shape(o2w) {
	nu = numu;    uorder = uo;
	umin = u0;    umax = u1;
	nv = numv;    vorder = vo;
	vmin = v0;    vmax = v1;
	isHomogeneous = homogeneous;
	isFilm = film;
	isBump = bump;
	isNoise = noise;
	if (isHomogeneous) {
		P = new Float[4*nu*nv];
		memcpy(P, p, 4*nu*nv*sizeof(Float));
	} else {
		P = new Float[3*nu*nv];
		memcpy(P, p, 3*nu*nv*sizeof(Float));
	}
	uknot = new Float[nu + uorder];
	memcpy(uknot, uk, (nu + uorder) * sizeof(Float));
	vknot = new Float[nv + vorder];
	memcpy(vknot, vk, (nv + vorder) * sizeof(Float));
}
NURBS::~NURBS() {
	delete[] P;
	delete[] uknot;
	delete[] vknot;
}
BBox NURBS::Bound() const {
	if (!isHomogeneous) {
		Float *pp = P;
		BBox bound = Point(pp[0], pp[1], pp[2]);
		for (int i = 0; i < nu*nv; ++i, pp += 3)
			bound = Union(bound, Point(pp[0], pp[1], pp[2]));
		return bound;
	} else {
		Float *pp = P;
		BBox bound = Point(pp[0] / pp[3], pp[1] / pp[3], pp[2] / pp[3]);
		for (int i = 0; i < nu*nv; ++i, pp += 4)
			bound = Union(bound, Point(pp[0] / pp[3], pp[1] / pp[3], pp[2] / pp[3]));
		return bound;
	}
}
BBox NURBS::BoundWorldSpace() const {
	if (!isHomogeneous) {
		Float *pp = P;
		Point pt = ObjectToWorld(Point(pp[0], pp[1], pp[2]));
		BBox bound = pt;
		for (int i = 0; i < nu*nv; ++i, pp += 3) {
			pt = ObjectToWorld(Point(pp[0], pp[1], pp[2]));
			bound = Union(bound, pt);
		}
		return bound;
	} else {
		Float *pp = P;
		Point pt = ObjectToWorld(Point(pp[0]/pp[3],
			pp[1]/pp[3], pp[2]/pp[3]));
		BBox bound = pt;
		for (int i = 0; i < nu*nv; ++i, pp += 4) {
			pt = ObjectToWorld(Point(pp[0]/pp[3],
				pp[1]/pp[3], pp[2]/pp[3]));
			bound = Union(bound, pt);
		}
		return bound;
	}
}

const Float NURBS::thickness[2501] = {
  0.726518, 0.683274, 0.754719, 0.773760, 0.800966, 0.740803, 0.782043,
  0.841286, 0.820000, 0.863679, 0.878628, 0.906053, 0.914199, 0.940399,
  0.967377, 0.983445, 1.000000, 0.981960, 0.951267, 0.943046, 0.894480,
  0.885248, 0.853411, 0.878482, 0.806842, 0.810620, 0.754570, 0.726506,
  0.807917, 0.697368, 0.721799, 0.743492, 0.697061, 0.621388, 0.663166,
  0.624559, 0.598712, 0.660301, 0.509070, 0.594956, 0.526334, 0.559455,
  0.452697, 0.548406, 0.371956, 0.373805, 0.480007, 0.487397, 0.184586,
  0.461076, 0.684106, 0.621703, 0.669073, 0.759618, 0.799183, 0.762020,
  0.744274, 0.763161, 0.836543, 0.820868, 0.861932, 0.919049, 0.933395,
  0.948956, 0.954717, 0.980390, 1.000000, 0.982303, 0.960249, 0.944642,
  0.921245, 0.870069, 0.876138, 0.884861, 0.868357, 0.837907, 0.775822,
  0.803443, 0.775699, 0.719533, 0.744042, 0.639850, 0.670121, 0.714295,
  0.690881, 0.584776, 0.537296, 0.649031, 0.524417, 0.579269, 0.609195,
  0.521220, 0.517917, 0.542116, 0.422549, 0.521446, 0.367901, 0.464309,
  0.413978, 0.452503, 0.677174, 0.650314, 0.771452, 0.785135, 0.708678,
  0.815209, 0.813178, 0.813660, 0.849028, 0.842790, 0.895013, 0.889347,
  0.917824, 0.930479, 0.952594, 0.977480, 1.000000, 0.979129, 0.952085,
  0.950141, 0.900198, 0.877004, 0.879726, 0.876132, 0.871214, 0.852156,
  0.816887, 0.766354, 0.786645, 0.721726, 0.687876, 0.653632, 0.692970,
  0.610184, 0.564267, 0.659521, 0.507836, 0.633743, 0.471186, 0.590213,
  0.390294, 0.591854, 0.330125, 0.465760, 0.331826, 0.333357, 0.477761,
  0.444607, 0.406463, 0.465025, 0.700859, 0.734482, 0.648568, 0.681938,
  0.762401, 0.721207, 0.745208, 0.788814, 0.836853, 0.879294, 0.884244,
  0.911331, 0.896903, 0.927615, 0.968000, 0.976530, 1.000000, 0.975392,
  0.967972, 0.922783, 0.932505, 0.872800, 0.840443, 0.884444, 0.847175,
  0.785762, 0.759490, 0.767404, 0.801080, 0.775833, 0.698778, 0.707671,
  0.692223, 0.651742, 0.670840, 0.524893, 0.606068, 0.627873, 0.512275,
  0.630852, 0.537788, 0.579410, 0.537327, 0.492698, 0.270558, 0.483940,
  0.485693, 0.488739, 0.460660, 0.433729, 0.741877, 0.732598, 0.766993,
  0.790626, 0.778483, 0.779698, 0.759611, 0.845494, 0.866050, 0.822447,
  0.888357, 0.885586, 0.924864, 0.930601, 0.963647, 0.977984, 1.000000,
  0.976898, 0.949179, 0.943743, 0.907174, 0.907026, 0.880638, 0.886707,
  0.820229, 0.824056, 0.819760, 0.727264, 0.756529, 0.751498, 0.770897,
  0.692452, 0.651483, 0.710353, 0.627813, 0.494422, 0.566382, 0.599711,
  0.622843, 0.469402, 0.558717, 0.495701, 0.475260, 0.401647, 0.493691,
  0.310573, 0.434449, 0.195257, 0.331238, 0.398903, 0.705603, 0.719317,
  0.696825, 0.691325, 0.772415, 0.793151, 0.820686, 0.803104, 0.857171,
  0.835508, 0.900864, 0.908457, 0.925690, 0.931712, 0.960638, 0.974805,
  1.000000, 0.983175, 0.963406, 0.924705, 0.900700, 0.892566, 0.884432,
  0.866234, 0.852300, 0.798745, 0.824235, 0.753967, 0.737654, 0.692262,
  0.774590, 0.600952, 0.725014, 0.628715, 0.573510, 0.607539, 0.674616,
  0.652739, 0.611457, 0.581642, 0.407825, 0.542499, 0.447934, 0.525123,
  0.541557, 0.519922, 0.491938, 0.272107, 0.441939, 0.355482, 0.643908,
  0.750513, 0.767600, 0.780958, 0.759777, 0.788074, 0.759436, 0.797059,
  0.864669, 0.859466, 0.877779, 0.914855, 0.919839, 0.933000, 0.952393,
  0.981686, 1.000000, 0.975596, 0.961961, 0.939772, 0.931569, 0.910082,
  0.856990, 0.837395, 0.839882, 0.777950, 0.755835, 0.763497, 0.736958,
  0.786175, 0.767489, 0.717261, 0.726124, 0.706660, 0.639292, 0.585825,
  0.577686, 0.610901, 0.582717, 0.476948, 0.418615, 0.364088, 0.353173,
  0.497021, 0.399012, 0.351328, 0.344759, 0.220689, 0.335868, 0.457793,
  0.692835, 0.660145, 0.740590, 0.770092, 0.758482, 0.734503, 0.832147,
  0.830641, 0.841405, 0.836051, 0.883357, 0.896169, 0.931329, 0.936655,
  0.954137, 0.978094, 1.000000, 0.981297, 0.967805, 0.934720, 0.933265,
  0.877756, 0.874397, 0.814004, 0.859851, 0.841716, 0.776995, 0.727498,
  0.797086, 0.765872, 0.638514, 0.692166, 0.742607, 0.692016, 0.579165,
  0.614974, 0.661760, 0.472997, 0.449848, 0.511391, 0.467189, 0.536419,
  0.549607, 0.338115, 0.278122, 0.253262, 0.387041, 0.456347, 0.416079,
  0.320484, 0.604721, 0.712869, 0.730592, 0.743830, 0.770888, 0.763684,
  0.778306, 0.849292, 0.838105, 0.832317, 0.886241, 0.901271, 0.924490,
  0.942753, 0.953855, 0.982831, 1.000000, 0.974982, 0.966993, 0.928129,
  0.919263, 0.884091, 0.896853, 0.843648, 0.827697, 0.848630, 0.783110,
  0.809928, 0.777690, 0.661530, 0.681360, 0.736641, 0.698960, 0.708917,
  0.601804, 0.558399, 0.528863, 0.602356, 0.453325, 0.576935, 0.579365,
  0.549440, 0.388362, 0.431934, 0.383144, 0.400209, 0.306757, 0.342436,
  0.370341, 0.202284, 0.688625, 0.603850, 0.702162, 0.723723, 0.734132,
  0.803344, 0.772785, 0.832532, 0.844079, 0.884278, 0.890210, 0.913412,
  0.927202, 0.933773, 0.962719, 0.973440, 1.000000, 0.982934, 0.962236,
  0.943996, 0.920734, 0.894776, 0.871117, 0.873809, 0.860984, 0.834697,
  0.757998, 0.712961, 0.693008, 0.762092, 0.668171, 0.716842, 0.709131,
  0.671153, 0.520360, 0.495254, 0.556845, 0.523153, 0.591717, 0.609765,
  0.428966, 0.456922, 0.520149, 0.546030, 0.479869, 0.360258, 0.496858,
  0.264647, 0.227584, 0.341140, 0.725199, 0.724221, 0.767071, 0.692510,
  0.746638, 0.800550, 0.831147, 0.824167, 0.857076, 0.884891, 0.861521,
  0.917579, 0.922006, 0.950687, 0.959888, 0.980512, 1.000000, 0.982606,
  0.965430, 0.942707, 0.925830, 0.919849, 0.901472, 0.846537, 0.865224,
  0.827682, 0.812889, 0.805251, 0.789050, 0.773412, 0.682321, 0.714880,
  0.675220, 0.592070, 0.686342, 0.678659, 0.662877, 0.611904, 0.539865,
  0.564390, 0.542181, 0.533930, 0.489835, 0.344796, 0.461285, 0.243296,
  0.454771, 0.385576, 0.451258, 0.344929, 0.603137, 0.698464, 0.721441,
  0.664191, 0.792598, 0.766536, 0.812879, 0.815585, 0.805351, 0.853639,
  0.862624, 0.895110, 0.894560, 0.945887, 0.961422, 0.982706, 1.000000,
  0.979453, 0.959542, 0.951907, 0.897352, 0.872956, 0.887891, 0.866494,
  0.814235, 0.790672, 0.812228, 0.818445, 0.785517, 0.735079, 0.711126,
  0.744745, 0.739059, 0.722326, 0.699886, 0.664757, 0.563693, 0.587577,
  0.488115, 0.534778, 0.608135, 0.491397, 0.563817, 0.562270, 0.334824,
  0.407571, 0.473736, 0.452888, 0.467727, 0.248970, 0.717016, 0.756445,
  0.694297, 0.726658, 0.767079, 0.785831, 0.811513, 0.787112, 0.838217,
  0.849471, 0.888573, 0.917236, 0.906471, 0.943035, 0.966180, 0.975152,
  1.000000, 0.976457, 0.962730, 0.942658, 0.926489, 0.904596, 0.846041,
  0.864135, 0.844932, 0.793415, 0.736653, 0.761368, 0.681924, 0.785451,
  0.733514, 0.735763, 0.731903, 0.720541, 0.569652, 0.609853, 0.597529,
  0.634024, 0.588595, 0.626878, 0.420057, 0.442241, 0.563430, 0.421231,
  0.533223, 0.483903, 0.392653, 0.300615, 0.476834, 0.186353, 0.729465,
  0.717407, 0.747596, 0.785526, 0.714976, 0.753126, 0.760072, 0.788605,
  0.849491, 0.832959, 0.852459, 0.884229, 0.934612, 0.920733, 0.951831,
  0.975308, 1.000000, 0.981383, 0.966877, 0.925040, 0.924322, 0.872657,
  0.873267, 0.859750, 0.834761, 0.799239, 0.770651, 0.817608, 0.767685,
  0.692437, 0.635112, 0.689416, 0.723411, 0.698231, 0.612149, 0.695753,
  0.479054, 0.584888, 0.482641, 0.574427, 0.563594, 0.452839, 0.564734,
  0.496905, 0.391794, 0.251694, 0.470596, 0.461649, 0.417841, 0.395029,
  0.733798, 0.669046, 0.751385, 0.710779, 0.792151, 0.739501, 0.808714,
  0.768256, 0.800182, 0.831048, 0.862397, 0.877763, 0.923413, 0.922981,
  0.966521, 0.983643, 1.000000, 0.983507, 0.959371, 0.931701, 0.923335,
  0.880156, 0.884869, 0.858325, 0.861804, 0.851612, 0.802618, 0.788690,
  0.784695, 0.698592, 0.682571, 0.718783, 0.685994, 0.551644, 0.703889,
  0.588644, 0.530988, 0.584699, 0.537082, 0.490916, 0.559614, 0.533829,
  0.464839, 0.389493, 0.491645, 0.367120, 0.300097, 0.451135, 0.271946,
  0.395486, 0.577886, 0.744150, 0.761946, 0.752471, 0.736478, 0.797044,
  0.802648, 0.846678, 0.817165, 0.840567, 0.847448, 0.900714, 0.933049,
  0.943166, 0.960611, 0.982769, 1.000000, 0.982117, 0.956615, 0.935482,
  0.896367, 0.868755, 0.840540, 0.859102, 0.821517, 0.836976, 0.827597,
  0.780812, 0.756032, 0.670523, 0.668832, 0.697527, 0.735906, 0.559413,
  0.545572, 0.564435, 0.638231, 0.602130, 0.447534, 0.626254, 0.594504,
  0.399794, 0.473670, 0.521548, 0.508234, 0.230249, 0.422692, 0.445553,
  0.429521, 0.455125, 0.599099, 0.737721, 0.771096, 0.675300, 0.729562,
  0.747028, 0.826821, 0.774301, 0.857571, 0.842450, 0.856571, 0.917956,
  0.929231, 0.924844, 0.967756, 0.982944, 1.000000, 0.981756, 0.959111,
  0.937965, 0.925721, 0.890350, 0.885741, 0.828218, 0.797630, 0.808306,
  0.755644, 0.796285, 0.702687, 0.781575, 0.731086, 0.649300, 0.704841,
  0.664841, 0.602724, 0.688935, 0.647298, 0.482148, 0.427927, 0.543410,
  0.494551, 0.415740, 0.467390, 0.349461, 0.380616, 0.400132, 0.202888,
  0.423199, 0.212900, 0.327200, 0.596007, 0.736871, 0.755023, 0.761979,
  0.800849, 0.814126, 0.744029, 0.767319, 0.842137, 0.833048, 0.901148,
  0.875740, 0.915254, 0.942753, 0.962589, 0.981821, 1.000000, 0.979020,
  0.965553, 0.942243, 0.933927, 0.883837, 0.895162, 0.850603, 0.829958,
  0.774647, 0.826687, 0.766865, 0.786408, 0.786376, 0.761628, 0.742651,
  0.710987, 0.681529, 0.642882, 0.642295, 0.600480, 0.595923, 0.594814,
  0.398487, 0.574458, 0.523743, 0.365467, 0.343042, 0.339583, 0.490650,
  0.488116, 0.445267, 0.239916, 0.390781, 0.646992, 0.626302, 0.765537,
  0.675453, 0.757548, 0.751457, 0.764668, 0.798628, 0.851690, 0.885574,
  0.860620, 0.900345, 0.929777, 0.950746, 0.951282, 0.977711, 1.000000,
  0.981348, 0.966293, 0.930750, 0.926403, 0.906075, 0.897109, 0.878945,
  0.859236, 0.764468, 0.808650, 0.792216, 0.787964, 0.771372, 0.686150,
  0.619471, 0.731093, 0.593477, 0.571170, 0.591351, 0.588275, 0.552885,
  0.590225, 0.425246, 0.474178, 0.577959, 0.506548, 0.326756, 0.514266,
  0.484380, 0.368014, 0.335760, 0.324267, 0.255159, 0.695186, 0.617404,
  0.766817, 0.730877, 0.798424, 0.797687, 0.795986, 0.817411, 0.821472,
  0.837626, 0.862549, 0.903694, 0.911169, 0.926203, 0.964436, 0.979800,
  1.000000, 0.976366, 0.966085, 0.926533, 0.910773, 0.881276, 0.840301,
  0.883371, 0.821104, 0.844062, 0.813996, 0.798860, 0.713503, 0.663371,
  0.772331, 0.736270, 0.603215, 0.712019, 0.578271, 0.695293, 0.607095,
  0.612530, 0.552353, 0.604904, 0.598717, 0.547901, 0.454145, 0.362265,
  0.520955, 0.298971, 0.447933, 0.320120, 0.307058, 0.350558, 0.673009,
  0.719674, 0.747417, 0.734703, 0.764837, 0.707805, 0.773843, 0.787188,
  0.854106, 0.855974, 0.861780, 0.907110, 0.906931, 0.935111, 0.960285,
  0.979236, 1.000000, 0.974201, 0.949822, 0.948037, 0.932745, 0.919570,
  0.881262, 0.843074, 0.809539, 0.806373, 0.793707, 0.813067, 0.780129,
  0.772547, 0.703122, 0.634222, 0.667801, 0.575899, 0.657209, 0.692020,
  0.502869, 0.645433, 0.469664, 0.600452, 0.470546, 0.472486, 0.468219,
  0.468593, 0.501115, 0.530372, 0.511948, 0.219403, 0.158176, 0.150072,
  0.733098, 0.749950, 0.636297, 0.729046, 0.714931, 0.744164, 0.824437,
  0.830412, 0.796650, 0.857251, 0.874278, 0.897956, 0.923876, 0.948620,
  0.961228, 0.976501, 1.000000, 0.981589, 0.949087, 0.951270, 0.918994,
  0.893872, 0.870845, 0.884814, 0.867596, 0.775455, 0.833396, 0.814784,
  0.716274, 0.785249, 0.769422, 0.622560, 0.724248, 0.724517, 0.641577,
  0.494925, 0.526136, 0.529988, 0.586372, 0.477832, 0.573083, 0.349168,
  0.531916, 0.416376, 0.527830, 0.350194, 0.415415, 0.429952, 0.479190,
  0.378549, 0.700628, 0.689581, 0.628787, 0.791670, 0.742452, 0.818588,
  0.749888, 0.795991, 0.860931, 0.843668, 0.857654, 0.910302, 0.914926,
  0.920369, 0.963605, 0.983260, 1.000000, 0.983238, 0.952287, 0.943904,
  0.902926, 0.917539, 0.889690, 0.881859, 0.819808, 0.824195, 0.758338,
  0.811047, 0.721413, 0.677602, 0.713215, 0.745847, 0.687162, 0.653282,
  0.694504, 0.684583, 0.575972, 0.657996, 0.474356, 0.454877, 0.450539,
  0.462413, 0.569785, 0.492453, 0.400145, 0.451664, 0.405172, 0.340004,
  0.360587, 0.398384, 0.578392, 0.720939, 0.735783, 0.762957, 0.762390,
  0.816311, 0.770302, 0.834507, 0.856114, 0.857462, 0.895116, 0.880715,
  0.912337, 0.934570, 0.962580, 0.976219, 1.000000, 0.973763, 0.951163,
  0.937424, 0.906557, 0.893915, 0.894636, 0.828185, 0.788522, 0.781148,
  0.821839, 0.783310, 0.762816, 0.702345, 0.675295, 0.706133, 0.733006,
  0.709912, 0.585118, 0.603583, 0.642772, 0.575082, 0.447408, 0.531435,
  0.606758, 0.579960, 0.514041, 0.388594, 0.346840, 0.347583, 0.384117,
  0.366788, 0.289907, 0.367373, 0.730632, 0.702211, 0.773446, 0.758562,
  0.776173, 0.822936, 0.825989, 0.809490, 0.841233, 0.846361, 0.887675,
  0.910721, 0.932324, 0.940995, 0.961618, 0.974994, 1.000000, 0.978067,
  0.961473, 0.922249, 0.930045, 0.900878, 0.900994, 0.854719, 0.859904,
  0.780333, 0.818619, 0.746041, 0.771114, 0.655145, 0.758527, 0.695902,
  0.683596, 0.703176, 0.557415, 0.544157, 0.648491, 0.653681, 0.572173,
  0.486444, 0.450789, 0.409529, 0.320974, 0.330090, 0.497144, 0.451769,
  0.298978, 0.430712, 0.475334, 0.423144, 0.711912, 0.730561, 0.689920,
  0.757649, 0.709819, 0.723293, 0.824906, 0.847422, 0.824436, 0.851191,
  0.898620, 0.878182, 0.932049, 0.936510, 0.967115, 0.983880, 1.000000,
  0.981889, 0.966763, 0.925660, 0.893337, 0.879149, 0.867239, 0.817191,
  0.818911, 0.765478, 0.813741, 0.777651, 0.783119, 0.750862, 0.683103,
  0.689881, 0.647197, 0.671992, 0.544415, 0.613857, 0.630186, 0.644801,
  0.518501, 0.629920, 0.450322, 0.410011, 0.361078, 0.325474, 0.462829,
  0.256629, 0.241743, 0.244956, 0.435893, 0.469782, 0.635637, 0.735580,
  0.659464, 0.748093, 0.791375, 0.769820, 0.809459, 0.815226, 0.866132,
  0.845516, 0.843701, 0.879340, 0.928801, 0.935120, 0.964136, 0.982815,
  1.000000, 0.978880, 0.963512, 0.927068, 0.914888, 0.917998, 0.869588,
  0.860531, 0.792220, 0.851566, 0.806584, 0.728841, 0.693801, 0.716953,
  0.652650, 0.671070, 0.601562, 0.612905, 0.698536, 0.684260, 0.631035,
  0.577388, 0.525701, 0.553104, 0.582632, 0.471148, 0.530728, 0.472491,
  0.500101, 0.404174, 0.359991, 0.397531, 0.453179, 0.157874, 0.709562,
  0.651300, 0.628259, 0.663189, 0.784589, 0.715489, 0.838005, 0.799596,
  0.808289, 0.817805, 0.887510, 0.887728, 0.914021, 0.946424, 0.960747,
  0.977133, 1.000000, 0.976557, 0.964236, 0.920118, 0.929357, 0.901818,
  0.873224, 0.858666, 0.798030, 0.779621, 0.835255, 0.775437, 0.780996,
  0.770804, 0.723706, 0.683430, 0.742918, 0.665560, 0.628714, 0.657315,
  0.615341, 0.559845, 0.418816, 0.590919, 0.506486, 0.534380, 0.320405,
  0.531687, 0.406612, 0.439174, 0.228500, 0.191817, 0.437278, 0.424162,
  0.583473, 0.705822, 0.693879, 0.730516, 0.684358, 0.770083, 0.806123,
  0.852984, 0.865155, 0.840447, 0.891771, 0.894733, 0.925589, 0.945627,
  0.967121, 0.974834, 1.000000, 0.980129, 0.959957, 0.938110, 0.909188,
  0.918136, 0.848733, 0.843768, 0.857729, 0.817635, 0.830551, 0.809081,
  0.761000, 0.756468, 0.732946, 0.706585, 0.617872, 0.719737, 0.709032,
  0.551765, 0.565223, 0.587762, 0.464800, 0.462969, 0.581255, 0.576587,
  0.507819, 0.444098, 0.458611, 0.433102, 0.409512, 0.502713, 0.179047,
  0.137292, 0.668814, 0.670519, 0.773568, 0.742453, 0.772862, 0.802014,
  0.839558, 0.817614, 0.842133, 0.860104, 0.850847, 0.876845, 0.896532,
  0.932732, 0.948396, 0.973699, 1.000000, 0.978839, 0.961137, 0.943495,
  0.923933, 0.891956, 0.879121, 0.836873, 0.860498, 0.768372, 0.823218,
  0.759049, 0.689871, 0.776040, 0.698302, 0.696999, 0.610442, 0.628275,
  0.711913, 0.663142, 0.484432, 0.662969, 0.491325, 0.557416, 0.518515,
  0.456247, 0.527846, 0.470639, 0.472442, 0.485116, 0.417775, 0.226181,
  0.221044, 0.184976, 0.712507, 0.732934, 0.715833, 0.696191, 0.805123,
  0.750645, 0.773476, 0.838258, 0.846938, 0.845608, 0.882790, 0.918261,
  0.930245, 0.945218, 0.954958, 0.982556, 1.000000, 0.978116, 0.964994,
  0.944869, 0.933514, 0.887589, 0.841657, 0.867247, 0.862032, 0.831672,
  0.785190, 0.766012, 0.785822, 0.740692, 0.720003, 0.695009, 0.634698,
  0.724774, 0.669555, 0.556868, 0.538970, 0.656399, 0.639468, 0.604453,
  0.454005, 0.497096, 0.542396, 0.304128, 0.332917, 0.251046, 0.488274,
  0.416634, 0.317927, 0.380384, 0.644883, 0.663515, 0.767589, 0.713959,
  0.799358, 0.802593, 0.837174, 0.794805, 0.801160, 0.872596, 0.893795,
  0.905184, 0.903905, 0.920622, 0.961539, 0.982934, 1.000000, 0.982724,
  0.965044, 0.928042, 0.925275, 0.870705, 0.888688, 0.853701, 0.799857,
  0.845828, 0.796201, 0.819957, 0.755183, 0.777716, 0.730018, 0.734850,
  0.605544, 0.657214, 0.544533, 0.498286, 0.532886, 0.453625, 0.502611,
  0.495055, 0.570498, 0.345982, 0.572252, 0.550030, 0.273084, 0.423283,
  0.465254, 0.298471, 0.299275, 0.350600, 0.669288, 0.756875, 0.730704,
  0.673226, 0.744920, 0.798012, 0.835144, 0.846448, 0.849214, 0.851617,
  0.890101, 0.882105, 0.903023, 0.947969, 0.963306, 0.976888, 1.000000,
  0.983283, 0.954753, 0.924546, 0.906695, 0.866953, 0.851795, 0.829905,
  0.866855, 0.783150, 0.814602, 0.795106, 0.750999, 0.790677, 0.666686,
  0.746077, 0.738996, 0.699060, 0.706629, 0.585567, 0.587191, 0.647147,
  0.475664, 0.410315, 0.456893, 0.576656, 0.407942, 0.436764, 0.483506,
  0.523957, 0.441384, 0.376758, 0.450542, 0.453699, 0.692303, 0.611491,
  0.768545, 0.763566, 0.716061, 0.810580, 0.830471, 0.846680, 0.837590,
  0.843329, 0.897094, 0.906995, 0.903036, 0.947467, 0.958115, 0.975067,
  1.000000, 0.982964, 0.967806, 0.933201, 0.933558, 0.885880, 0.855299,
  0.831801, 0.852536, 0.845298, 0.750226, 0.758001, 0.735778, 0.786098,
  0.677129, 0.749647, 0.741454, 0.588856, 0.656482, 0.543785, 0.677002,
  0.570430, 0.418722, 0.527508, 0.571930, 0.573766, 0.349001, 0.548631,
  0.474283, 0.422183, 0.519433, 0.448534, 0.310162, 0.468166, 0.585758,
  0.659790, 0.677143, 0.694760, 0.751077, 0.817013, 0.738670, 0.822639,
  0.814248, 0.848426, 0.877599, 0.897812, 0.908252, 0.938021, 0.962105,
  0.982905, 1.000000, 0.980842, 0.955439, 0.943287, 0.895766, 0.869955,
  0.877680, 0.828490, 0.868348, 0.773939, 0.813094, 0.818759, 0.794963,
  0.791635, 0.767605, 0.748536, 0.636206, 0.586425, 0.547058, 0.652579,
0.483451, 0.481559, 0.487568, 0.480657, 0.512078, 0.540877, 0.533347,
0.511629, 0.306165, 0.369473, 0.415006, 0.440496, 0.472735, 0.437487,
0.650690, 0.756359, 0.765154, 0.773346, 0.699096, 0.809951, 0.832407,
0.838948, 0.857328, 0.874314, 0.892421, 0.907089, 0.924307, 0.951810,
0.965169, 0.981846, 1.000000, 0.982946, 0.965572, 0.949896, 0.901051,
0.888342, 0.884230, 0.880629, 0.787039, 0.835413, 0.762663, 0.785776,
0.747838, 0.677617, 0.705346, 0.731630, 0.593812, 0.619065, 0.645315,
0.545133, 0.500086, 0.561748, 0.634267, 0.607371, 0.416230, 0.532066,
0.478897, 0.521762, 0.473491, 0.352879, 0.388287, 0.312992, 0.218598,
0.260655, 0.623516, 0.658135, 0.730322, 0.771229, 0.700133, 0.788378,
0.795951, 0.798441, 0.804054, 0.881549, 0.873622, 0.901654, 0.923794,
0.939231, 0.965498, 0.975453, 1.000000, 0.983464, 0.960904, 0.941510,
0.929271, 0.908706, 0.857878, 0.840723, 0.835815, 0.844553, 0.804925,
0.817482, 0.703856, 0.762403, 0.669829, 0.670469, 0.738097, 0.700789,
0.548246, 0.507887, 0.567992, 0.591061, 0.519240, 0.536505, 0.499465,
0.551469, 0.315139, 0.288312, 0.342795, 0.503695, 0.271430, 0.472271,
0.432691, 0.398170, 0.663393, 0.702740, 0.702767, 0.751198, 0.789138,
0.811407, 0.790127, 0.802941, 0.857384, 0.861848, 0.841631, 0.918029,
0.931843, 0.951202, 0.963842, 0.983615, 1.000000, 0.974430, 0.959908,
0.934565, 0.921923, 0.917128, 0.846488, 0.864591, 0.868360, 0.796332,
0.790036, 0.721155, 0.703204, 0.687002, 0.753439, 0.702764, 0.706394,
0.589196, 0.552408, 0.611484, 0.678502, 0.583028, 0.614547, 0.587839,
0.377140, 0.564929, 0.521182, 0.547890, 0.503891, 0.384465, 0.485003,
0.451577, 0.343213, 0.276918, 0.598727, 0.759821, 0.677952, 0.686429,
0.767662, 0.736487, 0.792304, 0.854313, 0.816901, 0.862958, 0.852977,
0.868329, 0.897552, 0.948101, 0.952217, 0.976053, 1.000000, 0.975941,
0.963385, 0.922650, 0.933186, 0.910160, 0.897866, 0.864520, 0.848697,
0.829176, 0.836819, 0.773788, 0.743103, 0.726337, 0.753369, 0.694404,
0.662854, 0.549511, 0.660649, 0.691590, 0.523588, 0.457220, 0.638648,
0.522550, 0.540835, 0.362121, 0.465526, 0.498261, 0.529389, 0.452845,
0.483400, 0.247680, 0.439244, 0.342556, 0.623445, 0.724435, 0.677037,
0.664402, 0.724782, 0.812679, 0.814549, 0.787653, 0.821627, 0.816436,
0.885468, 0.872171, 0.919547, 0.951880, 0.949176, 0.975162, 1.000000,
0.977186, 0.951750, 0.949860, 0.926612, 0.911062, 0.902479, 0.827800,
0.831693, 0.846213, 0.821912, 0.744804, 0.807162, 0.743679, 0.749097,
0.644708, 0.624864, 0.724237, 0.585784, 0.610280, 0.654397, 0.649288,
0.589830, 0.428140, 0.603384, 0.425597, 0.378501, 0.434322, 0.354359,
0.336168, 0.405835, 0.267457, 0.360226, 0.393843, 0.596482, 0.643045,
0.709306, 0.663880, 0.734352, 0.810973, 0.832493, 0.768187, 0.793797,
0.882209, 0.884709, 0.912427, 0.897819, 0.947546, 0.964440, 0.977828,
1.000000, 0.980625, 0.952919, 0.949650, 0.926378, 0.876519, 0.852238,
0.880681, 0.842278, 0.806342, 0.749070, 0.715860, 0.766046, 0.751764,
0.748449, 0.719329, 0.718764, 0.588530, 0.670450, 0.519768, 0.677460,
0.594744, 0.461953, 0.399550, 0.507657, 0.548986, 0.550948, 0.464209,
0.441384, 0.438442, 0.482711, 0.447120, 0.214803, 0.131000, 0.698979,
0.739519, 0.658259, 0.768924, 0.776752, 0.797007, 0.751442, 0.839583,
0.855602, 0.870428, 0.869489, 0.897028, 0.911047, 0.925678, 0.961812,
0.983894, 1.000000, 0.980725, 0.961094, 0.934523, 0.924077, 0.867209,
0.845155, 0.854423, 0.841728, 0.824808, 0.839297, 0.761080, 0.732991,
0.678859, 0.699023, 0.752454, 0.735827, 0.659877, 0.668833, 0.633335,
0.520049, 0.632130, 0.515654, 0.613920, 0.506269, 0.536138, 0.430201,
0.518519, 0.506696, 0.520680, 0.461056, 0.497481, 0.358491, 0.259606,
0.644499, 0.625628, 0.686968, 0.718950, 0.763087, 0.809589, 0.833687,
0.821351, 0.806344, 0.835564, 0.887553, 0.900647, 0.901634, 0.941663,
0.947100, 0.982614, 1.000000, 0.976178, 0.961615, 0.938355, 0.897572,
0.869153, 0.850228, 0.851638, 0.859129, 0.848283, 0.778684, 0.779797,
0.794908, 0.774175, 0.754492, 0.634299, 0.733926, 0.566863, 0.638353,
0.606505, 0.663969, 0.556959, 0.640307, 0.399928, 0.542590, 0.519432,
0.491734, 0.514684, 0.316187, 0.436476, 0.407718, 0.298488, 0.427961,
0.180218, 0.710961, 0.739094, 0.658262, 0.775540, 0.715463, 0.817447,
0.817311, 0.819487, 0.832322, 0.862063, 0.866443, 0.881615, 0.926288,
0.929508, 0.954529, 0.975925, 1.000000, 0.975611, 0.961118, 0.942315,
0.904376, 0.879246, 0.855520, 0.873840, 0.865637, 0.799890, 0.775012,
0.769352, 0.767736, 0.656875, 0.722478, 0.663082, 0.723561, 0.684680,
0.549301, 0.494046, 0.612928, 0.632711, 0.559213, 0.628315, 0.471641,
0.563968, 0.375232, 0.565837, 0.551235, 0.397077, 0.272304, 0.417696,
0.379857, 0.401774, 0.625672, 0.731816, 0.762991, 0.722054, 0.754685,
0.803956, 0.815657, 0.838893, 0.806303, 0.838755, 0.893335, 0.907831,
0.920688, 0.941894, 0.956199, 0.981086, 1.000000, 0.983286, 0.957936,
0.923261, 0.932446, 0.909862, 0.898182, 0.817362, 0.854966, 0.846996,
0.795611, 0.813197, 0.757863, 0.658141, 0.728411, 0.726288, 0.716609,
0.646702, 0.539332, 0.556860, 0.520702, 0.623794, 0.640754, 0.494831,
0.611227, 0.545373, 0.582512, 0.456371, 0.365364, 0.361777, 0.250706,
0.468289, 0.251815, 0.341118, 0.732869, 0.620080, 0.674839, 0.769194,
0.702302, 0.818826, 0.810053, 0.826039, 0.859896, 0.823700, 0.885196,
0.891655, 0.929833, 0.935096, 0.964768, 0.982930, 1.000000, 0.983605,
0.961465, 0.940402, 0.909141, 0.902217, 0.859903, 0.840187, 0.791123,
0.817988, 0.806726, 0.730642, 0.734804, 0.771971, 0.737939, 0.648261,
0.728778, 0.711758, 0.707172, 0.692564, 0.650309, 0.593147, 0.568941,
0.555932, 0.554076, 0.387255, 0.569842, 0.441316, 0.434595, 0.469932,
0.324455, 0.197305, 0.398550, 0.438869, 0.689062, 0.748865, 0.704913,
0.771064, 0.707532, 0.771991, 0.771333, 0.837164, 0.844906, 0.864028,
0.879470, 0.882365, 0.929362, 0.935167, 0.948695, 0.982013, 1.000000,
0.982957, 0.955269, 0.949248, 0.916614, 0.918076, 0.903863, 0.847536,
0.829280, 0.817936, 0.737688, 0.795421, 0.762625, 0.758487, 0.720440,
0.620351, 0.686620, 0.716772, 0.693850, 0.651928, 0.562996, 0.475345,
0.535439, 0.615691, 0.555618, 0.580796, 0.347585, 0.439066, 0.354978,
0.298405, 0.272374, 0.397864, 0.146897, 0.318768, 0.667321, 0.687368,
0.701964, 0.729197, 0.787205, 0.804738, 0.833162, 0.842146, 0.831945,
0.854141, 0.875885, 0.916919, 0.920003, 0.951533, 0.965341, 0.978400,
1.000000, 0.975592, 0.959035, 0.922762, 0.895611, 0.876023, 0.901604,
0.827884, 0.843027, 0.785138, 0.775305, 0.798882, 0.785219, 0.707962,
0.655476, 0.638251, 0.709062, 0.660746, 0.653651, 0.611348, 0.544880,
0.574978, 0.493108, 0.574451, 0.599337, 0.539119, 0.497390, 0.430407,
0.478024, 0.382117, 0.466178, 0.313050, 0.365143, 0.223916, 0.655678,
0.707165, 0.691299, 0.710765, 0.779386, 0.812975, 0.797298, 0.853416,
0.843167, 0.836082, 0.861793, 0.907656, 0.914291, 0.950745, 0.952102,
0.983644, 1.000000, 0.979253, 0.958376, 0.943605, 0.899366, 0.889989,
0.866490, 0.870317, 0.855471, 0.852659, 0.739666, 0.775743, 0.725462,
0.746674, 0.741086, 0.715251, 0.600407, 0.725467, 0.700620, 0.666283,
0.656587, 0.552791, 0.605967, 0.508007, 0.539897, 0.595117, 0.322814,
0.559772, 0.534529, 0.320042, 0.499767, 0.310328, 0.400184, 0.275066,
0.736172, 0.730204, 0.738178, 0.698278, 0.745089, 0.762541, 0.760013,
0.812451, 0.861770, 0.853797, 0.902920, 0.898881, 0.894894, 0.923753,
0.959075, 0.983520, 1.000000, 0.977583, 0.955692, 0.942433, 0.925730,
0.916390, 0.883176, 0.872150, 0.863558, 0.816377, 0.833059, 0.803940,
0.787469, 0.743848, 0.642187, 0.725314, 0.635108, 0.691459, 0.700966,
0.644354, 0.486148, 0.466499, 0.434615, 0.612117, 0.517505, 0.345830,
0.437751, 0.463202, 0.301083, 0.496639, 0.381179, 0.494637, 0.194393,
0.411957 };




void NURBS::Refine(vector<Shape *> &refined) const {
	int diceu = DICEU, dicev = DICEV;
	// create a list of thicknesses (really just a thickness function)
	Float *tfcn;
	if(isFilm) tfcn = new Float[diceu*dicev];
	else tfcn = NULL;

	Float *ueval = new Float[diceu];
	Float *veval = new Float[dicev];
	Point *evalPs = new Point[diceu*dicev];
	Point *evalNs = new Point[diceu*dicev];
	int i;

	for (i = 0; i < diceu; ++i)
		ueval[i] = Lerp((Float)i / (Float)(diceu-1), umin, umax);
	for (i = 0; i < dicev; ++i)
		veval[i] = Lerp((Float)i / (Float)(dicev-1), vmin, vmax);
	memset(evalPs, 0, diceu*dicev*sizeof(Point));
	memset(evalNs, 0, diceu*dicev*sizeof(Point));
	Homogeneous3 *Pw = (Homogeneous3 *)P;
	if (!isHomogeneous) {
		Pw = (Homogeneous3 *)alloca(nu*nv*sizeof(Homogeneous3));
		for (int i = 0; i < nu*nv; ++i) {
			Pw[i].x = P[3*i];
			Pw[i].y = P[3*i+1];
			Pw[i].z = P[3*i+2];
			Pw[i].w = 1.;
		}
	}
	for (int v = 0; v < dicev; ++v) {
		for (int u = 0; u < diceu; ++u) {
			Vector dPdu, dPdv;
			Point pt = NURBSEvaluateSurface(uorder, uknot, nu, ueval[u],
				vorder, vknot, nv, veval[v], Pw, &dPdu, &dPdv);
			evalPs[v*diceu + u].x = pt.x;
			evalPs[v*diceu + u].y = pt.y;
			evalPs[v*diceu + u].z = pt.z;
			// KSK: add calculation for thickness function
			// here...maybe thicker in middle and thin at edges
			if(isFilm) {
			  Float rnum = rand() / (RAND_MAX + 1.);
#define THICK3

			  // decent, along one axial edge
#ifdef THICK1
			  Float mid = fabs(diceu / 2 - u) / (0.5 * DICEU. * rnum + 0.75 * DICEU);
#endif

			  // along the other edge...not so good
#ifdef THICK2			 
			  Float mid = fabs(dicev / 2 - v) / (0.5 * DICEU * rnum + DICEU);
#endif

			  // best one so far?
#ifdef THICK3
			  Float mid = fabs(diceu / 3 - u) / (0.5 * DICEU * rnum +
							      0.75 * DICEU);
#endif
			  // a sort of gaussian - doesn't look good
#ifdef THICK4
			  Float mid0 = fabs(diceu / 2 - u) / (10. * rnum +
							      15.);
			  Float mid1 = fabs(dicev / 2 - v) / (10. * rnum +
							      15.);
			  Float mid = pow(.2,-(mid0 + mid1));
#endif
			  // adding two numbers together - looks okay
#ifdef THICK5
			  Float mid0 = fabs(diceu / 2 - u) / (10. * rnum +
							      15.);
			  Float mid1 = fabs(dicev / 2 - v) / (10. * rnum +
							      20.);
			  Float mid = mid0 + mid1;
#endif			 
			  tfcn[v*diceu + u] = thickness[v*diceu + u];//1. - mid;
			  //printf("%f, ", 1. - mid);
			}

	#if 0
			Normal N = Normal(Cross(dPdu, dPdv).Hat());
			nrms[i][j][0] = N.x;
			nrms[i][j][1] = N.y;
			nrms[i][j][2] = N.z;
	#endif
		}
	}
	int nTris = 2*(diceu-1)*(dicev-1);
	int *vertices = new int[3 * nTris];
	int *vertp = vertices;
	for (int v = 0; v < dicev-1; ++v) {
		for (int u = 0; u < diceu-1; ++u) {
#define VN(u,v) ((v)*diceu+(u))
			*vertp++ = VN(u,   v);
			*vertp++ = VN(u+1, v);
			*vertp++ = VN(u+1, v+1);
	
			*vertp++ = VN(u,   v);
			*vertp++ = VN(u+1, v+1);
			*vertp++ = VN(u,   v+1);
#undef VN
		}
	}
	refined.push_back(new TriangleMesh(ObjectToWorld, nTris, diceu*dicev, vertices,
					   evalPs, tfcn, diceu, isBump, isNoise));
	delete[] ueval;
	delete[] veval;
	delete[] evalPs;
	delete[] evalNs;
}
