#include "scene.h"
#include "camera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
  int Scene::numIntersectTests = 0;
  int Scene::numShadings = 0;
Scene::~Scene() {
	delete camera;
	delete sampler;
	delete image;
	delete integrator;
}
Spectrum Scene::L(const Ray &ray) const {
	return integrator->L(this, ray, NULL);
}
bool Scene::IntersectP(const Ray &ray) const {
  numIntersectTests++;
	return prims->IntersectP(ray);
}
Scene::Scene(Camera *cam, Integrator *in, Image *img, Sampler *s,
		const vector<Primitive *> &pr,
		const vector<Light *> &lts) {
	lights = lts;
	prims = new GridAccelerator(pr);
	camera = cam;
	image = img;
	sampler = s;
	integrator = in;
   StatsRegisterCounter(STATS_BASIC, "Scene", "Intersection Tests",
			&numIntersectTests);
   StatsRegisterCounter(STATS_BASIC, "Scene", "Point Shadings",
			&numShadings);
}

Float Scene:: Visibility(Surf& surf, int numLevels) {
  Point P = surf.dgGeom.P;
  int n = 0;
  Ray fromLens;
  Float delta = 1./((float) numLevels);
  Float u, v;
  Surf compSurf;

  for (int uLevel = 0; uLevel < numLevels; uLevel++) {
    for (int vLevel = 0; vLevel < numLevels; vLevel++) {
      u = (uLevel + RandomFloat())*delta;
      v = (vLevel + RandomFloat())*delta;
      camera->LensToPoint(u, v, P, fromLens);
      if (Intersect(fromLens, &compSurf)) {
	if (surf.primitive == compSurf.primitive) {
	  n++;
	}
      }
    }
  }

  return ((Float) n + 1.)/((Float) numLevels*numLevels + 1.);
}

Float Scene::SceneLeakage(Ray& ray, int numLevels) {
  int n = 0;
  Ray fromLens;
  Float delta = 1./((float) numLevels);
  Float u, v;
  Surf compSurf;

  for (int uLevel = 0; uLevel < numLevels; uLevel++) {
    for (int vLevel = 0; vLevel < numLevels; vLevel++) {
      u = (uLevel + RandomFloat())*delta;
      v = (vLevel + RandomFloat())*delta;
      camera->FromLensParallelTo(u, v, ray.D, fromLens);
      if (Intersect(fromLens, &compSurf)) {
	Float screenz = camera->WorldToScreen(compSurf.dgGeom.P).z;
	if (screenz <= 1.) {
	  continue;
	}
      }
      n++;
    }
  }

  return ((Float) n + 1.)/((Float) numLevels*numLevels + 1.);
}

void Scene::TraceThrough(Point& imagePoint,
			 Ray& ray, Surf& hit, Float wt) {
  Float alpha;
  Spectrum L;
  if (imagePoint.z > 1.) {
    L = 0.;
    alpha = 0.;
    assert(false);
  } else {
    numShadings++;
    L = integrator->Shade(this, ray, &alpha, hit);
    Float rgb[3];
    L.ConvertToRGB(rgb);
    if ((rgb[0] + rgb[1] + rgb[2]) < 0.00001) {
      return;
    }
  }

  image->AddSample(imagePoint, L, alpha, wt);
}

bool Scene::RandomShadingOfPoint(Point& P, Surf& hit,
				 Spectrum& L, Float& alpha) {
  Ray fromLens;

  // any point on the aperture will gives us a useful direction, whether
  // it has visibility or not, but we *must* use Shade rather than L here.
  camera->LensToPoint(RandomFloat(), RandomFloat(), P, fromLens);
  L = integrator->Shade(this, fromLens, &alpha, hit);

  Float rgb[3];
  L.ConvertToRGB(rgb);
  if ((rgb[0] + rgb[1] + rgb[2]) < 0.00001) {
    return false;
  }

  return true;
}

void Scene::HandleFoundPoint(DofMethod method,
			     Float xSource, Float ySource, Float screenz,
			     Surf hit, Float wt) {
  Ray fromLens;
  Surf compSurf;
  Point P = hit.dgGeom.P;
  Float u, v, xRaster, yRaster;
  int numLevels = (int) ceil(sqrt(camera->AreaOfConfusion(P.z)) + 0.00001);
  assert(numLevels > 0);
  Float delta = 1./((float) numLevels);
  Float alpha;
  Spectrum L;
  
  if (RandomFloat() > 1./((float) numLevels*numLevels)) return;
  
  if (method == spotted) {
    numShadings++;
    if (!RandomShadingOfPoint(P, hit, L, alpha)) return;
  }
  
  wt /= Visibility(hit);

  for (int uLevel = 0; uLevel < numLevels; uLevel++) {
    for (int vLevel = 0; vLevel < numLevels; vLevel++) {
      u = (uLevel + RandomFloat())*delta;
      v = (vLevel + RandomFloat())*delta;
      camera->LensToPoint(u, v, P, fromLens);
      if (Intersect(fromLens, &compSurf)) {
	if (hit.primitive == compSurf.primitive) {
	  camera->TraceToFilm(fromLens, xRaster, yRaster);
	  Point imagePoint(xRaster, yRaster, screenz);
	  if (method == disks) {
	    TraceThrough(imagePoint, fromLens, compSurf, wt);
	  } else {
	    image->AddSample(imagePoint, L, alpha, wt);
	  }
	}
      }
    }
  }
}

void Scene::HandleMiss(Ray ray, Float wt) {
  Ray fromLens;
  Surf compSurf;
  Float u, v, xRaster, yRaster;
  int numLevels = (int) ceil(sqrt(camera->ConfusionAtInfinity()) + 0.00001);
  assert(numLevels > 0);
  Float delta = 1./((float) numLevels);
  Spectrum L;
  Float leakage = SceneLeakage(ray);
  Float product = leakage*numLevels;

  if (RandomFloat() > .8/(product*product)) return;

  wt *= leakage*1.25;

  for (int uLevel = 0; uLevel < numLevels; uLevel++) {
    for (int vLevel = 0; vLevel < numLevels; vLevel++) {
      u = (uLevel + RandomFloat())*delta;
      v = (vLevel + RandomFloat())*delta;
      camera->FromLensParallelTo(u, v, ray.D, fromLens);
      if (Intersect(fromLens, &compSurf)) {
	Float screenz = camera->WorldToScreen(compSurf.dgGeom.P).z;
	if (screenz <= 1.) {
	  continue;
	}
      }
      camera->TraceToFilm(fromLens, xRaster, yRaster);
      Point Praster(xRaster, yRaster, 1.);
      image->AddSample(Praster, L, 0., wt);
    }
  }
}

void Scene::Render() {
  camera->Calibrate(this, integrator);
	cerr << "Rendering: ";
	ApplyRenderAlgorithm(stratified);
	sampler->Reset(3, 3, 2);
	ApplyRenderAlgorithm(spotted);
	image->Write();
	cerr << endl;
}

void Scene::ApplyRenderAlgorithm(DofMethod method, Float wt) {
  Float sample[5];
  for (int lastCount = numIntersectTests;
       sampler->GetNextImageSample(sample);
       sampler->UseCapacity(numIntersectTests - lastCount),
	 lastCount = numIntersectTests) {
    Ray ray;
    if (!camera->GenerateRay(sample, ray))
      continue;
    static int eyeRaysTraced = 0;
    if (eyeRaysTraced == 0)
      StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
			   &eyeRaysTraced);
    ++eyeRaysTraced;
    if (eyeRaysTraced % 10000 == 0) cerr << '+';
    
    if (method == stratified) {
      Float alpha;
      Spectrum L = integrator->L(this, ray, &alpha);
      numShadings++;
      Float screenz = camera->WorldToScreen(ray(ray.maxt)).z;
      if (screenz > 1.) {
	L = 0.;
	alpha = 0.;
      }
      Point Praster(sample[0], sample[1], screenz);
      image->AddSample(Praster, L, alpha, wt);
    } else {
      
      Surf hit;
      if (Intersect(ray, &hit)) {
	Float screenz = camera->WorldToScreen(hit.dgGeom.P).z;
	if (screenz > 1.) {
	  HandleMiss(ray, wt);
	} else {
	  HandleFoundPoint(method, sample[0], sample[1], screenz, hit, wt);
	}
      } else {
	HandleMiss(ray, wt);
      }
    }
  }
}

bool Scene::Intersect(const Ray &ray, Surf *surf) const {
  numIntersectTests++;
	return prims->Intersect(ray, surf);
}
