#include "ri.h"
#include "mbdofcamera.h"
#include "film.h"
#include "scene.h"

MbDOFPerspectiveCamera::
MbDOFPerspectiveCamera (const Transform *world2cam, int nTransforms, Float hither, Float yon,
						Float shutterOpen, float shutterClose, Float fstop, Float focalLen, Float focalDist,
						RtToken mode, Float fov, Film *film)
	: Camera(world2cam[0],
	  Perspective(fov, 1. /*scene->image->PixelAspectRatio*/,
				  hither, yon) * Scale(-1, -1, 1), hither, yon, film)
{
	WorldToCamera0 = world2cam[0];
	if(nTransforms > 1) {
		WorldToCamera1 = world2cam[1];
	} else {
		WorldToCamera1 = world2cam[0];
	}

	ShutterStart = shutterOpen;
	ShutterEnd = shutterClose;
	FStop = fstop;
	FocalLength = focalLen;
	FocalDistance = focalDist;
	Mode = mode;

	// Camera adjustment and exposure settings are done below in AdjustAutomatically, which
	// is only called after the scene is created, since the automatic controls obviously depend
	// upon the scene.
}


bool MbDOFPerspectiveCamera::GenerateRay(Float sample[5], Ray &ray) const
{
	// sample[0],sample[1]: point in screen space we want to see
	// sample[2],sample[3]: uniformly distributed numbers in [0,1] that correspond to points in aperture
	// sample[4]: uniformly distributed number in [0,1] that corresponds to time of ray

	if (sample[0] < film->GetImage()->SampleCropLeft ||
	    sample[0] > film->GetImage()->SampleCropRight ||
	    sample[1] < film->GetImage()->SampleCropBottom ||
	    sample[1] > film->GetImage()->SampleCropTop)
		return false;

	if (FocalLength == 0. && FocalDistance == 0.) {  // if this is a pinhole camera
		GenerateRay (0., 0., RasterToCamera(Point(sample[0],sample[1],-1)), ShutterEnd*sample[4], ray);
	} else {  // we have a positive aperture
		// framework doesn't allow us to rejection sample for a circular aperture, so we construct circular distribution
		// directly.
		//Float lensx = 2*sample[2]-1, lensy = 2*sample[3]-1;
		Float theta = 2*M_PI*sample[2], r = sqrt(sample[3]);
		Float lensx = r*cos(theta), lensy = r*sin(theta);
		Float apertureRadius = .5*FocalLength/FStop;
		Point Pcamera = RasterToCamera(Point(sample[0],sample[1],-1));
		double scaleToFocus = FocalDistance/Pcamera.z;
		GenerateRay (apertureRadius*lensx, apertureRadius*lensy,
					 Point(scaleToFocus*Pcamera.x, scaleToFocus*Pcamera.y, scaleToFocus*Pcamera.z), ShutterEnd*sample[4], ray);
	}
	return true;
}


void MbDOFPerspectiveCamera::GenerateRay (Float aperture_x, Float aperture_y, Point pointOnRay, Float time, Ray &ray) const
{
	// construct ray in camera space
	ray.O = Point (aperture_x, aperture_y, 0);
	ray.D = pointOnRay - ray.O;
	if (ray.D.z <= 0)  // make sure rays shooting forward in case pointOnRay is bad
		Error ("Problem generating rays from MbDOFPerspectiveCamera: FocalDistance=%g, FocalLength=%g, Fstop=%g, ClipHither=%g\n",
			   FocalDistance, FocalLength, FStop, ClipHither);
	ray.D.Normalize();

	// now transform to world space at the given time
	ray.O = Lerp (time, WorldToCamera0.InverseTransform(ray.O), WorldToCamera1.InverseTransform(ray.O));
	ray.D = CameraToWorld (ray.D);  // ray direction independent of time since just translating
}


void MbDOFPerspectiveCamera::
EstimateDistanceAndExposure (const Scene *scene, Float timelimit, Float windowFrac[4], int probeRes[3], Float &exposure, Float *zsamples)
{
	Float maxColor = 0;
	Float minDepth = RI_INFINITY;

	Float deltaX = (windowFrac[1]-windowFrac[0])*(film->GetImage()->SampleCropRight-film->GetImage()->SampleCropLeft)/probeRes[0];
	Float deltaY = (windowFrac[3]-windowFrac[2])*(film->GetImage()->SampleCropTop-film->GetImage()->SampleCropBottom)/probeRes[1];
	Float Xstart = film->GetImage()->SampleCropLeft
				   + (film->GetImage()->SampleCropRight-film->GetImage()->SampleCropLeft)*windowFrac[0];
	Float Ystart = film->GetImage()->SampleCropBottom
				   + (film->GetImage()->SampleCropTop-film->GetImage()->SampleCropBottom)*windowFrac[2];

	int sample = 0;
	for (int i=0; i < probeRes[0]; ++i) {
		for (int j=0; j < probeRes[1]; ++j) {
			for (int time_i=0; time_i < probeRes[2]; ++time_i) {
				Float x = Xstart + (i+RandomFloat())*deltaX;
				Float y = Ystart + (j+RandomFloat())*deltaY;
				Float t = timelimit*(time_i+RandomFloat())/probeRes[3];
				Ray ray;
				GenerateRay (0, 0, RasterToCamera(Point (x, y, -1)), t, ray);
				Float thisMaxColor = scene->L(ray,true).MaxNorm();
				Float depth = WorldToCamera(ray.D).z * ray.maxt;  // relying on camera only translating, not rotating

				if (depth < minDepth)
					minDepth = depth;
				if (zsamples)
					zsamples[sample++] = depth;
				if (depth > ClipYon) continue;
				if (thisMaxColor > maxColor)
					maxColor = thisMaxColor;
			}
		}
	}

	FocalDistance = minDepth;
	exposure = .95/(maxColor*film->image->Gain); // so max RGB value in output (before scale/dither) should be roughly .95
}


void MbDOFPerspectiveCamera::AdjustAutomatically (const Scene *scene)
{
	// basic idea: send a few probe rays in, around the center of the image, at varying times.
	// For all modes we should use the minimum (maybe get rid of outliers first?) z to determine FocalDistance.
	// Then we figure out Fstop/ShutterEnd:
	// - for shutter priority, adjust Fstop so that maximum (maybe minus a few outliers) exposure is in good range.
	// - for aperture priority, adjust ShutterEnd the same way
	// - for programmed, figure out ShutterEnd/(Fstop*Fstop) the same way, then figure out balance between the two
	//    to minimize overall blur. That is, figure out how to get reduce variance the most by shrinking aperture
	//    (eliminating the wide aperture samples) or decreasing shutter (eliminating the later time samples).

	// note we send out a ray with the following code:
	//   Float alpha;
	//   Spectrum L = scene->integrator->L(scene, ray, &alpha);
	// Also note we shouldn't use the GenerateRay method until FStop, FocusDistance, etc. are properly set.
	// We can send out a few pinhole rays at time 0 to get an idea of z values and brightness, and then set
	// FocusDistance and exposure. To start FocusDistance with something reasonable, get the BBox of the scene,
	// find furthest point (max z in camera space)... then keep taking mins on FocusDistance.

	if (Mode == LRT_MANUAL) {
		// no adjustments to be made
		if (ShutterEnd > 1)
			Warning("(Manual mode) ShutterEnd=%g which is greater than 1, limit of camera motion",ShutterEnd);

	} else if (Mode == LRT_SHUTTER_PRIORITY) {
		if (ShutterEnd > 1)
			Warning("(Shutter priority mode) ShutterEnd=%g which is greater than 1, limit of camera motion",ShutterEnd);
		Float exposure;
		Float windowFrac[4] = {.4, .6, .4, .6};
		int probeRes[3] = {5, 5, 4};
		EstimateDistanceAndExposure (scene, ShutterEnd, windowFrac, probeRes, exposure, NULL);
		FStop = sqrt(ShutterEnd/exposure);

	} else if (Mode == LRT_APERTURE_PRIORITY) {
		Float exposure;
		Float windowFrac[4] = {.4, .6, .4, .6};
		int probeRes[3] = {10, 10, 1};
		EstimateDistanceAndExposure (scene, 0., windowFrac, probeRes, exposure, NULL);
		ShutterEnd = exposure*FStop*FStop;
		if (ShutterEnd > 1) {
			Info("(Aperture priority mode) clamping ShutterEnd from %g to 1, limit of camera motion", ShutterEnd);
			ShutterEnd = 1;
		}

	} else if (Mode == LRT_PROGRAMMED) {
		// first get the FocalDistance
		Float exposure;
		Float windowFrac[4] = {.4, .6, .4, .6};
		int probeRes[3] = {10, 10, 1};
		Float zsamples[200];
		EstimateDistanceAndExposure (scene, 0., windowFrac, probeRes, exposure, zsamples);
		Float saveFocalDistance = FocalDistance;

		// now do a broader sample to get more zvalues (and more accurate exposure)
		windowFrac[0] = .1; windowFrac[1] = .9; windowFrac[2] = .1, windowFrac[3] = .9;
		EstimateDistanceAndExposure (scene, 0., windowFrac, probeRes, exposure, zsamples+100);
		FocalDistance = saveFocalDistance; // restore focus

		// figure out aperture we want to minimize total blur at each sample
		Float B = 0, C = 0;
		Float sprime = 1./(1./FocalDistance+1./FocalLength);
		Float v = (WorldToCamera1(Point(0,0,0))-WorldToCamera0(Point(0,0,0))).Length(); // speed of camera move
		Float vef2 = v*exposure*FocalLength*FocalLength;
		for (int i = 0; i < 200; ++i) {
			Float z = zsamples[i];
			if (z > ClipYon) continue;
			Float zprime = 1./(1./z+1./FocalLength);
			Float c = fabs((sprime-zprime)/zprime);
			Float weight = (i < 100) ? 0.2 : 1.0;   // weight under-sampled outer part of frame more
			B += weight * (2 * vef2 * sprime/(z+sprime) * c);
			C += weight * (M_PI * c*c);
		}
		// B/a+C*a*a is now the total weighted blurred area of these samples; seek aperture a to minimize this
		Float a = pow(B/(2*C),1./3.);
		ShutterEnd = exposure*FocalLength*FocalLength/(a*a);
		if (ShutterEnd > 1) {
			Warning ("(Programmed mode) Clamping ShutterEnd from %g to 1, limit of camera motion", ShutterEnd);
			ShutterEnd = 1;
		}
		FStop = sqrt(ShutterEnd/exposure);

	} else
		Warning("MbDOFPerspectiveCamera - unknown camera mode, assuming manual");

	Info("FStop=%g, FocalLength=%g, FocalDistance=%g, ShutterEnd=%g",FStop,FocalLength,FocalDistance,ShutterEnd);

	// now that Fstop and Shutter have been set, compute exposure
	if (ShutterEnd > ShutterStart && FStop < RI_INFINITY) // only do physical exposure for physically meaningful cameras
		film->image->Gain *= ShutterEnd/(FStop*FStop);
}

