#include <tiffio.h>


struct RGBA {
	int r, g, b, a;
	RGBA() : r(0), g(0), b(0), a(0) {}
	RGBA(int R, int G, int B, int A) : r(R), g(G), b(B), a(A) {}
};

RGBA *TIFFRead(const char *name, int *xSize, int *ySize) {
	RGBA *pixels = NULL;
	float *fbuf = NULL;
	unsigned char *ubuf = NULL;

	TIFF *tiff = TIFFOpen(name, "r");
	if (!tiff) {
//		Error("Unable to open TIFF %s", name);
		return NULL;
	}
	short int nSamples;
	TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, xSize);
	TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, ySize);
	TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nSamples);
	short int bitsPerSample, sampleFormat;
	if (!TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bitsPerSample)) {
		//Error("TIFFRead: bits per sample not set in TIFF");
		delete[] pixels;
		delete[] ubuf;
		delete[] fbuf;
		TIFFClose(tiff);
		return NULL;
	}
	if (!TIFFGetField(tiff, TIFFTAG_SAMPLEFORMAT, &sampleFormat)) {
		if (bitsPerSample == 32)
			sampleFormat = SAMPLEFORMAT_IEEEFP;
		else
			sampleFormat = SAMPLEFORMAT_UINT;
	}
	
	if (bitsPerSample == 32 && sampleFormat != SAMPLEFORMAT_IEEEFP) {
		//Error("TIFFRead: 32 bit TIFF not stored in floating point format");
		delete[] pixels;
		delete[] ubuf;
		delete[] fbuf;
		TIFFClose(tiff);
		return NULL;
	}
	else {
		if (bitsPerSample != 8) {
			//Error("TIFFRead: more than 8 bits per sample unsupported");
			delete[] pixels;
			delete[] ubuf;
			delete[] fbuf;
			TIFFClose(tiff);
			return NULL;
		}
		if (sampleFormat != SAMPLEFORMAT_UINT) {
			//Error("TIFFRead: 8 bit TIFFs must be stored as unsigned ints");
			delete[] pixels;
			delete[] ubuf;
			delete[] fbuf;
			TIFFClose(tiff);
			return NULL;
		}
	}
	
	if (nSamples * *xSize != TIFFScanlineSize(tiff)) {
		//Error("TIFFRead: RGB not interleaved in TIFF %s", name);
		delete[] pixels;
		delete[] ubuf;
		delete[] fbuf;
		TIFFClose(tiff);
		return NULL;
	}
	u_short *mapR = 0, *mapG = 0, *mapB = 0;
	if (nSamples == 1) {
		short photoType;
		TIFFGetField(tiff, TIFFTAG_PHOTOMETRIC, &photoType);
		if (photoType != PHOTOMETRIC_PALETTE) {
			//Error("TIFFRead: colormap not found in one-sample image");
			delete[] pixels;
			delete[] ubuf;
			delete[] fbuf;
			TIFFClose(tiff);
			return NULL;
		}
		TIFFGetField(tiff, TIFFTAG_COLORMAP, &mapR, &mapG, &mapB);
	}
	pixels = new RGBA[*xSize * *ySize];
	RGBA *pixelp = pixels;
	
	if (bitsPerSample == 32) fbuf = new float[nSamples * *xSize];
	else                     ubuf = new u_char[nSamples * *xSize];
	for (int y = 0; y < *ySize; ++y) {
		if (fbuf) {
			float *fbufp = fbuf;
			if (TIFFReadScanline(tiff, fbuf, y, 1) == -1) {
				delete[] pixels;
				delete[] ubuf;
				delete[] fbuf;
				TIFFClose(tiff);
				return NULL;
			}
			for (int x = 0; x < *xSize; ++x) {
				*pixelp = RGBA(255*fbufp[0], 255*fbufp[1], 255*fbufp[2], 255*fbufp[3]);
				++pixelp;
				fbufp += nSamples;
			}
		}
		else {
			u_char *ubufp = ubuf;
			if (TIFFReadScanline(tiff, ubuf, y, 1) == -1) {
				delete[] pixels;
				delete[] ubuf;
				delete[] fbuf;
				TIFFClose(tiff);
				return NULL;
			}
			for (int x = 0; x < *xSize; ++x) {
				if (nSamples == 1) {
				}
				else {
					*pixelp = RGBA(ubufp[0], ubufp[1], ubufp[2], ubufp[3]);
				}
				++pixelp;
				ubufp += nSamples;
			}
		}
	}
	delete[] ubuf;
	delete[] fbuf;
	TIFFClose(tiff);
	return pixels;
}
void TIFFWrite8Bit(const char *name, RGBA *rgba, int XRes, int YRes) {
	TIFF *tiff = TIFFOpen(name, "w");
	if (!tiff) {
		//Error("Unable to open TIFF %s for writing", name);
		return;
	}
	
	int sampleCount = 4;
	TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, sampleCount);
	short int extra[] = { EXTRASAMPLE_ASSOCALPHA };
	TIFFSetField(tiff, TIFFTAG_EXTRASAMPLES, (short)1, extra);
	TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, XRes);
	TIFFSetField(tiff, TIFFTAG_IMAGELENGTH, YRes);
	TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
	TIFFSetField(tiff, TIFFTAG_ROWSPERSTRIP, 1L);
	TIFFSetField(tiff, TIFFTAG_XRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_YRESOLUTION, 1.0);
	TIFFSetField(tiff, TIFFTAG_RESOLUTIONUNIT, 1);
	TIFFSetField(tiff, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
	TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(tiff, TIFFTAG_ORIENTATION, (int)ORIENTATION_TOPLEFT);
	u_char *buf = new u_char[sampleCount * XRes];
	RGBA *pixelp = rgba;
	
	for (int y = 0; y < YRes; ++y) {
		u_char *bufp = buf;
		for (int x = 0; x < XRes; ++x) {
			if (pixelp) {
				*bufp = (u_char)(pixelp->r);
				++bufp;
				*bufp = (u_char)(pixelp->g);
				++bufp;
				*bufp = (u_char)(pixelp->b);
				++bufp;
				*bufp = (u_char)(pixelp->a);
				++bufp;
				++pixelp;
			}
		}
		TIFFWriteScanline(tiff, buf, y, 1);
	}
	delete[] buf;
	TIFFClose(tiff);
}

int comp (int base, float baseA, int ground, float groundA, int shad, float shadA, int obj, float objA)
{
	if (objA == 0 && shadA == 0 && groundA == 0) return base;
	float c;
	if (shadA == 0 || groundA == 0) c = obj + (1-objA)*base;
	else c = obj + (1-objA)*(.5*(shad/shadA-ground/groundA)+base);
	if (c <= 0) return 0;
	if (c >= 255) return 255;
	return (int)c;
}

int main ()
{
	int xsize, ysize;
	RGBA *base = TIFFRead ("base.tiff", &xsize, &ysize);
	RGBA *ground = TIFFRead ("ground.tiff", &xsize, &ysize);
	RGBA *shadows = TIFFRead ("shadows.tiff", &xsize, &ysize);
	RGBA *object = TIFFRead ("object.tiff", &xsize, &ysize);

	for (int i = 0; i < xsize*ysize; ++i) {
		base[i].r = comp (base[i].r, float(base[i].a)/255., ground[i].r, float(ground[i].a)/255.,
						  shadows[i].r, float(shadows[i].a)/255., object[i].r, float(object[i].a)/255.);
		base[i].g = comp (base[i].g, float(base[i].a)/255., ground[i].g, float(ground[i].a)/255.,
						  shadows[i].g, float(shadows[i].a)/255., object[i].g, float(object[i].a)/255.);
		base[i].b = comp (base[i].b, float(base[i].a)/255., ground[i].b, float(ground[i].a)/255.,
						  shadows[i].b, float(shadows[i].a)/255., object[i].b, float(object[i].a)/255.);
	}
	TIFFWrite8Bit ("comp.tiff",base,xsize,ysize);
}

