#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

using namespace std;

const float persp = 2.5;
const int illumx = 1500;
const int illumy = 750;
const int samples = 10;

struct Vector {
	float x, y, z;
	Vector() : x(0), y(0), z(0) {}
	Vector (float xx, float yy, float zz) : x(xx), y(yy), z(zz) {}
	Vector (float heading, float pitch)
	{
		x = cos(heading)*cos(pitch);
		y = sin(pitch);
		z = sin(heading)*cos(pitch);
	}
};

struct RGBA {
	float r, g, b, a;
	RGBA() : r(0), g(0), b(0), a(0) {}
	RGBA(float rr, float gg, float bb, float aa) : r(rr), g(gg), b(bb), a(aa) {}
	RGBA &operator+= (const RGBA &q) {
		r += q.r; g += q.g; b += q.b; a += q.a;
		return *this;
	}
};
RGBA operator* (float s, const RGBA & p)
{
	RGBA result;
	result.r = s*p.r;
	result.g = s*p.g;
	result.b = s*p.b;
	result.a = s*p.a;
	return result;
}

struct image {
	int w, h;
	RGBA *pixels;
	image() : w(0), h(0), pixels(0) {}
	image(int ww, int hh) : w(ww), h(hh), pixels(new RGBA[ww*hh]) {}
	~image() {delete[] pixels; pixels=0;}
	RGBA &operator () (int i, int j)
	{
		return pixels[i+w*j];
	}
	RGBA bilerp (float x, float y)
	{
		RGBA result;
		int i = (int)floorf(x), j = (int)floorf(y);
		if (i > x) --i;
		if (j > y) --j;
		float fx = x-i, fy = y-j;
		if (i < 0 || i >= w-1 || j < 0 || j >= h-1) return result; 
		result += (1-fx)*(1-fy)*pixels[i+w*j];
		result += fx*(1-fy)*pixels[i+1+w*j];
		result += (1-fx)*fy*pixels[i+w*(j+1)];
		result += fx*fy*pixels[i+1+w*(j+1)];
		return result;
	}
};


void readcams (Vector *campos, Vector *camangles, char **photonames, float *camweights)
{
	FILE *fp = fopen ("caminfo.txt", "r");
	int numcams;
	fscanf(fp, "%d", &numcams);
	for (int i = 0; i < numcams; ++i) {
		fscanf(fp, "%f %f %f\n", &campos[i].x, &campos[i].y, &campos[i].z);
		fscanf(fp, "%f %f %f\n", &camangles[i].x, &camangles[i].y, &camangles[i].z);
		camangles[i].x *= M_PI/180.;
		camangles[i].y *= M_PI/180.;
		camangles[i].z *= M_PI/180.;
		char name[100];
		fgets(name,100,fp);
		int len = strlen(name);
		name[len-4] = 'p';
		name[len-3] = 'p';
		name[len-2] = 'm';
		name[len-1] = 0;
		photonames[i] = strdup(name);
		int numpts;
		fscanf(fp, "%d", &numpts);
		int junk0;
		float junk1, junk2;
		for (int j = 0; j < numpts; ++j)
			fscanf(fp, "%d %f %f", &junk0, &junk1, &junk2);
	}
	fclose(fp);
	fp = fopen ("camweights.txt","r");
	for (int i = 0; i < numcams; ++i) {
		fscanf(fp,"%f",&camweights[i]);
	}
	fclose(fp);
}


void readimage (image *&img, const char *name)
{
	char magic0, magic1;
	int width, height, maxdepth;
	FILE *fp = fopen(name,"rb");
	fscanf(fp,"%c%c %d %d %d",&magic0,&magic1,&width,&height,&maxdepth);
	fgetc(fp);
	img = new image(width,height);
	unsigned char rgb[3];
	for (int j = 0; j < height; ++j)
		for (int i = 0; i < width; ++i) {
			fread (rgb, sizeof(unsigned char), 3, fp);
			if (rgb[0] == maxdepth && rgb[1] == 0 && rgb[2] == 0) {
				(*img)(i,j) = RGBA();
			} else {
				(*img)(i,j) = RGBA(rgb[0]/(float)maxdepth,rgb[1]/(float)maxdepth,rgb[2]/(float)maxdepth,1);
			}
		}
	fclose(fp);
}


unsigned char inline quantize (float x, float min, float max)
{
	if (x < 0) return 0;
	else if (x/max > 1) return 255;
	else return (unsigned char)(255.*x/max);
}


void writeimage (image *img, const char *name)
{
	FILE *fp = fopen(name,"w");
	fprintf(fp,"P6\n%d %d\n%d\n",img->w,img->h,255);
	unsigned char rgb[3];
	float maxrgb = 0, minrgb = 1e38;
	for (int j = 0; j < img->h; ++j)
		for (int i = 0; i < img->w; ++i) {
			RGBA rgb = (*img)(i,j);
			if (rgb.a > 0) {
				if (rgb.r/rgb.a > maxrgb) maxrgb = rgb.r/rgb.a;
				if (rgb.g/rgb.a > maxrgb) maxrgb = rgb.g/rgb.a;
				if (rgb.b/rgb.a > maxrgb) maxrgb = rgb.b/rgb.a;
				if (rgb.r/rgb.a < minrgb) minrgb = rgb.r/rgb.a;
				if (rgb.g/rgb.a < minrgb) minrgb = rgb.g/rgb.a;
				if (rgb.b/rgb.a < minrgb) minrgb = rgb.b/rgb.a;
			}
		}
	float bettermax = 0, bettermin = 0;
	int nmax = 0, nmin = 0;
	for (int j = 0; j < img->h; ++j)
		for (int i = 0; i < img->w; ++i) {
			RGBA rgb = (*img)(i,j);
			if (rgb.a <= 0) continue;
			float b = (rgb.r+rgb.b+rgb.b)/rgb.a;
			if (b > .1*maxrgb) {
				bettermax += b;
				++nmax;
			}
		}
	maxrgb = bettermax/nmax;
	printf ("min: %g   to max: %g\n", minrgb, maxrgb);

	for (int j = 0; j < img->h; ++j)
		for (int i = 0; i < img->w; ++i) {
			RGBA p = (*img)(i,j);
			if (p.a == 0) {
				rgb[0] = 255; rgb[1] = 0; rgb[2] = 0;
			} else {
				rgb[0] = quantize(p.r/p.a,minrgb,maxrgb);
				rgb[1] = quantize(p.g/p.a,minrgb,maxrgb);
				rgb[2] = quantize(p.b/p.a,minrgb,maxrgb);
			}
			fwrite(rgb,sizeof(unsigned char),3,fp);
		}
	fclose(fp);
}


void jitter (float &f, float dx)
{
	f += (dx*rand())/RAND_MAX - dx/2;
}


#define NUMHEADING 8
float lookup_heading[NUMHEADING+1] = {-M_PI, -.75*M_PI, -.5*M_PI, -.25*M_PI, 0, .25*M_PI, .5*M_PI, .75*M_PI, M_PI};
float lookup_depth[NUMHEADING+1] = {30, 40, 50, 20, 50, 15, 50, 40, 30};


bool intersect (Vector D, Vector &hit)
{
	float t, mint = 1e38;
	// first with ground
	if (D.y < 0) {
		t = -2/D.y;
		if (t < mint) mint = t;
	}
	// next with sky
	if (D.y > 0) {
		t = 100/D.y;
		if (t < mint) mint = t;
	}
	// and now with cylinder
	float xzmag = sqrt(D.x*D.x+D.z*D.z);
	if (xzmag > 0) {
		float heading = atan2 (D.z, D.x);
		int i;
		for (i = 1; i < NUMHEADING; ++i)
			if (lookup_heading[i-1] < heading && heading <= lookup_heading[i]) break;
		float f = (heading-lookup_heading[i-1])/(lookup_heading[i]-lookup_heading[i-1]);
		float depth = (1-f)*lookup_depth[i-1]+f*lookup_depth[i];
		t = depth/xzmag;
		if (t < mint) mint = t;
	}
	if (mint == 1e38) return false;
	hit.x = mint*D.x;
	hit.y = 2+mint*D.y;
	hit.z = mint*D.z;
	return true;
}


Vector rotateheading (Vector &p, float heading)
{
	Vector r;
	float ch = cos(heading), sh = sin(heading);
	r.x = ch*p.x + sh*p.z;
	r.y = p.y;
	r.z = -sh*p.x + ch*p.z;
	return r;
}


Vector rotatepitch (Vector &p, float pitch)
{
	Vector r;
	float cp = cos(pitch), sp = sin(pitch);
	r.x = cp*p.x - sp*p.y;
	r.y = sp*p.x + cp*p.y;
	r.z = p.z;
	return r;
}


Vector rotateroll (Vector &p, float roll)
{
	Vector r;
	float cr = cos(roll), sr = sin(roll);
	r.x = p.x;
	r.y = cr*p.y - sr*p.z;
	r.z = sr*p.y + cr*p.z;
	return r;
}


RGBA findinphoto (Vector p, Vector &campos, Vector &camangles, image *photo)
{
	// transform to camera's frame
	p.x -= campos.x; p.y -= campos.y; p.z -= campos.z;
	p = rotateheading (p, camangles.x);
	p = rotatepitch (p, camangles.y);
	p = rotateroll (p, camangles.z);
	// check if behind camera
	if (p.x <= 0) return RGBA(0,0,0,0);
	float projx = persp*p.z/p.x;
	float projy = persp*p.y/p.x;
	float screenx = .5*photo->w*(projx+1);
	float screeny = -photo->w*.5*projy+.5*photo->h;
	return photo->bilerp(screenx,screeny);
}

void projectphoto (image *photo, Vector &campos, Vector &camangles, image *illum)
{
	for (int j = 0; j < illum->h; ++j)
		for (int i = 0; i < illum->w; ++i)
			for (int r = 0; r < samples; ++r) {
				float heading = 2*M_PI*(i/(float)illum->w);
				float pitch = -M_PI*((j/(float)illum->h)-.5);
				jitter (heading, 2*M_PI/(float)illum->w);
				jitter (pitch, M_PI/(float)illum->h);
				Vector direction(heading,pitch), hit;
				if (intersect(direction,hit))
					(*illum)(i,j) += findinphoto (hit, campos, camangles, photo); 
			}
}


float inline sqr(float x) {return x*x;}

// returns CIE ckear-sky model for my time and place and coordinate system,
// in kilo-candelas per metre squares (luminance)
float sky (float heading, float pitch)
{
	float alpha = heading-.651, zeta = M_PI/2-pitch;
	if (zeta < 0) return 0;
	float gamma = acos(.8894164*cos(zeta)+.4570979*sin(zeta)*cos(alpha));
	float luminance = 12.8 * (.91+10*exp(-3*gamma)+.45*sqr(cos(gamma)))*(1-exp(-.32/cos(zeta)))+1;
	if (luminance < 0) return 0;
	else return luminance;
}


void setskymodel (image *illum)
{
	float maxluminance = 0;
	for (int j = 0; j <= illum->h/2; ++j)
		for (int i = 0; i < illum->w; ++i) {
			float heading = 2*M_PI*(i/(float)illum->w);
			float pitch = -M_PI*((j/(float)illum->h)-.5);
			float luminance = sky(heading,pitch);
			if (luminance > maxluminance) maxluminance = luminance;
		}
	for (int j = 0; j <= illum->h/2; ++j)
		for (int i = 0; i < illum->w; ++i) {
			float heading = 2*M_PI*(i/(float)illum->w);
			float pitch = -M_PI*((j/(float)illum->h)-.5);
			float ratio = sky(heading,pitch)/maxluminance;
			if (ratio > 0)
				(*illum)(i,j) = RGBA(.6*ratio,.8*ratio,ratio,1);
		}
}
 

void combineillum (image *total, image *local, float weight)
{
	float scale = 0;
	int numcomps = 0;
	for (int j = 0; j < illumy; ++j)
		for (int i = 0; i < illumx; ++i) {
			RGBA loc = (*local)(i,j), tot = (*total)(i,j);
			if (loc.a > 0 && tot.a > 0) {
				loc = (1./loc.a) * loc;
				tot = (1./tot.a) * tot;
				float maxloc = loc.r; if (loc.g>maxloc)maxloc=loc.g; if (loc.b>maxloc)maxloc=loc.b;
				float maxtot = tot.r; if (tot.g>maxtot)maxtot=tot.g; if (tot.b>maxtot)maxtot=tot.b;
				if (fabs(loc.r/maxloc-tot.r/maxtot)+fabs(loc.g/maxloc-tot.g/maxtot)+fabs(loc.b/maxloc-tot.b/maxtot) < .4) {
					if (maxloc/maxtot < 10 && maxloc/maxtot > 1) {
						scale += maxloc/maxtot;
						++numcomps;
					}
				}
			}
		}
	scale = numcomps/scale;
	printf ("scaling by %f (based on %d pixels)\n", scale, numcomps);
	for (int j = 0; j < illumy; ++j)
		for (int i = 0; i < illumx; ++i) {
			RGBA loc = (*local)(i,j);
			(*total)(i,j) += weight*RGBA(scale*loc.r, scale*loc.g, scale*loc.b, loc.a);
		}
}


int main()
{
	Vector campos[72], camangles[72];
	float camweights[72];
	char *photonames[72];

	// read in cameras
	readcams (campos, camangles, photonames, camweights);
	image *illum = new image(illumx,illumy), *localillum = new image(illumx,illumy);
	memset(illum->pixels,0,sizeof(RGBA)*illum->h*illum->w);
	setskymodel (illum);
		for (int i = 0; i < 72; ++i) {
			if (i == 53 || i == 54) continue;
			image *photo = 0;
			printf ("reading %s\n", photonames[i]);
			readimage (photo, photonames[i]);
			memset(localillum->pixels,0,sizeof(RGBA)*localillum->h*localillum->w);
			projectphoto (photo, campos[i], camangles[i], localillum);
			delete photo;
			combineillum (illum, localillum, camweights[i]);
			char illumname[20];
			sprintf(illumname, "illum%d.ppm", i);
			printf ("writing %s\n", illumname);
			writeimage (illum, illumname);
		}
	delete illum;
	delete localillum;
	return 0;
}


