
#include "lrt.h"
#include "primitives.h"
#include "accel.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "shading.h"
#include "transport.h"

#include "photonmap.h"
#include "photon.h"

Integrator::~Integrator()
{
}

Spectrum ColorIntegrator::Integrate(const Ray & ray,
									HitInfo * hitInfo, Float * hitDist,
									Float * alpha) const
{
	if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
		*alpha = 1.;
		const Spectrum *Csp = hitInfo->GetColor(RI_CS);
		if (Csp) {
			return *Csp;
		} else {
			return hitInfo->hitPrim->attributes->Color;
		}
	} else {
		*alpha = 0.;
		return Spectrum(0, 0, 0);
	}
}

Spectrum RayCastingIntegrator::Integrate(const Ray & ray,
					 HitInfo * hitInfo,
					 Float * hitDist, Float * alpha) const
{
  if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    Spectrum L(0.);
    *alpha = 1.;
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) {
      L +=
	hitInfo->hitPrim->attributes->LightShader->
	Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;
    BRDF *brdf =
      hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    
    Point Pw =
      hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
    const list < Light * >&lights =
      hitInfo->hitPrim->attributes->Lights;
    list < Light * >::const_iterator iter = lights.begin();
    while (iter != lights.end()) {
      
      const Light *lt = *iter;
      Point Plight;
      Spectrum dE = lt->dE(Pw, &Plight);
      if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
	Vector wi = (Plight - Pw).Hat();
	const Normal & Nw = shadeContext.Ns;
	L += dE * brdf->fr(wi) * Dot(wi, Nw);
      }
      
      ++iter;
    }
    
    delete brdf;
    
    return L;
  } else {
    *alpha = 0.;
    return Spectrum(0.);
  }
}

Spectrum WhittedIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
									  Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
		Spectrum L(0.);
		*alpha = 1.;

		ShadeContext shadeContext(hitInfo, -ray.D);

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

		if (++RayDepth < 5) {
			Ray newRay;
			newRay.O = Pw;
			for (int i = 0; i < brdf->SpecularComponents(); ++i) {

				Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
				HitInfo hitInfo2;
				Float alpha2, maxt = INFINITY;
				L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);

			}
		}
		--RayDepth;

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

int WhittedIntegrator::RayDepth = 0;

Spectrum MCIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
								 Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) 
	{
		ShadeContext shadeContext(hitInfo, -ray.D);
		Spectrum L(0.);
		*alpha = 1.;

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		if (hitInfo->hitPrim->attributes->Sampling ==
			PrimitiveAttributes::SampleSurface) 
    {
  	  Vector wi;
			Float pdf;
			Spectrum brdfFr;
			Float u[2] = {RandomFloat(),RandomFloat()};

			brdfFr = brdf->Sample(u,&wi,&pdf);

			// check for pdf = 0.
			if (pdf != 0.)
      { 
  		  Point Pw =
  			  hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
  	 
			  const Normal & Nw = shadeContext.Ns;
  			Float dotWiNw= Dot(wi,Nw);
  			if (dotWiNw >= 0.)
  			{ 
  		    HitInfo hitInfoTemp; 
  		    Spectrum Li = Integrate(Ray(Pw,wi), &hitInfoTemp,
  			    hitDist, alpha);
          // 4*cos(thetaH) taken into account in pdf
    	    L += Li * (brdfFr/pdf) * dotWiNw; 
    	    //L += Li * dotWiNw; 
	  		}
	  	}
    }
		else if (hitInfo->hitPrim->attributes->Sampling ==
					 PrimitiveAttributes::SampleLight) 
	  {
  		Point Pw =
  			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
  		const list < Light * >&lights =
  			hitInfo->hitPrim->attributes->Lights;
  		list < Light * >::const_iterator iter = lights.begin();
		  // for each light...
  		while (iter != lights.end()) 
			{
  			const Light *lt = *iter;
  			Point Plight;
				int NumSamples = lt->NumSamples();
				Spectrum lightRad(0.);
			  
				for (int i=0; i<NumSamples; i++)
				{
    			Spectrum dE = lt->dE(Pw, &Plight);
    			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) 
					{
    				Vector wi = (Plight - Pw).Hat();
    				const Normal & Nw = shadeContext.Ns;
    				lightRad += dE * brdf->fr(wi) * Dot(wi, Nw); 
    			}
				}
				L += lightRad/NumSamples;
  
  			++iter;
  		}
		}
		else if (hitInfo->hitPrim->attributes->Sampling ==
					 PrimitiveAttributes::SampleCombination) {
			// YOUR CODE HERE: integrate by sampling a little bit
			// from both and then combining the results intelligently
		}

    /* specular transmission is handled by the BRDF */
		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

#include "sl.h"


static void null_shader(RmanShaderParams & params)
{
	params.Ci = SLTriple(0);
	params.Oi = SLTriple(1);
}

RendermanShader::RendermanShader(const char *name)
{
	char path[4096];
	this->name = strdup(name);
	char *shader_dir = getenv("LRT_SHADER_DIR");
	if (!shader_dir)
		shader_dir = "/home/humper/work/litrt/src/shaders";

	sprintf(path, "%s/%s.so", shader_dir, name);
	dso = new DSO(path);

	already_warned = false;

#define RMAN_MUST_HAVE( var, sym ) \
	void *var = dso->GetSym( sym ); \
	if (!var) { \
		Warning( "No %s in %s", sym, path );\
		type = RMAN_BROKEN_SHADER; \
		shader = null_shader; \
		return; \
	}

	RMAN_MUST_HAVE(type_ptr, "shader_type");
	RMAN_MUST_HAVE(shader_ptr, "shader");

#undef RMAN_MUST_HAVE

	type = *((RmanShaderType *) type_ptr);
	shader = (shader_func) shader_ptr;
}

void RendermanShader::Run(RmanShaderParams & params)
{
	switch (type) {
	case RMAN_SURFACE_SHADER:
		params.Cs =
			SLTriple(params.shadeContext.
					 InitializeColor(RI_CS, Spectrum(1.0)));
		params.Os = SLTriple(Spectrum(1, 1, 1));
		params.N = SLTriple(params.shadeContext.Ns);
		params.P =
			SLTriple(params.hitInfo->hitPrim->attributes->
					 ObjectToWorld(params.hitInfo->Pobj));
		params.I = SLTriple(params.shadeContext.wo);
		params.Ci = SLTriple(0, 0, 0);
		params.Os = SLTriple(1, 1, 1);
		shader(params);
		break;
	case RMAN_BROKEN_SHADER:
		if (!already_warned) {
			already_warned = 1;
			Warning("Shader %s is broken.", name);
		}
		params.Ci = SLTriple(RandomFloat(), RandomFloat(), RandomFloat());
		params.Oi = SLTriple(1, 1, 1);
		break;
	default:
		if (!already_warned) {
			already_warned = 1;
			Warning("I don't know how to handle shader %s", name);
		}
		break;
	}
}

Spectrum RendermanIntegrator::Integrate(const Ray & ray,
					HitInfo * hitInfo, Float * hitDist,
					Float * alpha) const
{
  
  if (scene->Intersect(ray, 0., hitDist, hitInfo) &&
      hitInfo->hitPrim->attributes->Surface != NULL) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    RmanShaderParams params(hitInfo->hitPrim->attributes->Surface->
			    surfaceFunction, shadeContext,
			    *scene->sampler, hitInfo);
    
    *alpha = 1;
    
    RendermanShader *shader = NULL;
    Assert(shader != NULL);
    /* XXX MMP: shader creation and management like this should happen up at
       the RI level, I think, so that this guy can just grab a useful Shader *
       from the HitInfo.  I've commented this out for now so that I can clean
       up stuff in shading.nw for 348 assignment 3. */
#if 0
    RendermanShader *shader =
      (RendermanShader *) shaders.Search(hitInfo->attributes->
					 Surface->Name());
    if (!shader) {
      shader =
	new RendermanShader(hitInfo->attributes->Surface->Name());
      shaders.Add(hitInfo->attributes->Surface->Name(), shader);
    }
#endif
    shader->Run(params);
    
    return params.Ci.MakeSpectrum();
  } else {
    *alpha = 0;
		return Spectrum(0.);
  }
}

int PhotonIntegrator::RayDepth = 0;
Spectrum PhotonIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
									  Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
		Spectrum L(0.);
		*alpha = 1.;

		ShadeContext shadeContext(hitInfo, -ray.D);

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

    if (brdf->TransmissionComponents() == 0)
    {      
      Float fRadius              = 0.0f;  
      const int   iExpectedPhotonAccum = 100;
      bool        bTimeToGiveUp        = false;
      int         iAttempts            = 0;

      // get photons based on target.. increase range until target found
      vector< pair<Float,Photon *> >* photonPairs = NULL;

      do
      {
        fRadius += 0.5f;
        if(photonPairs != NULL) delete photonPairs;
        photonPairs = 
          scene->causticPhotonMap->
            getNearestPhotons(Pw,iExpectedPhotonAccum, fRadius);
        bTimeToGiveUp = (iAttempts > 5);
        iAttempts++;
      
        //cerr << "photonPair size: " << photonPairs->size() << " ";
      }
      while((photonPairs->size() < (unsigned int)iExpectedPhotonAccum) 
        && !bTimeToGiveUp);

   
      int size = photonPairs->size();
      assert(size >= 0);
    
      // figure out what to do when the size return is less than 
      if (size > 0)
      {
        int numbOfPhotons = iExpectedPhotonAccum;

        if (size < numbOfPhotons)
        { numbOfPhotons =  size; }
        
        Float fRadiusMax2 = 
            (*photonPairs)[numbOfPhotons-1].first;
        Float fRadiusMax = sqrt(fRadiusMax2);
        
        for (int i=0; i< numbOfPhotons; i++)
        {
   	      // the necessary elements
          Float   fDist2 = (*photonPairs)[i].first;
          Photon *photon = (*photonPairs)[i].second;
          //Float   fK     = photon->k;
          Float   fK     = 8.0;
      	  Vector  wi     = photon->ray.D;
        	Float   fDp    = sqrt(fDist2);
        	Float   filterWeight  = max(0.0f,(1-fDp)/(fK*fRadiusMax));
	
 	        //L += (brdf->photonfr(wi) * photon->power * filterWeight) /
 	        L += (brdf->fr(wi) * photon->power * filterWeight) /
	        (( 1 - 2.0f/(3*fK) ) * M_PI * fRadiusMax2);
        }
      }
      delete photonPairs;
    }
		else
		{

		if (++RayDepth < 5) {
			Ray newRay;
			newRay.O = Pw;
			for (int i = 0; i < brdf->SpecularComponents(); ++i) {

				Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
				HitInfo hitInfo2;
				Float alpha2, maxt = INFINITY;
				L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);

			}
		}
		--RayDepth;
		}

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}


#if 0
Spectrum PhotonIntegrator::Integrate(const Ray & ray,HitInfo * hitInfo, Float * hitDist,
  Float * alpha) const
{
  if (scene->Intersect(ray, 0., hitDist, hitInfo)) 
  {
    ShadeContext shadeContext(hitInfo, -ray.D);
    Spectrum L(0.);
    *alpha = 1.;
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) 
    {
      L +=hitInfo->hitPrim->attributes->LightShader->Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface) return L;
    BRDF *brdf = hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    Point Pw   = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);

		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) 
    {
      const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) 
      {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}
			++iter;
		}
    
    if (false)
    {      
      Float fRadius              = 1.0f;  
      const int   iExpectedPhotonAccum = 20;
      bool        bTimeToGiveUp        = false;
      int         iAttempts            = 0;


      // get photons based on target.. increase range until target found
      vector< pair<Float,Photon *> >* photonPairs = NULL;
      photonPairs = scene->causticPhotonMap->
        getNearestPhotons(Pw,iExpectedPhotonAccum, fRadius);

      assert(photonPairs->size() >= 0);
    
      // figure out what to do when the size return is less than 
      if (photonPairs->size() > 0)
      {
        int size = photonPairs->size();
        for (int i=0; i< size; i++)
        {
   	      // the necessary elements
          Float   fDist2 = (*photonPairs)[i].first;
          Photon *photon = (*photonPairs)[i].second;
          Float   fK     = photon->k;
      	  Vector  wi     = photon->ray.D;
        	Float   fDp    = sqrt(fDist2);
        	Float   filterWeight  = max(0.0f,(1-fDp)/(fK*fRadius));
	
          // i am not tagging photons because they can have
          // multiple influence on other portions of the image
          //if (brdf->photonfr(wi) > 0.0) { cerr << ">"; }
 	        L += (brdf->photonfr(wi) * photon->power * filterWeight) /
	        (( 1 - 2.0f/(3*fK) ) * M_PI * fDist2);
        }
      }
      delete photonPairs;
    }
    else
    {
      // if no photons, check for transmission components
      if (++RayDepth < 5) 
      {
	      Ray newRay;
      	newRay.O = Pw;
      	for (int i = 0; i < brdf->SpecularComponents(); ++i) 
      	{
      	  Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
      	  HitInfo hitInfo2;
      	  Float alpha2, maxt = INFINITY;
      	  L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);
      	}
      }
      --RayDepth;
    }
    delete brdf;
    return L;
  } 
  else 
  {
    *alpha = 0.;
    return Spectrum(0.);
  }
}
#endif


#if 0
Spectrum PhotonIntegrator::Integrate(const Ray & ray,HitInfo * hitInfo, Float * hitDist,
				     Float * alpha) const
{
  if (scene->Intersect(ray, 0., hitDist, hitInfo)) 
  {
    ShadeContext shadeContext(hitInfo, -ray.D);
    Spectrum L(0.);
    *alpha = 1.;
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) 
    {
      L +=hitInfo->hitPrim->attributes->LightShader->Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;
    BRDF *brdf = hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    Point Pw   = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);

    // compute direct light contribution first! (nz)
    //   the direct light contribution is from the one (and only)
    //   directional light source

    // also, you would usually sample the light sourc & or 
    // the brdf ... ie. use mult-importance sampling ...
    // however, for our scene, we won't need this, i don't think
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) 
      {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}
    
    //if (brdf->TransmissionComponents() == 0)
    if (false)
    {      
    // this is the radius of our accumulation sphere
    //Float fRadius              = 0.0f; // this is not going to happen 
    //Float fRadius              = 3.0f;  
    Float fRadius              = 1.0f;  
    //const int   iExpectedPhotonAccum = int(30 + (30 * RandomFloat())); // target PhotonNum
    //const int   iExpectedPhotonAccum = 100;
    const int   iExpectedPhotonAccum = 20;
    bool        bTimeToGiveUp        = false;
    int         iAttempts            = 0;


    // get photons based on target.. increase range until target found
    vector< pair<Float,Photon *> >* photonPairs = NULL;
    photonPairs = scene->causticPhotonMap->getNearestPhotons(Pw,iExpectedPhotonAccum, fRadius);
#if 0
    do
    {
      fRadius += 2.0f;
      if(photonPairs != NULL) delete photonPairs;
      photonPairs = scene->causticPhotonMap->getNearestPhotons(Pw,iExpectedPhotonAccum, fRadius);
      bTimeToGiveUp = (iAttempts > 5);
      //bTimeToGiveUp = (iAttempts >= 0);
      iAttempts++;
      
      //cerr << "photonPair size: " << photonPairs->size() << " ";
    }
    while((photonPairs->size() ==0) && !bTimeToGiveUp);
    //while((photonPairs->size() < (unsigned int)iExpectedPhotonAccum) && !bTimeToGiveUp);
#endif    

    assert(photonPairs->size() >= 0);
    
    // figure out what to do when the size return is less than 
    if (photonPairs->size() > 0)
    {
      int size = photonPairs->size();
      for (int i=0; i< size; i++)
      {

	// the necessary elements
	Float   fDist2 = (*photonPairs)[i].first;
	Photon *photon = (*photonPairs)[i].second;
	Float   fK     = photon->k;
	Vector  wi     = photon->ray.D;
	Float   fDp    = sqrt(fDist2);
	Float   filterWeight  = max(0.0f,(1-fDp)/(fK*fRadius));
	
	// i am not tagging photons because they can have
	// multiple influence on other portions of the image
  //if (brdf->photonfr(wi) > 0.0) { cerr << ">"; }
	L += (brdf->photonfr(wi) * photon->power * filterWeight) /
	     (( 1 - 2.0f/(3*fK) ) * M_PI * fDist2);
      }
    }
    delete photonPairs;
    }
    else
    {
      // if no photons, check for transmission components
      if (++RayDepth < 5) 
      {
	Ray newRay;
	newRay.O = Pw;
	for (int i = 0; i < brdf->SpecularComponents(); ++i) 
	{
	  Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
	  HitInfo hitInfo2;
	  Float alpha2, maxt = INFINITY;
	  L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);
	}
      }
      --RayDepth;
    }
    delete brdf;
    return L;
  } 
  else 
  {
    *alpha = 0.;
    return Spectrum(0.);
  }
}


#endif




Spectrum PhotonDebugIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
									  Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) 
  {
		Spectrum L(0.);
		*alpha = 1.;
		
		ShadeContext shadeContext(hitInfo, -ray.D);

		if (!hitInfo->hitPrim->attributes->Surface)
			return L; // returns 0.
		
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);

    // check for photons
    vector< pair<Float,Photon *> >* photonPairs = 
      scene->causticPhotonMap->getNearestPhotons(Pw, 1 , 1.0);

    // debug here
		//cerr << "Point:" << Pw << endl;
		//printFoundPhoton(*photonPairs); 
		//getchar();

    assert(photonPairs->size() >= 0);

		//if (photonPairs->size() > 1)
		//{ cerr << "photons found: " << photonPairs->size()<<endl; }

    if (photonPairs->size() > 0)
    {
		  int size = photonPairs->size();
      for (int i=0; i< size; i++)
			{
        Photon *photon = (*photonPairs)[i].second;
        if (photon->tagged != true)
        { 
          photon->tagged = true;
			    taggedPhotons++;
          L += Spectrum(1.0f);  // is this the right color value
					break;
        }
      }
    }
		else
		{
      // if no photons, check for transmission components
      if (++RayDepth < 5) 
      {
  			Ray newRay;
  			newRay.O = Pw;
  			for (int i = 0; i < brdf->TransmissionComponents(); ++i) 
        {
  				Spectrum kernel = brdf->SampleTransmission(i, &newRay.D);
  				HitInfo hitInfo2;
  				Float alpha2, maxt = INFINITY;
  				L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);
  
  			}
  		}
		  --RayDepth;
    }
	  
		delete photonPairs;
		
		delete brdf;

		return L;
	} 
  else 
  {
		*alpha = 0.;
		return Spectrum(0.);
	}
}
		
void PhotonDebugIntegrator::printFoundPhoton( 
  vector< pair<Float,Photon *> > &photonPairs) const
{
   int size = photonPairs.size();
	 for (int i=0; i< size; i++)
	 {
	   Float dist2 = photonPairs[i].first;
	   Photon* photon = photonPairs[i].second;
		 cerr << i << ":" ;
		 cerr << " distance2:" << dist2; 
	   cerr << " " << photon->ray.O << endl; 
   }
}

int PhotonDebugIntegrator::RayDepth = 0;

int PhotonDebugIntegrator::taggedPhotons = 0;
