
#include "lrt.h"
#include "primitives.h"
#include "accel.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "shading.h"
#include "transport.h"

Integrator::~Integrator()
{
}

Spectrum ColorIntegrator::Integrate(const Ray & ray,
				    HitInfo * hitInfo, Float * hitDist,
				    Float * alpha) const
{
  if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
    *alpha = 1.;
    const Spectrum *Csp = hitInfo->GetColor(RI_CS);
    if (Csp) {
      return *Csp;
    } else {
      return hitInfo->hitPrim->attributes->Color;
    }
  } else {
    *alpha = 0.;
    return Spectrum(0, 0, 0);
  }
}

Spectrum RayCastingIntegrator::Integrate(const Ray & ray,
					 HitInfo * hitInfo,
					 Float * hitDist, Float * alpha) const
{
  if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    Spectrum L(0.);
    *alpha = 1.;
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) {
      L +=
	hitInfo->hitPrim->attributes->LightShader->
	Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;
    BRDF *brdf =
      hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    
    Point Pw =
      hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
    const list < Light * >&lights =
      hitInfo->hitPrim->attributes->Lights;
    list < Light * >::const_iterator iter = lights.begin();
    while (iter != lights.end()) {
      
      const Light *lt = *iter;
      Point Plight;
      Spectrum dE = lt->dE(Pw, &Plight);
      if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
	Vector wi = (Plight - Pw).Hat();
	const Normal & Nw = shadeContext.Ns;
	L += dE * brdf->fr(wi) * Dot(wi, Nw);
      }
      
      ++iter;
    }
    
    delete brdf;
    
    return L;
  } else {
    *alpha = 0.;
    return Spectrum(0.);
  }
}

Spectrum WhittedIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
				      Float * hitDist, Float * alpha) const
{
  if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
    Spectrum L(0.);
    *alpha = 1.;
    
    ShadeContext shadeContext(hitInfo, -ray.D);
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) {
      L +=
	hitInfo->hitPrim->attributes->LightShader->
	Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;
    BRDF *brdf =
      hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    
    Point Pw =
      hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
    const list < Light * >&lights =
      hitInfo->hitPrim->attributes->Lights;
    list < Light * >::const_iterator iter = lights.begin();
    while (iter != lights.end()) {
      
      const Light *lt = *iter;
      Point Plight;
      Spectrum dE = lt->dE(Pw, &Plight);
      if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
	Vector wi = (Plight - Pw).Hat();
	const Normal & Nw = shadeContext.Ns;
	L += dE * brdf->fr(wi) * Dot(wi, Nw);
      }
      
      ++iter;
    }
    
    if (++RayDepth < 5) {
      Ray newRay;
      newRay.O = Pw;
      for (int i = 0; i < brdf->SpecularComponents(); ++i) {
	
	Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
	HitInfo hitInfo2;
	Float alpha2, maxt = INFINITY;
	L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);
	
      }
    }
    --RayDepth;
    
    delete brdf;
    
    return L;
  } else {
    *alpha = 0.;
    return Spectrum(0.);
  }
}

int WhittedIntegrator::RayDepth = 0;
int MCIntegrator::RayDepth = 0;

Spectrum MCIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
				 Float * hitDist, Float * alpha) const
{
  if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    Spectrum L(0.);
    *alpha = 1.;
    
	 int hitLightSource = 0;
    if (hitInfo->hitPrim->attributes->LightShader != NULL) {
      L +=
	hitInfo->hitPrim->attributes->LightShader->
	Le(shadeContext);
		// ray intersects with a light source
		hitLightSource = 1;
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;

    BRDF *brdf = hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    
    if (hitInfo->hitPrim->attributes->Sampling ==
	PrimitiveAttributes::SampleSurface) {
      // YOUR CODE HERE: integrate by sampling the BRDF

      if (++RayDepth < 5) {
	Float u[2] = { RandomFloat(), RandomFloat() };
	Float pdf;
	Vector wi;
	const Normal &Nw = shadeContext.Ns;
	Spectrum val = brdf->Sample( u, &wi, &pdf );
	Spectrum Li;
	if ( pdf != 0.0 ) {
	  Ray newRay;
	  newRay.O = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
	  newRay.D = wi;
	  HitInfo hitInfo2;
	  Float alpha2, maxt = INFINITY;
	  // Compute incident radiance
	  Li = Integrate( newRay, &hitInfo2, &maxt, &alpha2 );

	  L += Li * val * 4 *  Dot(wi, Nw) / pdf;
	}
      }
      --RayDepth;

    }
    else if (hitInfo->hitPrim->attributes->Sampling ==
	     PrimitiveAttributes::SampleLight) {
      // YOUR CODE HERE: integrate by sampling the Light source
      // for each light...
      //    for i = 0 to light->NumSamples()
      //        compute (estimate of rendering equation) / NumSamples

      Point Pw = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
      const list<Light *>&lights = hitInfo->hitPrim->attributes->Lights;
      list<Light *>::const_iterator iter = lights.begin();
      while (iter != lights.end()) {
	const Light *lt = *iter;
	Spectrum Ll = 0.0;
	for ( int index = 0; index < lt->NumSamples(); index++ ) {
	  Point Plight;
	  Spectrum dE = lt->dE(Pw, &Plight);
	  if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
	    Vector wi = (Plight - Pw).Hat();
	    const Normal & Nw = shadeContext.Ns;
	    Ll += brdf->fr(wi) * Dot(wi, Nw) * dE;
	  }
	}
	Ll /= lt->NumSamples();
	L += Ll;
	
	++iter;
      }

    }
    else if (hitInfo->hitPrim->attributes->Sampling ==
	     PrimitiveAttributes::SampleCombination) {
      // YOUR CODE HERE: integrate by sampling a little bit
      // from both and then combining the results intelligently
    }
    
    delete brdf;
    
    // Ray Marching for atmospheric effects
    if ( strcmp( scene->atmosAttributes->type, LRT_DUST ) == 0 ) {
      float index;
      float RAY_MARCH_STEP = *hitDist / 40.0;
      
      // if we hit a light source, then we don't need to scatter
      // the ray
      if (hitLightSource) {
	L = ComputeRayLuminance(ray, L, RAY_MARCH_STEP);
	return L;
      } 
      for ( index = *hitDist; index > 0; index -= RAY_MARCH_STEP ) {
	Point Pw = ray(index);
	
	const list<Light *>&lights = hitInfo->hitPrim->attributes->Lights;
	list<Light *>::const_iterator iter = lights.begin();
	while (iter != lights.end()) {
	  const Light *lt = *iter;
	  Point Plight;
	  Spectrum dE = lt->dE(Pw, &Plight);
	  Ray lightRay;
	  lightRay.O = Pw;
	  lightRay.D = Plight - Pw;

	  float factor = RAY_MARCH_STEP / 2.0;
	  float op = opacity(Pw);

	  if (op > 0.0) {
		 Spectrum baseColor = Spectrum(0.5,0.5,0.5);
		 if ( scene->Unoccluded(Pw, Plight) ) {
			L = (L + (0.5) * factor * op *
				  phase(ray,lightRay)*ComputeRayLuminance(lightRay, dE, 1.0)) * (1 - factor*op) + factor*op*baseColor*5.0;
		 }
	  }

	  ++iter;
	}
      }
    }
    
    return L;
  } else {
    *alpha = 0.;
    return Spectrum(0.);
  }
}

float MCIntegrator::opacity(Point p) const {
  // return scene->atmosAttributes->opacity;
  return (scene->atmosDensity->grid(p));
}


float MCIntegrator::phase(const Ray &eyeRay, const Ray &lightRay) const {
  Vector eyeDir = Vector(eyeRay.D.Hat());
  Vector lightDir = Vector(lightRay.D.Hat());
  float costheta = Dot(eyeDir,lightDir);

  /*
  // Rayleigh phase function
  float phase = (0.75)*(1 + costheta*costheta)*0.05;
  */

  // Henyey-Greenstein function
  float g = 0.9;
  float g2 = g*g;
  float phase = pow(((1.0-g2))/(1.0 - 2*g*costheta + g2),1.5);
  // float phase = ((1.0 - g2)*0.5*g)/(1.0 - 2*g*costheta + g2);

  // fprintf(stderr,"costheta: %f, phase: %f\n",costheta,phase);
  return phase;
} 

#include "sl.h"

Spectrum MCIntegrator::ComputeRayLuminance( const Ray &lightRay, Spectrum dE, float step ) const {
  Spectrum L = dE;
  float index;
  step = 1.0;
  float factor = step / 2.0;
  
  for ( index = lightRay.D.Length(); index > 0; index -= step ) {
    Point pw = lightRay(index);
    float op = opacity(pw);
	 if (op > 0.0) {
		L = L * (1 - factor * op);
	 }
  }
  
  return L;
}

static void null_shader(RmanShaderParams & params)
{
  params.Ci = SLTriple(0);
  params.Oi = SLTriple(1);
}

RendermanShader::RendermanShader(const char *name)
{
  char path[4096];
  this->name = strdup(name);
  char *shader_dir = getenv("LRT_SHADER_DIR");
  if (!shader_dir)
    shader_dir = "/home/humper/work/litrt/src/shaders";
  
  sprintf(path, "%s/%s.so", shader_dir, name);
  dso = new DSO(path);
  
  already_warned = false;
  
#define RMAN_MUST_HAVE( var, sym ) \
	void *var = dso->GetSym( sym ); \
	if (!var) { \
		Warning( "No %s in %s", sym, path );\
		type = RMAN_BROKEN_SHADER; \
		shader = null_shader; \
		return; \
	}
  
  RMAN_MUST_HAVE(type_ptr, "shader_type");
  RMAN_MUST_HAVE(shader_ptr, "shader");
  
#undef RMAN_MUST_HAVE
  
  type = *((RmanShaderType *) type_ptr);
  shader = (shader_func) shader_ptr;
}

void RendermanShader::Run(RmanShaderParams & params)
{
  switch (type) {
  case RMAN_SURFACE_SHADER:
    params.Cs =
      SLTriple(params.shadeContext.
	       InitializeColor(RI_CS, Spectrum(1.0)));
    params.Os = SLTriple(Spectrum(1, 1, 1));
    params.N = SLTriple(params.shadeContext.Ns);
    params.P =
      SLTriple(params.hitInfo->hitPrim->attributes->
	       ObjectToWorld(params.hitInfo->Pobj));
    params.I = SLTriple(params.shadeContext.wo);
    params.Ci = SLTriple(0, 0, 0);
    params.Os = SLTriple(1, 1, 1);
    shader(params);
    break;
  case RMAN_BROKEN_SHADER:
    if (!already_warned) {
      already_warned = 1;
      Warning("Shader %s is broken.", name);
    }
    params.Ci = SLTriple(RandomFloat(), RandomFloat(), RandomFloat());
    params.Oi = SLTriple(1, 1, 1);
    break;
  default:
    if (!already_warned) {
      already_warned = 1;
      Warning("I don't know how to handle shader %s", name);
    }
    break;
  }
}

Spectrum RendermanIntegrator::Integrate(const Ray & ray,
					HitInfo * hitInfo, Float * hitDist,
					Float * alpha) const
{
  
  if (scene->Intersect(ray, 0., hitDist, hitInfo) &&
      hitInfo->hitPrim->attributes->Surface != NULL) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    RmanShaderParams params(hitInfo->hitPrim->attributes->Surface->
			    surfaceFunction, shadeContext,
			    *scene->sampler, hitInfo);
    
    *alpha = 1;
    
    RendermanShader *shader = NULL;
    Assert(shader != NULL);
    /* XXX MMP: shader creation and management like this should happen up at
       the RI level, I think, so that this guy can just grab a useful Shader *
       from the HitInfo.  I've commented this out for now so that I can clean
       up stuff in shading.nw for 348 assignment 3. */
#if 0
    RendermanShader *shader =
      (RendermanShader *) shaders.Search(hitInfo->attributes->
					 Surface->Name());
    if (!shader) {
      shader =
	new RendermanShader(hitInfo->attributes->Surface->Name());
      shaders.Add(hitInfo->attributes->Surface->Name(), shader);
    }
#endif
    shader->Run(params);
    
    return params.Ci.MakeSpectrum();
  } else {
    *alpha = 0;
    return Spectrum(0.);
  }
}
