
#include "image.h"
#include "options.h"
#include <unistd.h>
#include <errno.h>

Image::Image(const Options & opt)
:options(opt)
{

	XBase = options.RasterCropLeft;
	XDelta = options.RasterCropRight - XBase;
	YBase = options.RasterCropBottom;
	YDelta = options.RasterCropTop - YBase;

	if (opt.DisplayMode == RI_RGB || opt.DisplayMode == RI_RGBA ||
		opt.DisplayMode == RI_RGBZ || opt.DisplayMode == RI_RGBAZ) {
		Pixels = new Spectrum[XDelta * YDelta];
	} else
		Pixels = NULL;

	if (opt.DisplayMode == RI_RGBA || opt.DisplayMode == RI_RGBAZ ||
		opt.DisplayMode == RI_A || opt.DisplayMode == RI_AZ) {
		Alphas = new Float[XDelta * YDelta];
		for (int i = 0; i < XDelta * YDelta; ++i)
			Alphas[i] = 0.;
	} else
		Alphas = NULL;

	if (opt.DisplayMode == RI_RGBAZ || opt.DisplayMode == RI_RGBZ ||
		opt.DisplayMode == RI_AZ || opt.DisplayMode == RI_Z) {
		Depths = new Float[XDelta * YDelta];
		for (int i = 0; i < XDelta * YDelta; ++i)
			Depths[i] = 0.;
	} else
		Depths = NULL;

	WeightSums = new Float[XDelta * YDelta];
	for (int i = 0; i < XDelta * YDelta; ++i)
		WeightSums[i] = 0.;
}

Image::~Image()
{
	delete[]Pixels;
	delete[]Alphas;
	delete[]Depths;
	delete[]WeightSums;
}

void Image::AddSampleBasic(const Point & Praster,
						   const Spectrum & radiance, Float alpha)
{

	if (Praster.x < XBase || Praster.x >= XBase + XDelta ||
		Praster.y < YBase || Praster.y >= YBase + YDelta)
		return;

	int x = int (Praster.x), y = int (Praster.y);
	int offset = (y - YBase) * XDelta + (x - XBase);
	if (Pixels)
		Pixels[offset] += radiance;
	if (Alphas)
		Alphas[offset] += alpha;
	if (Depths)
		Depths[offset] += Praster.z;
	WeightSums[offset] += 1;

}

void Image::Write() const
{

	Float *RGBOut = NULL, *AlphaOut = NULL, *DepthOut = NULL;
	if (Pixels) {
		RGBOut = new Float[3 * XDelta * YDelta];
		for (int offset = 0; offset < XDelta * YDelta; ++offset)
			Pixels[offset].ConvertToRGB(&RGBOut[3 * offset]);
	}
	if (Alphas) {
		AlphaOut = new Float[XDelta * YDelta];
		memcpy(AlphaOut, Alphas, XDelta * YDelta * sizeof(Float));
	}
	if (Depths) {
		DepthOut = new Float[XDelta * YDelta];
		memcpy(DepthOut, Depths, XDelta * YDelta * sizeof(Float));
	}

	for (int offset = 0; offset < XDelta * YDelta; ++offset) {
		if (WeightSums[offset] == 0.)
			continue;

		Float invWt = 1. / WeightSums[offset];
		if (RGBOut) {
			RGBOut[3 * offset] *= invWt;
			RGBOut[3 * offset + 1] *= invWt;
			RGBOut[3 * offset + 2] *= invWt;
		}
		if (AlphaOut)
			AlphaOut[offset] *= invWt;
		if (DepthOut)
			DepthOut[offset] *= invWt;
	}

	if (RGBOut && AlphaOut) {
		for (int i = 0; i < XDelta * YDelta; ++i) {
			for (int j = 0; j < 3; ++j)
				RGBOut[3 * i + j] *= AlphaOut[i];
		}
	}

	if ((RGBOut || AlphaOut) &&
		(options.Gain != 1. || options.Gamma != 1.)) {
		Float gain = options.Gain;
		Float invGamma = 1. / options.Gamma;
		if (RGBOut)
			for (int offset = 0; offset < 3 * XDelta * YDelta; ++offset)
				RGBOut[offset] = pow(RGBOut[offset] * gain, invGamma);
		if (AlphaOut)
			for (int offset = 0; offset < XDelta * YDelta; ++offset)
				AlphaOut[offset] = pow(AlphaOut[offset] * gain, invGamma);

	}

	if (options.Imager != RI_NULL)
		ApplyImager(options.Imager, RGBOut, AlphaOut, DepthOut,
					XDelta * YDelta);

	if (options.ColorQuantOne != 0 && RGBOut)
		scaleAndDither(options.ColorQuantOne, options.ColorQuantMin,
					   options.ColorQuantMax, options.ColorQuantDither,
					   RGBOut, 3 * XDelta * YDelta);
	if (options.ColorQuantOne != 0 && AlphaOut)
		scaleAndDither(options.ColorQuantOne, options.ColorQuantMin,
					   options.ColorQuantMax, options.ColorQuantDither,
					   AlphaOut, XDelta * YDelta);
	if (options.DepthQuantOne != 0 && DepthOut)
		scaleAndDither(options.DepthQuantOne, options.DepthQuantMin,
					   options.DepthQuantMax, options.DepthQuantDither,
					   DepthOut, XDelta * YDelta);

	if (options.ColorQuantOne != 0.)
		TIFFWrite8Bit(options.DisplayName, RGBOut, AlphaOut, DepthOut,
					  XDelta, YDelta);
	else
		TIFFWriteFloat(options.DisplayName, RGBOut, AlphaOut, DepthOut,
					   XDelta, YDelta);

	if (options.ImageViewer != NULL && !fork())
		if (execlp
			(options.ImageViewer, options.ImageViewer, options.DisplayName,
			 NULL) == -1) {
			Error("Unable to run image viewing program %s: %s",
				  options.ImageViewer, strerror(errno));
			exit(1);
		}

	delete[]RGBOut;
	delete[]AlphaOut;
	delete[]DepthOut;

}

void Image::scaleAndDither(Float one, Float min, Float max,
						   Float ditheramp, Float * pixels, int nSamples)
{
	for (int i = 0; i < nSamples; ++i) {
		pixels[i] =
			Round(pixels[i] * one + RandomFloat(-ditheramp, ditheramp));
		pixels[i] = Clamp(pixels[i], min, max);
	}
}

void Image::AddSample(const Point & Praster, const Spectrum & radiance,
					  Float alpha)
{

	Float XWidth = options.FilterXWidth;
	Float YWidth = options.FilterYWidth;
	Float HalfXWidth = XWidth / 2.;
	Float HalfYWidth = YWidth / 2.;
	int x0 = max((int) ceil(Praster.x - HalfXWidth), XBase);
	int x1 = min((int) floor(Praster.x + HalfXWidth), XBase + XDelta);
	int y0 = max((int) ceil(Praster.y - HalfYWidth), YBase);
	int y1 = min((int) floor(Praster.y + HalfYWidth), YBase + YDelta);

	for (int y = y0; y < y1; ++y)
		for (int x = x0; x < x1; ++x) {

			Float fx = x - Praster.x;
			Float fy = y - Praster.y;
			if (fx < -HalfXWidth || fx > HalfXWidth || fy < -HalfYWidth ||
				fy > HalfYWidth) {
				cerr <<
					"Filter madness!  Please send the following output as well as "
					<< endl;
				cerr <<
					"The input file you're using to mmp@graphics.  --thanks!"
					<< endl;
				fprintf(stderr,
						"%10.10f %10.10f, %10.10f %10.10f, (%10.10f, %10.10f) %d %d %d %d %d %d\n",
						fx, HalfXWidth, fy, HalfYWidth, Praster.x,
						Praster.y, x, x0, x1, y, y0, y1);
			}

			Float filterWt = options.Filter(fx, fy, XWidth, YWidth);
			if (filterWt > 0.) {
				int offset = (y - YBase) * XDelta + (x - XBase);
				if (Pixels)
					Pixels[offset] += radiance * filterWt;
				if (Alphas)
					Alphas[offset] += alpha * filterWt;
				if (Depths)
					Depths[offset] += Praster.z * filterWt;
				WeightSums[offset] += filterWt;
			}

		}

}

void ApplyImager(const char *name, Float * RGBPixels, Float * alphas,
				 Float * depths, int nPixels)
{
}
