
#include "heightfield.h"
#include "trimesh.h"

Heightfield::Heightfield(int x, int y, float *zs, PrimitiveAttributes * a,
						 SurfaceFunction * sf)
:Primitive(a, sf)
{
	nx = x;
	ny = y;
	z = new float[nx * ny];
	memcpy(z, zs, nx * ny * sizeof(float));

	fprintf(stderr,"initially nx = %d and ny = %d\n",nx,ny);

	// set up my grid of meshes
	M = new (TriangleMesh*)[(nx - 1) * (ny - 1)];
	Point *P = new Point[4];
	int tempx = x;
	int tempy = y;
	int *vert = new int[6];
	fprintf(stderr,"now (0) nx = %d and ny = %d\n",nx,ny);
	nx = tempx;
	ny = tempy;
	if (vert == 0)
	  fprintf(stderr,"cannot alloc memory\n");
	int pos = 0;	

	// we only need 1 array of triangle vert indicies
	// specify vertices counter-clockwise
	vert[0] = 0;
	vert[1] = 2;
	vert[2] = 1;
	vert[3] = 0;
	vert[4] = 3;
	vert[5] = 2;

	for (int y = 0; y < (ny-1); y++) {
	  // fprintf(stderr,"y is %d\n",y);
	  for (int x = 0; x < (nx-1); x++) {
		 // fprintf(stderr,"got here, x:%d y:%d nx:%d ny:%d ??\n",x,y,nx,ny);
		 // define the vertices in each mesh of 2 triangles
		 P[0].x = (float) x / (float)(nx - 1);
		 P[0].y = (float) y / (float)(ny - 1);
		 P[0].z = z[x + (y*nx)];

		 // upper-right corner
		 P[1].x = (float) (x+1) / (float)(nx - 1);
		 P[1].y = P[0].y;
		 P[1].z = z[x+1 + (y*nx)];
		 
		 // lower-right corner
		 P[2].x = P[1].x;
		 P[2].y = (float) (y+1) / (float)(ny - 1);
		 P[2].z = z[x+1 + ((y+1)*nx)];
		 
		 // lower-left corner
		 P[3].x = P[0].x;
		 P[3].y = P[2].y;
		 P[3].z = z[x + ((y+1)*nx)];
		 M[pos++] = new TriangleMesh(2,4,vert,P,attributes,surfaceFunction);
	  }
	}

	fprintf(stderr,"now (1) nx = %d and ny = %d\n",nx,ny);
}

Heightfield::~Heightfield()
{
	delete[]z;
	//for (int i=0;i<((nx-1)*(ny-1));i++) 
	//  delete M[i];
	delete[]M;
}

BBox Heightfield::BoundObjectSpace() const
{
	float minz = z[0], maxz = z[0];
	for (int i = 1; i < nx * ny; ++i) {
		if (z[i] < minz)
			minz = z[i];
		if (z[i] > maxz)
			maxz = z[i];
	}
	return BBox(Point(0, 0, minz), Point(1, 1, maxz));
}

bool Heightfield::CanIntersect() const
{
	return true;
}

// helper functions
Float Heightfield::xVal2x(int x) const
{
  return ((Float)x/(Float)(nx-1));
}

Float Heightfield::yVal2y(int y) const
{
  return ((Float)y/(Float)(ny-1));
}

bool Heightfield::IntersectClosest(const Ray &ray, Float mint, Float *maxt, HitInfo *hit) const
{
  //fprintf(stderr,"got here! (1)\n");
  //fprintf(stderr,"now (2) nx = %d and ny = %d\n",nx,ny);
  Float maxValT = *maxt;

  // compute the intersection of the ray and the bounding box
  BBox bbox = this->BoundObjectSpace(); 
  Float rayT = *maxt;
  if (bbox.Inside(ray(mint)))
	 rayT = mint;
  else if (!bbox.IntersectP(ray, &rayT))
	 return false;
  Point gridIntersect = ray(rayT);

  // find the nearest square (transform ray to object space?)
  int xVal = (int)floor(gridIntersect.x * (nx-1));
  int yVal = (int)floor(gridIntersect.y * (ny-1));
  
  // assert that xVal and yVal values are legit?
  // reset ny to ny-1 and nx to nx-1?
  if (xVal == (nx - 1))
	 xVal--;
  if (xVal < 0)
	 xVal = 0;
  if (yVal == (ny - 1))
	 yVal--;
  if (yVal < 0)
	 yVal = 0;

  //Assert(xVal >= 0 && xVal < (nx-1));
  //Assert(yVal >= 0 && yVal < (ny-1));
  //fprintf(stderr,"nx: %d, ny: %d, xVal: %d, yVal: %d\n",nx,ny,xVal,yVal);
  
  Float NextXCrossing, DeltaX;
  int StepX, OutX;
  if (fabs(ray.D.x) < RI_EPSILON) {
	 NextXCrossing = INFINITY;
	 DeltaX = 0;
	 OutX = -1;
  } else if (ray.D.x > 0) {
	 NextXCrossing = rayT + (xVal2x(xVal+1) - gridIntersect.x) / ray.D.x;
	 DeltaX = (1/(float)nx) / ray.D.x;
	 StepX = 1;
	 OutX = nx-1;
  } else {
	 NextXCrossing = rayT + (xVal2x(xVal) - gridIntersect.x) / ray.D.x;
	 DeltaX = -(1/(float)nx) / ray.D.x;
	 StepX = -1;
	 OutX = -1;
  }

  Float NextYCrossing, DeltaY;
  int StepY, OutY;
  
  if (fabs(ray.D.y) < RI_EPSILON) {
	 NextYCrossing = INFINITY;
	 DeltaY = 0;
	 OutY = -1;
  } else if (ray.D.y < 0) {
	 NextYCrossing = rayT + (yVal2y(yVal) - gridIntersect.y) / ray.D.y;
	 DeltaY = -(1/(float)ny) / ray.D.y;
	 StepY = -1;
	 OutY = -1;
  } else {
	 NextYCrossing = rayT + (yVal2y(yVal + 1) - gridIntersect.y) / ray.D.y;
	 DeltaY = (1/(float)ny) / ray.D.y;
	 StepY = 1;
	 OutY = ny-1;
  }
  
  // fprintf(stderr,"got here (2)\n");

  // Go through the grid; stopping when we find an intersection
  int hitSomething = 0;
  while (1) {
	 // intersect with the mesh in the current square
	 TriangleMesh *region = M[yVal * (nx-1) + xVal];

	 //if (!region)
	 //fprintf(stderr,"NULL pointer\n");

	 vector<Primitive *>refined;
	 region->Refine(&refined);
	 // fprintf(stderr,"No of primitives %d",refined.size());
	 Primitive *t1 = refined[0];
	 Primitive *t2 = refined[1];

	 // fprintf(stderr,"got here (3)\n");
	 bool i1 = t1->IntersectClosest(ray,mint,maxt,hit);
	 if (hit == NULL) {
		hitSomething = 1;
		break;
	 }
	 bool i2 = t2->IntersectClosest(ray,mint,maxt,hit);
	 if (i1 || i2)
	 {
		hitSomething = 1;
		break;
	 }

	 // free up the triangle primitives
	 delete t1;
	 delete t2;
	 
	 // take a step
	 // fprintf(stderr,"got here (4)\n");
	 if (NextXCrossing < NextYCrossing) {
		xVal += StepX;
		if (xVal == OutX)
		  break;
		rayT = NextXCrossing;
		NextXCrossing += DeltaX;
	 }
	 else {
		yVal += StepY;
		if (yVal == OutY)
		  break;
		rayT = NextYCrossing;
		NextYCrossing += DeltaY;
	 }

	 // is this necessary?
	 if (rayT > maxValT)
		break;
  }  
  return hitSomething;
}


void Heightfield::Refine(vector < Primitive * >*refined) const
{
  fprintf(stderr,"Ran refine method!!!\n");
	int ntris = 2 * (nx - 1) * (ny - 1);
	int *verts = new int[3 * ntris];
	Point *P = new Point[nx * ny];
	int x, y;

	P = new Point[nx * ny];
	int pos = 0;
	for (y = 0; y < ny; ++y) {
		for (x = 0; x < nx; ++x) {
			P[pos].x = (float) x / (float) (nx - 1);
			P[pos].y = (float) y / (float) (ny - 1);
			P[pos].z = z[pos];
			++pos;
		}
	}

	int *vp = verts;
	for (y = 0; y < ny - 1; ++y) {
		for (x = 0; x < nx - 1; ++x) {
#define VERT(x,y) ((x)+(y)*nx)
			*vp++ = VERT(x, y);
			*vp++ = VERT(x + 1, y);
			*vp++ = VERT(x + 1, y + 1);

			*vp++ = VERT(x, y);
			*vp++ = VERT(x + 1, y + 1);
			*vp++ = VERT(x, y + 1);
		}
#undef VERT
	}

	refined->push_back(new TriangleMesh(ntris, nx * ny, verts, P,
										attributes,
										new
										SurfaceFunction
										(*surfaceFunction)));
	delete[]P;
	delete[]verts;
}
