
#include "lrt.h"
#include "camera.h"
#include "image.h"
#include "options.h"
#include "primitives.h"
#include "transport.h"

void Camera::RenderImage(Scene & scene, const Integrator & integrator) const
{
	Float sample[5];
	JitterSampler sampler(options);
	Image image(options);

	static int eyeRaysTraced = 0;
	if (eyeRaysTraced == 0)
		StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
							 &eyeRaysTraced);
	cerr << "Rendering: ";

	while (sampler.GetNextImageSample(sample)) {

		if (sample[0] < options.SampleCropLeft ||
			sample[0] > options.SampleCropRight ||
			sample[1] < options.SampleCropBottom ||
			sample[1] > options.SampleCropTop) continue;

		Ray ray;
		Point Pcamera;
		switch (options.ProjectionType) {
		case Options::Orthographic:
			Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
			ray = Ray(Pcamera, Vector(0, 0, 1));
			break;
		case Options::Perspective:
			Pcamera = RasterToCamera(Point(sample[0], sample[1], 0));
			ray =
				Ray(Point(0, 0, 0),
					Vector(Pcamera.x, Pcamera.y, Pcamera.z));
			ray.D *= invClipHither;
			break;
		}
		ray = CameraToWorld(ray);

		HitInfo hitInfo;
		Float alpha, hitDist = INFINITY;
		Spectrum L =
			integrator.Integrate(ray, scene, sampler, &hitInfo, &hitDist,
								 &alpha);

		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0)
			cerr << '+';

		Float screenz = WorldToScreen(ray(hitDist)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image.AddSample(Praster, L, alpha);

	}
	image.Write();
	cerr << endl;
}

Camera::Camera(const Options & opt)
:options(opt)
{

	WorldToCamera = opt.WorldToCamera;
	CameraToWorld = Transform(WorldToCamera.GetInverse());

	CameraToScreen = opt.CameraToScreen;
	ScreenToCamera = Transform(CameraToScreen.GetInverse());

	ScreenToRaster =
		Scale(opt.XResolution, opt.YResolution, 1.) *
		Scale(1, -1, 1) *
		Scale(1. / (opt.ScreenRight - opt.ScreenLeft),
			  1. / (opt.ScreenTop - opt.ScreenBottom), 1.) *
		Translate(Vector(-opt.ScreenLeft, -opt.ScreenTop, 0.));
	RasterToScreen = Transform(ScreenToRaster.GetInverse());

	WorldToScreen = CameraToScreen * WorldToCamera;
	RasterToCamera = ScreenToCamera * RasterToScreen;

	invClipHither = 1. / options.ClipHither;
}
