#include "density.h"

Density::Density(int n, BBox bounds) {
  // store the number of points in each dimension
  points = n;
  points2 = n*n;

  noise = new float[n*n*n];
  gridBoundary = new BBox(bounds.pMin,bounds.pMax);
  voxelX = (bounds.pMax.x - bounds.pMin.x)/(float)(n-1);
  voxelY = (bounds.pMax.y - bounds.pMin.y)/(float)(n-1);
  voxelZ = (bounds.pMax.z - bounds.pMin.z)/(float)(n-1);
}


// 3D random number generation borrowed from the following source
// http://freespace.virgin.net/hugo.elias/models/m_perlin.htm
float Density::noise3D(int x, int y, int z) {
  int n;
  n = x + y * 83 + z * 6907;
  n = (n<<13) ^ n;
  return ( 1.0 - ( (n * (n * n * 15731 + 789221) + 1376312589) & 0x7fffffff) / 1073741824.0);
}

float Density::smoothNoise3D(int x, int y, int z) {
  return noise3D(x,y,z);
}

void Density::generatePerlinGrid(float mean) {

  // number of octaves = 4 and amplitude = 1
  float a1 = mean;
  float a2 = a1 / 2.0;
  float a3 = a2 / 2.0;
  float a4 = a3 / 2.0;
  int counter = 0;

  for (int i=0;i<points;i++) {
	 for (int j=0;j<points;j++) {
		for (int k=0;k<points;k++) {
		  float noise1 = noise3D(i/8,j/8,k/8);
		  float noise2 = noise3D(i/4,j/4,k/4);
		  float noise3 = noise3D(i/2,j/2,k/2);
		  float noise4 = noise3D(i,j,k);
		  noise[counter++] = (a1 * noise1 + a2 * noise2 + a3 * noise3
			 + a4 * noise4) + (mean/1.414);
		}		  
	 }
  }

}

float Density::interpolate(float a, float b, float x) {
  return  a*(1-x) + b*x;
}

float Density::interpolate3D(float x, float y, float z, float corners[]) {
  float e1 = interpolate(corners[2],corners[0],x);
  float e2 = interpolate(corners[3],corners[1],x);
  float e3 = interpolate(e1,e2,y);
  float e4 = interpolate(corners[6],corners[4],x);
  float e5 = interpolate(corners[7],corners[5],x);
  float e6 = interpolate(e4,e5,y);
  return interpolate(e3,e6,z);
}

#ifndef LOOKUP
#define LOOKUP(x,y,z) noise[x*points2 + y*points + z]
#endif

float Density::grid(Point p) {
  // assume all points are inside the box
  if (!gridBoundary->Inside(p)) {	 
	 return 0.0;
  }
  
  // convert to grid co-ordinates (0..n-1)
  // if grid-coordinates are exactly n-1 we get a seg fault
  // so we do the following trick
  float eX = ((p.x - gridBoundary->pMin.x)*0.999)/voxelX;
  float eY = ((p.y - gridBoundary->pMin.y)*0.999)/voxelY;
  float eZ = ((p.z - gridBoundary->pMin.z)*0.999)/voxelZ;
  int x = (int)floor(eX);
  int y = (int)floor(eY);
  int z = (int)floor(eZ);

  // index into the grid array
  float corners[8];
  corners[0] = LOOKUP((x+1), y, z);
  corners[1] = LOOKUP((x+1), (y+1), z);
  corners[2] = LOOKUP(x, y, z);
  corners[3] = LOOKUP(x, (y+1), z);
  corners[4] = LOOKUP((x+1), y, (z+1));
  corners[5] = LOOKUP((x+1), y+1, (z+1));
  corners[6] = LOOKUP(x, y, (z+1));
  corners[7] = LOOKUP(x, (y+1), (z+1));
  
  float result = interpolate3D((eX - (float)x), (eY - (float)y), 
										 (eZ - (float)z), corners);

  /*
  fprintf(stderr,"point (%f,%f,%f) corresponds to co-ordinates (%d,%d,%d)
  value is %f\n", p.x,p.y,p.z,x,y,z,result);
  */

  return result;
}
























/*
    fprintf(stderr, "point (%f,%f,%f) is not inside the noise
 grid\n",p.x,p.y,p.z);
	 fprintf(stderr,"(%f,%f,%f) -
	 (%f,%f,%f)\n",gridBoundary->pMin.x,gridBoundary->pMin.y,gridBoundary->pMin.x,gridBoundary->pMax.x,gridBoundary->pMax.y,gridBoundary->pMax.z);
*/



