
#include "lrt.h"
#include "primitives.h"
#include "accel.h"
#include "camera.h"
#include "mbdofcamera.h"
#include "color.h"
#include "light.h"
#include "sampling.h"
#include "shading.h"
#include "transport.h"
#include "nurbs.h"
#include "scene.h"
#include "image.h"
#include "quadrics.h"
#include "trimesh.h"
#include "heightfield.h"
#include <ctype.h>
#ifdef _WIN32
#include <malloc.h>
#else
#include <alloca.h>
#endif

#include <stdarg.h>

#define STATE_UNINITIALIZED  0
#define STATE_BEGIN          1
#define STATE_WORLD_BEGIN    2
static int currentApiState = STATE_UNINITIALIZED;

static const char *missingStateCall[] =
	{ "RiEnd()", "RiBegin()", "RiWorldBegin()" };

#define TOKEN_HASH_SIZE 1024
static list<pair<RtToken, int> > TokenTable[TOKEN_HASH_SIZE];

#define TOKEN_TYPE_ERROR       -1
#define TOKEN_TYPE_UNIFORM (1<<0)
#define TOKEN_TYPE_VERTEX  (1<<1)
#define TOKEN_TYPE_FLOAT   (1<<2)
#define TOKEN_TYPE_POINT   (1<<3)
#define TOKEN_TYPE_VECTOR  (1<<4)
#define TOKEN_TYPE_NORMAL  (1<<5)
#define TOKEN_TYPE_STRING  (1<<6)
#define TOKEN_TYPE_COLOR   (1<<7)
#define TOKEN_TYPE_VOID    (1<<8)
#define TOKEN_TYPE_HPOINT  (1<<9)

static int argsCount;
static int argsAllocated;
static RtToken *argTokens;
static RtPointer *argParams;

#define MOTION_LEVELS 2
static int motionLevel = 0;
static bool inMotionBlock = false;
static Transform curTransform[MOTION_LEVELS];

static vector < Primitive * >primitives;
static list < Light * >lights;
static list<list<Light *> > lightStack;
static PrimitiveAttributes *curPrimitiveAttributes;

static LightAttributes *curLightAttributes;
static MaterialAttributes *curMaterialAttributes;

static list < char >hierarchicalState;
static list < Transform > transformStack[MOTION_LEVELS];

static list < PrimitiveAttributes * >geomAttrStack;
static list < LightAttributes * >lightAttrStack;
static list < MaterialAttributes * >surfaceAttrStack;

#define VERIFY_STATE(s, func) \
	if (currentApiState != s) { \
		Error("Must have called %s before calling %s().  Ignoring.", \
			missingStateCall[s], func); \
		return; \
	} \
	else						/* swallow trailing semicolon */

RtToken RI_A, RI_ABORT, RI_AMBIENTLIGHT, RI_AMPLITUDE, RI_AZ,
	RI_BACKGROUND, RI_BEAMDISTRIBUTION, RI_BICUBIC, RI_BILINEAR, RI_BLACK,
	RI_BUMPY, RI_CAMERA, RI_CLAMP, RI_COMMENT, RI_CONEANGLE,
	RI_CONEDELTAANGLE, RI_CONSTANT, RI_CS, RI_DEPTHCUE, RI_DIFFERENCE,
	RI_DISTANCE, RI_DISTANTLIGHT, RI_FILE, RI_FLATNESS, RI_FOG, RI_FOV,
	RI_FRAMEBUFFER, RI_FROM, RI_HANDLER, RI_HIDDEN, RI_IDENTIFIER,
	RI_IGNORE, RI_INSIDE, RI_INTENSITY, RI_INTERSECTION, RI_KA, RI_KD,
	RI_KR, RI_KS, RI_LH, RI_LIGHTCOLOR, RI_MATTE, RI_MAXDISTANCE, RI_METAL,
	RI_MINDISTANCE, RI_N, RI_NAME, RI_NONPERIODIC, RI_NP, RI_OBJECT,
	RI_ORIGIN, RI_ORTHOGRAPHIC, RI_OS, RI_OUTSIDE, RI_P, RI_PAINT,
	RI_PAINTEDPLASTIC, RI_PERIODIC, RI_PERSPECTIVE, RI_PLASTIC,
	RI_POINTLIGHT, RI_PRIMITIVE, RI_PRINT, RI_PW, RI_PZ, RI_RASTER, RI_RGB,
	RI_RGBA, RI_RGBAZ, RI_RGBZ, RI_RH, RI_ROUGHNESS, RI_S, RI_SCREEN,
	RI_SHADINGGROUP, RI_SHINYMETAL, RI_SMOOTH, RI_SPECULARCOLOR,
	RI_SPOTLIGHT, RI_ST, RI_STRUCTURE, RI_T, RI_TEXTURENAME, RI_TO,
	RI_TRIMDEVIATION, RI_UNION, RI_WORLD, RI_Z;

RtToken LRT_IMAGEVIEWER;
RtToken LRT_RENDER, LRT_DISPLAY;
RtToken LRT_INTEGRATOR, LRT_SHADOWS, LRT_COLOR;
RtToken LRT_WHITTED, LRT_RMAN, LRT_MONTECARLO;
RtToken LRT_SURFACE, LRT_COMBINATION, LRT_SAMPLE;
RtToken LRT_LIGHT, LRT_RAYCAST, LRT_NSAMPLES;
RtToken LRT_VIEWST, LRT_GLASS, LRT_INDEX, LRT_KT;
RtToken LRT_CHECKERED, LRT_CHECKCOLOR1, LRT_CHECKCOLOR2,
	LRT_CHECKFREQUENCY;
RtToken LRT_DIFFUSE;
RtToken LRT_TYPE, LRT_SHUTTER;
RtToken LRT_PINHOLE, LRT_MBDOF;
RtToken LRT_IRIS, LRT_STRIPE;
RtToken LRT_IRIS_RATE, LRT_STRIPE_WIDTH, LRT_STRIPE_DIRECTION;
RtToken LRT_DOWN, LRT_UP, LRT_LEFT, LRT_RIGHT;

RtToken LRT_CLEAR, LRT_DUST;

RtToken LRT_FORMAT, LRT_BUCKETSIZE, LRT_GRIDSIZE, LRT_TEXTUREMEMORY;
RtToken LRT_ZTHRESHOLD, LRT_EXTREMEDISPLACEMENT, LRT_EYESPLITS;
RtToken LRT_GEOMMEMORY, LRT_BIAS0, LRT_BIAS1, LRT_SHADER, LRT_TEXTURE;
RtToken LRT_VFXMASTER, LRT_VFXINSTANCE, LRT_ARCHIVE, LRT_RESOURCE;
RtToken LRT_MINSAMPLES, LRT_MAXSAMPLES, LRT_MAX_RAYLEVEL, LRT_BRANCH_RATIO;
RtToken LRT_MAX_BRANCH_RATIO, LRT_MINSHADOWBIAS, LRT_STEPS,
	LRT_MINPATCHSAMPLES;
RtToken LRT_VERBOSITY, LRT_INCLUDE, LRT_REFRESH, LRT_FULLCOLOR;
RtToken LRT_SETPOINTS, LRT_NAME, LRT_SHADINGGROUP, LRT_BINARY;
RtToken LRT_COORDINATESYSTEM, LRT_SPHERE, LRT_SENSE, LRT_AVERAGECOLOR;
RtToken LRT_EMISSIONCOLOR, LRT_PATCHSIZE, LRT_ELEMSIZE, LRT_MINSIZE;
RtToken LRT_ZONAL, LRT_CASTS_SHADOWS, LRT_PATCH_MAXLEVEL,
	LRT_PATCH_MINLEVEL;
RtToken LRT_PATCH_MULTIPLIER, LRT_TRUEDISPLACEMENT, LRT_CACHE;
RtToken LRT_UDIVISIONS, LRT_VDIVISIONS, LRT_ORIGIN;
RtToken LRT_MERGE, LRT_COMPRESSION, LRT_RESOLUTION, LRT_RESOLUTIONUNIT;
RtToken LRT_JITTER, LRT_PDISC, LRT_FLATNESS, LRT_WIDTH, LRT_CONSTANTWIDTH;

u_int hashTokenString(const char *name)
{
	u_int hashValue = 0;
	while (*name) {
		hashValue <<= 1;
		hashValue ^= *name;
		++name;
	}
	return hashValue % TOKEN_HASH_SIZE;
}

static int stringToType(const char *strType);
static void typeToString(int type, char string[]);

static int stringToType(const char *strType)
{
	if (strType == NULL)
		return TOKEN_TYPE_VOID;
	int type;

	const char *strp = strType;
	while (*strp && isspace(*strp))
		++strp;
	if (!*strp)
		return TOKEN_TYPE_ERROR;
	if (strncmp("uniform", strp, strlen("uniform")) == 0) {
		type = TOKEN_TYPE_UNIFORM;
		strp += strlen("uniform");
	} else if (strncmp("constant", strp, strlen("constant")) == 0) {
		type = TOKEN_TYPE_UNIFORM;
		strp += strlen("constant");
	} else if (strncmp("vertex", strp, strlen("vertex")) == 0) {
		type = TOKEN_TYPE_VERTEX;
		strp += strlen("vertex");
	} else
		type = TOKEN_TYPE_UNIFORM;

	while (*strp && isspace(*strp))
		++strp;
	if (!*strp)
		return TOKEN_TYPE_ERROR;

#define TRY_DECODING_TYPE(name, mask) \
		if (strncmp(name, strp, strlen(name)) == 0) { \
			type |= mask; strp += strlen(name); \
		}

	     TRY_DECODING_TYPE("float",  TOKEN_TYPE_FLOAT)
	else TRY_DECODING_TYPE("point",  TOKEN_TYPE_POINT)
	else TRY_DECODING_TYPE("hpoint", TOKEN_TYPE_HPOINT)
	else TRY_DECODING_TYPE("vector", TOKEN_TYPE_VECTOR)
	else TRY_DECODING_TYPE("normal", TOKEN_TYPE_NORMAL)
	else TRY_DECODING_TYPE("string", TOKEN_TYPE_STRING)
	else TRY_DECODING_TYPE("color",  TOKEN_TYPE_COLOR)
	else {
		Error("RiDeclare: unable to decode type \"%s\"", strType);
		return TOKEN_TYPE_ERROR;
	}

	while (*strp && isspace(*strp))
		++strp;
	if (*strp)
		Warning("RiDeclare: unknown text at end of declared type \"%s\".",
				strType);

	return type;
}

static void typeToString(int type, char *string)
{
	if (type == TOKEN_TYPE_VOID) {
		string = strcat(string, "RI_NULL");
		return;
	}
	if (type & TOKEN_TYPE_UNIFORM)
		string = strcat(string, "uniform ");
	else if (type & TOKEN_TYPE_VERTEX)
		string = strcat(string, "vertex ");
	else
		Severe("typeToString: unknown type %d", type);

	if (type & TOKEN_TYPE_FLOAT)
		string = strcat(string, "float");
	else if (type & TOKEN_TYPE_POINT)
		string = strcat(string, "point");
	else if (type & TOKEN_TYPE_HPOINT)
		string = strcat(string, "hpoint");
	else if (type & TOKEN_TYPE_VECTOR)
		string = strcat(string, "vector");
	else if (type & TOKEN_TYPE_NORMAL)
		string = strcat(string, "normal");
	else if (type & TOKEN_TYPE_STRING)
		string = strcat(string, "string");
	else if (type & TOKEN_TYPE_COLOR)
		string = strcat(string, "color");
	else
		Severe("typeToString: unknown type %d", type);
}

static RtToken lookupToken(const char *name)
{
	if (name == NULL)
		return RI_NULL;
	u_int offset = hashTokenString(name);
	list<pair<RtToken, int> >::iterator iter = TokenTable[offset].begin();
	while (iter != TokenTable[offset].end()) {
		if (strcmp(iter->first, name) == 0)
			return iter->first;
		++iter;
	}
	return RI_NULL;
}

static bool lookupTokenAndType(const char *name, RtToken * token,
							   int *type)
{
	Assert(name != NULL);

	bool tryInline = false;

	const char *p = name;
	while (*p) {
		if (isspace(*p)) {
			tryInline = true;
			break;
		}
		++p;
	}

	if (tryInline) {

		const char *end = &name[strlen(name) - 1];
		while (!isspace(*end))
			--end;
		char *buf = (char *) alloca(end - name + 1);
		strncpy(buf, name, end - name);
		buf[end - name] = '\0';

		*type = stringToType(buf);
		*token = (char *) end + 1;
		if (*type != TOKEN_TYPE_ERROR)
			return true;

	}

	u_int offset = hashTokenString(name);
	list<pair<RtToken, int> >::iterator iter = TokenTable[offset].begin();
	while (iter != TokenTable[offset].end()) {
		if (iter->first == name) {
			*token = iter->first;
			*type = iter->second;
			return true;
		}
		++iter;
	}
	iter = TokenTable[offset].begin();
	while (iter != TokenTable[offset].end()) {
		if (strcmp(iter->first, name) == 0) {
			*token = iter->first;
			*type = iter->second;
			return true;
		}
		++iter;
	}
	return false;

}

static void vectorizeParameters(va_list args)
{
	argsCount = 0;
	const char *name;
	while ((name = va_arg(args, const char *)) != RI_NULL) {
		if (argsCount >= argsAllocated) {

			int newArgs;
			if (argsAllocated == 0)
				newArgs = 10;
			else
				newArgs = 2 * argsAllocated;
			argTokens =
				(RtToken *) realloc(argTokens, newArgs * sizeof(RtToken));
			argParams =
				(RtPointer *) realloc(argParams,
									  newArgs * sizeof(RtPointer));
			argsAllocated = newArgs;

		}

		argTokens[argsCount] = (RtToken) name;
		argParams[argsCount] = va_arg(args, RtPointer);
		if (argParams[argsCount] == RI_NULL)
			Error("Null parameter value for argument %s?!?", name);
		else
			++argsCount;

	}
}

static Float InitializeFloat(RtToken tokenName, Float defaultValue,
							 int nArgs, RtToken tokens[],
							 RtPointer params[])
{

	RtToken tok;
	int type;
	if (!lookupTokenAndType(tokenName, &tok, &type))
		Severe("Token %s unknown in InitializeFloat() in RI", tokenName);
	if (!(type & TOKEN_TYPE_FLOAT) && !(type & TOKEN_TYPE_UNIFORM)) {
		char buf[128];
		typeToString(type, buf);
		Severe("Token %s declared to be non-uniform float type %s",
			   tokenName, buf);
	}

	for (int i = 0; i < nArgs; ++i)
		if (tok == tokens[i])
			return *((Float *) params[i]);
	for (int i = 0; i < nArgs; ++i)
		if (strcmp(tok, tokens[i]) == 0)
			return *((Float *) params[i]);
	return defaultValue;
}

void reportUnusedParameters(const char *funcName, int nArgs,
							RtToken tokens[], ...)
{

	vector < RtToken > usedParameters;
	va_list args;
	va_start(args, tokens);
	RtToken token;
	while ((token = va_arg(args, RtToken)) != RI_NULL)
		usedParameters.push_back(token);
	va_end(args);

	for (int i = 0; i < nArgs; ++i) {
		for (u_int j = 0; j < usedParameters.size(); ++j)
			if (tokens[i] == usedParameters[j])
				goto tokenUsed;
		for (u_int j = 0; j < usedParameters.size(); ++j)
			if (strcmp(tokens[i], usedParameters[j]) == 0)
				goto tokenUsed;
		Warning("%s(): parameter \"%s\" is unknown and will be ignored.",
				funcName, argTokens[i]);
	  tokenUsed:;
	}

}

static SurfaceFunction *initSurfaceFunction(int nVertices, int n,
											RtToken tokens[],
											RtPointer params[])
{
	SurfaceFunction *surffunc = new SurfaceFunction(nVertices);
	for (int i = 0; i < n; ++i) {
		if (strcmp(tokens[i], RI_P) == 0)
			continue;
		if (strcmp(tokens[i], RI_PW) == 0)
			continue;

		int paramType;
		RtToken token;
		if (lookupTokenAndType(tokens[i], &token, &paramType)) {
			if (paramType & TOKEN_TYPE_UNIFORM) {

				if (paramType & TOKEN_TYPE_FLOAT)
					surffunc->AddUniformFloat(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_POINT)
					surffunc->AddUniformPoint(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_HPOINT)
					surffunc->AddUniformHPoint(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_VECTOR)
					surffunc->AddUniformVector(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_NORMAL)
					surffunc->AddUniformNormal(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_COLOR)
					surffunc->AddUniformColor(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_STRING)
					surffunc->AddUniformString(token,
											   *((const char **)
												 params[i]));

			} else if (nVertices == 0) {

				Warning
					("Ignoring vertex data '%s' being bound to light or surface shader",
					 token);

			} else {
				Assert(paramType & TOKEN_TYPE_VERTEX);

				if (paramType & TOKEN_TYPE_FLOAT)
					surffunc->AddVertexFloat(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_POINT)
					surffunc->AddVertexPoint(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_HPOINT)
					surffunc->AddVertexHPoint(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_VECTOR)
					surffunc->AddVertexVector(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_NORMAL)
					surffunc->AddVertexNormal(token, (Float *) params[i]);
				else if (paramType & TOKEN_TYPE_COLOR)
					surffunc->AddVertexColor(token, (Float *) params[i]);

			}
		} else
			Warning("Unknown parameter '%s'.  Ignoring it.", tokens[i]);

	}
	return surffunc;
}

static void AddPrimitive(Primitive * prim)
{
	if (!prim)
		return;
	primitives.push_back(prim);
	if (curPrimitiveAttributes->LightShader)
		curPrimitiveAttributes->LightShader->SetGeometry(prim);
}

RtVoid RiWorldEnd()
{

	while (hierarchicalState.size()) {
		char c = hierarchicalState.back();
		if (c == 't')
			Error("Missing end to RiTransformBegin");
		else if (c == 'a')
			Error("Missing end to RiAttributeBegin");
		else
			Severe("Internal error in gfx state management");
		hierarchicalState.pop_back();
	}

	scene->AddPrimitives(primitives);
	scene->Render();

	curPrimitiveAttributes->Dereference();
	curPrimitiveAttributes = NULL;
	curMaterialAttributes->Dereference();
	curMaterialAttributes = NULL;
	currentApiState = STATE_BEGIN;
	for (u_int i = 0; i < primitives.size(); ++i)
		delete primitives[i];
	primitives.erase(primitives.begin(), primitives.end());
	list < Light * >::iterator iter = lights.begin();
	while (iter != lights.end()) {
		delete *iter;
		++iter;
	}
	lights.erase(lights.begin(), lights.end());

	StatsPrint(STATS_DETAILED, stderr);

}

RtToken RiDeclare(char *name, char *type)
{

	int tokenType = stringToType(type);
	if (tokenType == TOKEN_TYPE_ERROR)
		return RI_NULL;

	u_int hashValue = hashTokenString(name);
	list<pair<RtToken, int> >::iterator iter = TokenTable[hashValue].begin();
	while (iter != TokenTable[hashValue].end()) {
		if (strcmp(iter->first, name) == 0) {

			if (iter->second != tokenType) {
				char str1[80], str2[80];
				typeToString(iter->second, str1);
				typeToString(tokenType, str2);
				Warning("RiDeclare: Token '%s' redeclared from %s to %s.",
						name, str1, str2);
			}

			return iter->first;
		}
		++iter;
	}

	char *newToken = Strdup(name);
	TokenTable[hashValue].push_front(make_pair(newToken, tokenType));
	return newToken;

}

RtVoid RiBegin(RtToken tok)
{
	if (currentApiState != STATE_UNINITIALIZED) {
		Error("RiBegin() has already been called.");
		return;
	}
	currentApiState = STATE_BEGIN;


	RI_A = RiDeclare("a", RI_NULL);
	RI_ABORT = RiDeclare("abort", RI_NULL);
	RI_AMBIENTLIGHT = RiDeclare("ambientlight", RI_NULL);
	RI_AMPLITUDE = RiDeclare("amplitude", RI_NULL);
	RI_AZ = RiDeclare("az", RI_NULL);
	RI_BACKGROUND = RiDeclare("background", RI_NULL);
	RI_BEAMDISTRIBUTION = RiDeclare("beamdistribution", RI_NULL);
	RI_BICUBIC = RiDeclare("bicubic", RI_NULL);
	RI_BILINEAR = RiDeclare("bilinear", RI_NULL);
	RI_BLACK = RiDeclare("black", RI_NULL);
	RI_BUMPY = RiDeclare("bumpy", RI_NULL);
	RI_CAMERA = RiDeclare("camera", RI_NULL);
	RI_CLAMP = RiDeclare("clamp", RI_NULL);
	RI_COMMENT = RiDeclare("comment", RI_NULL);
	RI_CONEANGLE = RiDeclare("coneangle", "float");
	RI_CONEDELTAANGLE = RiDeclare("conedeltaangle", "float");
	RI_CONSTANT = RiDeclare("constant", RI_NULL);
	RI_CS = RiDeclare("Cs", "vertex color");
	RI_DEPTHCUE = RiDeclare("depthcue", RI_NULL);
	RI_DIFFERENCE = RiDeclare("difference", RI_NULL);
	RI_DISTANCE = RiDeclare("distance", RI_NULL);
	RI_DISTANTLIGHT = RiDeclare("distantlight", RI_NULL);
	RI_FILE = RiDeclare("file", RI_NULL);
	RI_FLATNESS = RiDeclare("flatness", RI_NULL);
	RI_FOG = RiDeclare("fog", RI_NULL);
	RI_FOV = RiDeclare("fov", "float");
	RI_FRAMEBUFFER = RiDeclare("framebuffer", RI_NULL);
	RI_FROM = RiDeclare("from", "point");
	RI_HANDLER = RiDeclare("handler", RI_NULL);
	RI_HIDDEN = RiDeclare("hidden", RI_NULL);
	RI_IDENTIFIER = RiDeclare("identifier", "string");
	RI_IGNORE = RiDeclare("ignore", RI_NULL);
	RI_INSIDE = RiDeclare("inside", RI_NULL);
	RI_INTENSITY = RiDeclare("intensity", "float");
	RI_INTERSECTION = RiDeclare("intersection", RI_NULL);
	RI_KA = RiDeclare("Ka", "float");
	RI_KD = RiDeclare("Kd", "float");
	RI_KR = RiDeclare("Kr", "float");
	RI_KS = RiDeclare("Ks", "float");
	RI_LH = RiDeclare("lh", RI_NULL);
	RI_LIGHTCOLOR = RiDeclare("lightcolor", "color");
	RI_MATTE = RiDeclare("matte", RI_NULL);
	RI_MAXDISTANCE = RiDeclare("maxdistance", "float");
	RI_METAL = RiDeclare("metal", RI_NULL);
	RI_MINDISTANCE = RiDeclare("mindistance", "float");
	RI_N = RiDeclare("N", "vertex normal");
	RI_NAME = RiDeclare("name", "string");
	RI_NONPERIODIC = RiDeclare("nonperiodic", RI_NULL);
	RI_NP = RiDeclare("Np", RI_NULL);
	RI_OBJECT = RiDeclare("object", RI_NULL);
	RI_ORIGIN = RiDeclare("origin", "point");
	RI_ORTHOGRAPHIC = RiDeclare("orthographic", RI_NULL);
	RI_OS = RiDeclare("Os", "float");
	RI_OUTSIDE = RiDeclare("outside", RI_NULL);
	RI_P = RiDeclare("P", "vertex point");
	RI_PAINT = RiDeclare("paint", RI_NULL);
	RI_PAINTEDPLASTIC = RiDeclare("paintedplastic", RI_NULL);
	RI_PERIODIC = RiDeclare("periodic", RI_NULL);
	RI_PERSPECTIVE = RiDeclare("perspective", RI_NULL);
	RI_PLASTIC = RiDeclare("plastic", RI_NULL);
	RI_POINTLIGHT = RiDeclare("pointlight", RI_NULL);
	RI_PRIMITIVE = RiDeclare("primitive", RI_NULL);
	RI_PRINT = RiDeclare("print", RI_NULL);
	RI_PW = RiDeclare("Pw", "vertex hpoint");
	RI_PZ = RiDeclare("Pz", "vertex float");
	RI_RASTER = RiDeclare("raster", RI_NULL);
	RI_RGB = RiDeclare("rgb", RI_NULL);
	RI_RGBA = RiDeclare("rgba", RI_NULL);
	RI_RGBAZ = RiDeclare("rgbaz", RI_NULL);
	RI_RGBZ = RiDeclare("rgbz", RI_NULL);
	RI_RH = RiDeclare("rh", RI_NULL);
	RI_ROUGHNESS = RiDeclare("roughness", "float");
	RI_S = RiDeclare("s", "vertex float");
	RI_SCREEN = RiDeclare("screen", RI_NULL);
	RI_SHADINGGROUP = RiDeclare("shadinggroup", RI_NULL);
	RI_SHINYMETAL = RiDeclare("shinymetal", RI_NULL);
	RI_SMOOTH = RiDeclare("smooth", RI_NULL);
	RI_SPECULARCOLOR = RiDeclare("specularcolor", "color");
	RI_SPOTLIGHT = RiDeclare("spotlight", RI_NULL);
	//RI_ST               = RiDeclare("st", "vertex float[2]");  // XXX
	RI_STRUCTURE = RiDeclare("structure", RI_NULL);
	RI_T = RiDeclare("t", "vertex float");
	RI_TEXTURENAME = RiDeclare("texturename", "string");
	RI_TO = RiDeclare("to", "point");
	RI_TRIMDEVIATION = RiDeclare("trimdeviation", "float");
	RI_UNION = RiDeclare("union", RI_NULL);
	RI_WORLD = RiDeclare("world", RI_NULL);
	RI_Z = RiDeclare("z", RI_NULL);

	LRT_RAYCAST = RiDeclare("raycast", RI_NULL);
	LRT_CHECKCOLOR1 = RiDeclare("checkcolor1", "color");
	LRT_CHECKCOLOR2 = RiDeclare("checkcolor2", "color");
	LRT_CHECKERED = RiDeclare("checkered", RI_NULL);
	LRT_CHECKFREQUENCY = RiDeclare("checkfrequency", "float");
	LRT_COLOR = RiDeclare("color", RI_NULL);
	LRT_DISPLAY = RiDeclare("display", RI_NULL);
	LRT_GLASS = RiDeclare("glass", RI_NULL);
	LRT_IMAGEVIEWER = RiDeclare("imageviewer", "string");
	LRT_INDEX = RiDeclare("index", "float");
	LRT_INTEGRATOR = RiDeclare("integrator", "string");
	LRT_KT = RiDeclare("Kt", "float");
	LRT_LIGHT = RiDeclare("light", RI_NULL);
	LRT_NSAMPLES = RiDeclare("nsamples", "float");
	LRT_RENDER = RiDeclare("render", RI_NULL);
	LRT_SHADOWS = RiDeclare("shadows", "string");
	LRT_VIEWST = RiDeclare("viewst", RI_NULL);
	LRT_WHITTED = RiDeclare("whitted", RI_NULL);
	LRT_MONTECARLO = RiDeclare("montecarlo", RI_NULL);
	LRT_RMAN = RiDeclare("renderman", RI_NULL);
	LRT_DIFFUSE = RiDeclare("diffuse", RI_NULL);
	LRT_SURFACE = RiDeclare("surface", RI_NULL);
	LRT_COMBINATION = RiDeclare("combination", RI_NULL);
	LRT_SAMPLE = RiDeclare("sample", RI_NULL);
	LRT_PINHOLE = RiDeclare("pinhole", "string");
	LRT_MBDOF = RiDeclare("mbdof", "string");
	LRT_IRIS = RiDeclare("iris", "string");
	LRT_STRIPE = RiDeclare("stripe", "string");
	LRT_TYPE = RiDeclare("type", "string");
	LRT_SHUTTER = RiDeclare("shutter", "string");
	LRT_IRIS_RATE = RiDeclare("iris_rate", "float");
	LRT_STRIPE_WIDTH = RiDeclare("stripe_width", "float");
	LRT_STRIPE_DIRECTION = RiDeclare("stripe_direction", "string");
	LRT_DOWN = RiDeclare("down", "string");
	LRT_UP = RiDeclare("up", "string");
	LRT_LEFT = RiDeclare("left", "string");
	LRT_RIGHT = RiDeclare("right", "string");

	LRT_CLEAR = RiDeclare( "clear", "string" );
	LRT_DUST = RiDeclare( "dust", "string" );

	LRT_FORMAT = RiDeclare("format", "string");
	LRT_BUCKETSIZE = RiDeclare("bucketsize", /*int */ "float" /*[2] */ );
	LRT_GRIDSIZE = RiDeclare("gridsize", /*int */ "float" /*[2] */ );
	LRT_TEXTUREMEMORY = RiDeclare("texturememory", /*int */ "float");
	LRT_ZTHRESHOLD = RiDeclare("zthreshold", "point");
	LRT_EXTREMEDISPLACEMENT =
		RiDeclare("extremedisplacement", /*int */ "float");
	LRT_EYESPLITS = RiDeclare("eyesplits", /*int */ "float");
	LRT_GEOMMEMORY = RiDeclare("geommemory", /*int */ "float");
	LRT_BIAS0 = RiDeclare("bias0", "float");
	LRT_BIAS1 = RiDeclare("bias1", "float");
	LRT_SHADER = RiDeclare("shader", "string");
	LRT_TEXTURE = RiDeclare("texture", "string");
	LRT_VFXMASTER = RiDeclare("vfxmaster", "string");
	LRT_VFXINSTANCE = RiDeclare("vfxinstance", "string");
	LRT_ARCHIVE = RiDeclare("archive", "string");	/* Added for PRMan 3.8 */
	LRT_RESOURCE = RiDeclare("resource", "string");	/* Missing item added. */
	LRT_MINSAMPLES = RiDeclare("minsamples", /*int */ "float");
	LRT_MAXSAMPLES = RiDeclare("maxsamples", /*int */ "float");
	LRT_MAX_RAYLEVEL = RiDeclare("max_raylevel", /*int */ "float");
	LRT_BRANCH_RATIO = RiDeclare("branch_ratio", /*int */ "float");
	LRT_MAX_BRANCH_RATIO = RiDeclare("max_branch_ratio", /*int */ "float");
	LRT_MINSHADOWBIAS = RiDeclare("minshadowbias", "float");
	LRT_STEPS = RiDeclare("steps", /*int */ "float");
	LRT_MINPATCHSAMPLES = RiDeclare("minpatchsamples", /*int */ "float");
	LRT_VERBOSITY = RiDeclare("verbosity", "string");
	LRT_INCLUDE = RiDeclare("include", "string");
	LRT_REFRESH = RiDeclare("refresh", /*int */ "float");	/* See [PIXA93b]. */
	LRT_FULLCOLOR = RiDeclare("fullcolor", /*int */ "float");	/* See [PIXA93b]. */
	LRT_SETPOINTS = RiDeclare("setpoints", /*int */ "float" /*[2] */ );	/* See [PIXA98]. */
	LRT_NAME = RiDeclare("name", "string");
	/* already declared as RI_NULL?? */
	/*LRT_SHADINGGROUP      = RiDeclare( "shadinggroup", "string" ); */

	LRT_BINARY = RiDeclare("binary", /*int */ "float");
	LRT_COORDINATESYSTEM = RiDeclare("coordinatesystem", "string");
	LRT_SPHERE = RiDeclare("sphere", "float");
	LRT_SENSE = RiDeclare("sense", "string");
	LRT_AVERAGECOLOR = RiDeclare("averagecolor", "color");
	LRT_EMISSIONCOLOR = RiDeclare("emissioncolor", "color");
	LRT_PATCHSIZE = RiDeclare("patchsize", /*int */ "float");
	LRT_ELEMSIZE = RiDeclare("elemsize", /*int */ "float");
	LRT_MINSIZE = RiDeclare("minsize", /*int */ "float");
	LRT_ZONAL = RiDeclare("zonal", "string");
	LRT_CASTS_SHADOWS = RiDeclare("casts_shadows", "string");
	LRT_PATCH_MAXLEVEL = RiDeclare("patch_maxlevel", /*int */ "float");
	LRT_PATCH_MINLEVEL = RiDeclare("patch_minlevel", /*int */ "float");	/* 2.3.5 */
	LRT_PATCH_MULTIPLIER = RiDeclare("patch_multiplier", /*int */ "float");	/* 2.3.6? */
	LRT_TRUEDISPLACEMENT = RiDeclare("truedisplacement", /*int */ "float");	/* 2.3.5 */
	LRT_CACHE = RiDeclare("cache", "string");
	LRT_UDIVISIONS = RiDeclare("udivisions", /*int */ "float");	/* See [PIXA93b]. */
	LRT_VDIVISIONS = RiDeclare("vdivisions", /*int */ "float");	/* See [PIXA93b]. */

	/* already declared?? */
	/*LRT_ORIGIN                = RiDeclare( "origin", int "float[2]" ); */

	LRT_MERGE = RiDeclare("merge", /*int */ "float");
	LRT_COMPRESSION = RiDeclare("compression", "string");	/* See [PIXA93a]. */
	LRT_RESOLUTION = RiDeclare("resolution", /*int */ "float" /*[2] */ );	/* See [PIXA93a]. */
	LRT_RESOLUTIONUNIT = RiDeclare("resolutionunit", "string");	/* See [PIXA93a]. */
	LRT_JITTER = RiDeclare("jitter", /*int */ "float");
	LRT_PDISC = RiDeclare("pdisc", /*int */ "float");
/*LRT_FLATNESS          = RiDeclare( "flatness", "float" ); *//* already declared? */
	LRT_WIDTH = RiDeclare("width", "vertex float");
	LRT_CONSTANTWIDTH = RiDeclare("constantwidth", "constant float");


	for (int i = 0; i < MOTION_LEVELS; ++i)
		curTransform[i] = Transform();
	scene = new Scene;
	curLightAttributes = new LightAttributes;

	if (tok != RI_NULL)
		Warning("Unknown renderer name %s in RiBegin()", tok);
}

RtVoid RiEnd()
{
	currentApiState = STATE_UNINITIALIZED;

	argsAllocated = 0;
	if (argTokens) {
		free(argTokens);
		argTokens = NULL;
	}
	if (argParams) {
		free(argParams);
		argParams = NULL;
	}

	curLightAttributes->Dereference();
	curLightAttributes = NULL;
	delete scene;
	scene = NULL;
	StatsCleanup();
}

RtVoid RiFrameBegin(RtInt num)
{
}

RtVoid RiFrameEnd()
{
}

RtVoid RiWorldBegin()
{
	VERIFY_STATE(STATE_BEGIN, "RiWorldBegin");
	currentApiState = STATE_WORLD_BEGIN;


	for (int i = 0; i < MOTION_LEVELS; ++i) {
		scene->camera->WorldToCamera[i] = curTransform[i];
		curTransform[i] = Transform();
	}


	curPrimitiveAttributes = new PrimitiveAttributes;
	curPrimitiveAttributes->Surface =
		MaterialCreate("matte", new SurfaceFunction(0));
	curMaterialAttributes = new MaterialAttributes;
}

RtVoid RiPixelSamples(RtFloat x, RtFloat y)
{
	VERIFY_STATE(STATE_BEGIN, "RiPixelSamples");
	scene->sampler->PixelSamples[0] = max((Float) 1., x);
	scene->sampler->PixelSamples[1] = max((Float) 1., y);
}

RtVoid RiFormat(RtInt x, RtInt y, RtFloat pixelAspect)
{
	VERIFY_STATE(STATE_BEGIN, "RiFormat");

	if (x < 0)
		x = 640;
	if (y < 0)
		y = 480;
	if (pixelAspect < 0)
		pixelAspect = 1;

	if (x < 2) {
		Error("RiFormat: x resolution must be >= 2.  Value %d invalid", x);
		x = 2;
	}
	if (y < 2) {
		Error("RiFormat: y resolution must be >= 2.  Value %d invalid", y);
		y = 2;
	}


	scene->image->XResolution = x;
	scene->image->YResolution = y;
	scene->image->PixelAspectRatio = pixelAspect;
	scene->image->FrameAspectRatio = x * pixelAspect / y;
	if (scene->image->FrameAspectRatio >= 1) {
		scene->camera->ScreenLeft = -scene->image->FrameAspectRatio;
		scene->camera->ScreenRight = scene->image->FrameAspectRatio;
		scene->camera->ScreenBottom = -1;
		scene->camera->ScreenTop = 1;
	} else {
		scene->camera->ScreenLeft = -1;
		scene->camera->ScreenRight = 1;
		scene->camera->ScreenBottom = -1. / scene->image->FrameAspectRatio;
		scene->camera->ScreenTop = 1. / scene->image->FrameAspectRatio;
	}
}

RtVoid RiFrameAspectRatio(RtFloat aspect)
{
	VERIFY_STATE(STATE_BEGIN, "RiFrameAspectRatio");
	scene->image->FrameAspectRatio = aspect;
}

RtVoid RiScreenWindow(RtFloat left, RtFloat right, RtFloat bottom,
					  RtFloat top)
{
	VERIFY_STATE(STATE_BEGIN, "RiScreenWindow");
	if (left > right)
		swap(left, right);
	if (bottom > top)
		swap(bottom, top);
	scene->camera->ScreenLeft = left;
	scene->camera->ScreenRight = right;
	scene->camera->ScreenBottom = bottom;
	scene->camera->ScreenTop = top;
}

RtVoid RiCropWindow(RtFloat left, RtFloat right, RtFloat bottom,
					RtFloat top)
{
	VERIFY_STATE(STATE_BEGIN, "RiCropWindow");
	if (left > right)
		swap(left, right);
	if (bottom > top)
		swap(bottom, top);
	scene->image->CropLeft = left;
	scene->image->CropRight = right;
	scene->image->CropBottom = bottom;
	scene->image->CropTop = top;
}

RtVoid RiClipping(RtFloat hither, RtFloat yon)
{
	VERIFY_STATE(STATE_BEGIN, "RiClipping");
	if (hither < RI_EPSILON) {
		Warning("RiClipping: hither value of %f is too low", hither);
		hither = RI_EPSILON;
	}
	if (yon <= hither) {
		Warning("RiClipping: yon value must be greater than hither");
		yon = RI_INFINITY;
	}
	scene->camera->ClipHither = hither;
	scene->camera->ClipYon = yon;
}

RtVoid RiShutter(RtFloat time0, RtFloat time1)
{
	VERIFY_STATE(STATE_BEGIN, "RiShutter");
	scene->camera->ShutterStart = time0;
	scene->camera->ShutterEnd = time1;
}

RtVoid RiDepthOfField(RtFloat fstop, RtFloat focallen, RtFloat focaldist)
{
	VERIFY_STATE(STATE_BEGIN, "RiDepthOfField");
	scene->camera->FStop = fstop;
	scene->camera->FocalLength = focallen;
	scene->camera->FocalDistance = focaldist;
}

RtVoid RiProjection(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RiProjectionV(name, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiProjectionV(RtToken name, RtInt nArgs, RtToken tokens[],
					 RtPointer params[])
{
	VERIFY_STATE(STATE_BEGIN, "RiProjection");

	if (name == RI_ORTHOGRAPHIC || strcmp(name, RI_ORTHOGRAPHIC) == 0) {

		scene->camera->CameraToScreen = curTransform[0] *
			Orthographic(scene->camera->ClipHither,
						 scene->camera->ClipYon);
		scene->camera->ProjectionType = Camera::Orthographic;
		reportUnusedParameters("RiProjection", nArgs, tokens, RI_NULL);

	} else if (name == RI_PERSPECTIVE || strcmp(name, RI_PERSPECTIVE) == 0) {

		Float fov = InitializeFloat(RI_FOV, 90., nArgs, tokens, params);
		scene->camera->CameraToScreen = curTransform[0] *
			Perspective(fov, 1. / scene->image->PixelAspectRatio,
						scene->camera->ClipHither,
						scene->camera->ClipYon) * Scale(-1, -1, 1);
		scene->camera->ProjectionType = Camera::Perspective;
		reportUnusedParameters("RiProjection", nArgs, tokens, RI_FOV,
							   RI_NULL);

	} else if (name == RI_NULL) {

		scene->camera->CameraToScreen = curTransform[0];
		scene->camera->ProjectionType = Camera::Perspective;	// guess
		reportUnusedParameters("RiProjection", nArgs, tokens, RI_NULL);

	} else
		Error("Unknown projection %s specified in RiProjection()", name);

	curTransform[0] = Transform();
}

RtVoid RiExposure(RtFloat gain, RtFloat gamma)
{
	scene->image->Gain = gain;
	scene->image->Gamma = gamma;
}

RtVoid RiPixelFilter(RtFilterFunc filter, RtFloat xwidth, RtFloat ywidth)
{
	scene->sampler->Filter = filter;
	scene->sampler->FilterXWidth = xwidth;
	scene->sampler->FilterYWidth = ywidth;
}

RtFloat RiBoxFilter(RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth)
{
	return BoxFilter(x, y, xwidth, ywidth);
}

RtFloat RiTriangleFilter(RtFloat x, RtFloat y, RtFloat xwidth,
						 RtFloat ywidth)
{
	return TriangleFilter(x, y, xwidth, ywidth);
}

RtFloat RiGaussianFilter(RtFloat x, RtFloat y, RtFloat xwidth,
						 RtFloat ywidth)
{
	return GaussianFilter(x, y, xwidth, ywidth);
}

RtFloat RiMitchellFilter(RtFloat x, RtFloat y, RtFloat xwidth,
						 RtFloat ywidth)
{
	return MitchellFilter(x, y, xwidth, ywidth);
}

RtFloat RiCatmullRomFilter(RtFloat x, RtFloat y, RtFloat xwidth,
						   RtFloat ywidth)
{
	return GaussianFilter(x, y, xwidth, ywidth);
}

RtFloat RiSincFilter(RtFloat x, RtFloat y, RtFloat xwidth, RtFloat ywidth)
{
	return GaussianFilter(x, y, xwidth, ywidth);
}

RtVoid RiImager(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RiImagerV(name, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiImagerV(RtToken name, RtInt n, RtToken tokens[],
				 RtPointer params[])
{
	VERIFY_STATE(STATE_BEGIN, "RiImager");
	scene->image->Imager = name;
	reportUnusedParameters("RiImager", n, tokens, RI_NULL);
}

RtVoid RiQuantize(RtToken type, RtInt one, RtInt minimum, RtInt maximum,
				  RtFloat ditheramp)
{
	if (type == RI_RGBA || strcmp(type, RI_RGBA) == 0) {
		scene->image->ColorQuantOne = one;
		scene->image->ColorQuantMin = minimum;
		scene->image->ColorQuantMax = maximum;
		scene->image->ColorQuantDither = ditheramp;
	} else if (type == RI_Z || strcmp(type, RI_Z) == 0) {
		scene->image->DepthQuantOne = one;
		scene->image->DepthQuantMin = minimum;
		scene->image->DepthQuantMax = maximum;
		scene->image->DepthQuantDither = ditheramp;
	} else
		Error("Unknown type %s passed to RiQuantize()", type);
}

RtVoid RiDisplay(char *name, RtToken type, RtToken mode, ...)
{
	va_list args;
	va_start(args, mode);
	vectorizeParameters(args);
	RiDisplayV(name, type, mode, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiDisplayV(char *name, RtToken type, RtToken mode, int nArgs,
				  RtToken tokens[], RtPointer parameters[])
{

	if (type != RI_FILE && strcmp(type, RI_FILE) != 0) {
		Error("Display type %s not supported.", type);
		return;
	}
	RtToken displayMode = lookupToken(mode);
	if (displayMode == RI_NULL) {
		Error("Display mode %s unknown.", mode);
		return;
	}

	if (displayMode != RI_RGB && displayMode != RI_RGBA &&
		displayMode != RI_RGBZ && displayMode != RI_RGBAZ &&
		displayMode != RI_A && displayMode != RI_AZ && displayMode != RI_Z) {
		Error("Display mode %s not supported.", mode);
		return;
	}

	scene->image->DisplayType = type;
	scene->image->DisplayName = Strdup(name);
	scene->image->DisplayMode = displayMode;

	reportUnusedParameters("RiDisplay", nArgs, tokens, RI_NULL);

}

RtVoid RiHider(RtToken type, ...)
{
	va_list args;
	va_start(args, type);
	vectorizeParameters(args);
	RiHiderV(type, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiHiderV(RtToken type, RtInt nArgs, RtToken tokens[],
				RtPointer parms[])
{
	if (type == RI_HIDDEN || strcmp(type, RI_HIDDEN) == 0 ||
		type == RI_NULL) {
		for (int i = 0; i < nArgs; ++i) {
			if (tokens[i] == LRT_JITTER ||
				strcmp(tokens[i], LRT_JITTER) == 0) {
				Float on = *((RtFloat *) (parms[i]));
				scene->sampler->JitterSamples = on;
			}
		}
		reportUnusedParameters("RiHider", nArgs, tokens, LRT_JITTER,
							   RI_NULL);
	} else {
		if (type == RI_PAINT || strcmp(type, RI_PAINT) == 0)
			Error("Paint hider not supported by RiHider()");
		else
			Error("Unknown type %s passed to RiHider()", type);
		reportUnusedParameters("RiHider", nArgs, tokens, RI_NULL);
	}
}

RtVoid RiOption(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RiOptionV(name, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiOptionV(RtToken name, RtInt n, RtToken tokens[],
				 RtPointer parms[])
{
	if (name == LRT_DISPLAY || strcmp(name, LRT_DISPLAY) == 0) {
		for (int i = 0; i < n; i++) {
			if (tokens[i] == LRT_IMAGEVIEWER ||
				strcmp(tokens[i], LRT_IMAGEVIEWER) == 0) {
				char **arr = (char **) parms[i];
				scene->image->ImageViewer = Strdup(arr[0]);
			} else
				Warning("Ignoring unknown option 'display' token %s",
						tokens[i]);
		}
	} else if (name == LRT_RENDER || strcmp(name, LRT_RENDER) == 0) {

		for (int i = 0; i < n; ++i) {
			if (tokens[i] == LRT_INTEGRATOR ||
				strcmp(tokens[i], LRT_INTEGRATOR) == 0) {
				RtToken param = *((RtToken *) (parms[i]));
				if (strcmp(param, LRT_COLOR) == 0)
					scene->IlluminationIntegrator = LRT_COLOR;
				else if (strcmp(param, LRT_RAYCAST) == 0)
					scene->IlluminationIntegrator = LRT_RAYCAST;
				else if (strcmp(param, LRT_WHITTED) == 0)
					scene->IlluminationIntegrator = LRT_WHITTED;
				else if (strcmp(param, LRT_MONTECARLO) == 0)
					scene->IlluminationIntegrator = LRT_MONTECARLO;
				else if (strcmp(param, LRT_RMAN) == 0)
					scene->IlluminationIntegrator = LRT_RMAN;
				else
					Error
						("Unknown integrator type \'%s\' passed to RiOption",
						 param);
			} else
				Error("Unknown rendering option \"%s\" passed to RiOption",
					  tokens[i]);
		}

	} else if (name == RI_CAMERA || strcmp(name, RI_CAMERA) == 0) {

		for (int i = 0; i < n; ++i) {
			if (tokens[i] == LRT_TYPE || strcmp(tokens[i], LRT_TYPE) == 0) {
				RtToken param = *((RtToken *) (parms[i]));
				if (strcmp(param, LRT_PINHOLE) == 0) {
					delete scene->camera;
					scene->camera = new PinholeCamera;
				} else if (strcmp(param, LRT_MBDOF) == 0) {
					delete scene->camera;
					scene->camera = new MbDOFCamera;
				} else
					Error("Unknown camera type \'%s\' passed to RiOption",
						  param);
			} else if (tokens[i] == LRT_SHUTTER ||
					   strcmp(tokens[i], LRT_SHUTTER) == 0) {
				RtToken param = *((RtToken *) (parms[i]));
				if (strcmp(param, LRT_IRIS) == 0)
					scene->camera->ShutterType = LRT_IRIS;
				else if (strcmp(param, LRT_STRIPE) == 0)
					scene->camera->ShutterType = LRT_STRIPE;
				else
					Error("Unknown shutter type \'%s\' passed to RiOption",
						  param);
			} else if (tokens[i] == LRT_IRIS_RATE ||
					   strcmp(tokens[i], LRT_IRIS_RATE) == 0) {
				Float param = *((Float *) (parms[i]));
				scene->camera->IrisRate = param;
			}
				else if (tokens[i] == LRT_STRIPE_WIDTH ||
						 strcmp(tokens[i], LRT_STRIPE_WIDTH) == 0) {
				Float param = *((Float *) (parms[i]));
				scene->camera->StripeWidth = param;
			}
				else if (tokens[i] == LRT_STRIPE_DIRECTION ||
						 strcmp(tokens[i], LRT_STRIPE_DIRECTION) == 0) {
				RtToken param = *((RtToken *) (parms[i]));
				if (strcmp(param, LRT_DOWN) == 0)
					scene->camera->StripeDirection = LRT_DOWN;
				if (strcmp(param, LRT_UP) == 0)
					scene->camera->StripeDirection = LRT_UP;
				if (strcmp(param, LRT_LEFT) == 0)
					scene->camera->StripeDirection = LRT_LEFT;
				if (strcmp(param, LRT_RIGHT) == 0)
					scene->camera->StripeDirection = LRT_RIGHT;
				else
					Error("Unknown shutter type \'%s\' passed to RiOption",
						  param);
			} else
				Error("Unknown rendering option \"%s\" passed to RiOption",
					  tokens[i]);
		}

	} else
		Warning("RiOption: unknown option class %s", name);
}

RtVoid RiTransformBegin()
{
	for (int i = 0; i < MOTION_LEVELS; ++i)
		transformStack[i].push_back(curTransform[i]);
	hierarchicalState.push_back('t');
}

RtVoid RiTransformEnd()
{
	if (!transformStack[0].size() || hierarchicalState.back() != 't') {
		Error("Unmatched RiTransformEnd encountered.  Ignoring it.");
		return;
	}
	for (int i = 0; i < MOTION_LEVELS; ++i) {
		curTransform[i] = transformStack[i].back();
		transformStack[i].pop_back();
	}
	hierarchicalState.pop_back();
}

RtVoid RiAttributeBegin()
{
	RiTransformBegin();
	VERIFY_STATE(STATE_WORLD_BEGIN, "RiAttributeBegin");

	geomAttrStack.push_back(curPrimitiveAttributes);
	curPrimitiveAttributes =
		new PrimitiveAttributes(*curPrimitiveAttributes);
	lightAttrStack.push_back(curLightAttributes);
	curLightAttributes = new LightAttributes(*curLightAttributes);
	surfaceAttrStack.push_back(curMaterialAttributes);
	curMaterialAttributes = new MaterialAttributes(*curMaterialAttributes);
	lightStack.push_back(lights);

	hierarchicalState.push_back('a');
}

RtVoid RiAttributeEnd()
{
	VERIFY_STATE(STATE_WORLD_BEGIN, "RiAttributeEnd");
	if (!geomAttrStack.size() || hierarchicalState.back() != 'a') {
		Error("Unmatched RiAttributeEnd encountered.  Ignoring it.");
		return;
	}
	curPrimitiveAttributes->Dereference();
	curPrimitiveAttributes = geomAttrStack.back();
	geomAttrStack.pop_back();
	curLightAttributes->Dereference();
	curLightAttributes = lightAttrStack.back();
	lightAttrStack.pop_back();
	curMaterialAttributes->Dereference();
	curMaterialAttributes = surfaceAttrStack.back();
	surfaceAttrStack.pop_back();
	hierarchicalState.pop_back();
	lights = lightStack.back();
	lightStack.pop_back();
	RiTransformEnd();
}

RtVoid RiColor(RtColor Cs)
{

	if (curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}

	curPrimitiveAttributes->Color = Spectrum(Cs[0], Cs[1], Cs[2]);
}

RtVoid RiOpacity(RtColor Cs)
{

	if (curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}

	curPrimitiveAttributes->Opacity = Spectrum(Cs[0], Cs[1], Cs[2]);
}

RtVoid RiOrientation(RtToken orientation)
{
	RI_UNIMP();
}

RtVoid RiReverseOrientation()
{
	RI_UNIMP();
}

RtVoid RiSides(RtInt sides)
{
	RI_UNIMP();
}

RtVoid RiAttribute(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RiAttributeV(name, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiAttributeV(RtToken name, RtInt n, RtToken tokens[],
					RtPointer parms[])
{
	if (name == LRT_LIGHT || strcmp(name, LRT_LIGHT) == 0) {

		for (int i = 0; i < n; ++i) {
			if (tokens[i] == LRT_SHADOWS ||
				strcmp(tokens[i], LRT_SHADOWS) == 0) {
				const char *param = *((const char **) (parms[i]));
				if (strcmp(param, "on") == 0) {

					if (curLightAttributes->GetReferences() > 1) {
						LightAttributes *newAttr =
							new LightAttributes(*curLightAttributes);
						curLightAttributes->Dereference();
						curLightAttributes = newAttr;
					}

					curLightAttributes->CastsShadows = true;
				} else if (strcmp(param, "off") == 0) {

					if (curLightAttributes->GetReferences() > 1) {
						LightAttributes *newAttr =
							new LightAttributes(*curLightAttributes);
						curLightAttributes->Dereference();
						curLightAttributes = newAttr;
					}

					curLightAttributes->CastsShadows = false;
				} else
					Warning
						("Unknown on/off value for attribute shadows %s",
						 param);
			} else if (tokens[i] == LRT_NSAMPLES ||
					   strcmp(tokens[i], LRT_NSAMPLES) == 0) {
				Float count = *((RtFloat *) (parms[i]));
				if (count < 1.)
					Warning("Ignoring < 1 number of light samples %f",
							count);
				else {

					if (curLightAttributes->GetReferences() > 1) {
						LightAttributes *newAttr =
							new LightAttributes(*curLightAttributes);
						curLightAttributes->Dereference();
						curLightAttributes = newAttr;
					}

					curLightAttributes->NSamples = int (count);
				}
			} else
				Error("Ignoring unknown light attribute %s", tokens[i]);
		}

	} else if (name == LRT_RENDER || strcmp(name, LRT_RENDER) == 0) {

		for (int i = 0; i < n; ++i) {
			if (tokens[i] == LRT_SAMPLE ||
				strcmp(tokens[i], LRT_SAMPLE) == 0) {
				const char *param = *((const char **) (parms[i]));

				if (curPrimitiveAttributes->GetReferences() > 1) {
					PrimitiveAttributes *newAttr =
						new PrimitiveAttributes(*curPrimitiveAttributes);
					curPrimitiveAttributes->Dereference();
					curPrimitiveAttributes = newAttr;
				}

				if (strcmp(param, LRT_SURFACE) == 0)
					curPrimitiveAttributes->Sampling =
						PrimitiveAttributes::SampleSurface;
				else if (strcmp(param, LRT_LIGHT) == 0)
					curPrimitiveAttributes->Sampling =
						PrimitiveAttributes::SampleLight;
				else if (strcmp(param, LRT_COMBINATION) == 0)
					curPrimitiveAttributes->Sampling =
						PrimitiveAttributes::SampleCombination;
				else
					Warning
						("Unknown 'sample' value, '%s' for attribute render",
						 param);
			} else
				Error("Ignoring unknown rendering attribute '%s'",
					  tokens[i]);
		}

	} else
		Warning("Ignoring unknown attribute %s", name);
}

RtVoid RiSurface(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RiSurfaceV(name, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiSurfaceV(RtToken name, RtInt n, RtToken tokens[],
				  RtPointer params[])
{

	if (curTransform[0] != curMaterialAttributes->WorldToSurface &&
		curMaterialAttributes->GetReferences() > 1) {
		MaterialAttributes *newAttr =
			new MaterialAttributes(*curMaterialAttributes);
		curMaterialAttributes->Dereference();
		curMaterialAttributes = newAttr;
	}
	curMaterialAttributes->SurfaceToWorld = curTransform[0];
	curMaterialAttributes->WorldToSurface =
		Transform(curTransform[0].GetInverse());
	curMaterialAttributes->Reference();

	SurfaceFunction *surfData = initSurfaceFunction(0, n, tokens, params);

	if (curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}

	curPrimitiveAttributes->Surface = MaterialCreate(name, surfData);
}

RtVoid RiIdentity()
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] = Transform();

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiTransform(RtMatrix tr)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] = Transform(tr[0], tr[4], tr[8], tr[12],
									tr[1], tr[5], tr[9], tr[13],
									tr[2], tr[6], tr[10], tr[14],
									tr[3], tr[7], tr[11], tr[15]);

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiConcatTransform(RtMatrix tr)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] = curTransform[xform] *
		Transform(tr[0], tr[4], tr[8], tr[12],
				  tr[1], tr[5], tr[9], tr[13],
				  tr[2], tr[6], tr[10], tr[14],
				  tr[3], tr[7], tr[11], tr[15]);

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiPerspective(RtFloat fov)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}
	// XXX?
	curTransform[xform] = curTransform[xform] * Perspective(fov, 1.0,
															scene->camera->
															ClipHither,
															scene->camera->
															ClipYon);

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiTranslate(RtFloat dx, RtFloat dy, RtFloat dz)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] =
		curTransform[xform] * Translate(Vector(dx, dy, dz));

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiRotate(RtFloat angle, RtFloat dx, RtFloat dy, RtFloat dz)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] =
		curTransform[xform] * Rotate(angle, Vector(dx, dy, dz));

	if (inMotionBlock)
		++motionLevel;

}

RtVoid RiScale(RtFloat sx, RtFloat sy, RtFloat sz)
{

	int xform = 0;
	if (inMotionBlock)
		xform = motionLevel;
	if (motionLevel > MOTION_LEVELS) {
		Warning("Only %d motion levels are supported. Ignoring.",
				motionLevel);
		return;
	}

	curTransform[xform] = curTransform[xform] * Scale(sx, sy, sz);

	if (inMotionBlock)
		++motionLevel;

}


RtVoid RiCoordinateSystem(RtToken)
{
	RI_UNIMP();
}

RtVoid RiPolygon(RtInt nverts, ...)
{
	va_list args;
	va_start(args, nverts);
	vectorizeParameters(args);
	RiPolygonV(nverts, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiPolygonV(RtInt nverts, RtInt n, RtToken tokens[],
				  RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	Point *P = NULL;
	for (int i = 0; i < n; ++i) {
		if (tokens[i] == RI_P || strcmp(tokens[i], RI_P) == 0)
			P = (Point *) params[i];
	}

	SurfaceFunction *surffunc =
		initSurfaceFunction(nverts, n, tokens, params);
	RtInt *indices = new RtInt[3 * (nverts - 2)];
	for (int i = 0; i < nverts - 2; i++) {
		indices[3 * i] = 0;
		indices[3 * i + 1] = i + 2;
		indices[3 * i + 2] = i + 1;
	}
	AddPrimitive(new TriangleMesh(nverts - 2, nverts, indices, P,
								  curPrimitiveAttributes, surffunc));

}

RtVoid RiRectangleV(RtInt n, RtToken tokens[], RtPointer params[])
{
  
//    if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
//         lights != curPrimitiveAttributes->Lights) &&
//        curPrimitiveAttributes->GetReferences() > 1) {
//      PrimitiveAttributes *newAttr =
//        new PrimitiveAttributes(*curPrimitiveAttributes);
//      curPrimitiveAttributes->Dereference();
//      curPrimitiveAttributes = newAttr;
//    }
//    curPrimitiveAttributes->ObjectToWorld = curTransform[0];
//    curPrimitiveAttributes->WorldToObject =
//      Transform(curTransform[0].GetInverse());
//    curPrimitiveAttributes->Lights = lights;
//    curPrimitiveAttributes->Reference();
  
//    Point *P = NULL;
//    for (int i = 0; i < n; ++i) {
//      if (tokens[i] == RI_P || strcmp(tokens[i], RI_P) == 0)
//        P = (Point *) params[i];
//    }
  
//    SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);

//    AddPrimitive(new Rectangle(P, curPrimitiveAttributes, surffunc));
}

RtVoid RiPointsPolygons(RtInt npolys, RtInt nvertices[], RtInt vertices[],
						...)
{
	va_list args;
	va_start(args, vertices);
	vectorizeParameters(args);
	RiPointsPolygonsV(npolys, nvertices, vertices,
					  argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiPointsPolygonsV(RtInt npolys, RtInt nvertices[], RtInt vertices[],
						 RtInt n, RtToken tokens[], RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	Point *P = NULL;
	for (int i = 0; i < n; ++i) {
		if (tokens[i] == RI_P || strcmp(tokens[i], RI_P) == 0)
			P = (Point *) params[i];
	}

	int nVertices = 0;
	int nIndices = 0;
	int vIndex = 0;
	for (int i = 0; i < npolys; ++i) {
		nIndices += 3 * (nvertices[i] - 2);
		for (int j = 0; j < nvertices[i]; j++) {
			nVertices = max(nVertices, vertices[vIndex] + 1);
			vIndex++;
		}
	}

	SurfaceFunction *surffunc =
		initSurfaceFunction(nVertices, n, tokens, params);
	RtInt *indices = new RtInt[nIndices];

	int whichIndex = 0;
	vIndex = 0;
	for (int i = 0; i < npolys; ++i) {
		int anchorIndex = vIndex;
		for (int j = 0; j < nvertices[i] - 2; j++) {
			indices[whichIndex++] = vertices[anchorIndex];
			indices[whichIndex++] = vertices[vIndex + 1];
			indices[whichIndex++] = vertices[vIndex + 2];
			vIndex++;
		}
		vIndex += 2;
	}

	AddPrimitive(new TriangleMesh(nIndices / 3, nVertices, indices,
								  P, curPrimitiveAttributes, surffunc));
}

RtVoid RiSphere(RtFloat radius, RtFloat zmin, RtFloat zmax,
				RtFloat thetamax, ...)
{
	va_list args;
	va_start(args, thetamax);
	vectorizeParameters(args);
	RiSphereV(radius, zmin, zmax, thetamax, argsCount, argTokens,
			  argParams);
	va_end(args);
}

RtVoid RiSphereV(RtFloat radius, RtFloat zmin, RtFloat zmax,
				 RtFloat thetaMax, RtInt n, RtToken tokens[],
				 RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new Sphere(radius, zmin, zmax, thetaMax,
							curPrimitiveAttributes, surffunc));

}

RtVoid RiDisk(RtFloat height, RtFloat radius, RtFloat thetamax, ...)
{
	va_list args;
	va_start(args, thetamax);
	vectorizeParameters(args);
	RiDiskV(height, radius, thetamax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiDiskV(RtFloat height, RtFloat radius, RtFloat thetamax,
			   RtInt n, RtToken tokens[], RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new Disk(height, radius, thetamax,
						  curPrimitiveAttributes, surffunc));
}

RtVoid RiNuPatch(RtInt nu, RtInt uorder, RtFloat * uknot,
				 RtFloat umin, RtFloat umax, RtInt nv, RtInt vorder,
				 RtFloat * vknot, RtFloat vmin, RtFloat vmax, ...)
{
	va_list args;
	va_start(args, vmax);
	vectorizeParameters(args);
	RiNuPatchV(nu, uorder, uknot, umin, umax, nv, vorder, vknot, vmin,
			   vmax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiNuPatchV(RtInt nu, RtInt uorder, RtFloat * uknot,
				  RtFloat umin, RtFloat umax, RtInt nv, RtInt vorder,
				  RtFloat * vknot, RtFloat vmin, RtFloat vmax,
				  RtInt n, RtToken tokens[], RtPointer params[])
{
	SurfaceFunction *surffunc =
		initSurfaceFunction(nu * nv, n, tokens, params);

	float *P = NULL;
	bool isHomogeneous = false;
	for (int i = 0; i < n; ++i) {
		if (strcmp(tokens[i], RI_P) == 0) {
			P = (float *) params[i];
			isHomogeneous = false;
			break;
		}
		if (strcmp(tokens[i], RI_PW) == 0) {
			P = (float *) params[i];
			isHomogeneous = true;
			break;
		}
	}
	if (!P) {
		Error("No control points supplied with NURBS patch.");
		return;
	}

	AddPrimitive(new NURBS(nu, uorder, uknot, umin, umax,
						   nv, vorder, vknot, vmin, vmax, P, isHomogeneous,
						   curPrimitiveAttributes, surffunc));
	if (curPrimitiveAttributes->LightShader)
		Warning("Area lights not supported for NURBS");
}


RtVoid RiTrimCurve(RtInt nloops, RtInt * ncurves, RtInt * order,
				   RtFloat * knot, RtFloat * amin, RtFloat * amax,
				   RtInt * n, RtFloat * u, RtFloat * v, RtFloat * w)
{
	RI_UNIMP();
}

RtVoid RiBasis(RtBasis ubasis, RtInt ustep, RtBasis vbasis, RtInt vstep)
{
}


RtVoid RiPatch(RtToken type, ...)
{
	va_list args;
	va_start(args, type);
	vectorizeParameters(args);
	RiPatchV(type, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiPatchV(RtToken type, RtInt n, RtToken tokens[],
				RtPointer params[])
{
	RI_UNIMP();
}


RtVoid RiPatchMesh(RtToken type, RtInt nu, RtToken uwrap,
				   RtInt nv, RtToken vwrap, ...)
{
	va_list args;
	va_start(args, vwrap);
	vectorizeParameters(args);
	RiPatchMeshV(type, nu, uwrap, nv, vwrap, argsCount, argTokens,
				 argParams);
	va_end(args);
}

RtVoid RiPatchMeshV(RtToken type, RtInt nu, RtToken uwrap, RtInt nv,
					RtToken vwrap, RtInt n, RtToken tokens[],
					RtPointer params[])
{
	if (!strcmp(type, "bilinear") && n == 1 &&
		!strcmp(uwrap, "nonperiodic") && !strcmp(vwrap, "nonperiodic") &&
		!strcmp(tokens[0], "Pz")) {

		if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
			 lights != curPrimitiveAttributes->Lights) &&
			curPrimitiveAttributes->GetReferences() > 1) {
			PrimitiveAttributes *newAttr =
				new PrimitiveAttributes(*curPrimitiveAttributes);
			curPrimitiveAttributes->Dereference();
			curPrimitiveAttributes = newAttr;
		}
		curPrimitiveAttributes->ObjectToWorld = curTransform[0];
		curPrimitiveAttributes->WorldToObject =
			Transform(curTransform[0].GetInverse());
		curPrimitiveAttributes->Lights = lights;
		curPrimitiveAttributes->Reference();

		// special case for heightfield primitive
		SurfaceFunction *surffunc =
			initSurfaceFunction(nu * nv, n, tokens, params);
		AddPrimitive(new
					 Heightfield(nu, nv, (float *) params[0],
								 curPrimitiveAttributes, surffunc));
	} else {
		RI_UNIMP();
	}
}

RtVoid RiCone(RtFloat height, RtFloat radius, RtFloat tmax, ...)
{
	va_list args;
	va_start(args, tmax);
	vectorizeParameters(args);
	RiConeV(height, radius, tmax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiConeV(RtFloat height, RtFloat radius, RtFloat tmax,
			   RtInt n, RtToken tokens[], RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new
				 Cone(height, radius, tmax, curPrimitiveAttributes,
					  surffunc));
}

RtVoid RiCylinder(RtFloat radius, RtFloat zmin, RtFloat zmax,
				  RtFloat tmax, ...)
{
	va_list args;
	va_start(args, tmax);
	vectorizeParameters(args);
	RiCylinderV(radius, zmin, zmax, tmax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiCylinderV(RtFloat radius, RtFloat zmin, RtFloat zmax,
				   RtFloat tmax, RtInt n, RtToken tokens[],
				   RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new
				 Cylinder(radius, zmin, zmax, tmax, curPrimitiveAttributes,
						  surffunc));
}

RtVoid RiHyperboloid(RtPoint point1, RtPoint point2, RtFloat tmax, ...)
{
	va_list args;
	va_start(args, tmax);
	vectorizeParameters(args);
	RiHyperboloidV(point1, point2, tmax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiHyperboloidV(RtPoint p1, RtPoint p2, RtFloat tmax,
					  RtInt n, RtToken tokens[], RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new Hyperboloid(Point(p1[0], p1[1], p1[2]),
								 Point(p2[0], p2[1], p2[2]), tmax,
								 curPrimitiveAttributes, surffunc));
}

RtVoid RiParaboloid(RtFloat rmax, RtFloat zmin, RtFloat zmax,
					RtFloat tmax, ...)
{
	va_list args;
	va_start(args, tmax);
	vectorizeParameters(args);
	RiParaboloidV(rmax, zmin, zmax, tmax, argsCount, argTokens, argParams);
	va_end(args);
}

RtVoid RiParaboloidV(RtFloat rmax, RtFloat zmin, RtFloat zmax,
					 RtFloat tmax, RtInt n, RtToken tokens[],
					 RtPointer params[])
{

	if ((curTransform[0] != curPrimitiveAttributes->WorldToObject ||
		 lights != curPrimitiveAttributes->Lights) &&
		curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}
	curPrimitiveAttributes->ObjectToWorld = curTransform[0];
	curPrimitiveAttributes->WorldToObject =
		Transform(curTransform[0].GetInverse());
	curPrimitiveAttributes->Lights = lights;
	curPrimitiveAttributes->Reference();

	SurfaceFunction *surffunc = initSurfaceFunction(4, n, tokens, params);
	AddPrimitive(new
				 Paraboloid(rmax, zmin, zmax, tmax, curPrimitiveAttributes,
							surffunc));
}

RtVoid RiTorus(RtFloat majrad, RtFloat minrad, RtFloat phimin,
			   RtFloat phimax, RtFloat tmax, ...)
{
	RI_UNIMP();
}

RtVoid RiTorusV(RtFloat majrad, RtFloat minrad, RtFloat phimin,
				RtFloat phimax, RtFloat tmax, RtInt n, RtToken tokens[],
				RtPointer params[])
{
	RI_UNIMP();
}

RtVoid RiSolidBegin(RtToken)
{
	RI_UNIMP();
}

RtVoid RiSolidEnd()
{
	RI_UNIMP();
}

RtVoid RiGeneralPolygon(RtInt nloops, RtInt nverts[], ...)
{
	RI_UNIMP();
}

RtVoid RiGeneralPolygonV(RtInt nloops, RtInt nverts[], RtInt n,
						 RtToken tokens[], RtPointer parms[])
{
	RI_UNIMP();
}

RtVoid RiPointsGeneralPolygons(RtInt npolys, RtInt nloops[],
							   RtInt nverts[], RtInt verts[], ...)
{
	RI_UNIMP();
}

RtVoid RiPointsGeneralPolygonsV(RtInt npolys, RtInt nloops[],
								RtInt nverts[], RtInt verts[], RtInt n,
								RtToken tokens[], RtPointer parms[])
{
	RI_UNIMP();
}

RtObjectHandle RiObjectBegin()
{
	RI_UNIMP();
	return RtObjectHandle(NULL);
}

void RiObjectEnd()
{
	RI_UNIMP();
}

RtLightHandle RiLightSource(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RtLightHandle ret =
		RiLightSourceV(name, argsCount, argTokens, argParams);
	va_end(args);
	return ret;
}

RtLightHandle RiLightSourceV(RtToken name, RtInt n, RtToken tokens[],
							 RtPointer params[])
{
	SurfaceFunction *data = initSurfaceFunction(0, n, tokens, params);

	if (curTransform[0] != curLightAttributes->WorldToLight &&
		curLightAttributes->GetReferences() > 1) {
		LightAttributes *newAttr =
			new LightAttributes(*curLightAttributes);
		curLightAttributes->Dereference();
		curLightAttributes = newAttr;
	}
	curLightAttributes->LightToWorld = curTransform[0];
	curLightAttributes->WorldToLight =
		Transform(curTransform[0].GetInverse());
	curLightAttributes->Reference();

	Light *lt;
	lt = PointLightCreate(name, data, curLightAttributes);
	if (lt == RI_NULL) {

		curLightAttributes->Dereference();
		delete data;
		Error("RiLightSource: point light type '%s' unknown.", name);

	} else
		lights.push_back(lt);
	return lt;
}

RtLightHandle RiAreaLightSource(RtToken name, ...)
{
	va_list args;
	va_start(args, name);
	vectorizeParameters(args);
	RtLightHandle ret =
		RiAreaLightSourceV(name, argsCount, argTokens, argParams);
	va_end(args);
	return ret;
}

RtLightHandle RiAreaLightSourceV(RtToken name, RtInt n, RtToken tokens[],
								 RtPointer params[])
{
	SurfaceFunction *data = initSurfaceFunction(0, n, tokens, params);

	if (curTransform[0] != curLightAttributes->WorldToLight &&
		curLightAttributes->GetReferences() > 1) {
		LightAttributes *newAttr =
			new LightAttributes(*curLightAttributes);
		curLightAttributes->Dereference();
		curLightAttributes = newAttr;
	}
	curLightAttributes->LightToWorld = curTransform[0];
	curLightAttributes->WorldToLight =
		Transform(curTransform[0].GetInverse());
	curLightAttributes->Reference();

	AreaLight *lt = AreaLightCreate(name, data, curLightAttributes);

	if (curPrimitiveAttributes->GetReferences() > 1) {
		PrimitiveAttributes *newAttr =
			new PrimitiveAttributes(*curPrimitiveAttributes);
		curPrimitiveAttributes->Dereference();
		curPrimitiveAttributes = newAttr;
	}

	curPrimitiveAttributes->LightShader = lt;
	if (lt == RI_NULL) {

		curLightAttributes->Dereference();
		delete data;
		Error("RiAreaLightSource: area light type '%s' unknown.", name);

	} else
		lights.push_back(lt);
	return lt;
}

RtVoid RiIlluminate(RtLightHandle light, RtBoolean on)
{
	if (on) {

		list < Light * >::iterator iter = lights.begin();
		while (iter != lights.end()) {
			if (*iter == light) {
				Info("Trying to Illuminate light that is already on.");
				return;
			}
			++iter;
		}
		lights.push_front((Light *) light);

	} else {

		list < Light * >::iterator iter = lights.begin();
		while (iter != lights.end()) {
			if (*iter == light) {
				lights.erase(iter);
				return;
			}
			++iter;
		}
		Error("Trying to de-Illuminate an unknown light!");

	}
}

RtVoid RiMotionBegin(RtInt N, ...)
{
	RtFloat times[16];
	Assert(N < 16);
	va_list args;
	va_start(args, N);
	for (int i = 0; i < N; ++i)
		times[i] = va_arg(args, RtFloat);
	RiMotionBeginV(N, times);
}


RtVoid RiMotionBeginV(RtInt N, RtFloat times[])
{
	Assert(!inMotionBlock);
	inMotionBlock = true;
	motionLevel = 0;
	if (N > 2)
		Warning("Only two levels in motion block will be used.");
}

RtVoid RiMotionEnd()
{
	if (!inMotionBlock)
		Error("Unmatched MotionEnd statement");
	inMotionBlock = false;
}

RtVoid RiPixelVariance(RtFloat)
{
}

RtVoid RiTextureCoordinates(RtFloat s1, RtFloat t1, RtFloat s2, RtFloat t2,
							RtFloat s3, RtFloat t3, RtFloat s4, RtFloat t4)
{
	RI_UNIMP();
}

RtVoid RiShadingRate(RtFloat r)
{
}
RtVoid RiShadingInterpolation(RtToken type)
{
}

RtVoid RiRelativeDetail(RtFloat relativedetail)
{
}

RtVoid RiMatte(RtBoolean onoff)
{
	RI_UNIMP();
}

RtVoid RiBound(RtBound bound)
{
	RI_UNIMP();
}

RtVoid RiDetail(RtBound bound)
{
	RI_UNIMP();
}
extern RtVoid RiDetailRange(RtFloat minvis, RtFloat lowtran,
							RtFloat uptran, RtFloat maxvis)
{
	RI_UNIMP();
}

RtVoid RiGeometricApproximation(RtToken type, RtFloat value)
{
}
RtVoid RiGeometricRepresentation(RtToken type)
{
	RI_UNIMP();
}

RtPoint *RiTransformPoints(RtToken fromspace, RtToken tospace,
						   RtInt npoints, RtPoint * points)
{
	RI_UNIMP();
	return (RtPoint *) (NULL);
}

RtVoid RiSkew(RtFloat, RtFloat, RtFloat, RtFloat, RtFloat, RtFloat,
			  RtFloat)
{
	RI_UNIMP();
}

extern RtVoid RiDeformation(RtToken name, ...)
{
	RI_UNIMP();
}
extern RtVoid RiDeformationV(RtToken name, RtInt n, RtToken tokens[],
							 RtPointer parms[])
{
	RI_UNIMP();
}
extern RtVoid RiDisplacement(RtToken name, ...)
{
	RI_UNIMP();
}
extern RtVoid RiDisplacementV(RtToken name, RtInt n, RtToken tokens[],
							  RtPointer parms[])
{
	RI_UNIMP();
}

RtVoid RiGeometry(RtToken type, ...)
{
	RI_UNIMP();
}

RtVoid RiGeometryV(RtToken type, RtInt n, RtToken tokens[],
				   RtPointer params[])
{
	RI_UNIMP();
}

RtVoid RiMakeTexture(char *pic, char *tex, RtToken swrap, RtToken twrap,
					 RtFilterFunc filterfunc, RtFloat swidth,
					 RtFloat twidth, ...)
{
	RI_UNIMP();
}
RtVoid RiMakeTextureV(char *pic, char *tex, RtToken swrap, RtToken twrap,
					  RtFilterFunc filterfunc, RtFloat swidth,
					  RtFloat twidth, RtInt n, RtToken tokens[],
					  RtPointer params[])
{
	RI_UNIMP();
}
RtVoid RiMakeBump(char *pic, char *tex, RtToken swrap, RtToken twrap,
				  RtFilterFunc filterfunc, RtFloat swidth, RtFloat twidth,
				  ...)
{
	RI_UNIMP();
}
RtVoid RiMakeBumpV(char *pic, char *tex, RtToken swrap, RtToken twrap,
				   RtFilterFunc filterfunc, RtFloat swidth, RtFloat twidth,
				   RtInt n, RtToken tokens[], RtPointer params[])
{
	RI_UNIMP();
}
RtVoid RiMakeLatLongEnvironment(char *pic, char *tex,
								RtFilterFunc filterfunc, RtFloat swidth,
								RtFloat twidth, ...)
{
	RI_UNIMP();
}
RtVoid RiMakeLatLongEnvironmentV(char *pic, char *tex,
								 RtFilterFunc filterfunc, RtFloat swidth,
								 RtFloat twidth, RtInt n, RtToken tokens[],
								 RtPointer params[])
{
	RI_UNIMP();
}
RtVoid RiMakeCubeFaceEnvironment(char *px, char *nx, char *py,
								 char *ny, char *pz, char *nz,
								 char *tex, RtFloat fov,
								 RtFilterFunc filterfunc, RtFloat swidth,
								 RtFloat twidth, ...)
{
	RI_UNIMP();
}
RtVoid RiMakeCubeFaceEnvironmentV(char *px, char *nx, char *py,
								  char *ny, char *pz, char *nz,
								  char *tex, RtFloat fov,
								  RtFilterFunc filterfunc, RtFloat swidth,
								  RtFloat twidth, RtInt n,
								  RtToken tokens[], RtPointer params[])
{
	RI_UNIMP();
}
RtVoid RiMakeShadow(char *pic, char *tex, ...)
{
	RI_UNIMP();
}
RtVoid RiMakeShadowV(char *pic, char *tex,
					 RtInt n, RtToken tokens[], RtPointer params[])
{
	RI_UNIMP();
}

extern RtVoid RiColorSamples(RtInt N, RtFloat * nRGB, RtFloat * RGBn)
{
	RI_UNIMP();
}

RtVoid RiAtmosphere(RtToken name, ...)
{
	RI_UNIMP();
}
extern RtVoid RiAtmosphereV(RtToken name, RtFloat opacity, RtInt n, RtToken tokens[],
			    RtPointer parms[])
{
  scene->atmosAttributes->type = Strdup( name );
  scene->atmosAttributes->opacity = opacity;
}
extern RtVoid RiInterior(RtToken name, ...)
{
	RI_UNIMP();
}
extern RtVoid RiInteriorV(RtToken name, RtInt n, RtToken tokens[],
						  RtPointer parms[])
{
	RI_UNIMP();
}
extern RtVoid RiExterior(RtToken name, ...)
{
	RI_UNIMP();
}
extern RtVoid RiExteriorV(RtToken name, RtInt n, RtToken tokens[],
						  RtPointer parms[])
{
	RI_UNIMP();
}
