
#include "shading.h"
#include "color.h"
#include "reflection.h"
#include "texture.h"

ShadeContext::ShadeContext(const HitInfo * hi, const Vector & w)
:wo(w.Hat())
{
	hitInfo = hi;

	Ng = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->NgObj);
	Ng.Normalize();
	if (Dot(Ng, wo) < 0.) {
		Ng = -Ng;
		entering = false;
	} else {
		entering = true;
	}

	const Normal *Ni = hitInfo->GetNormal(RI_N);
	if (Ni != NULL) {
		Ns = hitInfo->hitPrim->attributes->ObjectToWorld(*Ni).Hat();
		if (Dot(Ns, wo) < 0.)
			Ns = -Ns;
	} else {
		Ns = Ng;
	}

}

Float ShadeContext::InitializeFloat(RtToken token, Float def) const
{
	const Float *ret = hitInfo->GetFloat(token);
	if (ret)
		return *ret;
	if (token == RI_S)
		return hitInfo->u;
	else if (token == RI_T)
		return hitInfo->v;
	return def;
}

Spectrum ShadeContext::InitializeColor(RtToken token, const Spectrum & def) const
{
	const Spectrum *ret = hitInfo->GetColor(token);
	if (ret)
		return *ret;
	if (token == RI_CS)
		return hitInfo->hitPrim->attributes->Color;
	else if (token == RI_OS)
		return hitInfo->hitPrim->attributes->Opacity;
	return def;
}

Point ShadeContext::InitializePoint(RtToken token, const Point & def) const
{
	const Point *ret = hitInfo->GetPoint(token);
	if (ret)
		return *ret;
	else
		return def;
}

const char *ShadeContext::InitializeString(RtToken token, const char *def) const
{
	const char *ret = hitInfo->GetString(token);
	if (ret)
		return ret;
	else
		return def;
}

Vector ShadeContext::InitializeVector(RtToken token, const Vector & def) const
{
	const Vector *ret = hitInfo->GetVector(token);
	if (ret)
		return *ret;
	else
		return def;
}

Normal ShadeContext::InitializeNormal(RtToken token, const Normal & def) const
{
	const Normal *ret = hitInfo->GetNormal(token);
	if (ret)
		return *ret;
	else
		return def;
}

StringHashTable ShadeContext::textures;


TextureMap *ShadeContext::GetTexture(const char *filename)
{
	TextureMap *ret = (TextureMap *) textures.Search(filename);
	if (!ret) {
		ret = new TextureMap(filename);
		textures.Add(filename, ret);
	}
	return ret;
}

const Primitive *ShadeContext::GetPrimitive() const
{
  return hitInfo->hitPrim;
}

Material::~Material()
{
	delete surfaceFunction;
}

BRDF *MatteSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	Float surfaceKd = sc.InitializeFloat(RI_KD, Kd);
	return CreateLambertian(surfaceKd * Cs, sc.Ns);
}

BRDF *PlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	return PlasticShade(sc, Cs);
}

BRDF *PlasticSurface::PlasticShade(const ShadeContext & sc,
								   const Spectrum & Cs) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Float surfKs = sc.InitializeFloat(RI_KS, Ks);
	Float surfRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum surfSpecColor =
		sc.InitializeColor(RI_SPECULARCOLOR, SpecularColor);

	ScatteringMixture *mix = CreateScatteringMixture();
	mix->AddFunction(CreateLambertian(surfKd * Cs, sc.Ns));
	mix->AddFunction(CreateBlinnGlossy(surfKs * surfSpecColor,
									   Clamp(surfRoughness, 0., 1.), sc.Ns,
									   sc.wo));
	return mix;

}

PaintedPlasticSurface::PaintedPlasticSurface(SurfaceFunction *data)
		: PlasticSurface(data) {
	texName = surfaceFunction->InitializeString(RI_TEXTURENAME,
								NULL);
}

BRDF *PaintedPlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum surfColor = sc.InitializeColor(RI_CS, Spectrum(1.));
	const char *surfTexName = sc.InitializeString(RI_TEXTURENAME, texName);
	TextureMap *tmap = sc.GetTexture(surfTexName);

	if (tmap) {
		Float s = sc.InitializeFloat(RI_S, 0.);
		Float t = sc.InitializeFloat(RI_T, 0.);
		surfColor *= tmap->Lookup(s, t);
	}
	return PlasticShade(sc, surfColor);
}

BRDF *CheckeredSurface::Shade(const ShadeContext & sc) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Spectrum surfCColor1 = sc.InitializeColor(LRT_CHECKCOLOR1,
											  CheckColor1);
	Spectrum surfCColor2 = sc.InitializeColor(LRT_CHECKCOLOR2,
											  CheckColor2);
	Float freq = sc.InitializeFloat(LRT_CHECKFREQUENCY,
									CheckFrequency);

	Point Pshade = sc.InitializePoint(RI_P, Point(0, 0, 0));
	Point Pcheck(freq * Pshade.x, freq * Pshade.y, freq * Pshade.z);
	if (Mod(Round(Pcheck.x) + Round(Pcheck.y) + Round(Pcheck.z), 2) == 0) {
		return CreateLambertian(surfKd * surfCColor1, sc.Ns);
	} else {
		return CreateLambertian(surfKd * surfCColor2, sc.Ns);
	}

}

BRDF *GlassSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeKt = sc.InitializeFloat(LRT_KT, Kt);
	Float shadeIndex = sc.InitializeFloat(LRT_INDEX, Index);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(0.));

	if (shadeKr == 0. && shadeKt == 0.)
		return NULL;

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(Cs,
												  sc.Ns, sc.wo), shadeKr);
	}
	if (shadeKt > 0.) {

		if (sc.entering) {
			mix->AddFunction(CreateSpecularTransmission(Cs,
														sc.Ns, sc.wo,
														1.0, shadeIndex), Kt);
		} else {
			mix->AddFunction(CreateSpecularTransmission(Cs,
														sc.Ns, sc.wo,
														shadeIndex, 1.0), Kt);
		}

	}
	return mix;
}

BRDF *ShinyMetalSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKs = sc.InitializeFloat(RI_KS, Ks);
	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKs > 0.) {
		mix->AddFunction(CreateBlinnGlossy(shadeKs * Cs,
										   shadeRoughness, sc.Ns, sc.wo));
	}
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(shadeKr * Cs,
												  sc.Ns, sc.wo));
	}
	return mix;
}

BRDF *BrushedMetalSurface::Shade(const ShadeContext & sc) const
{
  Float freq = sc.InitializeFloat(LRT_BRUSHFREQUENCY, frequency);
  const char *t = sc.InitializeString(LRT_BRUSHTYPE, typestr);
  int tInt;
  if(strcmp(t,"sphere") == 0)
    tInt = BRUSHED_SPHERE;
  else
    tInt = BRUSHED_DISK;
  Float u = sc.InitializeFloat(RI_S, 0.);
  Float v = sc.InitializeFloat(RI_T, 0.);
  const Primitive *p = sc.GetPrimitive();

  return CreateBrushedMetal(freq, tInt, u, v, sc.Ns, sc.wo, p);
  //ScatteringMixture *mix = CreateScatteringMixture();
  //mix->AddFunction(CreateBrushedMetal(freq, tInt, u, v, sc.Ns, sc.wo, p));
  //return mix;
}

BRDF *ViewSTSurface::Shade(const ShadeContext & sc) const
{
	Float sp = sc.InitializeFloat(RI_S, 0.);
	Float tp = sc.InitializeFloat(RI_T, 0.);
	return CreateLambertian(Spectrum(sp, tp, 0.), sc.Ns);
}

Material *MaterialCreate(const char *name, SurfaceFunction * bindings)
{
	if (strcmp(name, RI_MATTE) == 0)
		return new MatteSurface(bindings);
	else if (strcmp(name, RI_PLASTIC) == 0)
		return new PlasticSurface(bindings);
	else if (strcmp(name, RI_PAINTEDPLASTIC) == 0)
		return new PaintedPlasticSurface(bindings);
	else if (strcmp(name, RI_SHINYMETAL) == 0)
		return new ShinyMetalSurface(bindings);
	else if (strcmp(name, LRT_BRUSHEDMETAL) == 0)
		return new BrushedMetalSurface(bindings);
	else if (strcmp(name, LRT_VIEWST) == 0)
		return new ViewSTSurface(bindings);
	else if (strcmp(name, LRT_CHECKERED) == 0)
		return new CheckeredSurface(bindings);
	else if (strcmp(name, LRT_GLASS) == 0)
		return new GlassSurface(bindings);
	else {
		Warning("Unknown surface shader: %s," "returning \"matte\"", name);
		return new MatteSurface(bindings);
	}
}

SurfaceFunction::~SurfaceFunction()
{

	for (u_int i = 0; i < vertexPoints.size(); ++i)
		delete[]vertexPoints[i].second;

	for (u_int i = 0; i < uniformStrings.size(); ++i)
		delete uniformStrings[i].second;

	for (u_int i = 0; i < vertexFloats.size(); ++i)
		delete[]vertexFloats[i].second;
	for (u_int i = 0; i < vertexHPoints.size(); ++i)
		delete[]vertexHPoints[i].second;
	for (u_int i = 0; i < vertexVectors.size(); ++i)
		delete[]vertexVectors[i].second;
	for (u_int i = 0; i < vertexNormals.size(); ++i)
		delete[]vertexNormals[i].second;
	for (u_int i = 0; i < vertexColors.size(); ++i)
		delete[]vertexColors[i].second;

}

SurfaceFunction::SurfaceFunction(const SurfaceFunction & sf)
{
	nVertex = sf.nVertex;

	vector<pair<RtToken, Point> >::const_iterator uniformPointIter;
	for (uniformPointIter = sf.uniformPoints.begin();
		 uniformPointIter != sf.uniformPoints.end(); uniformPointIter++) {
		uniformPoints.push_back(make_pair((*uniformPointIter).first,
										  (*uniformPointIter).second));
	}

	vector<pair<RtToken, Point *> >::const_iterator vertexPointIter;
	for (vertexPointIter = sf.vertexPoints.begin();
		 vertexPointIter != sf.vertexPoints.end(); vertexPointIter++) {
		Point *values = new Point[nVertex];
		memcpy(values,
			   (*vertexPointIter).second, nVertex * sizeof(*values));
		vertexPoints.
			push_back(make_pair((*vertexPointIter).first, values));
	}

	vector<pair<RtToken, Float> >::const_iterator uniformFloatIter;
	for (uniformFloatIter = sf.uniformFloats.begin();
		 uniformFloatIter != sf.uniformFloats.end(); uniformFloatIter++) {
		uniformFloats.push_back(make_pair((*uniformFloatIter).first,
										  (*uniformFloatIter).second));
	}
	vector<pair<RtToken, Float> >::const_iterator uniformHPointIter;
	for (uniformHPointIter = sf.uniformHPoints.begin();
		 uniformHPointIter != sf.uniformHPoints.end(); uniformHPointIter++) {
		uniformHPoints.push_back(make_pair((*uniformHPointIter).first,
										   (*uniformHPointIter).second));
	}
	vector<pair<RtToken, Vector> >::const_iterator uniformVectorIter;
	for (uniformVectorIter = sf.uniformVectors.begin();
		 uniformVectorIter != sf.uniformVectors.end(); uniformVectorIter++) {
		uniformVectors.push_back(make_pair((*uniformVectorIter).first,
										   (*uniformVectorIter).second));
	}
	vector<pair<RtToken, Normal> >::const_iterator uniformNormalIter;
	for (uniformNormalIter = sf.uniformNormals.begin();
		 uniformNormalIter != sf.uniformNormals.end(); uniformNormalIter++) {
		uniformNormals.push_back(make_pair((*uniformNormalIter).first,
										   (*uniformNormalIter).second));
	}
	vector<pair<RtToken, Spectrum> >::const_iterator uniformColorIter;
	for (uniformColorIter = sf.uniformColors.begin();
		 uniformColorIter != sf.uniformColors.end(); uniformColorIter++) {
		uniformColors.push_back(make_pair((*uniformColorIter).first,
										  (*uniformColorIter).second));
	}
	vector<pair<RtToken, char *> >::const_iterator uniformStringIter;
	for (uniformStringIter = sf.uniformStrings.begin();
		 uniformStringIter != sf.uniformStrings.end(); uniformStringIter++) {
		uniformStrings.push_back(make_pair((*uniformStringIter).first,
										   Strdup((*uniformStringIter).
												  second)));
	}

	vector<pair<RtToken, Float *> >::const_iterator vertexFloatIter;
	for (vertexFloatIter = sf.vertexFloats.begin();
		 vertexFloatIter != sf.vertexFloats.end(); vertexFloatIter++) {
		Float *values = new Float[nVertex];
		memcpy(values, (*vertexFloatIter).second,
			   nVertex * sizeof(*values));
		vertexFloats.
			push_back(make_pair((*vertexFloatIter).first, values));
	}
	vector<pair<RtToken, Float *> >::const_iterator vertexHPointIter;
	for (vertexHPointIter = sf.vertexHPoints.begin();
		 vertexHPointIter != sf.vertexHPoints.end(); vertexHPointIter++) {
		Float *values = new Float[4 * nVertex];
		memcpy(values, (*vertexHPointIter).second,
			   4 * nVertex * sizeof(*values));
		vertexHPoints.
			push_back(make_pair((*vertexHPointIter).first, values));
	}
	vector<pair<RtToken, Vector *> >::const_iterator vertexVectorIter;
	for (vertexVectorIter = sf.vertexVectors.begin();
		 vertexVectorIter != sf.vertexVectors.end(); vertexVectorIter++) {
		Vector *values = new Vector[nVertex];
		memcpy(values, (*vertexVectorIter).second,
			   nVertex * sizeof(*values));
		vertexVectors.
			push_back(make_pair((*vertexVectorIter).first, values));
	}
	vector<pair<RtToken, Normal *> >::const_iterator vertexNormalIter;
	for (vertexNormalIter = sf.vertexNormals.begin();
		 vertexNormalIter != sf.vertexNormals.end(); vertexNormalIter++) {
		Normal *values = new Normal[nVertex];
		memcpy(values, (*vertexNormalIter).second,
			   nVertex * sizeof(*values));
		vertexNormals.
			push_back(make_pair((*vertexNormalIter).first, values));
	}
	vector<pair<RtToken, Spectrum *> >::const_iterator vertexColorIter;
	for (vertexColorIter = sf.vertexColors.begin();
		 vertexColorIter != sf.vertexColors.end(); vertexColorIter++) {
		Spectrum *values = new Spectrum[nVertex];
		memcpy(values, (*vertexColorIter).second,
			   nVertex * sizeof(*values));
		vertexColors.
			push_back(make_pair((*vertexColorIter).first, values));
	}

}

void SurfaceFunction::AddUniformPoint(RtToken name, const Float * data)
{
	uniformPoints.
		push_back(make_pair(name, Point(data[0], data[1], data[2])));
}

void SurfaceFunction::AddVertexPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Point *values = new Point[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexPoints.push_back(make_pair(name, values));
}

void SurfaceFunction::AddUniformFloat(RtToken name, const Float * data)
{
	uniformFloats.push_back(make_pair(name, *data));
}
void SurfaceFunction::AddUniformHPoint(RtToken name, const Float * data)
{
	uniformHPoints.push_back(make_pair(name, data[0]));
	uniformHPoints.push_back(make_pair(name, data[1]));
	uniformHPoints.push_back(make_pair(name, data[2]));
	uniformHPoints.push_back(make_pair(name, data[3]));
}
void SurfaceFunction::AddUniformVector(RtToken name, const Float * data)
{
	uniformVectors.
		push_back(make_pair(name, Vector(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformNormal(RtToken name, const Float * data)
{
	uniformNormals.
		push_back(make_pair(name, Normal(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformColor(RtToken name, const Float * data)
{
	uniformColors.
		push_back(make_pair(name, Spectrum(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformString(RtToken name, const char *data)
{
	uniformStrings.push_back(make_pair(name, Strdup(data)));
}

void SurfaceFunction::AddVertexFloat(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[nVertex];
	memcpy(values, data, nVertex * sizeof(Float));
	vertexFloats.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexHPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[4 * nVertex];
	memcpy(values, data, 4 * nVertex * sizeof(Float));
	vertexHPoints.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexVector(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Vector *values = new Vector[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexVectors.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexNormal(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Normal *values = new Normal[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexNormals.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexColor(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Spectrum *values = new Spectrum[nVertex];
	for (u_int i = 0; i < nVertex; ++i)
		values[i] =
			Spectrum(data[3 * i], data[3 * i + 1], data[3 * i + 2]);
	vertexColors.push_back(make_pair(name, values));
}

const Point *SurfaceFunction::GetUniformPoint(RtToken name) const
{
	for (u_int i = 0; i < uniformPoints.size(); ++i) {
		if (name == uniformPoints[i].first)
			return &uniformPoints[i].second;
	}
	return NULL;
}

const Point *SurfaceFunction::GetVertexPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		if (name == vertexPoints[i].first)
			return vertexPoints[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetUniformFloat(RtToken name) const
{
	for (u_int i = 0; i < uniformFloats.size(); ++i) {
		if (name == uniformFloats[i].first)
			return &uniformFloats[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetUniformVector(RtToken name) const
{
	for (u_int i = 0; i < uniformVectors.size(); ++i) {
		if (name == uniformVectors[i].first)
			return &uniformVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetUniformNormal(RtToken name) const
{
	for (u_int i = 0; i < uniformNormals.size(); ++i) {
		if (name == uniformNormals[i].first)
			return &uniformNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetUniformColor(RtToken name) const
{
	for (u_int i = 0; i < uniformColors.size(); ++i) {
		if (name == uniformColors[i].first)
			return &uniformColors[i].second;
	}
	return NULL;
}
const char *SurfaceFunction::GetUniformString(RtToken name) const
{
	for (u_int i = 0; i < uniformStrings.size(); ++i) {
		if (name == uniformStrings[i].first)
			return uniformStrings[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetVertexFloat(RtToken name) const
{
	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		if (name == vertexFloats[i].first)
			return vertexFloats[i].second;
	}
	return NULL;
}
const Float *SurfaceFunction::GetVertexHPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexHPoints.size(); ++i) {
		if (name == vertexHPoints[i].first)
			return vertexHPoints[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetVertexVector(RtToken name) const
{
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		if (name == vertexVectors[i].first)
			return vertexVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetVertexNormal(RtToken name) const
{
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		if (name == vertexNormals[i].first)
			return vertexNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetVertexColor(RtToken name) const
{
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		if (name == vertexColors[i].first)
			return vertexColors[i].second;
	}
	return NULL;
}

Point SurfaceFunction::InitializePoint(RtToken name, const Point & def) const
{
	const Point *ptr = GetUniformPoint(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Float SurfaceFunction::InitializeFloat(RtToken name, Float def) const
{
	const Float *ptr = GetUniformFloat(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Spectrum SurfaceFunction::InitializeColor(RtToken name,
										  const Spectrum & def) const
{
	const Spectrum *ptr = GetUniformColor(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Vector SurfaceFunction::InitializeVector(RtToken name, const Vector & def) const
{
	const Vector *ptr = GetUniformVector(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Normal SurfaceFunction::InitializeNormal(RtToken name, const Normal & def) const
{
	const Normal *ptr = GetUniformNormal(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}
const char *SurfaceFunction::InitializeString(RtToken name,
											  const char *def) const
{
	const char *ptr = GetUniformString(name);
	if (ptr != NULL)
		return ptr;
	else
		return def;
}

InterpolatedPrimData::InterpolatedPrimData(const SurfaceFunction * pd)
{
	primitiveData = pd;

	if (primitiveData->vertexPoints.size() > 0) {
		int nvp = primitiveData->vertexPoints.size();
		interpVertexPoints = new Point[nvp];
	} else
		interpVertexPoints = NULL;

	if (primitiveData->vertexFloats.size() > 0) {
		interpVertexFloats = new Float[primitiveData->vertexFloats.size()];
	} else {
		interpVertexFloats = NULL;
	}
	if (primitiveData->vertexVectors.size() > 0) {
		int nv = primitiveData->vertexVectors.size();
		interpVertexVectors = new Vector[nv];
	} else {
		interpVertexVectors = NULL;
	}
	if (primitiveData->vertexNormals.size() > 0) {
		int vn = primitiveData->vertexNormals.size();
		interpVertexNormals = new Normal[vn];
	} else {
		interpVertexNormals = NULL;
	}
	if (primitiveData->vertexColors.size() > 0) {
		int nc = primitiveData->vertexColors.size();
		interpVertexColors = new Spectrum[nc];
	} else {
		interpVertexColors = NULL;
	}

}

InterpolatedPrimData::~InterpolatedPrimData()
{

	delete[]interpVertexPoints;

	delete[]interpVertexFloats;
	delete[]interpVertexVectors;
	delete[]interpVertexNormals;
	delete[]interpVertexColors;

}

InterpolatedPrimData *SurfaceFunction::Interpolate(int n,
												   const int *offsets,
												   const Float * weights) const
{
	InterpolatedPrimData *ret = new InterpolatedPrimData(this);

	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		Point value(0, 0, 0);
		Point *pts = vertexPoints[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value.x += weights[j] * pts[o].x;
			value.y += weights[j] * pts[o].y;
			value.z += weights[j] * pts[o].z;
		}
		ret->interpVertexPoints[i] = value;
	}

	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		Float value = 0.;
		Float *f = vertexFloats[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * f[o];
		}
		ret->interpVertexFloats[i] = value;
	}
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		Vector value(0, 0, 0);
		Vector *vec = vertexVectors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * vec[o];
		}
		ret->interpVertexVectors[i] = value;
	}
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		Normal value(0, 0, 0);
		Normal *norm = vertexNormals[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * norm[o];
		}
		ret->interpVertexNormals[i] = value;
	}
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		Spectrum value(0.);
		Spectrum *cp = vertexColors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * cp[o];
		}
		ret->interpVertexColors[i] = value;
	}

	return ret;
}

const Point *InterpolatedPrimData::GetPoint(RtToken token) const
{
	const Point *ret = primitiveData->GetUniformPoint(token);
	if (ret)
		return ret;

	const vector<pair<RtToken, Point *> > &vpp =
	primitiveData->vertexPoints;
	for (u_int i = 0; i < vpp.size(); ++i) {
		if (token == vpp[i].first)
			return &interpVertexPoints[i];
	}
	return NULL;

}

const Float *InterpolatedPrimData::GetFloat(RtToken token) const
{
	const vector<pair<RtToken, Float *> > &vfp = primitiveData->vertexFloats;
	for (u_int i = 0; i < vfp.size(); ++i) {
		if (token == vfp[i].first)
			return &interpVertexFloats[i];
	}
	return NULL;
}
const Vector *InterpolatedPrimData::GetVector(RtToken token) const
{
	const vector<pair<RtToken, Vector *> > &vvp = primitiveData->vertexVectors;
	for (u_int i = 0; i < vvp.size(); ++i) {
		if (token == vvp[i].first)
			return &interpVertexVectors[i];
	}
	return NULL;
}
const Normal *InterpolatedPrimData::GetNormal(RtToken token) const
{
	const vector<pair<RtToken, Normal *> > &vnp = primitiveData->vertexNormals;
	for (u_int i = 0; i < vnp.size(); ++i) {
		if (token == vnp[i].first)
			return &interpVertexNormals[i];
	}
	return NULL;
}
const Spectrum *InterpolatedPrimData::GetColor(RtToken token) const
{
	const vector<pair<RtToken, Spectrum *> > &vcp = primitiveData->vertexColors;
	for (u_int i = 0; i < vcp.size(); ++i) {
		if (token == vcp[i].first)
			return &interpVertexColors[i];
	}
	return NULL;
}
