
#include "lrt.h"
#include "scene.h"
#include "camera.h"
#include "mbdofcamera.h"
#include "primitives.h"
#include "image.h"
#include "transport.h"
#include "accel.h"
#include "sampling.h"
#include "photon.h"
#include "geometry.h"

Scene *scene = NULL;
int Pphotons=10000, Pneighbors=20;
float Pmaxradius=.3, Pintensity=1.0;
int Pmode=0;

const int numSources = 6;
Point source[numSources] = {
  Point(-30, 0, -9), Point(-30, 3.5, -9), Point(-30, 8, -9),
  Point(4, 10, -11), Point(4, 10, -11), Point(4, 10, -11)
};

void Scene::Render()
{
	Integrator *integrator = NULL;
	if (IlluminationIntegrator == LRT_COLOR)
		integrator = new ColorIntegrator;
	else if (IlluminationIntegrator == LRT_RAYCAST)
		integrator = new RayCastingIntegrator;
	else if (IlluminationIntegrator == LRT_WHITTED)
		integrator = new WhittedIntegrator;
	else if (IlluminationIntegrator == LRT_MONTECARLO)
		integrator = new MCIntegrator;
	else if (IlluminationIntegrator == LRT_RMAN)
		integrator = new RendermanIntegrator;
	else {
		Severe("Unknown integrator \'%s\'", IlluminationIntegrator);
		return;
	}

	camera->FinalizeOptions();
	sampler->FinalizeValues();
	image->FinalizeValues();

	if(Pmode>0) {
	  cerr << "Photon mapping:";
	  Float u[2];
	  Spectrum s(1., 1., 1.);
	  for(int j=0; j<numSources; j++) {
	    int nx = int(sqrt(Pphotons)), ny = Pphotons/nx;
	    for(int i=0; i<nx; i++) {
	      for(int k=0; k<ny; k++) {
		if((j*Pphotons+i*nx+k)%10000==1)
		  cerr << '+';
		u[0] = (RandomFloat()+i)/nx;
		u[1] = (RandomFloat()+k)/ny;
		LaunchPhoton(source[j], s, u);
	      }
	    }
	  }
	  cerr << endl << "Using " << numPhotons << " photons in caustic map.\n";
	  CompactPhoton();
	  if(Pmode==1) // preview!
	    DebugPhoton();
	}

	cerr << "Rendering: ";
	Float sample[5];
	while (sampler->GetNextImageSample(sample)) {

		Ray ray;
		if (!camera->GenerateRay(sample, ray))
			continue;

		static int eyeRaysTraced = 0;
		if (eyeRaysTraced == 0)
			StatsRegisterCounter(STATS_BASIC, "Camera", "Eye Rays Traced",
								 &eyeRaysTraced);
		++eyeRaysTraced;
		if (eyeRaysTraced % 10000 == 0)
			cerr << '+';

		HitInfo hitInfo;
		Float alpha, hitDist = INFINITY;
		Spectrum L =
			integrator->Integrate(ray, &hitInfo, &hitDist, &alpha);

		Float screenz = camera->WorldToScreen(ray(hitDist)).z;
		if (screenz > 1.) {
			L = 0.;
			alpha = 0.;
		}
		Point Praster(sample[0], sample[1], screenz);
		image->AddSample(Praster, L, alpha);

	}
	image->Write();
	cerr << endl;
}

bool Scene::Intersect(const Ray & ray, Float mint, Float * maxt,
					  HitInfo * hit) const
{
	if (!accelerator)
		return false;
	return accelerator->IntersectClosest(ray, mint, maxt, hit);
}

Scene::Scene()
{

	IlluminationIntegrator = LRT_WHITTED;

	accelerator = NULL;
	camera = new PinholeCamera;
	image = new Image;
	sampler = new JitterSampler;
}

void Scene::AddPrimitives(const vector < Primitive * >&prim)
{
	primitives = prim;
	if (primitives.size() > 0)
		accelerator = new GridAccelerator(primitives);
}

Scene::~Scene()
{
	delete accelerator;
	delete camera;
	delete sampler;
	delete image;
//  for (u_int i = 0; i < primitives.size(); ++i)
//      delete primitives[i];
}

bool Scene::Unoccluded(const Point & p1, const Point & p2) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Finite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = 1. - tmin;
	if (accelerator->IntersectClosest(Ray(p1, p2 - p1), tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}

bool Scene::Unoccluded(const Ray & r) const
{
	if (!accelerator)
		return true;

	static int shadowRayChecks = 0, shadowRayOccluded = 0;
	if (shadowRayChecks == 0)
		StatsRegisterRatio(STATS_DETAILED, "Integration",
						   "Infinite Shadow Ray Checks",
						   &shadowRayOccluded, &shadowRayChecks);
	++shadowRayChecks;

	Float tmin = 1e-6;
	Float tmax = INFINITY;
	Ray ray = r;
	ray.D.Normalize();
	if (accelerator->IntersectClosest(ray, tmin, &tmax, NULL)) {
		++shadowRayOccluded;
		return false;
	}

	return true;
}
