
#ifndef REFLECTION_H
#define REFLECTION_H
#include "lrt.h"
#include "color.h"
#include "geometry.h"
#include "transform.h"

class BRDF {
  public:

	virtual ~ BRDF();

	virtual Spectrum fr(const Vector & wi) const = 0;

	virtual int SpecularComponents() const;

	virtual int SpecularNoiseSamples() const;

	virtual Spectrum SampleSpecular(Vector *wo) const;

	virtual Spectrum SampleSpecular(int component, Vector * wo) const;

	virtual Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const =
		0;

	virtual Float Pdf(const Vector & wi) const = 0;

	virtual int Photon(Spectrum & power, Vector & dir) const;
	// return 0 = terminate photon without storing
	// return 1 = diffuse: store.
	// return 2 = specular: transmit.
	// return 3 = glossy: store and transmit.
};

class Lambertian:public BRDF { public:
	 Lambertian(const Spectrum & reflectance, const Normal & normal) {
		R = reflectance;
		N = normal;
	} Spectrum fr(const Vector & wi) const;

	Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const;

	Float Pdf(const Vector & wi) const;
  private:

	Spectrum R;
	Normal N;

};

class BlinnGlossy:public BRDF { public:
	 BlinnGlossy(const Spectrum & reflectance, Float roughness,
				 const Normal & normal, const Vector & wo);

	Spectrum fr(const Vector & wi) const;

	Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const;

	Float Pdf(const Vector & wi) const;

  private:
	 Spectrum R;
	Float invRoughness;
	Vector wo;
	Normal N;
	Float costhetao;

};

class BrushedMetal:public BRDF
{
  public:

  BrushedMetal(Float freq, int tInt, Float uval, Float vval,
               const Normal & normal, const Vector & wo,
               const Primitive * p);

  Spectrum fr(const Vector & wi) const;

  int SpecularNoiseSamples() const;

  Spectrum SampleSpecular(int component, Vector *wi) const;
  int SpecularComponents() const;

  Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const {
		*pdf = 0.;
		return Spectrum(0.);
	}

  Float Pdf(const Vector & wi) const {
		return 0.;
  }

  private:

  const Transform objtoworld;
  Vector wo;
  Normal N;
  Vector reflectedDir;
};

class SpecularReflection:public BRDF { public:
	 SpecularReflection(const Spectrum & r,
						const Normal & N, const Vector & wi);

	Spectrum fr(const Vector &) const {
		return Spectrum(0.);
	} int SpecularComponents() const {
		return 1;
	} Spectrum SampleSpecular(int component, Vector * wi) const;

	Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const {
		*pdf = 0.;
		return Spectrum(0.);
	} Float Pdf(const Vector & wi) const {
		return 0.;
	}
	
	int Photon(Spectrum & power, Vector & dir) const;

 private:

	 Spectrum R;
	Vector reflectedDirection;

};

class SpecularTransmission:public BRDF {
  public:

	SpecularTransmission(const Spectrum & r,
						 const Normal & N,
						 const Vector & wo, Float indexi, Float indext);

	Spectrum fr(const Vector &) const {
		return Spectrum(0.);
	} int SpecularComponents() const {
		return 1;
	} Spectrum SampleSpecular(int component, Vector * wo) const;

	Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const {
		*pdf = 0.;
		return Spectrum(0.);
	} Float Pdf(const Vector & wi) const {
		return 0.;
	}
	
	int Photon(Spectrum & power, Vector & dir) const;
 private:

	 Spectrum R;
	Vector DirT;

};

class ScatteringMixture:public BRDF {
  public:

	ScatteringMixture():numSpecular(0), totalWeight(0.) {
	} void AddFunction(BRDF * func, Float weight = 1.0) {
		funcs.push_back(func);
		weights.push_back(weight);
		numSpecular += func->SpecularComponents();
		totalWeight += weight;
	} ~ScatteringMixture();

	int SpecularComponents() const {
		return numSpecular;
	} Spectrum fr(const Vector & wi) const;

	Spectrum SampleSpecular(Vector *wo) const;

	Spectrum SampleSpecular(int component, Vector * wo) const;

	Spectrum Sample(Float u[2], Vector * wi, Float * pdf) const;

	Float Pdf(const Vector & wi) const;

	int Photon(Spectrum & power, Vector & dir) const;

  private:

	vector < BRDF * >funcs;
	vector < Float > weights;
	int numSpecular;
	float totalWeight;
};

#if 0
class Lafortune:public BRDF { public:
	 Lafortune(int nLobes, const LafortuneLobe lobes[], const Vector & wi,
			   const Normal & N, const Vector & dPdu);

	~Lafortune() {
		delete[]lobes;
	} Spectrum fr(const Vector &) const;

  private:

	int nLobes;
	LafortuneLobe *lobes;
	Normal N;
	Vector wo;
	Vector dPdu, dPdv;

};
#endif

#endif // REFLECTION_H
