
#include "lrt.h"
#include "color.h"
#include "reflection.h"
#include "shading.h"
#include "quadrics.h"

BRDF::~BRDF()
{
}

Spectrum Lambertian::fr(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return R;
	} else {
		return Spectrum(0.);
	}
}

BlinnGlossy::BlinnGlossy(const Spectrum & reflectance,
						 Float roughness, const Normal & normal,
						 const Vector & w)
{
	R = reflectance;
	roughness /= 8.;			// hack to match BMRT,
	                                        // prman
	invRoughness = 1. / roughness;
	N = normal;
	wo = w;
	costhetao = Dot(wo, N);
}

Spectrum BlinnGlossy::fr(const Vector & wi) const
{
	Vector H = (wi + wo).Hat();
	Float costhetah = Dot(N, H);
	Float costhetai = Dot(N, wi);
	if (costhetah > 0. && costhetai > 0.) {
	  float p = pow(costhetah, invRoughness);
	  if(p>1)
	    p=1;
	  Spectrum ret = R * p / (costhetai * costhetao);
	  if(isnan(ret.s[0])) {
	    //cerr << ret << " :: " << costhetah << ", " << invRoughness << ','
	    // << pow(costhetah, invRoughness) << ',' << R << endl;
	    ret = Spectrum(0.);
	  }
	  return ret;
	} else {
		return Spectrum(0.);
	}
}

BrushedMetal::BrushedMetal(Float frequency, int type, Float u, Float v,
                           const Normal & normal,
                           const Vector & w,
                           const Primitive * p)
{
  N = normal;
  wo = w;
  Float theta = u*360; // assuming the disk or sphere is complete
  Float phi;
  Normal objN(normal);
  Vector R;
  if(type == BrushedMetalSurface::BRUSHED_DISK) {
    phi = 90*cos(2*M_PI*frequency*v);
    R = p->attributes->ObjectToWorld(Vector(1.,0.,0.));
    R = Rotate(theta,Vector(objN))(R); // rotate R around N by theta
  }
  else { // (type == BrushedMetalSurface::BRUSHED_SPHERE)
    // this had very well better be a sphere - otherwise BOOM!
    R = p->attributes->ObjectToWorld(Vector(0.,0.,1.));
    Float zmin = ((Sphere*)p)->zmin;
    Float zmax = ((Sphere*)p)->zmax;
    Float r = ((Sphere*)p)->radius;
    Float phimin = asin(zmin/r)+M_PI/2;
    Float phimax = asin(zmax/r)+M_PI/2;
    Float vnew = cos(phimin+v*(phimax-phimin));
    phi = 90*cos(2*M_PI*frequency*vnew);
  }

  Vector T = Cross(R,Vector(objN)).Hat();
  objN = Rotate(phi,T)(objN);
  Vector S = Cross(T, Vector(objN)).Hat();
  objN += Normal(0.2*(RandomFloat()*T+RandomFloat()*S));
  objN.Normalize();

  reflectedDir = -wo - 2. * Dot(objN, -wo) * Vector(objN);
}

Spectrum BrushedMetal::fr(const Vector & wi) const
{
  return Spectrum(0.);
}

int BrushedMetal::SpecularNoiseSamples() const
{
  return 1;
}

int BrushedMetal::SpecularComponents() const
{
  return 1;
}

Spectrum BrushedMetal::SampleSpecular(int component, Vector * wi) const
{
  Assert(component == 0);
  *wi = reflectedDir;
  return Spectrum(1.);
}

int BRDF::SpecularNoiseSamples() const
{
  return 1;
}

int BRDF::SpecularComponents() const
{
	return 0;
}

Spectrum BRDF::SampleSpecular(Vector *wo) const
{
  return SampleSpecular(0, wo);
}

Spectrum BRDF::SampleSpecular(int component, Vector * wo) const
{
	Severe("SampleSpecular() shouldn't be called for BRDFs "
		   "without specular components.");
	return Spectrum(0.);
}

int BRDF::Photon(Spectrum & power, Vector & dir) const
{
  return 1;
}

SpecularReflection::SpecularReflection(const Spectrum & r,
				       const Normal & N, const Vector & wo)
{
	R = r;
	reflectedDirection = -wo - 2. * Dot(N, -wo) * Vector(N);
}

int SpecularReflection::Photon(Spectrum & power, Vector & dir) const
{
  //Info("Specular Reflection");
  dir = reflectedDirection;
  power *= R;
  return 2;
}

Spectrum SpecularReflection::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = reflectedDirection;
	return R;
}

SpecularTransmission::SpecularTransmission(const Spectrum & r,
										   const Normal & N,
										   const Vector & wo, Float indexi,
										   Float indext)
{
	R = r;

	Float eta = indexi / indext;
	Float c1 = Dot(wo, N);
	Float c2 = 1 - (eta * eta * (1. - c1));
	if (c2 < 0.) {
		// total internal reflection
		R = 0.;
		DirT = wo;
	} else {
		c2 = sqrt(c2);
		DirT = -eta * wo + (eta * c1 - c2) * Vector(N);
	}

}

int SpecularTransmission::Photon(Spectrum & power, Vector & dir) const
{
  //Info("Specular Transmission");
  dir = DirT;
  power *= R;
  return 2;
}

Spectrum SpecularTransmission::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = DirT;
	return R;
}

ScatteringMixture::~ScatteringMixture()
{
	vector < BRDF * >::const_iterator iter;
	for (iter = funcs.begin(); iter != funcs.end(); ++iter)
		delete *iter;
}

Spectrum ScatteringMixture::fr(const Vector & wi) const
{
	Spectrum ret(0);
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		ret += (*weightIter) * (*funcIter)->fr(wi);
		funcIter++, weightIter++;
	}

	return ret;
}

int ScatteringMixture::Photon(Spectrum & power, Vector & dir) const
{
  float rand = RandomFloat()*totalWeight;
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (rand >= *weightIter)
			rand -= *weightIter;
		else {
		  //power *= *weightIter;
		  return (*funcIter)->Photon(power, dir);
		}
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in" " ScatteringMixture::Photon()");
	return 0;
}

Spectrum ScatteringMixture::SampleSpecular(Vector *wo) const
{
  float rand = RandomFloat()*totalWeight;
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (rand >= *weightIter)
			rand -= *weightIter;
		else {
		  return (*funcIter)->SampleSpecular(wo);
		}
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in" " ScatteringMixture::SampleSpecular()");
	return Spectrum(0.);
}

Spectrum ScatteringMixture::SampleSpecular(int component, Vector * wi) const
{
	Assert(component < SpecularComponents());
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (component >= (*funcIter)->SpecularComponents())
			component -= (*funcIter)->SpecularComponents();
		else
			return (*weightIter) *
				(*funcIter)->SampleSpecular(component, wi);
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in" " ScatteringMixture::SampleSpecular()");
	return Spectrum(0);
}

#if 0
Lafortune::Lafortune(int n, const LafortuneLobe l[], const Vector & w,
					 const Normal & norm, const Vector & dpdu)
{
	nLobes = n;
	lobes = new LafortuneLobe[n];
	for (int i = 0; i < nLobes; ++i)
		lobes[i] = l[i];
	wo = w;
	N = norm.Hat();
	dPdu = dpdu.Hat();
	dPdv = Cross((Vector) N, dPdu);
}
#endif

#if 0
Spectrum Lafortune::fr(const Vector & wi) const
{
	Spectrum value(0.);

	Float xd = Dot(dPdu, wi) * Dot(dPdu, wo);
	Float yd = Dot(dPdv, wi) * Dot(dPdv, wo);
	Float zd = Dot(N, wi) * Dot(N, wo);

	for (int i = 0; i < nLobes; ++i) {

		Spectrum dot =
			xd * lobes[i].Cx + yd * lobes[i].Cy + zd * lobes[i].Cz;
		value += dot.Pow(lobes[i].Exponent);

	}
	return value;
}
#endif

BRDF *CreateLambertian(const Spectrum & reflectance, const Normal & normal)
{
	return new Lambertian(reflectance, normal);
}

BRDF *CreateBlinnGlossy(const Spectrum & reflectance,
						Float roughness,
						const Normal & normal, const Vector & wo)
{
	return new BlinnGlossy(reflectance, roughness, normal, wo);
}

BRDF *CreateBrushedMetal(Float frequency, int type, Float u, Float v,
                         const Normal & normal, const Vector & wo,
                         const Primitive * p)
{
  return new BrushedMetal(frequency, type, u, v, normal, wo, p);
}

BRDF *CreateSpecularReflection(const Spectrum & r,
							   const Normal & N, const Vector & wo)
{
	return new SpecularReflection(r, N, wo);
}

BRDF *CreateSpecularTransmission(const Spectrum & r,
								 const Normal & N,
								 const Vector & wo,
								 Float indexi, Float indext)
{
	return new SpecularTransmission(r, N, wo, indexi, indext);
}

ScatteringMixture *CreateScatteringMixture()
{
	return new ScatteringMixture;
}

#if 0
BRDF *CreateLafortune(int nLobes, const LafortuneLobe lobes[],
					  const Vector & wi, const Normal & N,
					  const Vector & dPdu)
{
	Assert(1 == 0);
	return NULL;
	// need to implement the sampling methods
//  return new Lafortune(nLobes, lobes, wi, N, dPdu);
}
#endif

Spectrum Lambertian::Sample(Float u[2], Vector * wi, Float * pdf) const
{

	Float u0 = 2 * u[0] - 1.;
	Float u1 = 2 * u[1] - 1.;
	if (u0 * u0 + u1 * u1 > 1.) {
		*pdf = 0.;
		return Spectrum(0.);
	}

	Vector Du, Dv;
	if (N.x == 0. && N.y == 0.) {
		Du = Vector(1, 0, 0);
		Dv = Vector(0, 1, 0);
	} else {
		Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
		Dv = Cross(Du, Vector(N)).Hat();
	}

	*wi = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	*pdf = Dot(N, *wi) / M_PI;

	return fr(*wi);
}

Spectrum BlinnGlossy::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	// YOUR CODE HERE

        Vector H;
	Float u0 = 2 * u[0] - 1.;
	Float u1 = 2 * u[1] - 1.;
	if (u0 * u0 + u1 * u1 > 1.) {
		*pdf = 0.;
		return Spectrum(0.);
	}

	// Quick nasty hack ...
	Float rough = 8./invRoughness;
	Clamp(rough, 0., 1.);
	u0*=rough;
	u1*=rough;

	Vector Du, Dv;
	if (N.x == 0. && N.y == 0.) {
		Du = Vector(1, 0, 0);
		Dv = Vector(0, 1, 0);
	} else {
		Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
		Dv = Cross(Du, Vector(N)).Hat();
	}

	H = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	* wi = - wo - 2. * Dot(H, - wo) * Vector(H); // reflected through H

	*pdf = Dot(N, *wi) * Dot(H, N) / M_PI;
	//*pdf = Dot(N, *wi) / (M_PI * (1-cos(asin(rough))));

	return fr(*wi);
}

Spectrum ScatteringMixture::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	int fnum = int (RandomFloat() * funcs.size());
	funcs[fnum]->Sample(u, wi, pdf);
	*pdf /= funcs.size();
	return fr(*wi);
}

Float Lambertian::Pdf(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return 1. / (M_PI * costheta);
	} else {
		return 0.;
	}
}

Float BlinnGlossy::Pdf(const Vector & wi) const
{
	// YOUR CODE HERE
        Vector H = (wi+wo).Hat();
	return Dot(N, wi) * Dot(H, N) / M_PI;
}

Float ScatteringMixture::Pdf(const Vector & wi) const
{
	Float pdfSum = 0.;
	for (u_int i = 0; i < funcs.size(); ++i)
		pdfSum += funcs[i]->Pdf(wi);
	return pdfSum / funcs.size();
}
