#include "stdio.h"
#include "photon.h"
#include "primitives.h"
#include "scene.h"
#include "reflection.h"

int numPhotons=0;
const int gridSize = 50;
const float gridSizeF = 49.99999999999f;
float minx=INFINITY, maxx=-INFINITY, miny=INFINITY, maxy=-INFINITY;
photon map[100000];
int link[100000];
int neighbor[500];

int grid[gridSize][gridSize];
int gridCount[gridSize][gridSize];

void DebugPhoton()
{
  cerr << "Num Photons: " << numPhotons << endl;
  /*
  for(int i=0; i<numPhotons; i++)
    cerr << "Photon " << i << ": " << map[i].x 
	 << " == " << map[i].p << endl;
  */

  cerr << "x: [" << minx << ".." << maxx <<
    "], y: [" << miny << ".." << maxy << "]\n";
  cerr << "gridCount:\n";
  for(int y=gridSize-1; y>=0; y--) {
    for(int x=0; x<gridSize; x++)
      cerr << gridCount[x][y] << " ";
    cerr << endl;
  }
}

void LaunchPhoton(const Point & p, const Spectrum & power, const float *u)
{
  //Vector v = Point(4, 4, 5)-p;
  Vector v = Point(4, -3.65, -11)-p;
  Vector x;
  if(Dot(v, Vector(1,1,0))<.5) {
    x = Cross(v, Vector(1, 1, 0)).Hat();
  } else if(Dot(v, Vector(0,1,1))<.5){
    x = Cross(v, Vector(0, 1, 1)).Hat();
  } else {
    x = Cross(v, Vector(1, 0, 1)).Hat();
  }
  Vector y = Cross(v, x).Hat();
  x *= 2.65*2*(0.5-u[0]);
  y *= 2.65*2*(0.5-u[1]);
  Ray newRay(p, v+x+y);
  Spectrum s(power);
  TracePhoton(s, newRay);
}

void CompactPhoton()
{
  int i, j;
  int x, y;

  // clear grid array
  for(i=0; i<gridSize; i++)
    for(j=0; j<gridSize; j++)
      grid[i][j]=-1;

  // for every photon ...
  for(i=0; i<numPhotons; i++) {
    x=(int)(gridSizeF*(map[i].x.x-minx)/(maxx-minx));
    y=(int)(gridSizeF*(map[i].x.z-miny)/(maxy-miny));
    link[i] = -1;

    // insert photon at head of linked list for its grid square
    gridCount[x][y]++;
    if(grid[x][y]==-1){
      grid[x][y]=i;
    } else {
      link[i]=grid[x][y];
      grid[x][y]=i;
    }
  }
}

void StorePhoton(Point & p, Spectrum & power, const Vector & dir)
{
  map[numPhotons].x = p;
  map[numPhotons].p = power;
  map[numPhotons].d = dir;
  numPhotons++;
  if(p.x<minx)
    minx=p.x;
  if(p.x>maxx)
    maxx=p.x;
  if(p.z<miny)
    miny=p.z;
  if(p.z>maxy)
    maxy=p.z;
}

int NearestPhoton(Point & p, float maxNeighborhood)
{
  float dist, min = INFINITY;
  int minLoc=-1;
  int i, x, y;
  int l, r, t, b;

  l=(int) Clamp(gridSizeF*(p.x-maxNeighborhood-minx)/(maxx-minx),0, gridSizeF);
  r=(int) Clamp(gridSizeF*(p.x+maxNeighborhood-minx)/(maxx-minx),0, gridSizeF);
  t=(int) Clamp(gridSizeF*(p.z-maxNeighborhood-miny)/(maxy-miny),0, gridSizeF);
  b=(int) Clamp(gridSizeF*(p.z+maxNeighborhood-miny)/(maxy-miny),0, gridSizeF);

  for(x=l;x<=r;x++){
    for(y=t;y<=b;y++) {
      // loop over all grid squares in neighborhood

      for(i=grid[x][y]; i>-1; i=link[i]) {
	// test all photons in linked list for current grid square

	dist = Distance(map[i].x, p);
	if(dist < min) {
	  min = dist;
	  minLoc = i;
	}
      }
    }
  }

  return minLoc;
}

int RetrievePhotons(Point & p, float & maxNeighborhood, int numNeighbors)
{
  float dist[100000];
  float min = INFINITY;
  int minLoc=-1;
  float radius=0;
  int i, j, x, y;
  int l, r, t, b;
  int count=0;

  l=int(gridSizeF*(p.x-maxNeighborhood-minx)/(maxx-minx));
  if(l>=gridSize) return 0;
  r=int(gridSizeF*(p.x+maxNeighborhood-minx)/(maxx-minx));
  if(r<0) return 0;
  t=int(gridSizeF*(p.z-maxNeighborhood-miny)/(maxy-miny));
  if(t>=gridSize) return 0;
  b=int(gridSizeF*(p.z+maxNeighborhood-miny)/(maxy-miny));
  if(b<0) return 0;

  if(l<0) l=0;
  if(r>=gridSize) r=gridSize-1;
  if(t<0) t=0;
  if(b>=gridSize) b=gridSize-1;

  for(x=l;x<=r;x++){
    for(y=t;y<=b;y++) {
      // loop over all grid squares in neighborhood

      count+=gridCount[x][y];
      for(i=grid[x][y]; i>-1; i=link[i]) {
	// test all photons in linked list for current grid square
	dist[i] = Distance(map[i].x, p);
      }
    }
  }

  if(count<numNeighbors) {
    // There are fewer photons within the neighborhood than requested.
    // return all that there are.

    numNeighbors=0;
    for(x=l;x<=r;x++){
      for(y=t;y<=b;y++) {
	// loop over all grid squares in neighborhood
	for(i=grid[x][y]; i>-1; i=link[i]) {
	  if(dist[i]<maxNeighborhood) {
	    neighbor[numNeighbors++] = i;
	    if(dist[i]>radius) radius=dist[i];
	  }
	}
      }
    }
  } else {
    // There are more neighbors around than requested.
    // We want to select the numNeighbors best.
    for(i=0; i<numNeighbors; i++) {
      min = INFINITY;
      minLoc=0;
      for(x=l;x<=r;x++){
	for(y=t;y<=b;y++) {
	  // loop over all grid squares in neighborhood

	  for(j=grid[x][y]; j>-1; j=link[j]) {
	    // test all photons in linked list for current grid square
	    if(dist[j] < min) {
	      min = dist[j];
	      minLoc = j;
	    }
	  }
	}
      }
      if(min>maxNeighborhood) {
	numNeighbors=i;
	break;
      }
      neighbor[i]=minLoc;
      dist[minLoc]=INFINITY;
      radius=min;
    }
  }
  maxNeighborhood=radius;

  return numNeighbors;
}

void TracePhoton(Spectrum & power, const Ray & ray)
{
  static int photonDepth = 0;
  float maxDist = INFINITY;
  HitInfo hitInfo;
  if (scene->Intersect(ray, 1e-4, &maxDist, &hitInfo)) {
    ShadeContext shadeContext(&hitInfo, -ray.D);

    if (hitInfo.hitPrim->attributes->LightShader != NULL) {
      // Discard photons scattered back into a light source
      //cerr << "light source." << endl;
      return;
    }

    if (!hitInfo.hitPrim->attributes->Surface){
      // Discard photons that don't hit any surface
      //cerr << "null surface." << endl;
      return;
    }

    BRDF *brdf = hitInfo.hitPrim->attributes->Surface->Shade(shadeContext);
    Point Pw = hitInfo.hitPrim->attributes->ObjectToWorld(hitInfo.Pobj);

    if (++photonDepth < 8) {
      Ray newRay(Pw, ray.D);
      int result = brdf->Photon(power, newRay.D);

      if(result & 1) {
	// Surface says to store photon
	if(photonDepth>1) {
	  //cerr << "Storing photon w/ power " << power  << endl;
	  StorePhoton(Pw, power, ray.D);
	}
      } else {
	//cerr << "power " << power << endl;
      }
      if((result & 2) > 0) {
	// transmit photon
	TracePhoton(power, newRay);
      }
    }
    --photonDepth;
    delete brdf;
  } else {
    //cerr << "Photon missed." << endl;
  }
}
