#include "texture.h"
#include "shading.h"
#include "light.h"
#include "primitives.h"
#include "color.h"
#include "accel.h"

Light::~Light()
{
	delete surfaceFunction;
	attributes->Dereference();
}

IsotropicPointLight::IsotropicPointLight(SurfaceFunction * ud,
LightAttributes * attr):				 Light(ud,
											   attr)
{

	lightPos = surfaceFunction->InitializePoint(RI_FROM, Point(0, 0, 0));
	lightPos = attr->LightToWorld(lightPos);

	Spectrum lightColor = surfaceFunction->InitializeColor(RI_LIGHTCOLOR,
														   Spectrum(1.0));
	Float intensity = surfaceFunction->InitializeFloat(RI_INTENSITY, 1.0);
	I = lightColor * intensity;

}

Spectrum IsotropicPointLight::dE(const Point & Ps, Point * Plight, Float *u) const
{
	*Plight = lightPos;
	return I / DistanceSquared(lightPos, Ps);
}

DirectionalLight::DirectionalLight(SurfaceFunction * ud,
LightAttributes * attr):		   Light(ud, attr)
{
	Point pFrom, pTo;

	pFrom = surfaceFunction->InitializePoint(RI_FROM, Point(0, 0, 0));
	pTo = surfaceFunction->InitializePoint(RI_TO, Point(0, 0, 1));
	pFrom = attr->LightToWorld(pFrom);
	pTo = attr->LightToWorld(pTo);
	lightDir = (pFrom - pTo).Hat();

	Spectrum lightColor = surfaceFunction->InitializeColor(RI_LIGHTCOLOR,
														   Spectrum(1.0));
	Float intensity = surfaceFunction->InitializeFloat(RI_INTENSITY, 1.0);
	I = lightColor * intensity;

}

Spectrum DirectionalLight::dE(const Point & Ps, Point * Plight, Float *u) const
{
	*Plight = Ps + 1000. * lightDir;	// Ye Olde Hack O' Rama
	return I;
}

AmbientLight::AmbientLight(SurfaceFunction * ud, LightAttributes * attr)
:	attributes(attr), surfaceFunction(ud)
{

	Spectrum lightColor = surfaceFunction->InitializeColor(RI_LIGHTCOLOR,
														   Spectrum(1.0));
	Float intensity = surfaceFunction->InitializeFloat(RI_INTENSITY, 1.0);
	I = lightColor * intensity;

}

AmbientLight::~AmbientLight()
{
	delete surfaceFunction;
	attributes->Dereference();
}

Spectrum AmbientLight::Intensity(const Point &) const
{
	return I;
}

SpotLight::SpotLight(SurfaceFunction * ud, LightAttributes * attr)
:	Light(ud, attr)
{

	pFrom = surfaceFunction->InitializePoint(RI_FROM, Point(0, 0, 0));
	pTo = surfaceFunction->InitializePoint(RI_TO, Point(0, 0, 1));
	pFrom = attr->LightToWorld(pFrom);
	pTo = attr->LightToWorld(pTo);
	lightDir = (pFrom - pTo).Hat();

	coneAngle =
		Radians(surfaceFunction->InitializeFloat(RI_CONEANGLE, 30.));
	coneDeltaAngle =
		Radians(surfaceFunction->InitializeFloat(RI_CONEDELTAANGLE, 5.));
	beamDistribution =
		surfaceFunction->InitializeFloat(RI_BEAMDISTRIBUTION, 2.0);
	coneAngle = Clamp(coneAngle, 0, 2 * M_PI);
	coneDeltaAngle = Clamp(coneDeltaAngle, 0, coneAngle);

	Spectrum lightColor = surfaceFunction->InitializeColor(RI_LIGHTCOLOR,
														   Spectrum(1.0));
	Float intensity = surfaceFunction->InitializeFloat(RI_INTENSITY, 1.0);
	I = lightColor * intensity;

}

Spectrum SpotLight::dE(const Point & Ps, Point * Plight, Float *u) const
{
	*Plight = pFrom;
	Float cosangle = max(Dot((pFrom - Ps).Hat(), lightDir), (Float) 0.);
	Float atten = pow(cosangle, beamDistribution);
	atten *= SmoothStep(cos(coneAngle),
						cos(coneAngle - coneDeltaAngle), cosangle);
	//atten *= SmoothStep(cos(coneAngle+coneDeltaAngle),
	//                    cos(coneAngle),
	//                    cosangle);
	return atten * I / DistanceSquared(pFrom, Ps);
}

AreaLight::AreaLight(SurfaceFunction * ud, LightAttributes * attr)
:	Light(ud, attr)
{
	primitive = NULL;
}

DiffuseAreaLight::DiffuseAreaLight(SurfaceFunction * ud,
LightAttributes * attr):		   AreaLight(ud,
											 attr)
{
	Spectrum lightColor = surfaceFunction->InitializeColor(RI_LIGHTCOLOR,
														   Spectrum(1.0));
	Float intensity = surfaceFunction->InitializeFloat(RI_INTENSITY, 1.0);
	Lo = lightColor * intensity;
}

Spectrum DiffuseAreaLight::Le(const ShadeContext & sc) const
{
	return Lo;
}

Spectrum DiffuseAreaLight::dE(const Point & Pshade, Point * Plight, Float *u) const
{
	HitInfo hitInfo;
	Float pdf = primitive->Sample(u, &hitInfo);
	if (pdf != 0.) {
		*Plight = primitive->attributes->ObjectToWorld(hitInfo.Pobj);
		Normal N = primitive->attributes->ObjectToWorld(hitInfo.NgObj);
		Float costhetao = fabs(Dot(N.Hat(),
					   (Pshade - *Plight).Hat()));

		return Lo * costhetao / (DistanceSquared(Pshade, *Plight)
					 * pdf);
	} else {
		return Spectrum(0.);
	}
}

Light *PointLightCreate(const char *name,
						SurfaceFunction * data, LightAttributes * attr)
{
	if (strcmp(name, RI_POINTLIGHT) == 0)
		return new IsotropicPointLight(data, attr);
	else if (strcmp(name, RI_DISTANTLIGHT) == 0)
		return new DirectionalLight(data, attr);
	else if (strcmp(name, RI_SPOTLIGHT) == 0)
		return new SpotLight(data, attr);
	else
		return NULL;
}

AreaLight *AreaLightCreate(const char *name,
						   SurfaceFunction * data, LightAttributes * attr)
{
	if (strcmp(name, LRT_DIFFUSE) == 0)
		return new DiffuseAreaLight(data, attr);
	else
		return NULL;
}
