#include "heightfield.h"
#include "trimesh.h"
#include "geometry.h"

Heightfield::Heightfield(int x, int y, float *zs, PrimitiveAttributes * a,
                                                 SurfaceFunction * sf)
  :Primitive(a, sf)
{
  nx = x;
  ny = y;
  xSpacing = 1./(nx-1);
  ySpacing = 1./(ny-1);

  z = new float[nx * ny];
  memcpy(z, zs, nx * ny * sizeof(float));
}

 Heightfield::~Heightfield()
 {
         delete[] z;
 }

 BBox Heightfield::BoundObjectSpace() const
 {
         float minz = z[0], maxz = z[0];
         for (int i = 1; i < nx * ny; ++i) {
                 if (z[i] < minz)
                         minz = z[i];
                 if (z[i] > maxz)
                         maxz = z[i];
         }
         return BBox(Point(0, 0, minz), Point(1, 1, maxz));
 }

 bool Heightfield::CanIntersect() const
 {
         return true;
 }

 bool
 Heightfield::IntersectClosest(const Ray &ray, Float mint, Float *maxt,
                                  HitInfo *hit) const
 {
    static BBox bbox(BoundObjectSpace());

    Float T = *maxt;
    if(bbox.Inside(ray(mint)))
    {  // ray begins inside grid
      T = mint;
    }
    else if(!bbox.IntersectP(ray, &T))
    {
      return false; // does not intersect grid at all
    }

    Point isect = ray(T);

    int xCell = x2c(isect.x);
    if(xCell==nx-1) // we are on the edge
    {
      xCell--;
    }
    Assert( xCell >= 0 && xCell < nx-1);

    Float nextXcross, dx;
    int stepX, outX;
    if(fabs(ray.D.x) < RI_EPSILON) // ray is vertical
    {
      cout << "vertical" << endl;
      nextXcross = INFINITY;
      dx = 0;
      outX = -1;
    }
    else if(ray.D.x > 0)
    {
      float tmp = 1./ray.D.x;
      nextXcross = T + (c2x(xCell+1) - isect.x)*tmp;///ray.D.x;
      dx = xSpacing*tmp;///ray.D.x;
      stepX = 1;
      outX = nx-1;
    }
    else
    {
      float tmp = 1./ray.D.x;
      nextXcross = T + (c2x(xCell) - isect.x)*tmp;///ray.D.x;
      dx = -xSpacing*tmp;///ray.D.x;
      stepX = -1;
      outX = -1;
    }

    int yCell = y2c(isect.y);
    if(yCell==ny-1) // we are on the edge
    {
      yCell--;
    }
    Assert( yCell >= 0 && yCell < ny-1);

    Float nextYcross, dy;
    int stepY, outY;
    if(fabs(ray.D.y) < RI_EPSILON) // ray is horizontal
    {
      nextYcross = INFINITY;
      dy = 0;
      outY = -1;
    }
    else if(ray.D.y > 0)
    {
      float tmp = 1/ray.D.y;
      nextYcross = T + (c2y(yCell+1) - isect.y)*tmp;///ray.D.y;
      dy = ySpacing*tmp;///ray.D.y;
      stepY = 1;
      outY = ny-1;
    }
    else
    {
      float tmp = 1/ray.D.y;
      nextYcross = T + (c2y(yCell) - isect.y)*tmp;///ray.D.y;
      dy = -ySpacing*tmp;///ray.D.y;
      stepY = -1;
      outY = -1;
    }

    bool saveHit = false;
    Float nextCross = min(nextXcross, nextYcross);

    Point into = isect;
    Point out = ray(nextCross);

    while(true)
    {
      int offset = yCell*(ny) + xCell;
      Point p1(c2x(xCell),c2y(yCell),z[offset]);
      Point p2(c2x(xCell+1),c2y(yCell),z[offset+1]);
      Point p3(c2x(xCell+1),c2y(yCell+1),z[offset+ny+1]);
      Point p4(c2x(xCell),c2y(yCell+1),z[offset+ny]);

      // figure out the local z-extents of this cell
      Float top = max(max(max(p1.z,p2.z),p3.z),p4.z);
      Float bot = min(min(min(p1.z,p2.z),p3.z),p4.z);

      if(!((into.z > top && out.z > top) ||
           (into.z < bot && out.z < bot)))
      {
        if(IntersectTri(ray,mint,maxt,hit,p1,p2,p3))
        {
          saveHit = true;
          if(hit!=NULL) {
            //cout << "u,v: " << hit->u << " " << hit->v << endl;
            hit->u = c2x(xCell + hit->u+hit->v);
            hit->v = 1-c2y(yCell + hit->v);
          }
        }
        else if(IntersectTri(ray,mint,maxt,hit,p3,p4,p1))
        {
          saveHit = true;
          if(hit!=NULL) {
            //cout << "u,v: " << hit->u << " " << hit->v << endl;
            hit->u = c2x(xCell + 1-hit->u-hit->v);
            hit->v = 1-c2y(yCell + 1-hit->v);
          }
        }
        if(saveHit)
          break;
      }
     
      if(nextXcross < nextYcross)
      {  // step in x
        if(*maxt < nextXcross)
        {
          break;
        }
        xCell+= stepX;
        if(xCell==outX)
          break;
        nextXcross += dx;
      }
      else // step in y
      {
        if(*maxt < nextYcross)
        {
          break;
        }
        yCell+= stepY;
        if(yCell==outY)
          break;
        nextYcross += dy;
      }
      nextCross = min(nextXcross,nextYcross);
      into = out;
      out = ray(nextCross);
    }
    //cout << "returning from intersect" << endl;
    return saveHit;
 }


 bool
 Heightfield::IntersectTri(const Ray & ray, Float mint,
                                                                 Float 
 * maxt, HitInfo * hit, Point v0, Point v1, Point v2) const
 {
         static int triangleTests = 0, triangleHits = 0;
         if (triangleTests == 0)
                 StatsRegisterRatio(STATS_DETAILED, "Geometry",
                                                    "Triangle Ray 
 Intersections", &triangleHits,
                                                    &triangleTests);

         ++triangleTests;

         Vector E1 = v1 - v0;
         Vector E2 = v2 - v0;
         Vector S_1 = Cross(ray.D, E2);
         Float divisor = Dot(S_1, E1);
         if (divisor == 0.)
                 return false;
         Float invDivisor = 1. / divisor;

         Vector T = ray.O - v0;
         Float u = Dot(T, S_1) * invDivisor;
         if (u < 0. || u > 1.0)
                 return false;

         Vector S_2 = Cross(T, E1);
         Float v = Dot(ray.D, S_2) * invDivisor;
         if (v < 0 || u + v > 1.0)
                 return false;

         Float t = Dot(E2, S_2) * invDivisor;
         if (t < mint || t > *maxt)
                 return false;
         *maxt = t;
    ++triangleHits;

         if (hit != NULL) {

                 Normal N = Normal(Cross(E2, E1));
                 N.Normalize();
                 hit->RecordHit(ray(t), N, u, v, this);

                 int interpOffsets[3] ={0,1,2};
                         //{ vertexIndex0, vertexIndex1, vertexIndex2 };
                 Float interpWeights[3] = { 1. - u - v, u, v };
                 InterpolatedPrimData *interpolatedData;
                 interpolatedData =
                         surfaceFunction->Interpolate(3, interpOffsets,
  
                            interpWeights);
                 hit->SetInterpolatedData(interpolatedData);

         }
         return true;
 }
