
#include "trimesh.h"

TriangleMesh::TriangleMesh(int nt, int nv, int *vi, Point * P,
						   PrimitiveAttributes * a, SurfaceFunction * sf)
:Primitive(a, sf)
{
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	p = new Point[nverts];
	memcpy(p, P, nverts * sizeof(Point));
}

TriangleMesh::~TriangleMesh()
{
	delete[]vertexIndex;
	delete[]p;
}

BBox TriangleMesh::BoundObjectSpace() const
{
	BBox objectSpaceBounds(p[0], p[1]);
	for (int i = 2; i < nverts; i++)
		objectSpaceBounds = Union(objectSpaceBounds, p[i]);
	return objectSpaceBounds;
}

BBox TriangleMesh::BoundWorldSpace() const
{
	BBox worldBounds(attributes->ObjectToWorld(p[0]),
					 attributes->ObjectToWorld(p[1]));
	for (int i = 2; i < nverts; i++) {
		worldBounds = Union(worldBounds, attributes->ObjectToWorld(p[i]));
	}
	return worldBounds;
}

void TriangleMesh::Refine(vector < Primitive * >*refined) const
{
	refined->reserve(ntris);
	for (int i = 0; i < ntris; ++i)
		refined->push_back(new Triangle(this, i));
}

BBox Triangle::BoundObjectSpace() const
{
	BBox objectBounds(mesh->p[mesh->vertexIndex[triNum * 3]],
					  mesh->p[mesh->vertexIndex[triNum * 3 + 1]]);
	return Union(objectBounds, mesh->p[mesh->vertexIndex[triNum * 3 + 2]]);
}

BBox Triangle::BoundWorldSpace() const
{
	Transform o2w = mesh->attributes->ObjectToWorld;
	BBox objectBounds(o2w(mesh->p[mesh->vertexIndex[triNum * 3]]),
					  o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 1]]));
	return Union(objectBounds,
				 o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 2]]));
}

bool Triangle::IntersectClosest(const Ray & ray, Float mint,
								Float * maxt, HitInfo * hit) const
{

	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry",
						   "Triangle Ray Intersections", &triangleHits,
						   &triangleTests);

	++triangleTests;

	int *vptr = &(mesh->vertexIndex[3 * triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;

	const Point *p = mesh->p;
	Vector E1 = p[vertexIndex1] - p[vertexIndex0];
	Vector E2 = p[vertexIndex2] - p[vertexIndex0];
	Vector S_1 = Cross(ray.D, E2);
	Float divisor = Dot(S_1, E1);
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;

	Vector T = ray.O - mesh->p[vertexIndex0];
	Float u = Dot(T, S_1) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;

	Vector S_2 = Cross(T, E1);
	Float v = Dot(ray.D, S_2) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;

	Float t = Dot(E2, S_2) * invDivisor;
	if (t < mint || t > *maxt)
		return false;
	*maxt = t;
	++triangleHits;

	if (hit != NULL) {
	        Point _x = p[vertexIndex0];
		Vector _y = Vector(p[vertexIndex1].x, p[vertexIndex1].y, p[vertexIndex1].z);
		Vector _z = Vector(p[vertexIndex2].x, p[vertexIndex2].y, p[vertexIndex2].z);

		Normal N = Normal(Cross(E2, E1));
		N.Normalize();
		hit->RecordHit(ray(t), N, u, v, mesh);

		int interpOffsets[3] =
			{ vertexIndex0, vertexIndex1, vertexIndex2 };
		Float interpWeights[3] = { 1. - u - v, u, v };
		InterpolatedPrimData *interpolatedData;
		interpolatedData =
			mesh->surfaceFunction->Interpolate(3,
							   interpOffsets,
							   interpWeights);
		hit->SetInterpolatedData(interpolatedData);
		
		Point P = Point(_x.x*(1.-u-v),_x.y*(1.-u-v),_x.z*(1.-u-v))+(_y*u + _z*v) ;

		const Float* _s = interpolatedData->GetFloat(RI_S);
		const Float* _t = interpolatedData->GetFloat(RI_T);

		if (!_s || !_t) return true;

		hit->dPdS = Vector(1, 0, 0);
		hit->dPdT = Vector(0, 1, 0);
		if (u > .01)
		{
		  Float _u = u-.01;
		  Float _v = v;
		  Float iw2[3] = {1. - _u, _u, _v};
		  InterpolatedPrimData* id2 = mesh->surfaceFunction->Interpolate(3, interpOffsets, iw2);

		  Point p = Point(P.x*iw2[0],P.y*iw2[0],P.z*iw2[0])+_y*iw2[1] + _z*iw2[2];

		  hit->dPdU = ((p-P)/-.01).Hat();
		  hit->dPdV = Vector(Cross(Vector(N), hit->dPdU));
		  hit->dPdV.Normalize();
		  const Float* s = id2->GetFloat(RI_S);
		  const Float* t = id2->GetFloat(RI_T);
		  if (s && t)
		  {
		    if (*s==*_s) hit->dPdS = hit->dPdU;
		    else hit->dPdS = hit->dPdU * (-.01 / (*s - *_s));
		    if (*t==*_t) hit->dPdT = hit->dPdV;
		    else hit->dPdT = hit->dPdU * (-.01 / (*t - *_t));
		  }
		}
		else if (u+v < .99)
		{
		  Float _u = u+.01;
		  Float _v = v;
		  Float iw2[3] = {1. - _u, _u, _v};
		  InterpolatedPrimData* id2 = mesh->surfaceFunction->Interpolate(3, interpOffsets, iw2);

		  Point p =  Point(P.x*iw2[0],P.y*iw2[0],P.z*iw2[0])+_y*iw2[1] + _z*iw2[2];

		  hit->dPdU = ((p-P)/.01).Hat();
		  hit->dPdV = Vector(Cross(Vector(N), hit->dPdU));
		  hit->dPdV.Normalize();
		  const Float* s = id2->GetFloat(RI_S);
		  const Float* t = id2->GetFloat(RI_T);
		  if (s && t)
		  {
		    if (*s == *_s) hit->dPdS = hit->dPdU;
		    else hit->dPdS = hit->dPdU * (.01 / (*s - *_s));
		    if (*t == *_t) hit->dPdT = hit->dPdV;
		    hit->dPdT = hit->dPdU * (.01 / (*t - *_t));
		  }
		}
		else
		{
		  Float _u = u;
		  Float _v = v-.01;
		  Float iw2[3] = {1. - _u, _u, _v};
		  InterpolatedPrimData* id2 = mesh->surfaceFunction->Interpolate(3, interpOffsets, iw2);

		  Point p = Point(P.x*iw2[0],P.y*iw2[0],P.z*iw2[0])+_y*iw2[1] + _z*iw2[2];

		  hit->dPdV = ((p-P)/-.01).Hat();

		  hit->dPdU = Vector(Cross(Vector(N), hit->dPdV));
		  hit->dPdU.Normalize();
		  const Float* s = id2->GetFloat(RI_S);
		  const Float* t = id2->GetFloat(RI_T);
		  if (s && t)
		  {
		    if (*s == *_s) hit->dPdU = hit->dPdV;
		    else hit->dPdS = hit->dPdV * (.01 / (*s - *_s));
		    if (*t == *_t) hit->dPdV = hit->dPdU;
		    else hit->dPdT = hit->dPdV * (.01 / (*t - *_t));
		  }
		}

	}
	return true;
}
