
#ifndef SHADING_H
#define SHADING_H
#include "lrt.h"
#include "primitives.h"
#include "mysl.h"

class SurfaceFunction {
  public:

	SurfaceFunction(u_int nv) {
		nVertex = nv;
	} SurfaceFunction(const SurfaceFunction & sf);

	~SurfaceFunction();

	void AddUniformPoint(RtToken name, const Float * data);

	void AddVertexPoint(RtToken name, const Float * data);

	void AddUniformFloat(RtToken name, const Float * data);
	void AddUniformHPoint(RtToken name, const Float * data);
	void AddUniformVector(RtToken name, const Float * data);
	void AddUniformNormal(RtToken name, const Float * data);
	void AddUniformColor(RtToken name, const Float * data);
	void AddUniformString(RtToken name, const char *data);

	void AddVertexFloat(RtToken name, const Float * data);
	void AddVertexHPoint(RtToken name, const Float * data);
	void AddVertexVector(RtToken name, const Float * data);
	void AddVertexNormal(RtToken name, const Float * data);
	void AddVertexColor(RtToken name, const Float * data);

	const Point *GetUniformPoint(RtToken name) const;

	const Point *GetVertexPoint(RtToken name) const;

	const Float *GetUniformFloat(RtToken name) const;
	const Vector *GetUniformVector(RtToken name) const;
	const Normal *GetUniformNormal(RtToken name) const;
	const Spectrum *GetUniformColor(RtToken name) const;
	const char *GetUniformString(RtToken name) const;

	const Float *GetVertexFloat(RtToken name) const;
	const Float *GetVertexHPoint(RtToken name) const;
	const Vector *GetVertexVector(RtToken name) const;
	const Normal *GetVertexNormal(RtToken name) const;
	const Spectrum *GetVertexColor(RtToken name) const;

	Float InitializeFloat(RtToken name, Float def) const;
	Point InitializePoint(RtToken name, const Point & def) const;
	Vector InitializeVector(RtToken name, const Vector & def) const;
	Normal InitializeNormal(RtToken name, const Normal & def) const;
	Spectrum InitializeColor(RtToken name, const Spectrum & def) const;
	const char *InitializeString(RtToken name, const char *def) const;

	friend class InterpolatedPrimData;

	InterpolatedPrimData *Interpolate(int n, const int *offsets,
									  const Float * weights) const;

  private:

	u_int nVertex;

	vector<pair<RtToken, Point> > uniformPoints;

	vector<pair<RtToken, Point *> > vertexPoints;

	vector<pair<RtToken, Float> > uniformFloats;
	vector<pair<RtToken, Float> > uniformHPoints;
	vector<pair<RtToken, Vector> > uniformVectors;
	vector<pair<RtToken, Normal> > uniformNormals;
	vector<pair<RtToken, Spectrum> > uniformColors;
	vector<pair<RtToken, char *> > uniformStrings;

	vector<pair<RtToken, Float *> > vertexFloats;
	vector<pair<RtToken, Float *> > vertexHPoints;
	vector<pair<RtToken, Vector *> > vertexVectors;
	vector<pair<RtToken, Normal *> > vertexNormals;
	vector<pair<RtToken, Spectrum *> > vertexColors;

};

class InterpolatedPrimData { public:
	 InterpolatedPrimData(const SurfaceFunction * pd);

	~InterpolatedPrimData();

	friend class SurfaceFunction;
	friend class Triangle;

  private:

	 friend class HitInfo;

	const Point *GetPoint(RtToken token) const;

	const Float *GetFloat(RtToken token) const;
	const Vector *GetVector(RtToken token) const;
	const Normal *GetNormal(RtToken token) const;
	const Spectrum *GetColor(RtToken token) const;

	const SurfaceFunction *primitiveData;

	Point *interpVertexPoints;

	Float *interpVertexFloats;
	Vector *interpVertexVectors;
	Normal *interpVertexNormals;
	Spectrum *interpVertexColors;

};

class MaterialAttributes:public ReferenceCounted { public:
	 Transform WorldToSurface, SurfaceToWorld;


};

class ShadeContext { public:
	 ShadeContext(const HitInfo *, const Vector & wo);

	Float InitializeFloat(RtToken token, Float def) const;
	Point InitializePoint(RtToken token, const Point & def) const;
	Vector InitializeVector(RtToken token, const Vector & def) const;
	Normal InitializeNormal(RtToken token, const Normal & def) const;
	Spectrum InitializeColor(RtToken token, const Spectrum & def) const;
	const char *InitializeString(RtToken token, const char *def) const;

	static TextureMap *GetTexture(const char *texName, const char *filename = NULL);
	static CubeMap    *GetCube   (const char *cubeName,
				      const char *px = NULL, const char *nx = NULL,
				      const char *py = NULL, const char *ny = NULL,
				      const char *pz = NULL, const char *nz = NULL);

	const Vector wo;

	Normal Ns, Ng;
	bool entering;

	friend class SLMachine;

  private:
	 const HitInfo *hitInfo;

	static StringHashTable textures;
	static StringHashTable cubes;

};

class Material { public:
	 Material(SurfaceFunction * ud)
	:surfaceFunction(ud) {
	} virtual ~ Material();

	virtual BRDF *Shade(const ShadeContext & sc) const = 0;

	SurfaceFunction *surfaceFunction;


};

class SLMaterial: public Material
{
 public:
     SLMaterial( const char* name, SurfaceFunction * data);
     BRDF *Shade(const ShadeContext & sc) const;
     virtual ~SLMaterial();
 private:
     vector<union _operation> _codeBase;
};

class MatteSurface:public Material { public:
	 MatteSurface(SurfaceFunction * data)
	:Material(data) {
		Kd = surfaceFunction->InitializeFloat(RI_KD, 1.0);
	} BRDF *Shade(const ShadeContext & sc) const;

  private:

	Float Kd;

};

class PlasticSurface:public Material { public:
	 PlasticSurface(SurfaceFunction * data)
	:Material(data) {
		Kd = surfaceFunction->InitializeFloat(RI_KD, .5);
		Ks = surfaceFunction->InitializeFloat(RI_KS, .5);
		Roughness = surfaceFunction->InitializeFloat(RI_ROUGHNESS, .1);
		SpecularColor =
			surfaceFunction->InitializeColor(RI_SPECULARCOLOR,
											 Spectrum(1));
	} BRDF *Shade(const ShadeContext & sc) const;
	BRDF *PlasticShade(const ShadeContext & sc, const Spectrum & Cs) const;

  protected:

	Float Kd, Ks, Roughness;
	Spectrum SpecularColor;

};

class PaintedPlasticSurface:public PlasticSurface { public:
	 PaintedPlasticSurface(SurfaceFunction * data);

	BRDF *Shade(const ShadeContext & sc) const;

  private:
	 const char *texName;

};

class CheckeredSurface:public Material { public:
	 CheckeredSurface(SurfaceFunction * data)
	:Material(data) {
		Kd = surfaceFunction->InitializeFloat(RI_KD, 1.0);
		CheckColor1 =
			surfaceFunction->InitializeColor(LRT_CHECKCOLOR1,
											 Spectrum(1, 0, 0));
		CheckColor2 =
			surfaceFunction->InitializeColor(LRT_CHECKCOLOR2,
											 Spectrum(0, 0, 1));
		CheckFrequency =
			surfaceFunction->InitializeFloat(LRT_CHECKFREQUENCY, 1);
	} BRDF *Shade(const ShadeContext & sc) const;

  private:

	Float Kd, CheckFrequency;
	Spectrum CheckColor1, CheckColor2;

};

class GlassSurface:public Material { public:
	 GlassSurface(SurfaceFunction * data)
	:Material(data) {
		Kr = surfaceFunction->InitializeFloat(RI_KR, .5);
		Kt = surfaceFunction->InitializeFloat(LRT_KT, .5);
		Index = surfaceFunction->InitializeFloat(LRT_INDEX, 1.5);
	} BRDF *Shade(const ShadeContext & sc) const;

  private:

	Float Kr, Kt, Index;

};

class ShinyMetalSurface:public Material { public:
	 ShinyMetalSurface(SurfaceFunction * data)
	:Material(data) {
		Ks = surfaceFunction->InitializeFloat(RI_KS, 1.0);
		Kr = surfaceFunction->InitializeFloat(RI_KR, 1.0);
		Roughness = surfaceFunction->InitializeFloat(RI_ROUGHNESS, 0.1);
	} BRDF *Shade(const ShadeContext & sc) const;

  private:

	Float Ks, Kr, Roughness;

};

class ViewSTSurface:public Material { public:
	 ViewSTSurface(SurfaceFunction * data)
	:Material(data) {
	} BRDF *Shade(const ShadeContext & sc) const;

};

#endif // SHADING_H
