#include "lrt.h"
#include "color.h"
#include "reflection.h"
#include "transform.h"
#include "shading.h"
#include "scene.h"
#include "transport.h"
#include "texture.h"

extern double noise1(double);
extern double noise2(double*);
extern double noise3(double*);


BRDF::~BRDF()
{
}

Spectrum Lambertian::fr(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return R;
	} else {
		return Spectrum(0.);
	}
}

BlinnGlossy::BlinnGlossy(const Spectrum & reflectance,
			 Float roughness, const Normal & normal,
			 const Vector & w)
{
	R = reflectance;
	roughness /= 8.;			// hack to match BMRT, prman
	invRoughness = 1. / roughness;
	N = normal;
	wo = w;
	costhetao = Dot(wo, N);
}

Spectrum BlinnGlossy::fr(const Vector & wi) const
{
	Vector H = (wi + wo).Hat();
	Float costhetah = Dot(N, H);
	Float costhetai = Dot(N, wi);
	if (costhetah > 0. && costhetai > 0.) {
		return R * pow(costhetah, invRoughness) / (costhetai * costhetao);
	} else {
		return Spectrum(0.);
	}
}

int BRDF::SpecularComponents() const
{
	return 0;
}

Spectrum BRDF::SampleSpecular(int component, Vector * wo) const
{
	Severe("SampleSpecular() shouldn't be called for BRDFs "
		   "without specular components.");
	return Spectrum(0.);
}

SpecularReflection::SpecularReflection(const Spectrum & r,
				       const Normal & N, const Vector & wo)
{
	R = r;
	reflectedDirection = -wo - 2. * Dot(N, -wo) * Vector(N);
}

Spectrum SpecularReflection::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = reflectedDirection;
	return R;
}

SpecularTransmission::SpecularTransmission(const Spectrum & r, const Normal & N, 
					   const Vector & wo, Float indexi, Float indext)
{
	R = r;

	Float eta = indexi / indext;
	Float c1 = Dot(wo, N);
	Float c2 = 1 - (eta * eta * (1. - c1));
	if (c2 < 0.) {
		// total internal reflection
		R = 0.;
		DirT = wo;
	} else {
		c2 = sqrt(c2);
		DirT = -eta * wo + (eta * c1 - c2) * Vector(N);
	}

}

Spectrum SpecularTransmission::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = DirT;
	return R;
}

ScatteringMixture::~ScatteringMixture()
{
	vector < BRDF * >::const_iterator iter;
	for (iter = funcs.begin(); iter != funcs.end(); ++iter)
		delete *iter;
}

Spectrum ScatteringMixture::fr(const Vector & wi) const
{
	Spectrum ret(0);
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		ret += (*weightIter) * (*funcIter)->fr(wi);
		funcIter++, weightIter++;
	}
	return ret;
}

Spectrum ScatteringMixture::SampleSpecular(int component, Vector * wi) const
{
	Assert(component < SpecularComponents());
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (component >= (*funcIter)->SpecularComponents())
			component -= (*funcIter)->SpecularComponents();
		else
			return (*weightIter) *
				(*funcIter)->SampleSpecular(component, wi);
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in" " ScatteringMixture::SampleSpecular()");
	return Spectrum(0);
}

#if 0
Lafortune::Lafortune(int n, const LafortuneLobe l[], const Vector & w,
		     const Normal & norm, const Vector & dpdu)
{
	nLobes = n;
	lobes = new LafortuneLobe[n];
	for (int i = 0; i < nLobes; ++i)
		lobes[i] = l[i];
	wo = w;
	N = norm.Hat();
	dPdu = dpdu.Hat();
	dPdv = Cross((Vector) N, dPdu);
}
#endif

#if 0
Spectrum Lafortune::fr(const Vector & wi) const
{
	Spectrum value(0.);

	Float xd = Dot(dPdu, wi) * Dot(dPdu, wo);
	Float yd = Dot(dPdv, wi) * Dot(dPdv, wo);
	Float zd = Dot(N, wi) * Dot(N, wo);

	for (int i = 0; i < nLobes; ++i) {

		Spectrum dot =
			xd * lobes[i].Cx + yd * lobes[i].Cy + zd * lobes[i].Cz;
		value += dot.Pow(lobes[i].Exponent);

	}
	return value;
}
#endif

BRDF *CreateLambertian(const Spectrum & reflectance, const Normal & normal)
{
	return new Lambertian(reflectance, normal);
}

BRDF *CreateBlinnGlossy(const Spectrum & reflectance,
			Float roughness,
			const Normal & normal, const Vector & wo)
{
	return new BlinnGlossy(reflectance, roughness, normal, wo);
}

BRDF *CreateSpecularReflection(const Spectrum & r,
			       const Normal & N, const Vector & wo)
{
	return new SpecularReflection(r, N, wo);
}

BRDF *CreateSLMachine(const vector<union _operation> *code, const ShadeContext& sc)
{
  return new SLMachine(code, sc);
}

BRDF *CreateSpecularTransmission(const Spectrum & r,
				 const Normal & N,
				 const Vector & wo,
				 Float indexi, Float indext)
{
	return new SpecularTransmission(r, N, wo, indexi, indext);
}

ScatteringMixture *CreateScatteringMixture()
{
	return new ScatteringMixture;
}

#if 0
BRDF *CreateLafortune(int nLobes, const LafortuneLobe lobes[],
		      const Vector & wi, const Normal & N,
		      const Vector & dPdu)
{
	Assert(1 == 0);
	return NULL;
	// need to implement the sampling methods
//  return new Lafortune(nLobes, lobes, wi, N, dPdu);
}
#endif

Spectrum Lambertian::Sample(Float u[2], Vector * wi, Float * pdf) const
{

	Float u0 = 2 * u[0] - 1.;
	Float u1 = 2 * u[1] - 1.;
	if (u0 * u0 + u1 * u1 > 1.) {
		*pdf = 0.;
		return Spectrum(0.);
	}

	Vector Du, Dv;
	if (N.x == 0. && N.y == 0.) {
		Du = Vector(1, 0, 0);
		Dv = Vector(0, 1, 0);
	} else {
		Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
		Dv = Cross(Du, Vector(N)).Hat();
	}

	*wi = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	*pdf = Dot(N, *wi) / M_PI;

	return fr(*wi);
}

Spectrum BlinnGlossy::Sample(Float u[2], Vector * wi, Float * pdf) const
{
        float u0 = 2 * u[0] - 1.;
	float u1 = 2 * u[1] - 1.;

	if (u0 * u0 + u1 * u1 > 1.)
	{
	  *pdf = 0.;
	  return Spectrum(0.);
	}

	float r = sqrtf(u0*u0 + u1*u1);

	float _r = pow(r, invRoughness);

	u0 = u0*_r/r;
	u1 = u1*_r/r;

	Vector Du, Dv;

	if (N.x == 0 && N.y ==0)
	{
	  Du = Vector(1, 0, 0);
	  Dv = Vector(0, 1, 0);
	}
	else
	{
	  Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
	  Dv = Cross(Du, Vector(N)).Hat();
	}

	*wi = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	*pdf = pow(Dot(N, *wi), 1./invRoughness) / M_PI;

	return fr(*wi);
		  
	// YOUR CODE HERE
	// return Spectrum(0.);
}

Spectrum ScatteringMixture::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	int fnum = int (RandomFloat() * funcs.size());
	funcs[fnum]->Sample(u, wi, pdf);
	*pdf /= funcs.size();
	return fr(*wi);
}

Float Lambertian::Pdf(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return 1. / (M_PI * costheta);
	} else {
		return 0.;
	}
}

Float BlinnGlossy::Pdf(const Vector & wi) const
{
  return 0.;
}

Float ScatteringMixture::Pdf(const Vector & wi) const
{
	Float pdfSum = 0.;
	for (u_int i = 0; i < funcs.size(); ++i)
		pdfSum += funcs[i]->Pdf(wi);
	return pdfSum / funcs.size();
}

/*********************************************************************************************/

static inline void ClearVector(float* s)
{
  s[0] = s[1] = s[2] = s[3] = 0.0f;
}

Spectrum SLMachine::fr(const Vector &wi) const
{
  Spectrum black(0.);
  return fr(wi,  &black);
}

Spectrum SLMachine::fr (const Vector &wi, const Spectrum* lightdE) const
{
  Vector l = (*worldToObject)(wi.Hat());
  L[0] = l.x;
  L[1] = l.y;
  L[2] = l.z;
  
  const Vector* w = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformVector(RI_REGISTERR0);
  if (w)
  {
    r0[0] = w->x;
    r0[1] = w->y;
    r0[2] = w->z;
    r0[3] = 0;
  }
  else ClearVector(r0);
  
  w = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformVector(RI_REGISTERR1);
  if (w)
  {
    r1[0] = w->x;
    r1[1] = w->y;
    r1[2] = w->z;
    r1[3] = 0;
  }
  else ClearVector(r1);

  w = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformVector(RI_REGISTERR2);
  if (w)
  {
    r2[0] = w->x;
    r2[1] = w->y;
    r2[2] = w->z;
    r2[3] = 0;
  }
  else ClearVector(r2);

  w = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformVector(RI_REGISTERR3);
  if (w)
  {
    r3[0] = w->x;
    r3[1] = w->y;
    r3[2] = w->z;
    r3[3] = 0;
  }
  else ClearVector(r3);

  const Float* m = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_REGISTERS0);
  if (m)
    s0 = *m;
  else s0 = 0;

  m = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_REGISTERS1);
  if (m)
    s1 = *m;
  else s1 = 0;

  m = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_REGISTERS2);
  if (m)
    s2 = *m;
  else s2 = 0;

  m = savesc->hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_REGISTERS3);
  if (m)
    s3 = *m;
  else s3 = 0;

  lightdE->ConvertToRGB(dE);

  ip = -1;

  while (++ip < _codeBase->size())
  {
    union _operation op;
    op.chunk[0] = (*_codeBase)[ip].chunk[0];
    op.chunk[1] = (*_codeBase)[ip].chunk[1];
    
    switch(op.op_3.opcode)
    {
    case RET: return Spectrum(v0[0], v0[1], v0[2]); break;
    case CLAMP: PerformClamp(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case RND: PerformRnd(op.op_3.r_dst); break;
    case TURB: PerformTurb(op.op_3.r_dst, op.op_3.r_src0); break;
    case NORM: PerformNorm(op.op_3.r_dst, op.op_3.r_src0); break;
    case TRACE: PerformTrace(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case NOP: break;
    case DIV: PerformDiv(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case POW: PerformPow(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case ADD: PerformAdd(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case SUB: PerformSub(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case LI:  PerformLI(op.op_li_float.r_dst, op.op_li_float.f); break;
    case LIV: PerformLIV(op.op_li_float.r_dst, op.op_li_float.reserved, op.op_li_float.f); break;
    case MOV: PerformMov(op.op_3.r_dst, op.op_3.r_src0); break;
    case MUL: PerformMul(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case DP3: PerformDP3(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case EXP: PerformEXP(op.op_3.r_dst, op.op_3.r_src0); break;
    case FLOOR: PerformFloor(op.op_3.r_dst, op.op_3.r_src0); break;
    case CEIL: PerformCeil(op.op_3.r_dst, op.op_3.r_src0); break;
    case LOOKUP: PerformLookup(op.op_3.r_dst, op.op_3.r_src0, op.op_3.r_src1); break;
    case MOVC: PerformMOVC(op.op_copy_scalar.r_dst, op.op_copy_scalar.reserved, op.op_copy_scalar.r_src); break;
    case LC: PerformLC(op.op_copy_scalar.r_dst, op.op_copy_scalar.reserved, op.op_copy_scalar.r_src); break;
    case BLT: PerformBLT(op.op_branch.reg_val0, op.op_branch.reg_val1, op.op_branch.jumprel); break;
    case JMP: ip+=op.op_branch.jumprel; break;
    default:
      RI_UNIMP();
    }
  }

  return Spectrum(v0[0], v0[1], v0[2]);
}

SLMachine::SLMachine(const vector <union _operation> *_code, const ShadeContext &sc)
{
  savesc = const_cast<ShadeContext*>(&sc);
  _codeBase = const_cast<vector<union _operation>*>(_code);
  worldToObject = &sc.hitInfo->hitPrim->attributes->WorldToObject;

  Vector __x ( sc.Ns);
  Vector Norm = sc.hitInfo->hitPrim->attributes->WorldToObject(__x);
  N[0] = Norm.x;
  N[1] = Norm.y;
  N[2] = Norm.z;

  Vector wo = (*worldToObject)(sc.wo);
  V[0] = wo.x;
  V[1] = wo.y;
  V[2] = wo.z;
  V[3] = 1.;
  
  Point    pnt = sc.hitInfo->hitPrim->attributes->ObjectToWorld(sc.hitInfo->Pobj);
  P[0] = pnt.x;
  P[1] = pnt.y;
  P[2] = pnt.z;
  P[3] = 1.0;

  Spectrum col = sc.InitializeColor(RI_CS, Spectrum(0.));


  xy[0] = sc.hitInfo->Pobj.x;
  xy[1] = sc.hitInfo->Pobj.y;
  xy[2] = sc.hitInfo->Pobj.z;

  col.ConvertToRGB(C0);
  
  Spectrum sp = sc.InitializeColor(RI_SPECULARCOLOR, col);
  sp.ConvertToRGB(C1);

  Float s = sc.InitializeFloat(RI_S, 0);
  Float t = sc.InitializeFloat(RI_T, 0);

  const char *tex0 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT0);
  const char *tex1 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT1);
  const char *tex2 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT2);
  const char *tex3 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT3);
  const char *tex4 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT4);
  const char *tex5 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT5);
  const char *tex6 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT6);
  const char *tex7 = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformString(RI_REGISTERT7);

  ClearVector(t0);
  if (tex0)
  {
    Tex0 = sc.GetTexture(tex0);
    if (Tex0)
    {
      Spectrum c = Tex0->Lookup(s, t);
      c.ConvertToRGB(t0);
    }
  }

  ClearVector(t1);
  if (tex1)
  {
    Tex1 = sc.GetTexture(tex1);
    if (Tex1)
    {
      if (Tex1->Type()==TextureMap::COLOR)
      {
	Spectrum c = Tex1->Lookup(s, t);
	c.ConvertToRGB(t1);
      }
      else
      {
	Spectrum g; float temp[3];
	g = Tex1->Lookup(s, t);
	g.ConvertToRGB(temp);
	float _c = temp[0];
	g = Tex1->Lookup(s+1./Tex1->width, t);
	g.ConvertToRGB(temp);
	float _cx = temp[0];
	g = Tex1->Lookup(s,(t+1./Tex1->height));
	g.ConvertToRGB(temp);
	float _cy = temp[0];
	float dcx = _c - _cx;
	float dcy = _c - _cy;
	float recip = 1. / (dcx*dcx + dcy*dcy + 1);
	t1[0] = dcy * recip;
	t1[1] = -dcx * recip;
	t1[2] = recip;
      }
    }
  }

  ClearVector(t2);
  if (tex2)
  {
    Tex2 = sc.GetTexture(tex2);
    if (Tex2)
   {
      Spectrum c = Tex2->Lookup(s, t);
      c.ConvertToRGB(t2);
    }
  }

  ClearVector(t3);
  if (tex3)
  {
    Tex0 = sc.GetTexture(tex3);
    if (Tex3)
    {
      Spectrum c = Tex3->Lookup(s, t);
      c.ConvertToRGB(t3);
    }
  }

  ClearVector(t4);
  if (tex4)
  {
    Tex4 = sc.GetTexture(tex4);
    if (Tex4)
    {
      Spectrum c = Tex4->Lookup(s, t);
      c.ConvertToRGB(t4);
    }
  }

  ClearVector(t5);
  if (tex5)
  {
    Tex0 = sc.GetTexture(tex5);
    if (Tex5)
    {
      Spectrum c = Tex5->Lookup(s, t);
      c.ConvertToRGB(t5);
    }
  }

  ClearVector(t6);
  if (tex6)
  {
    Tex6 = sc.GetTexture(tex6);
    if (Tex6)
    {
      Spectrum c = Tex6->Lookup(s, t);
      c.ConvertToRGB(t6);
    }
  }
  ClearVector(t7);
  if (tex7)
  {
    Tex7 = sc.GetTexture(tex7);
    if (Tex7)
    {
      Spectrum c = Tex7->Lookup(s, t);
      c.ConvertToRGB(t7);
    }
  }

  const Float* m;

  ClearVector(v0);
  v1 = 0;
  m = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_ROUGHNESS);
  if (m) S = *m;
  else S = 0.1;
  S/=8.;
  S = 1./S;
  Ks = sc.InitializeFloat(RI_KS, 0.5);
  Kd = sc.InitializeFloat(RI_KD, 0.5);
  m = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(RI_KR);
  if (m) Kr = *m;
  else Kr = 0.;
  m = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(LRT_KT);
  if (m) Kt = *m;
  else Kt = 0.;
  Ka = sc.InitializeFloat(RI_KA, 0.0);

  uv[0] = s;
  uv[1] = t;

  Float index;
  m = sc.hitInfo->hitPrim->attributes->Surface->surfaceFunction->GetUniformFloat(LRT_INDEX);
  if (m) index = *m;
  else index = 1.0;

  if (sc.entering) I = 1.0 / index;
  else I = index / 1.0;

  
  Vector dU = ((*worldToObject)(sc.hitInfo->dPdU)).Hat();
  Vector dV = ((*worldToObject)(sc.hitInfo->dPdV)).Hat();
  Vector dS = ((*worldToObject)(sc.hitInfo->dPdS)).Hat();
  Vector dT = ((*worldToObject)(sc.hitInfo->dPdT)).Hat();
  dPdU[0] = dU.x;
  dPdU[1] = dU.y;
  dPdU[2] = dU.z;
  dPdV[0] = dV.x;
  dPdV[1] = dV.y;
  dPdV[2] = dV.z;
  dPdS[0] = dS.x;
  dPdS[1] = dS.y;
  dPdS[2] = dS.z;
  dPdT[0] = dT.x;
  dPdT[1] = dT.y;
  dPdT[2] = dT.z;
  
  //  cout << N[0] << " " << N[1] << " " << N[2] << endl;
  //  cout << "(" << L[0] << " " << L[1] << " " << L[2] << ")" << endl;

}

/*********************************************************************************
  Opcodes
 *********************************************************************************/
#define ISVECTOR(a) ( ((a) < 26) )

void SLMachine::PerformRnd(const unsigned char dst) const
{
  Float * d = GetDstReg(dst);
  if (!d) return;

  if (ISVECTOR(dst))
  {
    d[0] = ((float)(rand() % 1000000)) / 1000000.;
    d[1] = ((float)(rand() % 1000000)) / 1000000.;
    d[2] = ((float)(rand() % 1000000)) / 1000000.;
  }
  else *d =((float)(rand() % 1000000)) / 1000000.;
}

void SLMachine::PerformClamp(const unsigned char dst, const unsigned char
			     src0, const unsigned char src1) const
{
  if ( (ISVECTOR(dst) && !(ISVECTOR(src0) && ISVECTOR(src1))) ||
       (!ISVECTOR(dst) && (ISVECTOR(src0) || ISVECTOR(src1))) )
    { Warning("clamp error"); return; }

  Float * d = GetDstReg(dst);
  Float * r = GetSrcReg(src0);
  Float * s = GetSrcReg(src1);

  if (!d || !r || !s) return;

  if (d[0] > s[0]) d[0] = s[0];
  if (d[0] < r[0]) d[0] = r[0];

  if (ISVECTOR(dst))
  {
    if (d[1] > s[1]) d[1] = s[1];
    if (d[1] < r[1]) d[1] = r[1];
    if (d[2] > s[2]) d[2] = s[2];
    if (d[2] < r[2]) d[2] = r[2];
  }
}

void SLMachine::PerformTrace(const unsigned char dst, const unsigned char src0, 
			     const unsigned char src1) const
{
  if (!ISVECTOR(dst) || !ISVECTOR(src0) || !ISVECTOR(src1)) 
    { Warning("trace vector, vector, vector: %d, %d, %d",dst,src0,src1); return; }

  Float * pnt = GetSrcReg(src0);
  Float * vec = GetSrcReg(src1);
  Float * ret = GetDstReg(dst);
  
  if (!pnt || !vec || !ret) return;

  HitInfo d;
  Ray r;
  r.O.x = pnt[0];
  r.O.y = pnt[1];
  r.O.z = pnt[2];
  r.D.x = vec[0];
  r.D.y = vec[1];
  r.D.z = vec[2];

  Transform i = Transform(worldToObject->GetInverse());
  r.D = i(r.D);
  
  Float dist = INFINITY;
  Float alpha;

  Spectrum col = scene->integrator->Integrate(r, &d, &dist, &alpha);
  col.ConvertToRGB(ret);
}

void SLMachine::PerformTurb(const unsigned char dst, const unsigned char src) const
{
  if ( ISVECTOR(dst) ) { Warning("turb scalar, <vector/scalar>"); return; }
  Float * d = GetDstReg(dst);
  Float * s = GetSrcReg(src);
  if (!d || !s) return;
  
  if (ISVECTOR(src))
  {
    double vec[3];
    vec[0] = s[0];
    vec[1] = s[1];
    vec[2] = s[2];
    *d = noise3(vec);
  }
  else *d = noise1(*s);
}

void SLMachine::PerformNorm(const unsigned char dst, const unsigned char src) const
{
  if (! (ISVECTOR(dst) && ISVECTOR(src))) { Warning("norm vector, vector"); return; }
  
  Float* d = GetDstReg(dst);
  Float* s = GetSrcReg(src);

  if (!d || !s) return;

  float mag = 1./sqrtf(s[0]*s[0] + s[1]*s[1] + s[2]*s[2]);
  d[0] = s[0]*mag;
  d[1] = s[1]*mag;
  d[2] = s[2]*mag;
}
  

void SLMachine::PerformLI(const unsigned char dst, const float val) const
{
  if (ISVECTOR(dst)) { Warning("LI only for scalar registers"); return; }
  
  Float* _d = GetDstReg(dst);
  if (!_d) return;
  *_d = val;
}

void SLMachine::PerformPow(const unsigned char dst, const unsigned char
			   src, const unsigned char exp) const
{
  if (ISVECTOR(src) || ISVECTOR(dst)) { Warning("pow: scalar, scalar, scalar"); return; }
  
  Float *d = GetDstReg(dst);
  Float *b = GetSrcReg(src);
  Float *e = GetSrcReg(exp);

  if (!d || !b || !e) return;

  *d = pow(*b, *e);
}

void SLMachine::PerformLIV(const unsigned char dst, const unsigned char index, const float val) const
{
  if (!ISVECTOR(dst)) { Warning("LIV only for vector registers"); return; }
  
  Float* _d = GetDstReg(dst);
  if (!_d) return;
  if (index > 3) { Warning("invalid vector index detected!"); return; }
  _d[index] = val;
}

void SLMachine::PerformDP3(const unsigned char dst, const unsigned char
			   src0, const unsigned char src1) const
{
  if (ISVECTOR(dst) || !ISVECTOR(src0) || !ISVECTOR(src1)) 
    { Warning("DP3: scalar, vector, vector: %d %d %d",dst,src0,src1); return; }
  
  Float* _dst = GetDstReg(dst);
  Float* _src0 = GetSrcReg(src0);
  Float* _src1 = GetSrcReg(src1);

  if (!_dst || !_src0 || !_src1) return;

  *_dst = _src0[0]*_src1[0] + _src0[1]*_src1[1] + _src0[2]*_src1[2];
}

void SLMachine::PerformAdd(const unsigned char dst, const unsigned char
			   src0, const unsigned char src1) const
{
  if (! ( (ISVECTOR(src0) && ISVECTOR(src1) && ISVECTOR(dst)) || 
	 (!ISVECTOR(src0) &&!ISVECTOR(src1) &&!ISVECTOR(dst)) )) { Warning("Add operand error"); return; }

  Float* _dst = GetDstReg(dst);
  Float* _src0 = GetSrcReg(src0);
  Float* _src1 = GetSrcReg(src1);

  if (!_dst || !_src0 || !_src1) return;

  if (ISVECTOR(dst))
  {
    _dst[0] = _src0[0] + _src1[0];
    _dst[1] = _src0[1] + _src1[1];
    _dst[2] = _src0[2] + _src1[2];
  }
  else *_dst = *_src0 + *_src1;
}

void SLMachine::PerformFloor(const unsigned char dst, const unsigned char src) const
{
  if (ISVECTOR(src) || ISVECTOR(dst)) { Warning("floor scalar, scalar"); return; }

  Float *d = GetDstReg(dst);
  Float *s = GetSrcReg(src);

  if (!d || !s) return;

  *d = floorf(*s);
}

void SLMachine::PerformCeil(const unsigned char dst, const unsigned char src) const
{
  if (ISVECTOR(src) || ISVECTOR(dst)) { Warning("ceil scalar, scalar"); return; }

  Float *d = GetDstReg(dst);
  Float *s = GetSrcReg(src);

  if (!d || !s) return;

  *d = ceilf(*s);
}

void SLMachine::PerformDiv(const unsigned char dst, const unsigned char
			   src0, const unsigned char src1) const
{
  if ( ISVECTOR(src1) || (ISVECTOR(dst) && !ISVECTOR(src0))) { Warning("div format error"); return; }

  Float *d = GetDstReg(dst);
  Float *r = GetSrcReg(src0);
  Float *s = GetSrcReg(src1);

  if (!d || !r || !s) return;
  if (ISVECTOR(dst))
  {
    d[0] = r[0] / *s;
    d[1] = r[1] / *s;
    d[2] = r[2] / *s;
  }
  else *d = *r / *s;

}

void SLMachine::PerformSub(const unsigned char dst, const unsigned char
			   src0, const unsigned char src1) const
{
  if (! ( (ISVECTOR(src0) && ISVECTOR(src1) && ISVECTOR(dst)) || 
	 (!ISVECTOR(src0) &&!ISVECTOR(src1) &&!ISVECTOR(dst)) )) { Warning("Add operand error"); return; }


  Float* _dst = GetDstReg(dst);
  Float* _src0 = GetSrcReg(src0);
  Float* _src1 = GetSrcReg(src1);

  if (!_dst || !_src0 || !_src1) return;

  if (ISVECTOR(dst))
  {
    _dst[0] = _src0[0] - _src1[0];
    _dst[1] = _src0[1] - _src1[1];
    _dst[2] = _src0[2] - _src1[2];
  }
  else *_dst = *_src0 - *_src1;
}

void SLMachine::PerformEXP(const unsigned char dst, const unsigned char
			   exponent) const
{
  if (ISVECTOR(dst) || ISVECTOR(exponent)) { Warning("exp scalar, scalar"); return; }
  
  Float* d = GetDstReg(dst);
  Float* e = GetSrcReg(exponent);

  if (!d || !e) return;

  *d = (float) exp(*e);
}

void SLMachine::PerformBLT(const unsigned char a, const unsigned char b,
			   const short rel) const
{
  if (ISVECTOR(a) || ISVECTOR(b)) { Warning ("blt scalar, scalar, imm");  return; }

  Float* c = GetSrcReg(a);
  Float* d = GetSrcReg(b);

  if (!c || !d) return;


  if (*c < *d) ip+=rel;
}

void SLMachine::PerformLC(const unsigned char dst, const unsigned char
			  index, const unsigned char src) const
{
  if (ISVECTOR(src) || !ISVECTOR(dst) || index>3) { 
     Warning("lc {x|y|z|w} vector, scalar: %d %d %d", dst, index, src); return; }
  
  Float* d = GetDstReg(dst);
  Float* s = GetSrcReg(src);

  if (!d || !s) return;
  
  d[index] = *s;
}

void SLMachine::PerformMOVC(const unsigned char dst, const unsigned char
			    index, const unsigned char src) const
{
  if (ISVECTOR(dst) || !ISVECTOR(src) || index > 3) { Warning("movc {x|y|z|w} scalar, vector"); return; }
 
  Float *d = GetDstReg(dst);
  Float *s = GetSrcReg(src);
  
  if (!d || !s) return;

  *d = s[index];
}

void SLMachine::PerformLookup(const unsigned char dst, const unsigned char
			      texture, const unsigned char tc) const
{
  if (!ISVECTOR(dst) || !ISVECTOR(tc)) { Warning("lookup vector, t[N], vector"); return; }
  
  TextureMap* tMap;
  Float* d = GetDstReg(dst);
  Float* s = GetSrcReg(tc);

  switch(texture)
  {
  case rT0: tMap = Tex0; break;
  case rT1: tMap = Tex1; break;
  case rT2: tMap = Tex2; break;
  case rT3: tMap = Tex3; break;
  case rT4: tMap = Tex4; break;
  case rT5: tMap = Tex5; break;
  case rT6: tMap = Tex6; break;
  case rT7: tMap = Tex7; break;
  default:
    Warning("lookup vector, t[N], vector"); return;
  }

  if (!tMap) { Warning("lookup requires valid texture map"); return; }
  if (tMap)
  {
    if (tMap->Type()==TextureMap::COLOR)
    {
      Spectrum c = tMap->Lookup(s[0], s[1]);
      c.ConvertToRGB(d);
    }
    else
    {
      Spectrum g; float temp[3];
      g = tMap->Lookup(s[0], s[1]);
      g.ConvertToRGB(temp);
      float _c = temp[0];
      g = tMap->Lookup(s[0]+1./tMap->width, s[1]);
      g.ConvertToRGB(temp);
      float _cx = temp[0];
      g = tMap->Lookup(s[0],(s[1]+1./tMap->height));
      g.ConvertToRGB(temp);
      float _cy = temp[0];
      float dcx = _c - _cx;
      float dcy = _c - _cy;
      float recip = 1. / (dcx*dcx + dcy*dcy + 1);
      d[0] = dcy * recip;
      d[1] = -dcx * recip;
      d[2] = recip;
    }
  }
}

void SLMachine::PerformMul(const unsigned char dst, const unsigned char
			   src0, const unsigned char src1) const
{
  if ( ((ISVECTOR(src0) || ISVECTOR(src1)) && !ISVECTOR(dst)) ||
       (!ISVECTOR(src0) && !ISVECTOR(src1) && ISVECTOR(dst))) { Warning("Mul operand error"); return; }
  

  Float* d = GetDstReg(dst);
  Float* r = GetSrcReg(src0);
  Float* s = GetSrcReg(src1);
  
  if (!d || !r || !s) return;

  if (ISVECTOR(dst))
  {
    if (ISVECTOR(src0) && ISVECTOR(src1)) // perform cross product
    {
      d[0] = r[1] * s[2] - s[1] * r[2];
      d[1] = r[2] * s[0] - r[0] * s[2];
      d[2] = r[0] * s[1] - s[0] * r[1];
    }
    else if (!ISVECTOR(src0))
    {
      d[0] = s[0] * *r;
      d[1] = s[1] * *r;
      d[2] = s[2] * *r;
    }
    else
    {
      d[0] = r[0]* *s;
      d[1] = r[1]* *s;
      d[2] = r[2]* *s;
    }
  }
  else
  {
    *d = *r * *s;
  }
}

void SLMachine::PerformMov(const unsigned char dst, const unsigned char src) const
{
  if ((ISVECTOR(dst) && !ISVECTOR(src)) ||
      (!ISVECTOR(dst) && ISVECTOR(src))) { Warning("Error detected in mov opcode"); return; }
  

  Float* _dst = GetDstReg(dst);
  Float* _src = GetSrcReg(src);

  if (!_dst || !_src) return;

  if (ISVECTOR(dst))
  {
    _dst[0] = _src[0];
    _dst[1] = _src[1];
    _dst[2] = _src[2];
  }
  else
  {
    *_dst = *_src;
  }
}

Float* SLMachine::GetDstReg(const unsigned char dst) const
{
  Float *_dst;

  switch (dst)
  {
  case rC1:
  case rL:
  case rXY:
  case rV:
  case rDPDS:
  case rDPDT:
  case rDPDU:
  case rDPDV:
  case rUV:
  case rN:
  case rT0:
  case rT1:
  case rT2:
  case rT3:
  case rT4:
  case rT5:
  case rT6:
  case rT7:
  case rS:
  case rKD:
  case rKS:
  case rKR:
  case rKT:
  case rKA:
  case rINDEX:
  case rP:
  case rDE:
  case rC0: Warning("Can't write to register dst"); return NULL;
  case rR0: _dst = r0; break;
  case rR1: _dst = r1; break;
  case rR2: _dst = r2; break;
  case rR3: _dst = r3; break;
  case rV0: _dst = v0; break;
  case rV1: _dst = &v1; break;
  case rS0: _dst = &s0; break;
  case rS1: _dst = &s1; break;
  case rS2: _dst = &s2; break;
  case rS3: _dst = &s3; break;
  default:
    Warning("Not a register: %d", dst);
    return NULL;
  }

  return _dst;
}

Float* SLMachine::GetSrcReg(const unsigned char src) const
{
  Float* _src;

  switch (src)
  {
  case rC1:   _src = C1; break;
  case rL:    _src = L; break;
  case rV:    _src = V; break;
  case rXY:   _src = xy; break;
  case rDPDS: _src = dPdS; break;
  case rDPDT: _src = dPdT; break;
  case rDPDU: _src = dPdU; break;
  case rDPDV: _src = dPdV; break;
  case rUV:   _src = uv;   break;
  case rN:    _src = N; break;
  case rDE:   _src = dE; break;
  case rT0:   _src = t0; break;
  case rT1:   _src = t1; break;
  case rT2:   _src = t2; break;
  case rT3:   _src = t3; break;
  case rT4:   _src = t4; break;
  case rT5:   _src = t5; break;
  case rT6:   _src = t6; break;
  case rT7:   _src = t7; break;
  case rS:    _src = &S; break;
  case rKD:   _src = &Ks; break;
  case rKS:   _src = &Kd; break;
  case rC0:   _src = C0; break;
  case rR0:   _src = r0; break;
  case rR1:   _src = r1; break;
  case rR2:   _src = r2; break;
  case rR3:   _src = r3; break;
  case rV0:   _src = v0; break;
  case rV1:   _src = &v1; break;
  case rS0:   _src = &s0; break;
  case rS1:   _src = &s1; break;
  case rS2:   _src = &s2; break;
  case rS3:   _src = &s3; break;
  case rKT:   _src = &Kt; break;
  case rKA:   _src = &Ka; break;
  case rKR:   _src = &Kr; break;
  case rINDEX: _src = &I; break;
  case rP: _src = P; break;
  default:
    Warning("Not a register: %d", src);
    return NULL;
  }

  return _src;
}

#undef ISVECTOR
