#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <memory.h>
#include "mysl.h"

#include <vector.h>

vector<union _operation> operatorvec;

#define zero(a)  memset((a), 0x00, sizeof((a)))

union _operation operation;

//  define the registers our machine uses
//  C  = constant surface color, defined in RIB
//  L = light position
//  V = view vector
//  xy = point position (x, y, z, w), dPdU, dPdX
//  uv = surface parametrization (u, v, w)
//  n  = surface normal
//  S  = roughness
//  tN = textures
//  rN = temporary vector registers, used for computations
//  sN = temporary scalar registers, used for computations
//  vN = return values (diffuse, specular contributions)
//  sp = stack pointer
//  ra = return address
const char* registersstr[] = 
  { "C0", "C1", "L", "V", "Pw", "xy", "ds",
    "dt", "du", "dv", "uv", "N", "dE",
    "t0", "t1", "t2", "t3", "t4", "t5", 
    "t6", "t7", "r0", "r1", "r2", "r3", 
    "v0", "s0", "s1", "s2", "s3", "v1", 
    "Kd", "Ks", "S", "Kr", "Kt", "Ka", "I", NULL 
  };


//  define the allowable operations
//  nop = no operation
//  movc = copy a component of a vector into a scalar
//  lc = load a scalar into a component of a vector

const char* opsstr[] = 
  { "nop", "mov", "add", "mul", "dp3", 
    "pow", "sub", "rnd", "turb", "li", "ret", 
    "clamp", "liv", "norm", "trace", "blt",
    "jmp", "lookup", "movc", "exp", "lc", "div", 
    "floor", "ceil", NULL 
  };

const char* RType_Options[] = { "rgb", "a", NULL };
const char* RegP_Fields[]   = { "xyz", "dPdU", "dPdV", NULL };
const char* RegT_Options[]  = { "neighbor", "filter", NULL };
const char* Vector_Fields[] = { "x", "y", "z", "w", "xyz", "xyzw", NULL };


/**************************************************
 *  getNextToken()                                *
 **************************************************/
char* getNextToken(FILE* f)
{
  static char str[1024];
  
  if (!f) return NULL;
  
  zero(str);
  if (fscanf(f, "%s", str) == EOF) return NULL;

  return str;
}

char* getNextToken(char** istr)
{
  static char str[1024];
  int i=0;

  if (!*istr) return NULL;

  zero(str);
  if (sscanf(*istr, "%s", str)==EOF) return NULL;

  *istr += strlen(str)+1;

  while (str[i]==' ') i++;

  return &(str[i]);
}

char* getNextLine(FILE* f)
{
  static char str[1024];
  if (!f) return NULL;

  zero(str);
  if (fgets(str, 1024, f)) return str;

  return NULL;
}

int getOperand(char** istr)
{
  char* c = getNextToken(istr);
  if (!c) return -1;

  for (int i=0; i<NUM_OPS; i++)
    if (!strcmp(opsstr[i], c)) return i;

  return INVALID_OPERATION;
}

bool checkHeader(FILE *f)
{
  char hs[64];
  float ver;

  fscanf(f, "%s %f", hs, &ver);

  if (strcmp(hs, "sl") || ver!=1.0f) return false;

  return true;
}

/****************************************************
 *  Parse instruction functions.                    *
 ****************************************************/
bool ParseJmp(char** str, int opcode, union _operation* op)
{
  op->op_branch.opcode = opcode;
  op->op_branch.reg_val0 = INVALID_REGISTER;
  op->op_branch.reg_val1 = INVALID_REGISTER;
  op->op_branch.jumprel = 0xffff;
  
  char* c = getNextToken(str);
  if (!c) return false;

  if (opcode==BLT)
  {
  
    for (int i=0; i<NUM_REGISTERS; i++)
      if (strstr(c, registersstr[i])) op->op_branch.reg_val0 = i;

    c = getNextToken(str);
    if (!c) return false;

    for (int i=0; i<NUM_REGISTERS; i++)
      if (strstr(c, registersstr[i])) op->op_branch.reg_val1 = i;

    c = getNextToken(str);
    if (!c) return false;
  }
 
  op->op_branch.jumprel = (short) atoi(c);

  if ((op->op_branch.reg_val0 == INVALID_REGISTER ||
      op->op_branch.reg_val1 == INVALID_REGISTER) && opcode==BLT) return false;

  return true;
}

bool ParseCopy(char** str, int opcode, union _operation* op)
{
  op->op_copy_scalar.opcode = opcode;
  
  char* c = getNextToken(str);

  if (!c) return false;
  switch (c[0])
  {
  case 'x':
  case 'r':
  case 'X':
  case 'R': op->op_copy_scalar.reserved = 0; break;
  case 'y':
  case 'g':
  case 'Y':
  case 'G': op->op_copy_scalar.reserved = 1; break;
  case 'z':
  case 'b':
  case 'Z':
  case 'B': op->op_copy_scalar.reserved = 2; break;
  case 'w':
  case 'a':
  case 'W':
  case 'A': op->op_copy_scalar.reserved = 3; break;
  default: cout << "Bad vector operand" << endl; return false;
  }

  c = getNextToken(str);

  if (!c) return false;
  for (int i=0; i<NUM_REGISTERS; i++)
    if (strstr(c, registersstr[i])) op->op_copy_scalar.r_dst = i;

  c = getNextToken(str);
 
  if (!c) return false;
  for (int i=0; i<NUM_REGISTERS; i++)
    if (strstr(c, registersstr[i])) op->op_copy_scalar.r_src = i;


  if (op->op_copy_scalar.r_dst == INVALID_REGISTER ||
      op->op_copy_scalar.r_src == INVALID_REGISTER) return false;
  
  return true;
}

bool ParseRType(char** str, int opcode, union _operation* op)
{
  op->op_3.opcode = opcode;

  op->op_3.r_dst = INVALID_REGISTER;
  op->op_3.r_src0 = INVALID_REGISTER;
  op->op_3.r_src1 = INVALID_REGISTER;

  char* c = getNextToken(str);

  if (!c) return false;

  for (int i=0; i<NUM_REGISTERS; i++)
    if (strstr(c, registersstr[i])) op->op_3.r_dst = i;

  if (opcode!=RND)
  {
    c = getNextToken(str);
    for (int i=0; i<NUM_REGISTERS; i++)
      if (strstr(c, registersstr[i])) op->op_3.r_src0 = i;
  }

  if (!(opcode == MOV || opcode==RND || opcode==TURB || 
	opcode==NORM || opcode==EXP || opcode==FLOOR || opcode==CEIL))
  {
    c = getNextToken(str);
    for (int i=0; i<NUM_REGISTERS; i++)
      if (strstr(c, registersstr[i])) op->op_3.r_src1 = i;
  }

  if (  op->op_3.r_dst == INVALID_REGISTER || 

      (op->op_3.r_src0 == INVALID_REGISTER && opcode!=RND) || 
      (op->op_3.r_src1 == INVALID_REGISTER &&
        (opcode!=MOV && opcode!=TURB && opcode!=RND && 
	 opcode!=NORM && opcode!=EXP && opcode!=CEIL && opcode!=FLOOR)))
    return false;

  return true;
}

bool ParseImmType(char** str, int opcode, union _operation* op)
{
  op->op_li_float.opcode = opcode;
  op->op_li_float.r_dst = INVALID_REGISTER;

  char* c = getNextToken(str);

  if (!c) return false;

  if (opcode==LIV) 
  {
    switch (c[0])
    {
    case 'x':
    case 'r':
    case 'X':
    case 'R': op->op_li_float.reserved = 0; break;
    case 'y':
    case 'g':
    case 'Y':
    case 'G': op->op_li_float.reserved = 1; break;
    case 'z':
    case 'b':
    case 'Z':
    case 'B': op->op_li_float.reserved = 2; break;
    case 'w':
    case 'a':
    case 'W':
    case 'A': op->op_li_float.reserved = 3; break;
    default: cout << "Bad vector operand" << endl; return false;
    }

    c = getNextToken(str); 
    if (!c) return false; 
  }

  for (int i=0; i<NUM_REGISTERS; i++)
    if (strstr(c, registersstr[i])) op->op_li_float.r_dst = i;

  c = getNextToken(str);
  if (!c) return false;

  op->op_li_float.f = atof(c);

  if (op->op_li_float.r_dst == INVALID_REGISTER) return false;

  return true;
}

/**************************************************
 *  SLcompile                                     *
 **************************************************/

#define ERROR fclose(_if); fclose(_of); cout << "Error on line " << line << endl;

bool SLcompile(const char* on, const char* in)
{
  if (strcmp(on,in)==0) { cout << "File names identical!" << endl; return false; }
  FILE* _if = fopen(in, "r");
  FILE* _of = fopen(on, "wb");

  static int line = 0;

  operatorvec.clear();

  cout << "SL 1.0 assembler" << endl;

  if (!_if || !_of) { cout << "Can't open files!" << endl; return false; }

  char* s;

  if (!checkHeader(_if)) { cout << "Bad header." << endl; fclose(_if); fclose(_of); return false; }

  while ( !feof(_if) && (s = getNextLine(_if))!=NULL )
  {
    line++;
    if (s[0] == '#') continue; // comment
    int opcode = getOperand(&s);
    if (opcode == -1) continue;  // get rid of white space
    if (opcode == INVALID_OPERATION) { ERROR cout << "Invalid opcode" <<
					 endl; break;}

    union _operation op;
    
    switch(opcode)
    {
    case BLT:
    case JMP: if (!ParseJmp(&s, opcode, &op)) { ERROR break; } break;
    case LC:
    case MOVC: if (!ParseCopy(&s, opcode, &op)) { ERROR break; } break;
    case DIV:
    case LOOKUP:
    case EXP:
    case MOV:
    case NORM:
    case MUL:
    case ADD:
    case SUB:
    case TRACE:
    case POW:
    case TURB:
    case RND:
    case CLAMP:
    case FLOOR:
    case CEIL:
    case DP3: if (!ParseRType(&s, opcode, &op)) { ERROR break; } break;
    case RET:
    case NOP: op.op_3.opcode = opcode; break;
    case LIV:
    case LI:  if (!ParseImmType(&s, opcode, &op)) { ERROR break; } break;
    }

    operatorvec.push_back(op);

  }

  fclose(_if);

  for (unsigned int i=0; i<operatorvec.size(); i++) 
  {
    fwrite(&operatorvec[i], sizeof(union _operation), 1, _of);
  }
  fclose(_of);

  return true;
}
