%{
#include "ri.h"
#include "lrt.h"

#include <stdarg.h>

extern int yylex( void );
int line_num = 1;

#define YYMAXDEPTH 100000000

void yyerror( char *str ) {
	Error( "RIB parsing error on line %d\n\t%s", line_num,str );
}

void ParseError( const char *format, ... ) PRINTF_FORMAT;

void ParseError( const char *format, ... ) {
	char error[4096];
	va_list args;
	va_start( args, format );
	vsprintf( error, format, args );
	yyerror(error);
	va_end( args );
}

// RI says 64k is the maxiumum light handle number
#define MAX_LIGHT_HANDLE 65536
static RtLightHandle lights[MAX_LIGHT_HANDLE];

int cur_paramlist_allocated = 0;
int cur_paramlist_size = 0;
RtToken *cur_paramlist_tokens = NULL;
RtPointer *cur_paramlist_args = NULL;
RtInt *cur_paramlist_sizes = NULL;

#define CPS cur_paramlist_size
#define CPT cur_paramlist_tokens
#define CPA cur_paramlist_args
#define CPSZ cur_paramlist_sizes

typedef struct RIB_array {
	int element_size;
	int allocated;
	int nelems;
	RtVoid *array;
} RIB_array;

RIB_array *cur_array = NULL;

#define NA(r) ((RtFloat *) r->array)
#define SA(r) ((const char **) r->array)

void AddArrayElement( void *elem ) {
	if (cur_array->nelems >= cur_array->allocated) {
		cur_array->allocated = 2*cur_array->allocated + 1;
		cur_array->array = realloc( cur_array->array,
			cur_array->allocated*cur_array->element_size );
	}
	char *next = ((char *)cur_array->array) + cur_array->nelems *
		cur_array->element_size;
	memcpy( next, elem, cur_array->element_size );
	cur_array->nelems++;
}

RIB_array *ArrayDup( RIB_array *ra )
{
	RIB_array *ret = new RIB_array;
	ret->element_size = ra->element_size;
	ret->allocated = ra->allocated;
	ret->nelems = ra->nelems;
	ret->array = new char[ra->nelems * ra->element_size];
	memcpy( ret->array, ra->array, ra->nelems * ra->element_size );
	return ret;
}

void ArrayFree( RIB_array *ra )
{
	delete [] ((char *) ra->array);
	delete ra;
}

static RtFloat * FindBasis( const char *str )
{
	/*
	if (!strcmp( str, "bezier" )) return (RtFloat *) RiBezierBasis;
	else if (!strcmp( str, "b-spline" )) return (RtFloat *) RiBSplineBasis;
	else if (!strcmp( str, "catmull-rom" )) return (RtFloat *) RiCatmullRomBasis;
	else if (!strcmp( str, "hermite" )) return (RtFloat *) RiHermiteBasis;
	else if (!strcmp( str, "power" )) return (RtFloat *) RiPowerBasis;
	else ParseError( "Unknown basis: %s", str );
	*/

	Warning("Named bases aren't supported yet...");

	return NULL;
}
static bool VerifyArrayLength( RIB_array *arr, RtInt required,
	const char *command ) {
	if (arr->nelems != required) {
		ParseError( "%s requires a(n) %d element array!", command, required);
		return false;
	}
	return true;
}

static RtInt *MakeIntArray( RIB_array *arr )
{
	RtInt *ret = new RtInt[arr->nelems];
	for (int i = 0 ; i < arr->nelems ; i++)
		ret[i] = RtInt(NA(arr)[i]);
	return ret;
}

static RtFilterFunc FindFilterFunc( const char *name )
{
	if (!strcmp( name, "box" ))
		return RiBoxFilter;
	else if (!strcmp( name, "triangle" ))
		return RiTriangleFilter;
	else if (!strcmp( name, "catmull-rom" ))
		return RiCatmullRomFilter;
	else if (!strcmp( name, "sinc" ))
		return RiSincFilter;
	else if (!strcmp( name, "gaussian" ))
		return RiGaussianFilter;
	else
		ParseError( "Unknown filter function :%s", name );
	return RtFilterFunc(NULL);
}
%}

%union {
char string[1024];
RtFloat num;
RIB_array *ribarray;
}
%token <string> STRING ID
%token <num> NUM
%token LBRACK RBRACK DASH


%token AREALIGHTSOURCE ATMOSPHERE ATTRIBUTE ATTRIBUTEBEGIN ATTRIBUTEEND BASIS
%token BOUND CLIPPING COLOR COLORSAMPLES CONCATTRANSFORM
%token CONE BUTTERFLY COORDINATESYSTEM CROPWINDOW CYLINDER DECLARE
%token DEFORMATION DEPTHOFFIELD DETAIL DETAILRANGE DISK
%token DISPLACEMENT DISPLAY EXPOSURE EXTERIOR FORMAT
%token FRAMEASPECTRATIO FRAMEBEGIN FRAMEEND GENERALPOLYGON
%token GEOMETRICAPPROXIMATION GEOMETRY HIDER HYPERBOLOID
%token IDENTITY ILLUMINATE IMAGER INTERIOR LIGHTSOURCE
%token MAKEBUMP MAKECUBEFACEENVIRONMENT MAKELATLONGENVIRONMENT
%token MAKESHADOW MAKETEXTURE MATTE MOTIONBEGIN MOTIONEND
%token NUPATCH OBJECTBEGIN OBJECTEND OBJECTINSTANCE OPACITY
%token OPTION ORIENTATION PARABOLOID PATCH PATCHMESH PERSPECTIVE PIXELFILTER
%token PIXELSAMPLES PIXELVARIANCE POINTSGENERALPOLYGONS
%token POINTSPOLYGONS POLYGON PROJECTION QUANTIZE
%token RELATIVEDETAIL REVERSEORIENTATION ROTATE SCALE
%token SCREENWINDOW SHADINGINTERPOLATION SHADINGRATE SHUTTER
%token SIDES SKEW SOLIDBEGIN SOLIDEND SPHERE SURFACE
%token TEXTURECOORDINATES TORUS TRANSFORM TRANSFORMBEGIN
%token TRANSFORMEND TRANSLATE TRIMCURVE
%token VERSION WORLDBEGIN WORLDEND

%token HIGH_PRECEDENCE

%type<ribarray> array num_array string_array
%type<ribarray> real_num_array real_string_array
%%
start: ri_stmt_list

array_init: %prec HIGH_PRECEDENCE
{
	if (cur_array) ArrayFree( cur_array );
	cur_array = new RIB_array;
	cur_array->allocated = 0;
	cur_array->nelems = 0;
	cur_array->array = NULL;
}

string_array_init: %prec HIGH_PRECEDENCE
{
	cur_array->element_size = sizeof( const char * );
}

num_array_init: %prec HIGH_PRECEDENCE
{
	cur_array->element_size = sizeof( RtFloat );
}

array: string_array
{
	$$ = $1;
}
| num_array
{
	$$ = $1;
}

string_array: real_string_array
{
	$$ = $1;
}
| single_element_string_array
{
	$$ = ArrayDup(cur_array);
}

real_string_array: array_init LBRACK string_list RBRACK
{
	$$ = ArrayDup(cur_array);
}
single_element_string_array: array_init string_list_entry

string_list: string_list string_list_entry
| string_list_entry

string_list_entry: string_array_init STRING
{
	char *to_add = strdup($2);
	AddArrayElement( &to_add );
}

num_array: real_num_array
{
	$$ = $1;
}
| single_element_num_array
{
	$$ = ArrayDup(cur_array);
}

real_num_array: array_init LBRACK num_list RBRACK
{
	$$ = ArrayDup(cur_array);
}
single_element_num_array: array_init num_list_entry

num_list: num_list num_list_entry
	| num_list_entry

num_list_entry: num_array_init NUM
{
	RtFloat to_add = $2;
	AddArrayElement( &to_add );
}

paramlist: paramlist_init paramlist_contents

paramlist_init: %prec HIGH_PRECEDENCE
{
	cur_paramlist_size = 0;
}

paramlist_contents: paramlist_entry paramlist_contents
	|

paramlist_entry: STRING array
{
	RtPointer arg = new char[ $2->nelems * $2->element_size ];
	memcpy( arg, $2->array, $2->nelems * $2->element_size );
	if (cur_paramlist_size >= cur_paramlist_allocated) {
		cur_paramlist_allocated = 2*cur_paramlist_allocated + 1;
		cur_paramlist_tokens = (RtToken *) realloc( cur_paramlist_tokens, cur_paramlist_allocated*sizeof(RtToken) );
		cur_paramlist_args = (RtPointer *) realloc( cur_paramlist_args, cur_paramlist_allocated*sizeof(RtPointer) );
		cur_paramlist_sizes = (RtInt *) realloc( cur_paramlist_sizes, cur_paramlist_allocated*sizeof(RtInt) );
	}
	cur_paramlist_tokens[cur_paramlist_size] = $1;
	cur_paramlist_sizes[cur_paramlist_size] = $2->nelems;
	cur_paramlist_args[cur_paramlist_size++] = arg;
	ArrayFree( $2 );
}

ri_stmt_list: ri_stmt_list ri_stmt
	| ri_stmt

ri_stmt: VERSION NUM { printf ("RIB Version: %f\n", $2 ); }
| AREALIGHTSOURCE STRING NUM paramlist
{
    int num = (int)$3;
	if (num < 0 || num >= MAX_LIGHT_HANDLE)
		Error("Invalid light handle %d passed to AreaLightSource. Ignoring.", num);
	else
		lights[num] = RiAreaLightSourceV( $2, CPS, CPT, CPA );
}
| ATMOSPHERE STRING paramlist
{
	RiAtmosphereV( $2, CPS, CPT, CPA );
}
| ATTRIBUTE STRING paramlist
{
	RiAttributeV( $2, CPS, CPT, CPA );
}
| ATTRIBUTEBEGIN
{
	RiAttributeBegin();
}
| ATTRIBUTEEND
{
	RiAttributeEnd();
}
| BASIS STRING NUM STRING NUM
{
	RtFloat * basis1 = FindBasis( $2 );
	RtFloat * basis2 = FindBasis( $4 );
	RiBasis( basis1, RtInt($3), basis2, RtInt($5) );
}
| BASIS STRING NUM real_num_array NUM
{
	if (VerifyArrayLength( $4, 16, "Basis" )) {
		RtFloat * basis1 = FindBasis( $2 );
		RiBasis( basis1, RtInt($3), (RtFloat *) $4->array, RtInt($5) );
	}
	ArrayFree( $4 );
}
| BASIS real_num_array NUM STRING NUM
{
	if (VerifyArrayLength( $2, 16, "Basis" )) {
		RtFloat * basis2 = FindBasis( $4 );
		RiBasis( (RtFloat *) $2->array, RtInt($3), basis2, RtInt($5) );
	}
	ArrayFree( $2 );
}
| BASIS real_num_array NUM real_num_array NUM
{
	if (VerifyArrayLength($2, 16, "Basis") && VerifyArrayLength($4, 16, "Basis"))
		RiBasis( (RtFloat *) $2->array, RtInt($3), (RtFloat *) $4->array, RtInt($5) );
	ArrayFree( $2 );
	ArrayFree( $4 );
}
| BOUND NUM NUM NUM NUM NUM NUM
{
	RtBound bound;
	bound[0] = $2; bound[1] = $3; bound[2] = $4;
	bound[3] = $5; bound[4] = $6; bound[5] = $7;
	RiBound( bound );
}
| BOUND real_num_array
{
	if (VerifyArrayLength( $2, 6, "Bound" ))
		RiBound( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| CLIPPING NUM NUM
{
	RiClipping( $2, $3 );
}
| COLOR NUM NUM NUM
{
	RtBound color;
	color[0] = $2; color[1] = $3; color[2] = $4;
	RiColor( color );
}
| COLOR real_num_array
{
	if (VerifyArrayLength( $2, 3, "Color" ));
		RiColor( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| COLORSAMPLES real_num_array real_num_array
{
	if ($2->nelems != $3->nelems) {
		ParseError( "The two arrays should be the same size in ColorSamples" );
		goto noGoodColor;
	}
	if ($2->nelems % 3 != 0) {
		ParseError( "The two arrays should be the multiples of 3 in ColorSamples" );
		goto noGoodColor;
	}
	RiColorSamples( $2->nelems / 3, NA($2), NA($3) );
noGoodColor:
	ArrayFree( $2 );
	ArrayFree( $3 );
}
| CONE NUM NUM NUM paramlist
{
	RiConeV( $2, $3, $4, CPS, CPT, CPA );
}
| BUTTERFLY NUM NUM NUM NUM paramlist
{
  RiButterflyV( $2, $3, $4, $5, CPS, CPT, CPA );
}
| BUTTERFLY real_num_array paramlist
{
	VerifyArrayLength( $2, 4, "Butterfly" );
	RiButterflyV( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3], CPS, CPT, CPA );
}
| CONE real_num_array paramlist
{
	VerifyArrayLength( $2, 3, "Cone" );
	RiConeV( NA($2)[0], NA($2)[1], NA($2)[2], CPS, CPT, CPA );
}
| CONCATTRANSFORM num_array
{
	if (VerifyArrayLength( $2, 16, "ConcatTransform" ));
		RiConcatTransform( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| COORDINATESYSTEM STRING
{
	RiCoordinateSystem( $2 );
}
| CROPWINDOW NUM NUM NUM NUM
{
	RiCropWindow( $2, $3, $4, $5 );
}
| CROPWINDOW real_num_array
{
	if (VerifyArrayLength( $2, 4, "CropWindow" ))
		RiCropWindow( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3] );
	ArrayFree( $2 );
}
| CYLINDER NUM NUM NUM NUM paramlist
{
	RiCylinderV( $2, $3, $4, $5, CPS, CPT, CPA );
}
| CYLINDER real_num_array paramlist
{
	if (VerifyArrayLength( $2, 4, "Cylinder" ))
		RiCylinderV( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3], CPS, CPT, CPA );
	ArrayFree( $2 );
}
| DECLARE STRING STRING
{
	RiDeclare( $2, $3 );
}
| DEFORMATION STRING paramlist
{
	RiDeformationV( $2, CPS, CPT, CPA );
}
| DEPTHOFFIELD NUM NUM NUM
{
	RiDepthOfField( $2, $3, $4 );
}
| DEPTHOFFIELD DASH
{
	RiDepthOfField( RI_INFINITY, 0, 0 );
}
| DETAIL NUM NUM NUM NUM NUM NUM
{
	RtBound bound;
	bound[0] = $2; bound[1] = $3; bound[2] = $4;
	bound[3] = $5; bound[4] = $6; bound[5] = $7;
	RiDetail( bound );
}
| DETAIL real_num_array
{
	if (VerifyArrayLength( $2, 6, "Detail" ))
		RiDetail( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| DETAILRANGE NUM NUM NUM NUM
{
	RiDetailRange( $2, $3, $4, $5 );
}
| DETAILRANGE real_num_array
{
	if (VerifyArrayLength( $2, 4, "DetailRange" ))
		RiDetailRange( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3] );
	ArrayFree( $2 );
}
| DISK NUM NUM NUM paramlist
{
	RiDiskV( $2, $3, $4, CPS, CPT, CPA );
}
| DISK real_num_array paramlist
{
	if (VerifyArrayLength( $2, 3, "Disk" ))
		RiDiskV( NA($2)[0], NA($2)[1], NA($2)[2], CPS, CPT, CPA );
	ArrayFree( $2 );
}
| DISPLACEMENT STRING paramlist
{
	RiDisplacementV( $2, CPS, CPT, CPA );
}
| DISPLAY STRING STRING STRING paramlist
{
	RiDisplayV( $2, $3, $4, CPS, CPT, CPA );
}
| EXPOSURE NUM NUM
{
	RiExposure( $2, $3 );
}
| EXTERIOR STRING paramlist
{
	RiExteriorV( $2, CPS, CPT, CPA );
}
| FORMAT NUM NUM NUM
{
	RiFormat( RtInt($2), RtInt($3), $4 );
}
| FRAMEASPECTRATIO NUM
{
	RiFrameAspectRatio( $2 );
}
| FRAMEBEGIN NUM
{
	RiFrameBegin( RtInt($2) );
}
| FRAMEEND
{
	RiFrameEnd();
}
| GENERALPOLYGON num_array paramlist
{
	RtInt *verts = MakeIntArray( $2 );
	RiGeneralPolygonV( $2->nelems, verts, CPS, CPT, CPA );
	delete [] verts;
	ArrayFree( $2 );
}
| GEOMETRICAPPROXIMATION STRING NUM
{
	RiGeometricApproximation( $2, $3 );
}
| GEOMETRY STRING paramlist
{
	RiGeometryV( $2, CPS, CPT, CPA );
}
| HIDER STRING paramlist
{
	RiHiderV( $2, CPS, CPT, CPA );
}
| HYPERBOLOID NUM NUM NUM NUM NUM NUM NUM paramlist
{
	RtPoint point1, point2;
	point1[0] = $2; point1[1] = $3; point1[2] = $4;
	point2[0] = $5; point2[1] = $6; point2[2] = $7;
	RiHyperboloidV( point1, point2, $8, CPS, CPT, CPA );
}
| HYPERBOLOID real_num_array paramlist
{
	RtBound point1, point2;
	if (VerifyArrayLength( $2, 7, "Hyperboloid" )) {
		point1[0] = NA($2)[0]; point1[1] = NA($2)[1]; point1[2] = NA($2)[2];
		point2[0] = NA($2)[3]; point2[1] = NA($2)[4]; point2[2] = NA($2)[5];
		RiHyperboloidV( point1, point2, NA($2)[6], CPS, CPT, CPA );
	}
	ArrayFree( $2 );
}
| IDENTITY
{
	RiIdentity();
}
| ILLUMINATE NUM NUM
{
    int num = (int)$2;
	if (num < 0 || num >= MAX_LIGHT_HANDLE)
		Error("Invalid light handle %d passed to Illuminate. Ignoring.", num);
	else {
		RtLightHandle light = lights[num];
		if (light == (RtLightHandle)RI_NULL)
			ParseError( "Unknown light number: %d", num);
		else
			RiIlluminate( light, (RtBoolean) $3 );
	}
}
| IMAGER STRING paramlist
{
	RiImagerV( $2, CPS, CPT, CPA );
}
| INTERIOR STRING paramlist
{
	RiInteriorV( $2, CPS, CPT, CPA );
}
| LIGHTSOURCE STRING NUM paramlist
{
    int num = (int)$3;
	if (num < 0 || num >= MAX_LIGHT_HANDLE)
		Error("Invalid light handle %d passed to LightSource. Ignoring.", num);
	else
		lights[num] = RiLightSourceV( $2, CPS, CPT, CPA );
}
| MAKEBUMP STRING STRING STRING STRING STRING NUM NUM paramlist
{
	RtFilterFunc filterfunc = FindFilterFunc( $6 );
	RiMakeBumpV( $2, $3, $4, $5, filterfunc, $7, $8, CPS, CPT, CPA );
}
| MAKECUBEFACEENVIRONMENT STRING STRING STRING STRING STRING STRING STRING NUM STRING NUM NUM paramlist
{
	RtFilterFunc filterfunc = FindFilterFunc( $10 );
	RiMakeCubeFaceEnvironmentV( $2, $3, $4, $5, $6, $7, $8, $9, filterfunc,
		$11, $12, CPS, CPT, CPA );
}
| MAKELATLONGENVIRONMENT STRING STRING STRING NUM NUM paramlist
{
	RtFilterFunc filterfunc = FindFilterFunc( $4 );
	RiMakeLatLongEnvironmentV( $2, $3, filterfunc, $5, $5, CPS, CPT, CPA );
}
| MAKESHADOW STRING STRING paramlist
{
	RiMakeShadowV( $2, $3, CPS, CPT, CPA );
}
| MAKETEXTURE STRING STRING STRING STRING STRING NUM NUM paramlist
{
	RtFilterFunc filterfunc = FindFilterFunc( $6 );
	RiMakeTextureV( $2, $3, $4, $5, filterfunc, $7, $8, CPS, CPT, CPA );
}
| MATTE NUM
{
	RiMatte( $2 != 0. );
}
| MOTIONBEGIN num_array
{
	RiMotionBeginV( $2->nelems, NA($2) );
	ArrayFree( $2 );
}
| MOTIONEND
{
	RiMotionEnd();
}
| NUPATCH NUM NUM real_num_array NUM NUM NUM NUM real_num_array NUM NUM paramlist
{
	RiNuPatchV( RtInt($2), RtInt($3), NA($4), $5, $6,
				RtInt($7), RtInt($8), NA($9), $10, $11, CPS, CPT, CPA );
	ArrayFree($4);
	ArrayFree($9);
}
| OBJECTBEGIN NUM
{
	Warning( "Don't know how to parse ObjectInstance" );
}
| OBJECTEND
{
	RiObjectEnd();
}
| OBJECTINSTANCE NUM
{
	Warning( "Don't know how to parse ObjectInstance" );
}
| OPACITY NUM NUM NUM
{
	RtColor color;
	color[0] = $2; color[1] = $3; color[2] = $4;
	RiOpacity( color );
}
| OPACITY real_num_array
{
	if (VerifyArrayLength( $2, 3, "Opacity" ))
		RiOpacity( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| OPTION STRING paramlist
{
	RiOptionV( $2, CPS, CPT, CPA );
}
| ORIENTATION STRING
{
	RiOrientation( $2 );
}
| PARABOLOID NUM NUM NUM NUM paramlist
{
	RiParaboloidV( $2, $3, $4, $5, CPS, CPT, CPA );
}
| PARABOLOID real_num_array paramlist
{
	if (VerifyArrayLength( $2, 4, "Paraboloid" ))
		RiParaboloidV( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3], CPS, CPT, CPA );
	ArrayFree( $2 );
}
| PATCH STRING paramlist
{
	RiPatchV( $2, CPS, CPT, CPA );
}
| PATCHMESH STRING NUM STRING NUM STRING paramlist
{
	RiPatchMeshV( $2, RtInt($3), $4, RtInt($5), $6, CPS, CPT, CPA );
}
| PERSPECTIVE NUM
{
	RiPerspective( $2 );
}
| PIXELFILTER STRING NUM NUM
{
	RtFilterFunc func = FindFilterFunc( $2 );
	RiPixelFilter( func, $3, $4 );
}
| PIXELSAMPLES NUM NUM
{
	RiPixelSamples( $2, $3 );
}
| PIXELVARIANCE NUM
{
	RiPixelVariance( $2 );
}
| POINTSGENERALPOLYGONS num_array num_array num_array paramlist
{
	RtInt *nloops, *nvertices, *vertices;

	nloops = MakeIntArray( $2 );
	nvertices = MakeIntArray( $3 );
	vertices = MakeIntArray( $4 );

	RiPointsGeneralPolygonsV( $2->nelems, nloops, nvertices, vertices, CPS, CPT, CPA );

	delete [] nloops;
	delete [] nvertices;
	delete [] vertices;
	ArrayFree( $2 );
	ArrayFree( $3 );
	ArrayFree( $4 );
}
| POINTSPOLYGONS num_array num_array paramlist
{
	// Gotta convert these bad boys to arrays of integers.  that sucks.
	RtInt *nverts, *verts;

	nverts = MakeIntArray( $2 );
	verts = MakeIntArray( $3 );
	int tot = 0;
	for (int i = 0 ; i < $2->nelems ; i++)
		tot += nverts[i];
	if (tot != $3->nelems)
		ParseError("Wrong number of vertex elements in PointsPolygons. Ignoring.");
	else
		RiPointsPolygonsV( $2->nelems, nverts, verts, CPS, CPT, CPA );

	delete [] nverts;
	delete [] verts;
	ArrayFree( $2 );
	ArrayFree( $3 );
}
| POLYGON paramlist
{
	RtInt num_verts = -1;

	for (int i = 0 ; i < CPS ; i++) {
		if (CPT[i] == RI_P || strcmp(CPT[i], RI_P) == 0) {
			num_verts = CPSZ[i];
			if (num_verts % 3 != 0) {
				ParseError( "RI_P array must be 3*n in length." );
				goto noGoodPolygon;
			}
			num_verts /= 3;
			break;
		}
	}
	if (num_verts == -1)
		ParseError( "Polygon without an RI_P parameter???" );
	else
		RiPolygonV( num_verts, CPS, CPT, CPA );
noGoodPolygon:
	;
}
| PROJECTION STRING paramlist
{
	RiProjectionV( $2, CPS, CPT, CPA );
}
| QUANTIZE STRING NUM NUM NUM NUM
{
	RiQuantize( $2, RtInt($3), RtInt($4), RtInt($5), $6 );
}
| RELATIVEDETAIL NUM
{
	RiRelativeDetail( $2 );
}
| REVERSEORIENTATION
{
	RiReverseOrientation();
}
| ROTATE NUM NUM NUM NUM
{
	RiRotate( $2, $3, $4, $5 );
}
| SCALE NUM NUM NUM
{
	RiScale( $2, $3, $4 );
}
| SCREENWINDOW NUM NUM NUM NUM
{
	RiScreenWindow( $2, $3, $4, $5 );
}
| SCREENWINDOW real_num_array
{
	if (VerifyArrayLength( $2, 4, "ScreenWindow" ))
		RiScreenWindow( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3] );
	ArrayFree( $2 );
}
| SHADINGINTERPOLATION STRING
{
	RiShadingInterpolation( $2 );
}
| SHADINGRATE NUM
{
	RiShadingRate( $2 );
}
| SHUTTER NUM NUM
{
	RiShutter( $2, $3 );
}
| SIDES NUM
{
	RiSides( RtInt($2) );
}
| SKEW NUM NUM NUM NUM NUM NUM NUM
{
	RiSkew( $2, $3, $4, $5, $6, $7, $8 );
}
| SKEW real_num_array
{
	if (VerifyArrayLength( $2, 7, "Skew" ))
		RiSkew( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3],
				NA($2)[4], NA($2)[5], NA($2)[6] );
	ArrayFree( $2 );
}
| SOLIDBEGIN STRING
{
	RiSolidBegin( $2 );
}
| SOLIDEND
{
	RiSolidEnd();
}
| SPHERE NUM NUM NUM NUM paramlist
{
	RiSphereV( $2, $3, $4, $5, CPS, CPT, CPA );
}
| SPHERE real_num_array paramlist
{
	if (VerifyArrayLength( $2, 4, "Sphere" ))
		RiSphereV( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3], CPS, CPT, CPA );
	ArrayFree( $2 );
}
| SURFACE STRING paramlist
{
	RiSurfaceV( $2, CPS, CPT, CPA );
}
| TEXTURECOORDINATES NUM NUM NUM NUM NUM NUM NUM NUM
{
	RiTextureCoordinates( $2, $3, $4, $5,
						  $6, $7, $8, $9 );
}
| TEXTURECOORDINATES real_num_array
{
	if (VerifyArrayLength( $2, 8, "TextureCoordinates" ))
		RiTextureCoordinates( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3],
							  NA($2)[4], NA($2)[5], NA($2)[7], NA($2)[8] );
	ArrayFree( $2 );
}
| TORUS NUM NUM NUM NUM NUM paramlist
{
	RiTorus( $2, $3, $4, $5, $6, CPS, CPT, CPA );
}
| TORUS real_num_array paramlist
{
	if (VerifyArrayLength( $2, 5, "Torus" ))
		RiTorus( NA($2)[0], NA($2)[1], NA($2)[2], NA($2)[3], NA($2)[4],
			CPS, CPT, CPA );
    ArrayFree($2);
}
| TRANSFORM real_num_array
{
	if (VerifyArrayLength( $2, 16, "Transform" ))
		RiTransform( (RtFloat *) $2->array );
	ArrayFree( $2 );
}
| TRANSFORMBEGIN
{
	RiTransformBegin();
}
| TRANSFORMEND
{
	RiTransformEnd();
}
| TRANSLATE NUM NUM NUM
{
	RiTranslate( $2, $3, $4 );
}
| TRIMCURVE num_array num_array num_array num_array num_array num_array num_array num_array num_array
{
	RtInt *ncurves, *order, *n;

	ncurves = MakeIntArray( $2 );
	order = MakeIntArray( $3 );
	n = MakeIntArray( $7 );

	RiTrimCurve( $2->nelems, ncurves, order, NA($4), NA($5), NA($6), n, NA($8), NA($9), NA($10) );
	delete [] ncurves;
	delete [] order;
	delete [] n;

	ArrayFree( $2 );
	ArrayFree( $3 );
	ArrayFree( $4 );
	ArrayFree( $5 );
	ArrayFree( $6 );
	ArrayFree( $7 );
	ArrayFree( $8 );
	ArrayFree( $9 );
	ArrayFree( $10 );
}
| WORLDBEGIN
{
	RiWorldBegin();
}
| WORLDEND
{
	RiWorldEnd();
}
%%
