
#include "trimesh.h"

TriangleMesh::TriangleMesh(int nt, int nv, int *vi, Point * P,
						   PrimitiveAttributes * a, SurfaceFunction * sf)
:Primitive(a, sf)
{
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	p = new Point[nverts];
	memcpy(p, P, nverts * sizeof(Point));
}

TriangleMesh::~TriangleMesh()
{
	delete[]vertexIndex;
	delete[]p;
}

BBox TriangleMesh::BoundObjectSpace() const
{
	BBox objectSpaceBounds(p[0], p[1]);
	for (int i = 2; i < nverts; i++)
		objectSpaceBounds = Union(objectSpaceBounds, p[i]);
	return objectSpaceBounds;
}

BBox TriangleMesh::BoundWorldSpace() const
{
	BBox worldBounds(attributes->ObjectToWorld(p[0]),
					 attributes->ObjectToWorld(p[1]));
	for (int i = 2; i < nverts; i++) {
		worldBounds = Union(worldBounds, attributes->ObjectToWorld(p[i]));
	}
	return worldBounds;
}

void TriangleMesh::Refine(vector < Primitive * >*refined) const
{
	refined->reserve(ntris);
	for (int i = 0; i < ntris; ++i)
		refined->push_back(new Triangle(this, i));
}

BBox Triangle::BoundObjectSpace() const
{
	BBox objectBounds(mesh->p[mesh->vertexIndex[triNum * 3]],
					  mesh->p[mesh->vertexIndex[triNum * 3 + 1]]);
	return Union(objectBounds, mesh->p[mesh->vertexIndex[triNum * 3 + 2]]);
}

BBox Triangle::BoundWorldSpace() const
{
	Transform o2w = mesh->attributes->ObjectToWorld;
	BBox objectBounds(o2w(mesh->p[mesh->vertexIndex[triNum * 3]]),
					  o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 1]]));
	return Union(objectBounds,
				 o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 2]]));
}

bool Triangle::IntersectClosest(const Ray & ray, Float mint,
								Float * maxt, HitInfo * hit) const
{

	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry",
						   "Triangle Ray Intersections", &triangleHits,
						   &triangleTests);

	++triangleTests;

	int *vptr = &(mesh->vertexIndex[3 * triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;

	const Point *p = mesh->p;
	Vector E1 = p[vertexIndex1] - p[vertexIndex0];
	Vector E2 = p[vertexIndex2] - p[vertexIndex0];
	Vector S_1 = Cross(ray.D, E2);
	Float divisor = Dot(S_1, E1);
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;

	Vector T = ray.O - mesh->p[vertexIndex0];
	Float u = Dot(T, S_1) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;

	Vector S_2 = Cross(T, E1);
	Float v = Dot(ray.D, S_2) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;

	Float t = Dot(E2, S_2) * invDivisor;
	if (t < mint || t > *maxt)
		return false;
	*maxt = t;
	++triangleHits;

	if (hit != NULL) {

		Normal N = Normal(Cross(E2, E1));
		N.Normalize();
		hit->RecordHit(ray(t), N, u, v, mesh);

		int interpOffsets[3] =
			{ vertexIndex0, vertexIndex1, vertexIndex2 };
		Float interpWeights[3] = { 1. - u - v, u, v };
		InterpolatedPrimData *interpolatedData;
		interpolatedData =
			mesh->surfaceFunction->Interpolate(3, interpOffsets,
											   interpWeights);
		hit->SetInterpolatedData(interpolatedData);

	}
	return true;
}
