
#include "color.h"

#define CLAMP(v,l,h)    ((v)<(l) ? (l) : (v) > (h) ? (h) : (v))
#define IlluminantC     0.3101, 0.3162  /* For NTSC television */
#define IlluminantD65   0.3127, 0.3291  /* For EBU and SMPTE */

struct colourSystem {
    double xRed, yRed,
           xGreen, yGreen,
           xBlue, yBlue,
           xWhite, yWhite;
};

static struct colourSystem
                  /* xRed   yRed  xGreen yGreen  xBlue  yBlue  White point */
    NTSCsystem  =  { 0.67,  0.33,  0.21,  0.71,  0.14,  0.08,  IlluminantC   },
    EBUsystem   =  { 0.64,  0.33,  0.29,  0.60,  0.15,  0.06,  IlluminantD65 },
    SMPTEsystem =  { 0.630, 0.340, 0.310, 0.595, 0.155, 0.070, IlluminantD65 };

ostream & operator << (ostream & os, const Spectrum & s) {
	os << s.s[0] << ", " << s.s[1] << ", " << s.s[2];
	return os;
}

void Spectrum::ConvertToRGB(Float * result) const
{
	result[0] = s[0];
	result[1] = s[1];
	result[2] = s[2];
}

/* Spectrum::populateWaves()
	This function creates the values for the waves array by repeatedly
	calling the bb_spectrum function.
*/

void Spectrum::populateWaves()
{
	int i, lambda;
	for (i = 0, lambda = 380; lambda < 780; i++, lambda += (400 / SPEC)) {
		waves[i] = bb_spectrum(lambda);
	}
	monochromatic = 2;
}

void Spectrum::printRGB()
{
	printf("RGB = %f, %f, %f\n", s[0], s[1], s[2]);
}

void Spectrum::printWaves()
{
	int i, lambda;
	for (i = 0, lambda = 380; lambda < 780; i++, lambda += (400 / SPEC)) {
		printf("%d : %f\n", lambda, waves[i]);
	}
}

/*
Creates a spectrum with the same energy in "wavelength" as the original
spectrum, but with no energy in other wavelengths. The rgb values are
set to -2 because spectrum_to_rgb should always be called after this
constructor.
*/
Spectrum::Spectrum(Spectrum R, int wavelength)
{
	int i;
	for (i =0; i < 3; i++) s[i] = -2.0;
	for (i = 0; i < SPEC; i++) waves[i] = 0.0;
	waves[wavelength] = R.waves[wavelength];
	monochromatic = 1;
	wave_bucket = wavelength;
}

/* The functions below were modified from equations by the following
	author.
*/

/*
                     Colour Rendering of Spectra

                            by John Walker
                       http://www.fourmilab.ch/

                This program is in the public domain.

    For complete information about the techniques employed in this
    program, see the World-Wide Web document:

             http://www.fourmilab.ch/documents/specrend/

*/

/*                             XYZ_TO_RGB

    Given an additive tricolour system CS, defined by the  CIE  x  and  y
    chromaticities  of  its  three  primaries (z is derived trivially as
    1-(x+y)), and a desired chromaticity (XC,  YC,  ZC)  in  CIE  space,
    determine  the  contribution of each primary in a linear combination
    which  sums  to  the  desired  chromaticity.    If   the   requested
    chromaticity falls outside the Maxwell triangle (colour gamut) formed
    by the three primaries, one of the  r,  g,  or  b  weights  will  be
    negative.   Use  inside_gamut()  to  test  for  a  valid  colour  and
    constrain_rgb() to desaturate an outside-gamut colour to the  closest
    representation within the available gamut. */


void Spectrum::xyz_to_rgb()
{
	struct colourSystem *cs = &NTSCsystem;
    Float xr, yr, zr, xg, yg, zg, xb, yb, zb, d;
	Float xc = x;
	Float yc = y;
	Float zc = z;

    xr = cs->xRed;    yr = cs->yRed;    zr = 1 - (xr + yr);
    xg = cs->xGreen;  yg = cs->yGreen;  zg = 1 - (xg + yg);
    xb = cs->xBlue;   yb = cs->yBlue;   zb = 1 - (xb + yb);
    d = xr*yg*zb - xg*yr*zb - xr*yb*zg + xb*yr*zg + xg*yb*zr - xb*yg*zr;

    s[0] = (-xg*yc*zb + xc*yg*zb + xg*yb*zc - xb*yg*zc - xc*yb*zg + xb*yc*zg) / d;

    s[1] = (xr*yc*zb - xc*yr*zb - xr*yb*zc + xb*yr*zc + xc*yb*zr - xb*yc*zr) / d;

    s[2] = (xr*yg*zc - xg*yr*zc - xr*yc*zg + xc*yr*zg + xg*yc*zr - xc*yg*zr) / d;
}

/*                          SPECTRUM_TO_XYZ

    Calculate the CIE X, Y, and Z coordinates corresponding to a light
    source  with  spectral  distribution   given   by   the   function
    SPEC_INTENS,  which is called with a series of wavelengths between
    380 and 780 nm  (the  argument  is  expressed  in  meters),  which
    returns  emittance  at  that  wavelength  in arbitrary units.  The
    chromaticity coordinates of the spectrum are returned in the x, y,
    and z arguments which respect the identity:

            x + y + z = 1.
*/

void Spectrum::spectrum_to_xyz()
{
    int i;
    Float lambda, X = 0, Y = 0, Z = 0, XYZ;

    /* CIE colour matching functions xBar, yBar, and zBar for
       wavelengths from 380 through 780 nanometers, every 5
       nanometers.  For a wavelength lambda in this range:

            cie_colour_match[(lambda - 380) / 5][0] = xBar
            cie_colour_match[(lambda - 380) / 5][1] = yBar
            cie_colour_match[(lambda - 380) / 5][2] = zBar

       To  save  memory,  this  table can be declared as floats rather
       than Floats; (IEEE)  float  has  enough  significant  bits  to
       represent  the values.  It's declared as a Float here to avoid
       warnings about "conversion between floating-point  types"  from
       certain persnickety compilers. */

    static Float cie_colour_match[81][3] = {
        {0.0014,0.0000,0.0065}, {0.0022,0.0001,0.0105}, {0.0042,0.0001,0.0201},
        {0.0076,0.0002,0.0362}, {0.0143,0.0004,0.0679}, {0.0232,0.0006,0.1102},
        {0.0435,0.0012,0.2074}, {0.0776,0.0022,0.3713}, {0.1344,0.0040,0.6456},
        {0.2148,0.0073,1.0391}, {0.2839,0.0116,1.3856}, {0.3285,0.0168,1.6230},
        {0.3483,0.0230,1.7471}, {0.3481,0.0298,1.7826}, {0.3362,0.0380,1.7721},
        {0.3187,0.0480,1.7441}, {0.2908,0.0600,1.6692}, {0.2511,0.0739,1.5281},
        {0.1954,0.0910,1.2876}, {0.1421,0.1126,1.0419}, {0.0956,0.1390,0.8130},
        {0.0580,0.1693,0.6162}, {0.0320,0.2080,0.4652}, {0.0147,0.2586,0.3533},
        {0.0049,0.3230,0.2720}, {0.0024,0.4073,0.2123}, {0.0093,0.5030,0.1582},
        {0.0291,0.6082,0.1117}, {0.0633,0.7100,0.0782}, {0.1096,0.7932,0.0573},
        {0.1655,0.8620,0.0422}, {0.2257,0.9149,0.0298}, {0.2904,0.9540,0.0203},
        {0.3597,0.9803,0.0134}, {0.4334,0.9950,0.0087}, {0.5121,1.0000,0.0057},
        {0.5945,0.9950,0.0039}, {0.6784,0.9786,0.0027}, {0.7621,0.9520,0.0021},
        {0.8425,0.9154,0.0018}, {0.9163,0.8700,0.0017}, {0.9786,0.8163,0.0014},
        {1.0263,0.7570,0.0011}, {1.0567,0.6949,0.0010}, {1.0622,0.6310,0.0008},
        {1.0456,0.5668,0.0006}, {1.0026,0.5030,0.0003}, {0.9384,0.4412,0.0002},
        {0.8544,0.3810,0.0002}, {0.7514,0.3210,0.0001}, {0.6424,0.2650,0.0000},
        {0.5419,0.2170,0.0000}, {0.4479,0.1750,0.0000}, {0.3608,0.1382,0.0000},
        {0.2835,0.1070,0.0000}, {0.2187,0.0816,0.0000}, {0.1649,0.0610,0.0000},
        {0.1212,0.0446,0.0000}, {0.0874,0.0320,0.0000}, {0.0636,0.0232,0.0000},
        {0.0468,0.0170,0.0000}, {0.0329,0.0119,0.0000}, {0.0227,0.0082,0.0000},
        {0.0158,0.0057,0.0000}, {0.0114,0.0041,0.0000}, {0.0081,0.0029,0.0000},
        {0.0058,0.0021,0.0000}, {0.0041,0.0015,0.0000}, {0.0029,0.0010,0.0000},
        {0.0020,0.0007,0.0000}, {0.0014,0.0005,0.0000}, {0.0010,0.0004,0.0000},
        {0.0007,0.0002,0.0000}, {0.0005,0.0002,0.0000}, {0.0003,0.0001,0.0000},
        {0.0002,0.0001,0.0000}, {0.0002,0.0001,0.0000}, {0.0001,0.0000,0.0000},
        {0.0001,0.0000,0.0000}, {0.0001,0.0000,0.0000}, {0.0000,0.0000,0.0000}
    };

    for (i = 0, lambda = 380; lambda < 780; i++, lambda += (400 / SPEC)) {
        Float Me;

        Me = waves[i];
        X += Me * cie_colour_match[i][0];
        Y += Me * cie_colour_match[i][1];
        Z += Me * cie_colour_match[i][2];
    }
    XYZ = (X + Y + Z);
    x = X / XYZ;
    y = Y / XYZ;
    z = Z / XYZ;
}

/*                            BB_SPECTRUM

    Calculate, by Planck's radiation law, the emittance of a black body
    of temperature bbTemp at the given wavelength (in meters).  */

Float bbTemp = 5000;                 /* Hidden temperature argument
                                         to BB_SPECTRUM. */
Float Spectrum::bb_spectrum(Float wavelength)
{
    double wlm = wavelength * 1e-9;   /* Wavelength in meters */

    return (3.74183e-16 * pow(wlm, -5.0)) /
           (exp(1.4388e-2 / (wlm * bbTemp)) - 1.0);
}

/*                            INSIDE_GAMUT

    Test  whether  a requested colour is within the gamut achievable with
    the primaries of the current colour system.  This amounts  simply  to
    testing whether all the primary weights are non-negative. */

int Spectrum::inside_gamut()
{
    return s[0] >= 0 && s[1] >= 0 && s[2] >= 0;
}

/*                           CONSTRAIN_RGB

    If  the  requested RGB shade contains a negative weight for one of
    the primaries, it lies outside the colour gamut accessible from the
    given triple of primaries.  Desaturate it by mixing with the white
    point of the colour system so as to reduce  the  primary  with  the
    negative weight to  zero.   This  is  equivalent  to  finding  the
    intersection  on the CIE diagram of a line drawn between the white
    point and the  requested  colour  with  the  edge  of  the  Maxwell
    triangle formed by the three primaries. */

int Spectrum::constrain_rgb()
{
    /* Is the contribution of one of the primaries negative ? */

    if (!inside_gamut()) {
		Float r = s[0];
		Float g = s[1];
		Float b = s[2];
        double par, wr, wg, wb;

        /* Yes.  Find the RGB mixing weights of the white point (we
           assume the white point is in the gamut!). */

		wr = 1.0;
		wb = 1.0;
		wg = 1.0;

        /* Find the primary with negative weight and calculate the
           parameter of the point on the vector from the white point
           to the original requested colour in RGB space. */

        if (r < g && r < b) {
            par = wr / (wr - r);
        } else if (g < r && g < b) {
            par = wg / (wg - g);
        } else {
            par = wb / (wb - b);
        }

        /* Now finally calculate the gamut-constrained RGB weights. */

        s[0] = CLAMP(wr + par * (r - wr), 0, 1);
        s[1] = CLAMP(wg + par * (g - wg), 0, 1);
        s[2] = CLAMP(wb + par * (b - wb), 0, 1);

        return 1;                     /* Colour modified to fit RGB gamut */
    }
    return 0;                         /* Colour within RGB gamut */
}

