function D = wavedist(wspeed, res)

% wavedist: returns a 2-dimensional frequency distribution of simulated
% ocean waves.
%
% wspeed: windspeed (meters/sec)
%	- note: something strange about this factor - lower numbers give smaller,
%	        faster waves - higher numbers give slower waves
%
% res: resolution of frequency distribution
%
% Uses the Pierson-Moskowitz freqency spectrum for wave frequency distribution
% in the downwind direction, along with a directional attenuation function
% described in Mastin, et al. (Fourier Synthesis of Ocean Scenes)

g = 9.81; %m/s^2
u10 = wspeed; %meters/sec

% maximum wave frequency - depends on wind speed
% this is from the formula in the paper - I think something is
% a bit strange about it - lower wind speed numbers give smaller, choppier
% waves, so put in small wspeed values for small, higher freq. waves
fm = 0.13*g/u10;

D = zeros(res);

cx = res/2 + 0.5;
cy = cx;

scale = res/2;

% loop through array
for x = 1:res
   for y = 1:res
      
      dx = x - cx;
      dy = y - cy;
      
      % calculate current angle
      theta = atan(dx/dy);
      
      % change this to the angle away from the downwind direction
      % in this case, downwind direction is up and down (from (50.5,0) to (50.5,100)
      % note: in this case, the angle from the downwind direction really refers to the angle
      % from the downwind axis, so the angle starts at 0, goes up to pi/2, then back to 0 when
      % facing in the opposite direction. This passes frequencies in one direction while blocking
      % perpendicular frequencies.
      	if ( (y > cx) & (x < cx))
            theta = -theta;
         elseif ( (y < cx) & (x > cx) )
            theta = -theta;
         end
         
         % frequency is distance from center, scaled by resolution of array
       	f = sqrt(dx^2 + dy ^2)/scale;
          
         % exponential factor changes depending on current freq.
         if (f<fm)
      		mu = 4.06;
   		else
      		mu = -2.34;
   		end
         
         %factor used in determining directional weighting
   		p = 9.77 * (f/fm)^mu;
         
         % phillips constant - an oceanographic figure
         ph = 0.0081;
         
         % downwind power spectrum - Pierson-Moskowitz model
         fpm = (ph * g^2)./( (2*pi)^4 * f.^5 ) .* exp( -(5/4)*(fm./f).^4 );
         
         % attenuate downwind power spectrum by directional scaling factor
         D(y,x) = cos(theta/2)^(2*p) * fpm;
         
   end
end

% constrain values to lie between 0 and 1
maximum = max(max(D));
D = D ./ maximum;