
#include "image.h"
#include "camera.h"
#include "scene.h"
#include "geometry.h"
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

extern struct shmDB *shm_base;

Image::Image()
{
	Pixels = NULL;
	Alphas = NULL;
	Depths = NULL;
	WeightSums = NULL;

	XResolution = 640;
	YResolution = 480;

	CropLeft = CropBottom = 0.;
	CropRight = CropTop = 1.;

	DisplayMode = RI_RGBA;

	Gain = Gamma = 1.;

	Imager = RI_NULL;

	ColorQuantOne = 255;
	ColorQuantMin = 0;
	ColorQuantMax = 255;
	ColorQuantDither = 0.5;

	DepthQuantOne = 0;
	DepthQuantMin = 0;
	DepthQuantMax = 0;
	DepthQuantDither = 0.;

	DisplayType = RI_FILE;
	DisplayName = "out.tiff";

	ImageViewer = NULL;

	PixelAspectRatio = 1.;
	FrameAspectRatio = 4. / 3.;

}

void Image::FinalizeValues()
{

	int RasterCropLeft = (int) Clamp(ceil(XResolution * CropLeft),
									 0, XResolution);
	int RasterCropRight = (int) Clamp(ceil(XResolution * CropRight),
									  0, XResolution);
	int RasterCropBottom = (int) Clamp(ceil(YResolution * CropBottom),
									   0, YResolution);
	int RasterCropTop = (int) Clamp(ceil(YResolution * CropTop),
									0, YResolution);
	SampleCropLeft =
		Clamp(RasterCropLeft - scene->sampler->FilterXWidth, 0.,
			  XResolution);
	SampleCropRight =
		Clamp(RasterCropRight + scene->sampler->FilterXWidth, 0.,
			  XResolution);
	SampleCropBottom =
		Clamp(RasterCropBottom - scene->sampler->FilterYWidth, 0.,
			  YResolution);
	SampleCropTop =
		Clamp(RasterCropTop + scene->sampler->FilterYWidth, 0.,
			  YResolution);

	XBase = (int) Clamp(ceil(XResolution * CropLeft), 0, XResolution);
	XWidth =
		(int) Clamp(ceil(XResolution * CropRight), 0, XResolution) - XBase;
	YBase = (int) Clamp(ceil(YResolution * CropBottom), 0, YResolution);
	YWidth =
		(int) Clamp(ceil(YResolution * CropTop), 0, YResolution) - YBase;

	if (DisplayMode == RI_RGB || DisplayMode == RI_RGBA ||
		DisplayMode == RI_RGBZ || DisplayMode == RI_RGBAZ) {
		Pixels = new Spectrum[XWidth * YWidth];
	} else
		Pixels = NULL;

	if (DisplayMode == RI_RGBA || DisplayMode == RI_RGBAZ ||
		DisplayMode == RI_A || DisplayMode == RI_AZ) {
		Alphas = new Float[XWidth * YWidth];
		for (int i = 0; i < XWidth * YWidth; ++i)
			Alphas[i] = 0.;
	} else
		Alphas = NULL;

	if (DisplayMode == RI_RGBAZ || DisplayMode == RI_RGBZ ||
		DisplayMode == RI_AZ || DisplayMode == RI_Z) {
		Depths = new Float[XWidth * YWidth];
		for (int i = 0; i < XWidth * YWidth; ++i)
			Depths[i] = 0.;
	} else
		Depths = NULL;

	WeightSums = new Float[XWidth * YWidth];
	for (int i = 0; i < XWidth * YWidth; ++i)
		WeightSums[i] = 0.;

}

Image::~Image()
{
	delete[]Pixels;
	delete[]Alphas;
	delete[]Depths;
	delete[]WeightSums;
}

void Image::AddSampleBasic(const Point & Praster,
						   const Spectrum & radiance, Float alpha)
{

	if (Praster.x < XBase || Praster.x >= XBase + XWidth ||
		Praster.y < YBase || Praster.y >= YBase + YWidth)
		return;

	int x = int (Praster.x), y = int (Praster.y);
	int offset = (y - YBase) * XWidth + (x - XBase);
	if (Pixels)
		Pixels[offset] += radiance;
	if (Alphas)
		Alphas[offset] += alpha;
	if (Depths)
		Depths[offset] += Praster.z;
	WeightSums[offset] += 1;

}

void Image::Write() const
{

	Float *RGBOut = NULL, *AlphaOut = NULL, *DepthOut = NULL;
	if (Pixels) {
		RGBOut = new Float[3 * XWidth * YWidth];
		for (int offset = 0; offset < XWidth * YWidth; ++offset)
			Pixels[offset].ConvertToRGB(&RGBOut[3 * offset]);
	}
	if (Alphas) {
		AlphaOut = new Float[XWidth * YWidth];
		memcpy(AlphaOut, Alphas, XWidth * YWidth * sizeof(Float));
	}
	if (Depths) {
		DepthOut = new Float[XWidth * YWidth];
		memcpy(DepthOut, Depths, XWidth * YWidth * sizeof(Float));
	}

	for (int offset = 0; offset < XWidth * YWidth; ++offset) {
		if (WeightSums[offset] == 0.)
			continue;

		Float invWt = 1. / WeightSums[offset];
		if (RGBOut) {
			for (int c = 0; c < 3; ++c) {
				RGBOut[3 * offset + c] *= invWt;
				if (RGBOut[3 * offset + c] < 0.)
					RGBOut[3 * offset + c] = 0.;
			}
		}
		if (AlphaOut)
			AlphaOut[offset] *= invWt;
		if (DepthOut)
			DepthOut[offset] *= invWt;
	}

	if (RGBOut && AlphaOut) {
		for (int i = 0; i < XWidth * YWidth; ++i) {
			for (int j = 0; j < 3; ++j)
				RGBOut[3 * i + j] *= AlphaOut[i];
		}
	}

	if ((RGBOut || AlphaOut) && (Gain != 1. || Gamma != 1.)) {
		Float invGamma = 1. / Gamma;
		if (RGBOut)
			for (int offset = 0; offset < 3 * XWidth * YWidth; ++offset)
				RGBOut[offset] = pow(RGBOut[offset] * Gain, invGamma);
		if (AlphaOut)
			for (int offset = 0; offset < XWidth * YWidth; ++offset)
				AlphaOut[offset] = pow(AlphaOut[offset] * Gain, invGamma);

	}

	if (Imager != RI_NULL)
		ApplyImager(Imager, RGBOut, AlphaOut, DepthOut, XWidth * YWidth);

	if (ColorQuantOne != 0 && RGBOut)
		scaleAndDither(ColorQuantOne, ColorQuantMin,
					   ColorQuantMax, ColorQuantDither, RGBOut,
					   3 * XWidth * YWidth);
	if (ColorQuantOne != 0 && AlphaOut)
		scaleAndDither(ColorQuantOne, ColorQuantMin,
					   ColorQuantMax, ColorQuantDither, AlphaOut,
					   XWidth * YWidth);
	if (DepthQuantOne != 0 && DepthOut)
		scaleAndDither(DepthQuantOne, DepthQuantMin,
					   DepthQuantMax, DepthQuantDither, DepthOut,
					   XWidth * YWidth);

	if (ColorQuantOne != 0.)
		TIFFWrite8Bit(DisplayName, RGBOut, AlphaOut, DepthOut, XWidth,
					  YWidth);
	else
		TIFFWriteFloat(DisplayName, RGBOut, AlphaOut, DepthOut, XWidth,
					   YWidth);

	if (ImageViewer != NULL && !fork())
		if (execlp(ImageViewer, ImageViewer, DisplayName, NULL) == -1) {
			Error("Unable to run image viewing program %s: %s",
				  ImageViewer, strerror(errno));
			exit(1);
		}

	delete[]RGBOut;
	delete[]AlphaOut;
	delete[]DepthOut;

}

void Image::scaleAndDither(Float one, Float min, Float max,
						   Float ditheramp, Float * pixels, int nSamples)
{
	for (int i = 0; i < nSamples; ++i) {
		pixels[i] =
			Round(pixels[i] * one + RandomFloat(-ditheramp, ditheramp));
		pixels[i] = Clamp(pixels[i], min, max);
	}
}

void Image::AddSample(const Point & Praster, const Spectrum & radiance,
					  Float alpha)
{

	Float HalfXWidth = scene->sampler->FilterXWidth / 2.;
	Float HalfYWidth = scene->sampler->FilterYWidth / 2.;
	int x0 = max((int) ceil(Praster.x - HalfXWidth), XBase);
	int x1 = min((int) floor(Praster.x + HalfXWidth), XBase + XWidth);
	int y0 = max((int) ceil(Praster.y - HalfYWidth), YBase);
	int y1 = min((int) floor(Praster.y + HalfYWidth), YBase + YWidth);

	for (int y = y0; y < y1; ++y)
		for (int x = x0; x < x1; ++x) {

			Float fx = x - Praster.x;
			Float fy = y - Praster.y;

			Float filterWt =
				scene->sampler->Filter(fx, fy, XWidth, YWidth);
			int offset = (y - YBase) * XWidth + (x - XBase);
			if (Pixels)
				Pixels[offset] += radiance * filterWt;
			if (Alphas)
				Alphas[offset] += alpha * filterWt;
			if (Depths)
				Depths[offset] += Praster.z * filterWt;
			WeightSums[offset] += filterWt;
		
			if (shm_base->viewerpid != 0){
				float rgb_out[3];
				float invW;
	
				Pixels[offset].ConvertToRGB(rgb_out);
			
				if (WeightSums[offset] != 0){
					while (!shm_base->PickedUp){};
					shm_base->x = x - XBase;
					shm_base->y = y - YBase;
					invW = 1./WeightSums[offset];
					shm_base->r = rgb_out[0]*invW;
					shm_base->g = rgb_out[1]*invW;
					shm_base->b = rgb_out[2]*invW;
					shm_base->PickedUp = 0;
					kill(shm_base->viewerpid, SIGUSR1);
				}
			}
		}
}

void ApplyImager(const char *name, Float * RGBPixels, Float * alphas,
				 Float * depths, int nPixels)
{
}
