
#ifndef SL_H
#define SL_H

#include "lrt.h"
#include "geometry.h"
#include "color.h"
#include "primitives.h"

typedef float SLFloat;
typedef char *SLString;

class DSO { public:
	 DSO(const char *filename);
	~DSO();

	void *GetSym(const char *symbol) const;

  private:
#ifdef WIN32
	 HINSTANCE hinstLib;
#else
	void *hinstLib;
#endif

};

class SLTriple { public:
	 SLTriple():a(0), b(0), c(0) {
	} SLTriple(Float f):a(f), b(f), c(f) {
	}
  SLTriple(Float _a, Float _b, Float _c):a(_a), b(_b), c(_c) {
	}
	SLTriple(const Point & p):a(p.x), b(p.y), c(p.z) {
	}
	SLTriple(const Vector & v):a(v.x), b(v.y), c(v.z) {
	}
	SLTriple(const Normal & n):a(n.x), b(n.y), c(n.z) {
	}
	SLTriple(const Spectrum & s):a(s.s[0]), b(s.s[1]), c(s.s[2]) {
	}

	Point MakePoint() const {
		return Point(a, b, c);
	} Vector MakeVector() const {
		return Vector(a, b, c);
	} Normal MakeNormal() const {
		return Normal(a, b, c);
	} Spectrum MakeSpectrum() const {
		return Spectrum(a, b, c);
	} Float Length2() const {
		return a * a + b * b + c * c;
	} Float Length() const {
		return sqrt(Length2());
	} SLTriple Hat() const { Float invLen = 1.0 / Length();
		 return SLTriple(a * invLen, b * invLen, c * invLen);
	} SLTriple FaceForward(const SLTriple & t);

	SLTriple operator + (const SLTriple & t) const
	{ return SLTriple(a + t.a, b + t.b, c + t.c);
	} SLTriple & operator += (const SLTriple & t) {
		a += t.a;
		b += t.b;
		c += t.c;
		return (*this);
	}

	SLTriple operator - (const SLTriple & t) const
	{ return SLTriple(a - t.a, b - t.b, c - t.c);
	} SLTriple & operator -= (const SLTriple & t) {
		a -= t.a;
		b -= t.b;
		c -= t.c;
		return (*this);
	}


	SLTriple operator - () const { return SLTriple(-a, -b, -c);
	}
		SLTriple operator *(const SLTriple & t) const
	{ return SLTriple(a * t.a, b * t.b, c * t.c);
	} SLTriple & operator *= (const SLTriple & t) {
		a *= t.a;
		b *= t.b;
		c *= t.c;
		return (*this);
	}


	SLTriple operator / (const SLTriple & t) const
	{ return SLTriple(a / t.a, b / t.b, c / t.c);
	} SLTriple & operator /= (const SLTriple & t) {
		a /= t.a;
		b /= t.b;
		c /= t.c;
		return (*this);
	}


	bool operator == (const SLTriple & t) const {
		return a == t.a && b == t.b && c == t.c;
	}
		bool operator != (const SLTriple & t) const {
			  return a != t.a || b != t.b || c != t.c;
	} Float a, b, c;


};

struct RmanShaderParams {
	RmanShaderParams(SurfaceFunction * sf, const ShadeContext & sc,
					 Sampler & sg, HitInfo * hi):surfaceFunction(sf),
		shadeContext(sc), sampleGenerator(sg), hitInfo(hi) {
	} SLTriple Cs, Os;
	SLTriple P, dPdu, dPdv;
	SLTriple N, Ng;

	Float u, v, du, dv;
	Float s, t;

	SLTriple I;
	SLTriple Ci, Oi;

	SLTriple E;
	Float A;

	SurfaceFunction *surfaceFunction;
	const ShadeContext & shadeContext;
	Sampler & sampleGenerator;
	HitInfo *hitInfo;
};

class RendermanShader { public:
	 RendermanShader(const char *name);
	void Run(RmanShaderParams &);

  private:
	 char *name;
	DSO *dso;
	RmanShaderType type;
	typedef void (*shader_func) (RmanShaderParams &);
	shader_func shader;
	bool already_warned;

};

inline Float Dot(const SLTriple & t1, const SLTriple & t2)
{
	return t1.a * t2.a + t1.b * t2.b + t1.c * t2.c;
}

inline SLTriple Cross(const SLTriple & t1, const SLTriple & t2)
{
	return SLTriple((t1.b * t2.c) - (t1.c * t2.b),
					(t1.c * t2.a) - (t1.a * t2.c),
					(t1.a * t2.b) - (t1.b * t2.a));
}

extern SLTriple LrtSLAmbient(RmanShaderParams &);

#endif /* SL_H */
