
#include "shading.h"
#include "color.h"
#include "reflection.h"
#include "texture.h"

ShadeContext::ShadeContext(const HitInfo * hi, const Vector & w)
:wo(w.Hat())
{
	hitInfo = hi;

	Ng = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->NgObj);
	Ng.Normalize();
	if (Dot(Ng, wo) < 0.) {
		Ng = -Ng;
		entering = false;
	} else {
		entering = true;
	}

	const Normal *Ni = hitInfo->GetNormal(RI_N);
	if (Ni != NULL) {
		Ns = hitInfo->hitPrim->attributes->ObjectToWorld(*Ni).Hat();
		if (Dot(Ns, wo) < 0.)
			Ns = -Ns;
	} else {
		Ns = Ng;
	}

}

Float ShadeContext::InitializeFloat(RtToken token, Float def) const
{
	const Float *ret = hitInfo->GetFloat(token);
	if (ret)
		return *ret;
	if (token == RI_S)
		return hitInfo->u;
	else if (token == RI_T)
		return hitInfo->v;
	return def;
}

Spectrum ShadeContext::InitializeColor(RtToken token, const Spectrum & def) const
{
	const Spectrum *ret = hitInfo->GetColor(token);
	if (ret)
		return *ret;
	if (token == RI_CS)
		return hitInfo->hitPrim->attributes->Color;
	else if (token == RI_OS)
		return hitInfo->hitPrim->attributes->Opacity;
	return def;
}

Point ShadeContext::InitializePoint(RtToken token, const Point & def) const
{
	const Point *ret = hitInfo->GetPoint(token);
	if (ret)
		return *ret;
	else
		return def;
}

const char *ShadeContext::InitializeString(RtToken token, const char *def) const
{
	const char *ret = hitInfo->GetString(token);
	if (ret)
		return ret;
	else
		return def;
}

Vector ShadeContext::InitializeVector(RtToken token, const Vector & def) const
{
	const Vector *ret = hitInfo->GetVector(token);
	if (ret)
		return *ret;
	else
		return def;
}

Normal ShadeContext::InitializeNormal(RtToken token, const Normal & def) const
{
	const Normal *ret = hitInfo->GetNormal(token);
	if (ret)
		return *ret;
	else
		return def;
}

StringHashTable ShadeContext::textures;


TextureMap *ShadeContext::GetTexture(const char *filename)
{
	TextureMap *ret = (TextureMap *) textures.Search(filename);
	if (!ret) {
		ret = new TextureMap(filename);
		textures.Add(filename, ret);
	}
	return ret;
}

Material::~Material()
{
	delete surfaceFunction;
}

BRDF *MatteSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	Float surfaceKd = sc.InitializeFloat(RI_KD, Kd);
	return CreateLambertian(surfaceKd * Cs, sc.Ns);
}

BRDF *PlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	return PlasticShade(sc, Cs);
}

BRDF *PlasticSurface::PlasticShade(const ShadeContext & sc,
								   const Spectrum & Cs) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Float surfKs = sc.InitializeFloat(RI_KS, Ks);
	Float surfRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum surfSpecColor =
		sc.InitializeColor(RI_SPECULARCOLOR, SpecularColor);

	ScatteringMixture *mix = CreateScatteringMixture();
	mix->AddFunction(CreateLambertian(surfKd * Cs, sc.Ns));
	mix->AddFunction(CreateBlinnGlossy(surfKs * surfSpecColor,
									   Clamp(surfRoughness, 0., 1.), sc.Ns,
									   sc.wo));
	return mix;

}

PaintedPlasticSurface::PaintedPlasticSurface(SurfaceFunction *data)
		: PlasticSurface(data) {
	texName = surfaceFunction->InitializeString(RI_TEXTURENAME,
								NULL);
}

BRDF *PaintedPlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum surfColor = sc.InitializeColor(RI_CS, Spectrum(1.));
	const char *surfTexName = sc.InitializeString(RI_TEXTURENAME, texName);
	TextureMap *tmap = sc.GetTexture(surfTexName);

	if (tmap) {
		Float s = sc.InitializeFloat(RI_S, 0.);
		Float t = sc.InitializeFloat(RI_T, 0.);
		surfColor *= tmap->Lookup(s, t);
	}
	return PlasticShade(sc, surfColor);
}

BRDF *CheckeredSurface::Shade(const ShadeContext & sc) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Spectrum surfCColor1 = sc.InitializeColor(LRT_CHECKCOLOR1,
											  CheckColor1);
	Spectrum surfCColor2 = sc.InitializeColor(LRT_CHECKCOLOR2,
											  CheckColor2);
	Float freq = sc.InitializeFloat(LRT_CHECKFREQUENCY,
									CheckFrequency);

	Point Pshade = sc.InitializePoint(RI_P, Point(0, 0, 0));
	Point Pcheck(freq * Pshade.x, freq * Pshade.y, freq * Pshade.z);
	if (Mod(Round(Pcheck.x) + Round(Pcheck.y) + Round(Pcheck.z), 2) == 0) {
		return CreateLambertian(surfKd * surfCColor1, sc.Ns);
	} else {
		return CreateLambertian(surfKd * surfCColor2, sc.Ns);
	}

}

BRDF *GlassSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeKt = sc.InitializeFloat(LRT_KT, Kt);
	Float shadeIndex = sc.InitializeFloat(LRT_INDEX, Index);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(0.));

	if (shadeKr == 0. && shadeKt == 0.)
		return NULL;

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(shadeKr * Cs,
												  sc.Ns, sc.wo));
	}
	if (shadeKt > 0.) {

		if (sc.entering) {
			mix->AddFunction(CreateSpecularTransmission(Kt * Cs,
														sc.Ns, sc.wo,
														1.0, shadeIndex));
		} else {
			mix->AddFunction(CreateSpecularTransmission(Kt * Cs,
														sc.Ns, sc.wo,
														shadeIndex, 1.0));
		}

	}
	return mix;
}

BRDF *ShinyMetalSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKs = sc.InitializeFloat(RI_KS, Ks);
	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKs > 0.) {
		mix->AddFunction(CreateBlinnGlossy(shadeKs * Cs,
										   shadeRoughness, sc.Ns, sc.wo));
	}
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(shadeKr * Cs,
												  sc.Ns, sc.wo));
	}
	return mix;
}

//ours

const Float ButterflyWingSurface::colorCurves[95][4] =  {
    {360, 0.000129900000, 0.000003917000, 0.000606100000}, 
    {365, 0.000232100000, 0.000006965000, 0.001086000000}, 
    {370, 0.000414900000, 0.000012390000, 0.001946000000}, 
    {375, 0.000741600000, 0.000022020000, 0.003486000000}, 
    {380, 0.001368000000, 0.000039000000, 0.006450001000}, 
    {385, 0.002236000000, 0.000064000000, 0.010549990000}, 
    {390, 0.004243000000, 0.000120000000, 0.020050010000}, 
    {395, 0.007650000000, 0.000217000000, 0.036210000000}, 
    {400, 0.014310000000, 0.000396000000, 0.067850010000}, 
    {405, 0.023190000000, 0.000640000000, 0.110200000000}, 
    {410, 0.043510000000, 0.001210000000, 0.207400000000}, 
    {415, 0.077630000000, 0.002180000000, 0.371300000000}, 
    {420, 0.134380000000, 0.004000000000, 0.645600000000}, 
    {425, 0.214770000000, 0.007300000000, 1.039050100000}, 
    {430, 0.283900000000, 0.011600000000, 1.385600000000}, 
    {435, 0.328500000000, 0.016840000000, 1.622960000000}, 
    {440, 0.348280000000, 0.023000000000, 1.747060000000}, 
    {445, 0.348060000000, 0.029800000000, 1.782600000000}, 
    {450, 0.336200000000, 0.038000000000, 1.772110000000}, 
    {455, 0.318700000000, 0.048000000000, 1.744100000000}, 
    {460, 0.290800000000, 0.060000000000, 1.669200000000}, 
    {465, 0.251100000000, 0.073900000000, 1.528100000000}, 
    {470, 0.195360000000, 0.090980000000, 1.287640000000}, 
    {475, 0.142100000000, 0.112600000000, 1.041900000000}, 
    {480, 0.095640000000, 0.139020000000, 0.812950100000}, 
    {485, 0.057950010000, 0.169300000000, 0.616200000000}, 
    {490, 0.032010000000, 0.208020000000, 0.465180000000}, 
    {495, 0.014700000000, 0.258600000000, 0.353300000000}, 
    {500, 0.004900000000, 0.323000000000, 0.272000000000}, 
    {505, 0.002400000000, 0.407300000000, 0.212300000000}, 
    {510, 0.009300000000, 0.503000000000, 0.158200000000}, 
    {515, 0.029100000000, 0.608200000000, 0.111700000000}, 
    {520, 0.063270000000, 0.710000000000, 0.078249990000}, 
    {525, 0.109600000000, 0.793200000000, 0.057250010000}, 
    {530, 0.165500000000, 0.862000000000, 0.042160000000}, 
    {535, 0.225749900000, 0.914850100000, 0.029840000000}, 
    {540, 0.290400000000, 0.954000000000, 0.020300000000}, 
    {545, 0.359700000000, 0.980300000000, 0.013400000000}, 
    {550, 0.433449900000, 0.994950100000, 0.008749999000}, 
    {555, 0.512050100000, 1.000000000000, 0.005749999000}, 
    {560, 0.594500000000, 0.995000000000, 0.003900000000}, 
    {565, 0.678400000000, 0.978600000000, 0.002749999000}, 
    {570, 0.762100000000, 0.952000000000, 0.002100000000}, 
    {575, 0.842500000000, 0.915400000000, 0.001800000000}, 
    {580, 0.916300000000, 0.870000000000, 0.001650001000}, 
    {585, 0.978600000000, 0.816300000000, 0.001400000000}, 
    {590, 1.026300000000, 0.757000000000, 0.001100000000}, 
    {595, 1.056700000000, 0.694900000000, 0.001000000000}, 
    {600, 1.062200000000, 0.631000000000, 0.000800000000}, 
    {605, 1.045600000000, 0.566800000000, 0.000600000000}, 
    {610, 1.002600000000, 0.503000000000, 0.000340000000}, 
    {615, 0.938400000000, 0.441200000000, 0.000240000000}, 
    {620, 0.854449900000, 0.381000000000, 0.000190000000}, 
    {625, 0.751400000000, 0.321000000000, 0.000100000000}, 
    {630, 0.642400000000, 0.265000000000, 0.000049999990}, 
    {635, 0.541900000000, 0.217000000000, 0.000030000000}, 
    {640, 0.447900000000, 0.175000000000, 0.000020000000}, 
    {645, 0.360800000000, 0.138200000000, 0.000010000000}, 
    {650, 0.283500000000, 0.107000000000, 0.000000000000}, 
    {655, 0.218700000000, 0.081600000000, 0.000000000000}, 
    {660, 0.164900000000, 0.061000000000, 0.000000000000}, 
    {665, 0.121200000000, 0.044580000000, 0.000000000000}, 
    {670, 0.087400000000, 0.032000000000, 0.000000000000}, 
    {675, 0.063600000000, 0.023200000000, 0.000000000000}, 
    {680, 0.046770000000, 0.017000000000, 0.000000000000}, 
    {685, 0.032900000000, 0.011920000000, 0.000000000000}, 
    {690, 0.022700000000, 0.008210000000, 0.000000000000}, 
    {695, 0.015840000000, 0.005723000000, 0.000000000000}, 
    {700, 0.011359160000, 0.004102000000, 0.000000000000}, 
    {705, 0.008110916000, 0.002929000000, 0.000000000000}, 
    {710, 0.005790346000, 0.002091000000, 0.000000000000}, 
    {715, 0.004106457000, 0.001484000000, 0.000000000000}, 
    {720, 0.002899327000, 0.001047000000, 0.000000000000}, 
    {725, 0.002049190000, 0.000740000000, 0.000000000000}, 
    {730, 0.001439971000, 0.000520000000, 0.000000000000}, 
    {735, 0.000999949300, 0.000361100000, 0.000000000000}, 
    {740, 0.000690078600, 0.000249200000, 0.000000000000}, 
    {745, 0.000476021300, 0.000171900000, 0.000000000000}, 
    {750, 0.000332301100, 0.000120000000, 0.000000000000}, 
    {755, 0.000234826100, 0.000084800000, 0.000000000000}, 
    {760, 0.000166150500, 0.000060000000, 0.000000000000}, 
    {765, 0.000117413000, 0.000042400000, 0.000000000000}, 
    {770, 0.000083075270, 0.000030000000, 0.000000000000}, 
    {775, 0.000058706520, 0.000021200000, 0.000000000000}, 
    {780, 0.000041509940, 0.000014990000, 0.000000000000}, 
    {785, 0.000029353260, 0.000010600000, 0.000000000000}, 
    {790, 0.000020673830, 0.000007465700, 0.000000000000}, 
    {795, 0.000014559770, 0.000005257800, 0.000000000000}, 
    {800, 0.000010253980, 0.000003702900, 0.000000000000}, 
    {805, 0.000007221456, 0.000002607800, 0.000000000000}, 
    {810, 0.000005085868, 0.000001836600, 0.000000000000}, 
    {815, 0.000003581652, 0.000001293400, 0.000000000000}, 
    {820, 0.000002522525, 0.000000910930, 0.000000000000}, 
    {825, 0.000001776509, 0.000000641530, 0.000000000000}, 
    {830, 0.000001251141, 0.000000451810, 0.000000000000}, 
  };

int ButterflyWingSurface::initMatrix(Float XYZ2RGBmatrix[3][3]) {

  Float Yw;

  // use Yw = 100 for notready and Yw = 1 for notready2
  Yw = 1.;
  
  const Float xr = 0.62, yr = 0.33, xg = 0.21, yg = 0.685, xb = 0.15, yb =
    0.063;
  const Float xw = 0.33, yw = 0.33;
  
  Float D = xr*(yg - yb) + xg*(yb - yr) + xb*(yr - yg);
  Float Cr = (Yw/(yw*D)) * (xw*(yg - yb) - yw*(xg - xb) + xg*yb - xb*yg);
  Float Cg = (Yw/(yw*D)) * (xw*(yb - yr) - yw*(xb - xr) - xr*yb + xb*yr);
  Float Cb = (Yw/(yw*D)) * (xw*(yr - yg) - yw*(xr - xg) + xr*yg - xg*yr);
  
  Float Xr = xr*Cr;
  Float Yr = yr*Cr;
  Float Zr = (1. - xr - yr)*Cr;
  Float Xg = xg*Cg;
  Float Yg = yg*Cg;
  Float Zg = (1. - xg - yg)*Cg;
  Float Xb = xb*Cb;
  Float Yb = yb*Cb;
  Float Zb = (1. - xb - yb)*Cb;

  // Now calculate inverse matrix (XYZ to RGB)
  Float det =  Xr*(Yg*Zb-Yb*Zg)
    + Yr*(Zg*Xb-Xg*Zb)
    + Zr*(Xg*Yb-Yg*Xb);

  if (det != 0) {

    Float detInv = 1/det;

    XYZ2RGBmatrix[0][0] =  detInv * (Yg*Zb - Yb*Zg);
    XYZ2RGBmatrix[0][1] = -detInv * (Xg*Zb - Xb*Zg);
    XYZ2RGBmatrix[0][2] =  detInv * (Xg*Yb - Xb*Yg);
    XYZ2RGBmatrix[1][0] = -detInv * (Yr*Zb - Yb*Zr);
    XYZ2RGBmatrix[1][1] =  detInv * (Xr*Zb - Xb*Zr);
    XYZ2RGBmatrix[1][2] = -detInv * (Xr*Yb - Xb*Yr);
    XYZ2RGBmatrix[2][0] =  detInv * (Yr*Zg - Yg*Zr);
    XYZ2RGBmatrix[2][1] = -detInv * (Xr*Zg - Xg*Zr);
    XYZ2RGBmatrix[2][2] =  detInv * (Xr*Yg - Xg*Yr);

    //test matrix -- they are inverses!
 
    return 0;
  }
  else {
    return -1;
  }


}

BRDF *ButterflyWingSurface::Shade(const ShadeContext & sc) const
{

  Float surfKd = sc.InitializeFloat(RI_KD, Kd);
  //Float shadeKs = sc.InitializeFloat(RI_KS, Ks);
  //Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
  //Float shadeRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
  Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));

  const char *surfTexName = sc.InitializeString(RI_TEXTURENAME, texName);
  if (surfTexName != NULL) {
    TextureMap *tmap = sc.GetTexture(surfTexName);

    if (tmap) {
      Float s = sc.InitializeFloat(RI_S, 0.);
      Float t = sc.InitializeFloat(RI_T, 0.);
      Cs *= tmap->Lookup(s, t);
    }
  }
  // don't EVER return a mix since scattermixture::fr() won't work right if one of the elements of the mixture is an iridescence
  // also, don't create a scatter mixture that contains an iridescence
  return (CreateIridescence(surfKd * Cs, sc.Ns, XYZ2RGBmatrix, colorCurves));
}

BRDF *TexturedSurface::Shade(const ShadeContext & sc) const
{
  Spectrum surfColor = sc.InitializeColor(RI_CS, Spectrum(1.));

  const char *surfTexName = sc.InitializeString(RI_TEXTURENAME, texName);
  TextureMap *tmap = sc.GetTexture(surfTexName);

  if (tmap) {
    Float s = sc.InitializeFloat(RI_S, 0.);
    Float t = sc.InitializeFloat(RI_T, 0.);
    surfColor *= tmap->Lookup(s, t);
  }
  return (CreateTexturedBRDF(surfColor));
}




BRDF *ViewSTSurface::Shade(const ShadeContext & sc) const
{
	Float sp = sc.InitializeFloat(RI_S, 0.);
	Float tp = sc.InitializeFloat(RI_T, 0.);
	return CreateLambertian(Spectrum(sp, tp, 0.), sc.Ns);
}

Material *MaterialCreate(const char *name, SurfaceFunction * bindings)
{
	if (strcmp(name, RI_MATTE) == 0)
		return new MatteSurface(bindings);
	else if (strcmp(name, RI_PLASTIC) == 0)
		return new PlasticSurface(bindings);
	else if (strcmp(name, RI_PAINTEDPLASTIC) == 0)
		return new PaintedPlasticSurface(bindings);
	else if (strcmp(name, RI_SHINYMETAL) == 0)
		return new ShinyMetalSurface(bindings);
	//ours
	else if (strcmp(name, RI_BUTTERFLYWING) == 0)
		return new ButterflyWingSurface(bindings);
	else if (strcmp(name, RI_TEXTUREDSURFACE) == 0)
		return new TexturedSurface(bindings);
	else if (strcmp(name, LRT_VIEWST) == 0)
		return new ViewSTSurface(bindings);
	else if (strcmp(name, LRT_CHECKERED) == 0)
		return new CheckeredSurface(bindings);
	else if (strcmp(name, LRT_GLASS) == 0)
		return new GlassSurface(bindings);
	else {
		Warning("Unknown surface shader: %s," "returning \"matte\"", name);
		return new MatteSurface(bindings);
	}
}

SurfaceFunction::~SurfaceFunction()
{

	for (u_int i = 0; i < vertexPoints.size(); ++i)
		delete[]vertexPoints[i].second;

	for (u_int i = 0; i < uniformStrings.size(); ++i)
		delete uniformStrings[i].second;

	for (u_int i = 0; i < vertexFloats.size(); ++i)
		delete[]vertexFloats[i].second;
	for (u_int i = 0; i < vertexHPoints.size(); ++i)
		delete[]vertexHPoints[i].second;
	for (u_int i = 0; i < vertexVectors.size(); ++i)
		delete[]vertexVectors[i].second;
	for (u_int i = 0; i < vertexNormals.size(); ++i)
		delete[]vertexNormals[i].second;
	for (u_int i = 0; i < vertexColors.size(); ++i)
		delete[]vertexColors[i].second;

}

SurfaceFunction::SurfaceFunction(const SurfaceFunction & sf)
{
	nVertex = sf.nVertex;

	vector<pair<RtToken, Point> >::const_iterator uniformPointIter;
	for (uniformPointIter = sf.uniformPoints.begin();
		 uniformPointIter != sf.uniformPoints.end(); uniformPointIter++) {
		uniformPoints.push_back(make_pair((*uniformPointIter).first,
										  (*uniformPointIter).second));
	}

	vector<pair<RtToken, Point *> >::const_iterator vertexPointIter;
	for (vertexPointIter = sf.vertexPoints.begin();
		 vertexPointIter != sf.vertexPoints.end(); vertexPointIter++) {
		Point *values = new Point[nVertex];
		memcpy(values,
			   (*vertexPointIter).second, nVertex * sizeof(*values));
		vertexPoints.
			push_back(make_pair((*vertexPointIter).first, values));
	}

	vector<pair<RtToken, Float> >::const_iterator uniformFloatIter;
	for (uniformFloatIter = sf.uniformFloats.begin();
		 uniformFloatIter != sf.uniformFloats.end(); uniformFloatIter++) {
		uniformFloats.push_back(make_pair((*uniformFloatIter).first,
										  (*uniformFloatIter).second));
	}
	vector<pair<RtToken, Float> >::const_iterator uniformHPointIter;
	for (uniformHPointIter = sf.uniformHPoints.begin();
		 uniformHPointIter != sf.uniformHPoints.end(); uniformHPointIter++) {
		uniformHPoints.push_back(make_pair((*uniformHPointIter).first,
										   (*uniformHPointIter).second));
	}
	vector<pair<RtToken, Vector> >::const_iterator uniformVectorIter;
	for (uniformVectorIter = sf.uniformVectors.begin();
		 uniformVectorIter != sf.uniformVectors.end(); uniformVectorIter++) {
		uniformVectors.push_back(make_pair((*uniformVectorIter).first,
										   (*uniformVectorIter).second));
	}
	vector<pair<RtToken, Normal> >::const_iterator uniformNormalIter;
	for (uniformNormalIter = sf.uniformNormals.begin();
		 uniformNormalIter != sf.uniformNormals.end(); uniformNormalIter++) {
		uniformNormals.push_back(make_pair((*uniformNormalIter).first,
										   (*uniformNormalIter).second));
	}
	vector<pair<RtToken, Spectrum> >::const_iterator uniformColorIter;
	for (uniformColorIter = sf.uniformColors.begin();
		 uniformColorIter != sf.uniformColors.end(); uniformColorIter++) {
		uniformColors.push_back(make_pair((*uniformColorIter).first,
										  (*uniformColorIter).second));
	}
	vector<pair<RtToken, char *> >::const_iterator uniformStringIter;
	for (uniformStringIter = sf.uniformStrings.begin();
		 uniformStringIter != sf.uniformStrings.end(); uniformStringIter++) {
		uniformStrings.push_back(make_pair((*uniformStringIter).first,
										   Strdup((*uniformStringIter).
												  second)));
	}

	vector<pair<RtToken, Float *> >::const_iterator vertexFloatIter;
	for (vertexFloatIter = sf.vertexFloats.begin();
		 vertexFloatIter != sf.vertexFloats.end(); vertexFloatIter++) {
		Float *values = new Float[nVertex];
		memcpy(values, (*vertexFloatIter).second,
			   nVertex * sizeof(*values));
		vertexFloats.
			push_back(make_pair((*vertexFloatIter).first, values));
	}
	vector<pair<RtToken, Float *> >::const_iterator vertexHPointIter;
	for (vertexHPointIter = sf.vertexHPoints.begin();
		 vertexHPointIter != sf.vertexHPoints.end(); vertexHPointIter++) {
		Float *values = new Float[4 * nVertex];
		memcpy(values, (*vertexHPointIter).second,
			   4 * nVertex * sizeof(*values));
		vertexHPoints.
			push_back(make_pair((*vertexHPointIter).first, values));
	}
	vector<pair<RtToken, Vector *> >::const_iterator vertexVectorIter;
	for (vertexVectorIter = sf.vertexVectors.begin();
		 vertexVectorIter != sf.vertexVectors.end(); vertexVectorIter++) {
		Vector *values = new Vector[nVertex];
		memcpy(values, (*vertexVectorIter).second,
			   nVertex * sizeof(*values));
		vertexVectors.
			push_back(make_pair((*vertexVectorIter).first, values));
	}
	vector<pair<RtToken, Normal *> >::const_iterator vertexNormalIter;
	for (vertexNormalIter = sf.vertexNormals.begin();
		 vertexNormalIter != sf.vertexNormals.end(); vertexNormalIter++) {
		Normal *values = new Normal[nVertex];
		memcpy(values, (*vertexNormalIter).second,
			   nVertex * sizeof(*values));
		vertexNormals.
			push_back(make_pair((*vertexNormalIter).first, values));
	}
	vector<pair<RtToken, Spectrum *> >::const_iterator vertexColorIter;
	for (vertexColorIter = sf.vertexColors.begin();
		 vertexColorIter != sf.vertexColors.end(); vertexColorIter++) {
		Spectrum *values = new Spectrum[nVertex];
		memcpy(values, (*vertexColorIter).second,
			   nVertex * sizeof(*values));
		vertexColors.
			push_back(make_pair((*vertexColorIter).first, values));
	}

}

void SurfaceFunction::AddUniformPoint(RtToken name, const Float * data)
{
	uniformPoints.
		push_back(make_pair(name, Point(data[0], data[1], data[2])));
}

void SurfaceFunction::AddVertexPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Point *values = new Point[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexPoints.push_back(make_pair(name, values));
}

void SurfaceFunction::AddUniformFloat(RtToken name, const Float * data)
{
	uniformFloats.push_back(make_pair(name, *data));
}
void SurfaceFunction::AddUniformHPoint(RtToken name, const Float * data)
{
	uniformHPoints.push_back(make_pair(name, data[0]));
	uniformHPoints.push_back(make_pair(name, data[1]));
	uniformHPoints.push_back(make_pair(name, data[2]));
	uniformHPoints.push_back(make_pair(name, data[3]));
}
void SurfaceFunction::AddUniformVector(RtToken name, const Float * data)
{
	uniformVectors.
		push_back(make_pair(name, Vector(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformNormal(RtToken name, const Float * data)
{
	uniformNormals.
		push_back(make_pair(name, Normal(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformColor(RtToken name, const Float * data)
{
	uniformColors.
		push_back(make_pair(name, Spectrum(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformString(RtToken name, const char *data)
{
	uniformStrings.push_back(make_pair(name, Strdup(data)));
}

void SurfaceFunction::AddVertexFloat(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[nVertex];
	memcpy(values, data, nVertex * sizeof(Float));
	vertexFloats.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexHPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[4 * nVertex];
	memcpy(values, data, 4 * nVertex * sizeof(Float));
	vertexHPoints.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexVector(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Vector *values = new Vector[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexVectors.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexNormal(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Normal *values = new Normal[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexNormals.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexColor(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Spectrum *values = new Spectrum[nVertex];
	for (u_int i = 0; i < nVertex; ++i)
		values[i] =
			Spectrum(data[3 * i], data[3 * i + 1], data[3 * i + 2]);
	vertexColors.push_back(make_pair(name, values));
}

const Point *SurfaceFunction::GetUniformPoint(RtToken name) const
{
	for (u_int i = 0; i < uniformPoints.size(); ++i) {
		if (name == uniformPoints[i].first)
			return &uniformPoints[i].second;
	}
	return NULL;
}

const Point *SurfaceFunction::GetVertexPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		if (name == vertexPoints[i].first)
			return vertexPoints[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetUniformFloat(RtToken name) const
{
	for (u_int i = 0; i < uniformFloats.size(); ++i) {
		if (name == uniformFloats[i].first)
			return &uniformFloats[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetUniformVector(RtToken name) const
{
	for (u_int i = 0; i < uniformVectors.size(); ++i) {
		if (name == uniformVectors[i].first)
			return &uniformVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetUniformNormal(RtToken name) const
{
	for (u_int i = 0; i < uniformNormals.size(); ++i) {
		if (name == uniformNormals[i].first)
			return &uniformNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetUniformColor(RtToken name) const
{
	for (u_int i = 0; i < uniformColors.size(); ++i) {
		if (name == uniformColors[i].first)
			return &uniformColors[i].second;
	}
	return NULL;
}
const char *SurfaceFunction::GetUniformString(RtToken name) const
{
	for (u_int i = 0; i < uniformStrings.size(); ++i) {
		if (name == uniformStrings[i].first)
			return uniformStrings[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetVertexFloat(RtToken name) const
{
	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		if (name == vertexFloats[i].first)
			return vertexFloats[i].second;
	}
	return NULL;
}
const Float *SurfaceFunction::GetVertexHPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexHPoints.size(); ++i) {
		if (name == vertexHPoints[i].first)
			return vertexHPoints[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetVertexVector(RtToken name) const
{
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		if (name == vertexVectors[i].first)
			return vertexVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetVertexNormal(RtToken name) const
{
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		if (name == vertexNormals[i].first)
			return vertexNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetVertexColor(RtToken name) const
{
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		if (name == vertexColors[i].first)
			return vertexColors[i].second;
	}
	return NULL;
}

Point SurfaceFunction::InitializePoint(RtToken name, const Point & def) const
{
	const Point *ptr = GetUniformPoint(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Float SurfaceFunction::InitializeFloat(RtToken name, Float def) const
{
	const Float *ptr = GetUniformFloat(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Spectrum SurfaceFunction::InitializeColor(RtToken name,
										  const Spectrum & def) const
{
	const Spectrum *ptr = GetUniformColor(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Vector SurfaceFunction::InitializeVector(RtToken name, const Vector & def) const
{
	const Vector *ptr = GetUniformVector(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Normal SurfaceFunction::InitializeNormal(RtToken name, const Normal & def) const
{
	const Normal *ptr = GetUniformNormal(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}
const char *SurfaceFunction::InitializeString(RtToken name,
											  const char *def) const
{
	const char *ptr = GetUniformString(name);
	if (ptr != NULL)
		return ptr;
	else
		return def;
}

InterpolatedPrimData::InterpolatedPrimData(const SurfaceFunction * pd)
{
	primitiveData = pd;

	if (primitiveData->vertexPoints.size() > 0) {
		int nvp = primitiveData->vertexPoints.size();
		interpVertexPoints = new Point[nvp];
	} else
		interpVertexPoints = NULL;

	if (primitiveData->vertexFloats.size() > 0) {
		interpVertexFloats = new Float[primitiveData->vertexFloats.size()];
	} else {
		interpVertexFloats = NULL;
	}
	if (primitiveData->vertexVectors.size() > 0) {
		int nv = primitiveData->vertexVectors.size();
		interpVertexVectors = new Vector[nv];
	} else {
		interpVertexVectors = NULL;
	}
	if (primitiveData->vertexNormals.size() > 0) {
		int vn = primitiveData->vertexNormals.size();
		interpVertexNormals = new Normal[vn];
	} else {
		interpVertexNormals = NULL;
	}
	if (primitiveData->vertexColors.size() > 0) {
		int nc = primitiveData->vertexColors.size();
		interpVertexColors = new Spectrum[nc];
	} else {
		interpVertexColors = NULL;
	}

}

InterpolatedPrimData::~InterpolatedPrimData()
{

	delete[]interpVertexPoints;

	delete[]interpVertexFloats;
	delete[]interpVertexVectors;
	delete[]interpVertexNormals;
	delete[]interpVertexColors;

}

InterpolatedPrimData *SurfaceFunction::Interpolate(int n,
												   const int *offsets,
												   const Float * weights) const
{
	InterpolatedPrimData *ret = new InterpolatedPrimData(this);

	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		Point value(0, 0, 0);
		Point *pts = vertexPoints[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value.x += weights[j] * pts[o].x;
			value.y += weights[j] * pts[o].y;
			value.z += weights[j] * pts[o].z;
		}
		ret->interpVertexPoints[i] = value;
	}

	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		Float value = 0.;
		Float *f = vertexFloats[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * f[o];
		}
		ret->interpVertexFloats[i] = value;
	}
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		Vector value(0, 0, 0);
		Vector *vec = vertexVectors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * vec[o];
		}
		ret->interpVertexVectors[i] = value;
	}
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		Normal value(0, 0, 0);
		Normal *norm = vertexNormals[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * norm[o];
		}
		ret->interpVertexNormals[i] = value;
	}
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		Spectrum value(0.);
		Spectrum *cp = vertexColors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * cp[o];
		}
		ret->interpVertexColors[i] = value;
	}

	return ret;
}

const Point *InterpolatedPrimData::GetPoint(RtToken token) const
{
	const Point *ret = primitiveData->GetUniformPoint(token);
	if (ret)
		return ret;

	const vector<pair<RtToken, Point *> > &vpp =
	primitiveData->vertexPoints;
	for (u_int i = 0; i < vpp.size(); ++i) {
		if (token == vpp[i].first)
			return &interpVertexPoints[i];
	}
	return NULL;

}

const Float *InterpolatedPrimData::GetFloat(RtToken token) const
{
	const vector<pair<RtToken, Float *> > &vfp = primitiveData->vertexFloats;
	for (u_int i = 0; i < vfp.size(); ++i) {
		if (token == vfp[i].first)
			return &interpVertexFloats[i];
	}
	return NULL;
}
const Vector *InterpolatedPrimData::GetVector(RtToken token) const
{
	const vector<pair<RtToken, Vector *> > &vvp = primitiveData->vertexVectors;
	for (u_int i = 0; i < vvp.size(); ++i) {
		if (token == vvp[i].first)
			return &interpVertexVectors[i];
	}
	return NULL;
}
const Normal *InterpolatedPrimData::GetNormal(RtToken token) const
{
	const vector<pair<RtToken, Normal *> > &vnp = primitiveData->vertexNormals;
	for (u_int i = 0; i < vnp.size(); ++i) {
		if (token == vnp[i].first)
			return &interpVertexNormals[i];
	}
	return NULL;
}
const Spectrum *InterpolatedPrimData::GetColor(RtToken token) const
{
	const vector<pair<RtToken, Spectrum *> > &vcp = primitiveData->vertexColors;
	for (u_int i = 0; i < vcp.size(); ++i) {
		if (token == vcp[i].first)
			return &interpVertexColors[i];
	}
	return NULL;
}
