
#include "lrt.h"
#include "primitives.h"
#include "accel.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "shading.h"
#include "transport.h"

Integrator::~Integrator()
{
}

Spectrum ColorIntegrator::Integrate(const Ray & ray,
									HitInfo * hitInfo, Float * hitDist,
									Float * alpha) const
{
	if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
		*alpha = 1.;
		const Spectrum *Csp = hitInfo->GetColor(RI_CS);
		if (Csp) {
			return *Csp;
		} else {
			return hitInfo->hitPrim->attributes->Color;
		}
	} else {
		*alpha = 0.;
		return Spectrum(0, 0, 0);
	}
}

Spectrum RayCastingIntegrator::Integrate(const Ray & ray,
										 HitInfo * hitInfo,
										 Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
		ShadeContext shadeContext(hitInfo, -ray.D);
		Spectrum L(0.);
		*alpha = 1.;

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

Spectrum WhittedIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
									  Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
		Spectrum L(0.);
		*alpha = 1.;

		ShadeContext shadeContext(hitInfo, -ray.D);

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

		if (++RayDepth < 5) {
			Ray newRay;
			newRay.O = Pw;
			for (int i = 0; i < brdf->SpecularComponents(); ++i) {

				Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
				HitInfo hitInfo2;
				Float alpha2, maxt = INFINITY;
				L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);

			}
		}
		--RayDepth;

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

int WhittedIntegrator::RayDepth = 0;
int MCIntegrator::RayDepth = 0;
Spectrum MCIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
								 Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
		ShadeContext shadeContext(hitInfo, -ray.D);
		Spectrum L(0.);
		*alpha = 1.;

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		if (hitInfo->hitPrim->attributes->Sampling ==
			PrimitiveAttributes::SampleSurface) {
			// YOUR CODE HERE: integrate by sampling the BRDF
		  Point Pw =
		    hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		  


		  // here's the estimator
		  // 2*(s+2)*Li*cos(thetai)*cos(alpha)/(M_PI);
		    Spectrum MCestimator(0.);

		    if (++RayDepth < 3) {
		      int numSamples = 30;

		    // iterate over number of samples
		      for (int i=0; i<numSamples; i++) {
			
			Vector wi;
			Float pdf;
			Float randfloats[2] = {RandomFloat(), RandomFloat()};
			
			Spectrum kernel = brdf->Sample(randfloats, &wi, &pdf);
			
			if (pdf > 0) {
			  //fprintf(stderr, "got here\n");
			  HitInfo hitInfo2;
			  Float alpha2, maxt = INFINITY;
			  MCestimator += kernel *
			  Integrate(Ray(Point(Pw.x, Pw.y, Pw.z),
			  Vector(wi.x, wi.y, wi.z)),
			  &hitInfo2, &maxt, &alpha2);
			}
		      }
		      --RayDepth;
		      L += MCestimator / numSamples;
		    }
		}
		else if (hitInfo->hitPrim->attributes->Sampling ==
			 PrimitiveAttributes::SampleLight) {
		  // YOUR CODE HERE: integrate by sampling the Light source
		  // for each light...
		  //    for i = 0 to light->NumSamples()
		  //        compute (estimate of rendering equation)
		  // / NumSamples
		  

		  Point Pw =
		    hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		  
		  const list < Light * >&lights =
		    hitInfo->hitPrim->attributes->Lights;
		  list < Light * >::const_iterator iter = lights.begin();
		  while (iter != lights.end()) {
		    const Light *lt = *iter;
		    Point Plight;
		    Spectrum MCestimator(0.);

		    for (int i=0; i<lt->NumSamples(); i++) {
		      Spectrum dE = lt->dE(Pw, &Plight);
		      if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
			Vector wi = (Plight - Pw).Hat();
			const Normal & Nw = shadeContext.Ns;

			// I think dE() already multiplies dE by cos(theta')
			// and divides by dist^2 and pdf
			MCestimator += dE * brdf->fr(wi) * Dot(wi, Nw);
		      }
		    }
		    L += MCestimator / lt->NumSamples();		    
		    ++iter;
		  }
		  
		  
		}
		else if (hitInfo->hitPrim->attributes->Sampling ==
			 PrimitiveAttributes::SampleCombination) {
		  // YOUR CODE HERE: integrate by sampling a little bit
		  // from both and then combining the results intelligently
		}

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

#include "sl.h"


static void null_shader(RmanShaderParams & params)
{
	params.Ci = SLTriple(0);
	params.Oi = SLTriple(1);
}

RendermanShader::RendermanShader(const char *name)
{
	char path[4096];
	this->name = strdup(name);
	char *shader_dir = getenv("LRT_SHADER_DIR");
	if (!shader_dir)
		shader_dir = "/home/humper/work/litrt/src/shaders";

	sprintf(path, "%s/%s.so", shader_dir, name);
	dso = new DSO(path);

	already_warned = false;

#define RMAN_MUST_HAVE( var, sym ) \
	void *var = dso->GetSym( sym ); \
	if (!var) { \
		Warning( "No %s in %s", sym, path );\
		type = RMAN_BROKEN_SHADER; \
		shader = null_shader; \
		return; \
	}

	RMAN_MUST_HAVE(type_ptr, "shader_type");
	RMAN_MUST_HAVE(shader_ptr, "shader");

#undef RMAN_MUST_HAVE

	type = *((RmanShaderType *) type_ptr);
	shader = (shader_func) shader_ptr;
}

void RendermanShader::Run(RmanShaderParams & params)
{
	switch (type) {
	case RMAN_SURFACE_SHADER:
		params.Cs =
			SLTriple(params.shadeContext.
					 InitializeColor(RI_CS, Spectrum(1.0)));
		params.Os = SLTriple(Spectrum(1, 1, 1));
		params.N = SLTriple(params.shadeContext.Ns);
		params.P =
			SLTriple(params.hitInfo->hitPrim->attributes->
					 ObjectToWorld(params.hitInfo->Pobj));
		params.I = SLTriple(params.shadeContext.wo);
		params.Ci = SLTriple(0, 0, 0);
		params.Os = SLTriple(1, 1, 1);
		shader(params);
		break;
	case RMAN_BROKEN_SHADER:
		if (!already_warned) {
			already_warned = 1;
			Warning("Shader %s is broken.", name);
		}
		params.Ci = SLTriple(RandomFloat(), RandomFloat(), RandomFloat());
		params.Oi = SLTriple(1, 1, 1);
		break;
	default:
		if (!already_warned) {
			already_warned = 1;
			Warning("I don't know how to handle shader %s", name);
		}
		break;
	}
}

Spectrum RendermanIntegrator::Integrate(const Ray & ray,
										HitInfo * hitInfo, Float * hitDist,
										Float * alpha) const
{

	if (scene->Intersect(ray, 0., hitDist, hitInfo) &&
		hitInfo->hitPrim->attributes->Surface != NULL) {
		ShadeContext shadeContext(hitInfo, -ray.D);
		RmanShaderParams params(hitInfo->hitPrim->attributes->Surface->
								surfaceFunction, shadeContext,
								*scene->sampler, hitInfo);

		*alpha = 1;

		RendermanShader *shader = NULL;
		Assert(shader != NULL);
		/* XXX MMP: shader creation and management like this should happen up at
		   the RI level, I think, so that this guy can just grab a useful Shader *
		   from the HitInfo.  I've commented this out for now so that I can clean
		   up stuff in shading.nw for 348 assignment 3. */
#if 0
		RendermanShader *shader =
			(RendermanShader *) shaders.Search(hitInfo->attributes->
											   Surface->Name());
		if (!shader) {
			shader =
				new RendermanShader(hitInfo->attributes->Surface->Name());
			shaders.Add(hitInfo->attributes->Surface->Name(), shader);
		}
#endif
		shader->Run(params);

		return params.Ci.MakeSpectrum();
	} else {
		*alpha = 0;
		return Spectrum(0.);
	}
}
