#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>

#include "g_math.h"

/*

texel input options

- texel x-y-z size [0..] [0..] [0..]

- hair color [0,1] [0,1] [0,1]

- hair color noise [0,1] [0,1] [0,1]

- hair density [0,1]

- base hair height [0,1], height noise [0..1]

- base hair thickness [0, ...], thick noise [0..1]

// by default hair normal will be 0 1 0

- tangent noise [0..1],[0..1]

- hair optical density

*/

typedef struct
{
    vec3_t      tangent;
    float       density;
} cell_t;

typedef struct
{
    int         x_size;
    int         y_size;
    int         z_size;
    cell_t   ***cells;
    
    float       color[3];
    float       color_noise[3];
    float       height_base;
    float       height_noise;
    float       thickness_base;
    float       thickness_noise;
    float       tangent_noise_x, tangent_noise_z;
    float       optical_density;
    float       hair_density;
} texel_t;

//////////////////////////////////////////////////////////////////////
//  makefur code
//////////////////////////////////////////////////////////////////////

void
allocate_texel(texel_t *texel)
{
    int i, j;
        
    texel->cells = (cell_t ***) malloc(texel->x_size * sizeof(cell_t **));
    for (i = 0; i < texel->x_size; i++)
    {
        texel->cells[i] = (cell_t **) malloc(texel->y_size * sizeof(cell_t *));
        
        for (j = 0; j < texel->y_size; j++)
        {
            texel->cells[i][j] = (cell_t *) calloc(texel->z_size, sizeof(cell_t));
        }
    }
}

void
compute_texel_tangent(texel_t *texel)
{
    int i, j, k;
    
    // set tangent vectors -- all straight up for now
    for (i = 0; i < texel->x_size; i++)
    {
        for (j = 0; j < texel->y_size; j++)
        {
            for (k = 0; k < texel->z_size; k++)
            {
                VectorSet(0.0f, 1.0f, 0.0f, texel->cells[i][j][k].tangent);
            }
        }
    }    
}

void
read_input(char *in_filename, texel_t *texel)
{
    char buf[256];
    FILE *in_fp;
    
    in_fp = fopen(in_filename, "r");
    
    if (!in_fp)
    {
        printf("<*> couldn't open input file <%s> for reading\n", in_filename);
        exit(1);
    }
    
    printf("<*> reading texel parameters\n");
    
    // read texel x-y-z size
    fgets(buf, 255, in_fp);
    sscanf(buf, "%d %d %d", &(texel->x_size), &(texel->y_size), &(texel->z_size));
    printf("<*> - size (%d, %d, %d)\n", texel->x_size, texel->y_size, texel->z_size);
    
    // read texel color
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f %f %f", texel->color, texel->color + 1, texel->color + 2);
    printf("<*> - color (%.1f, %.1f, %.1f)\n", texel->color[0], texel->color[1], texel->color[2]);
    
    // read texel color noise
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f %f %f", texel->color_noise, texel->color_noise + 1, texel->color_noise + 2);
    printf("<*> - color noise (%.3f, %.3f, %.3f)\n", texel->color_noise[0], texel->color_noise[1], texel->color_noise[2]);    
    
    // read texel density
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f", &(texel->hair_density));
    printf("<*> - hair density %.4f\n", texel->hair_density);
    
    // read texel height, noise
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f %f", &(texel->height_base), &(texel->height_noise));
    printf("<*> - height %.3f + Random(%.3f, %.3f)\n", texel->height_base,
        -texel->height_noise, texel->height_noise);
    
    // read texel thickness, noise
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f %f", &(texel->thickness_base), &(texel->thickness_noise));
    printf("<*> - thickness %.3f + Random(%.3f, %.3f)\n", texel->thickness_base,
        -texel->thickness_noise, texel->thickness_noise);    
    
    // read tangent noise
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f %f", &(texel->tangent_noise_x), &(texel->tangent_noise_z));
    
    // read optical density
    fgets(buf, 255, in_fp);
    sscanf(buf, "%f", &(texel->optical_density));
    printf("<*> - optical density %.2f\n", texel->optical_density);
    
    printf("\n");    
    fclose(in_fp);
}

texel_t *
create_texel(char *in_filename)
{
    int h, num_hairs;
            
    // create new texel
    texel_t *texel = (texel_t *) malloc(sizeof(texel_t));
    
    // read in texel params
    read_input(in_filename, texel);

    // allocate texel storage
    allocate_texel(texel);
    
    // set texel tangents at every point
    compute_texel_tangent(texel);
    
    // compute total number of hairs to create in this texel
    num_hairs = (int) (texel->hair_density * texel->x_size * texel->z_size + 0.5f);
    
    // throw darts for now -- do poisson distribution later =(
    printf("<*> computing %d hairs ...\n", num_hairs);
    for (h = 0; h < num_hairs; h++)
    {
        float sample_x = RandomRange(0.0f, texel->x_size);
        float sample_z = RandomRange(0.0f, texel->z_size);
        int j;
        
        int i = (int) Clamp(sample_x, 0.5f, texel->x_size - 0.5f);
        int k = (int) Clamp(sample_z, 0.5f, texel->z_size - 0.5f);
        
        float hair_length = texel->height_base + RandomRange(-texel->height_noise, texel->height_noise);
        int len = (int) (hair_length * texel->y_size);
                        
        for (j = 0; j < len; j++)
        {            
            float thickness = texel->thickness_base + RandomRange(-texel->thickness_noise, texel->thickness_noise);
            texel->cells[i][j][k].density += thickness;
        }
    }
    
    return texel;
}

void
write_texel(texel_t *texel, const char *out_filename)
{
    FILE *out_fp;
    int i, j;
    
    // open output file
    out_fp = fopen(out_filename, "wb");
    
    if (!out_fp)
    {
        printf("<*> Error: couldn't open <%s> for writing\n", out_filename);
        exit(1);
    }
    
    // write out dimensions
    fwrite(&(texel->x_size), sizeof(int), 1, out_fp);
    fwrite(&(texel->y_size), sizeof(int), 1, out_fp);
    fwrite(&(texel->z_size), sizeof(int), 1, out_fp);
    
    // write color
    fwrite(texel->color, sizeof(float), 3, out_fp);
    
    // write color noise
    fwrite(texel->color_noise, sizeof(float), 3, out_fp);
    
    // write optical density
    fwrite(&(texel->optical_density), sizeof(float), 1, out_fp);
    
    // write out data
    for (i = 0; i < texel->x_size; i++)
    {
        for (j = 0; j < texel->y_size; j++)
        {
            fwrite(texel->cells[i][j], sizeof(cell_t), texel->z_size, out_fp);
        }
    }
    
    fclose(out_fp);
}

int
main(int argc, char **argv)
{
    texel_t *texel = NULL;
    
    srand(time(NULL));
    
    printf("<*> makefur 1.0 : texel generation program\n\n");
    
    if (argc != 3)
    {
        printf("<*> usage: makefur <input filename> <outfile filename>\n");
        return 0;
    }
    
    texel = create_texel(argv[1]);
    
    printf("<*> writing texel to file\n");
    write_texel(texel, argv[2]);
    
    printf("<*> done\n\n");
    
    return 0;
}
