
#include "trimesh.h"

TriangleMesh::TriangleMesh(int nt, int nv, int *vi, Point * P,
						   PrimitiveAttributes * a, SurfaceFunction * sf)
:Primitive(a, sf)
{
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	p = new Point[nverts];
	memcpy(p, P, nverts * sizeof(Point));
}

TriangleMesh::~TriangleMesh()
{
	delete[]vertexIndex;
	delete[]p;
}

BBox TriangleMesh::BoundObjectSpace() const
{
	BBox objectSpaceBounds(p[0], p[1]);
	for (int i = 2; i < nverts; i++)
		objectSpaceBounds = Union(objectSpaceBounds, p[i]);
	return objectSpaceBounds;
}

BBox TriangleMesh::BoundWorldSpace() const
{
	BBox worldBounds(attributes->ObjectToWorld(p[0]),
					 attributes->ObjectToWorld(p[1]));
	for (int i = 2; i < nverts; i++) {
		worldBounds = Union(worldBounds, attributes->ObjectToWorld(p[i]));
	}
	return worldBounds;
}

void TriangleMesh::Refine(vector < Primitive * >*refined) const
{    
    const char *texel_shader = attributes->Surface->surfaceFunction->GetUniformString(LRT_TEXEL_SHADER);
    
    refined->reserve(ntris);
    
    //
    //  determine which type of triangle mesh to build
    //
    
    if (!texel_shader)
    {
        printf("<*> building regular triangle mesh\n");
    	for (int i = 0; i < ntris; ++i)
    	{
    		refined->push_back(new Triangle(this, i));
    	}
	}
    else
    {
        printf("<*> building furry triangles\n");
        
        FurryMesh *furmesh = new FurryMesh(
            ntris, nverts, vertexIndex, p, attributes->Surface->surfaceFunction);

    	for (int i = 0; i < ntris; ++i)
    	{
    		refined->push_back(new FurryTriangle(this, i, furmesh, attributes, surfaceFunction));
    	}
    }
}

BBox Triangle::BoundObjectSpace() const
{
	BBox objectBounds(mesh->p[mesh->vertexIndex[triNum * 3]],
					  mesh->p[mesh->vertexIndex[triNum * 3 + 1]]);
	return Union(objectBounds, mesh->p[mesh->vertexIndex[triNum * 3 + 2]]);
}

BBox Triangle::BoundWorldSpace() const
{
	Transform o2w = mesh->attributes->ObjectToWorld;
	BBox objectBounds(o2w(mesh->p[mesh->vertexIndex[triNum * 3]]),
					  o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 1]]));
	return Union(objectBounds,
				 o2w(mesh->p[mesh->vertexIndex[triNum * 3 + 2]]));
}

bool Triangle::IntersectClosest(const Ray & ray, Float mint,
								Float * maxt, HitInfo * hit) const
{

	static int triangleTests = 0, triangleHits = 0;
	if (triangleTests == 0)
		StatsRegisterRatio(STATS_DETAILED, "Geometry",
						   "Triangle Ray Intersections", &triangleHits,
						   &triangleTests);

	++triangleTests;

	int *vptr = &(mesh->vertexIndex[3 * triNum]);
	int vertexIndex0 = *vptr++;
	int vertexIndex1 = *vptr++;
	int vertexIndex2 = *vptr++;

	const Point *p = mesh->p;
	Vector E1 = p[vertexIndex1] - p[vertexIndex0];
	Vector E2 = p[vertexIndex2] - p[vertexIndex0];
	Vector S_1 = Cross(ray.D, E2);
	Float divisor = Dot(S_1, E1);
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;

	Vector T = ray.O - mesh->p[vertexIndex0];
	Float u = Dot(T, S_1) * invDivisor;
	if (u < 0. || u > 1.0)
		return false;

	Vector S_2 = Cross(T, E1);
	Float v = Dot(ray.D, S_2) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return false;

	Float t = Dot(E2, S_2) * invDivisor;
	if (t < mint || t > *maxt)
		return false;
	*maxt = t;
	++triangleHits;

	if (hit != NULL) {

		Normal N = Normal(Cross(E2, E1));
		N.Normalize();
		hit->RecordHit(ray(t), N, u, v, mesh);

		int interpOffsets[3] =
			{ vertexIndex0, vertexIndex1, vertexIndex2 };
		Float interpWeights[3] = { 1. - u - v, u, v };
		InterpolatedPrimData *interpolatedData;
		interpolatedData =
			mesh->surfaceFunction->Interpolate(3,
											   interpOffsets,
											   interpWeights);
		hit->SetInterpolatedData(interpolatedData);

	}
	return true;
}

//////////////////////////////////////////////////////////////////////
//  FurryMesh code
//////////////////////////////////////////////////////////////////////

FurryMesh::FurryMesh(
    int     num_tris,
    int     num_verts,
    int    *indices,
    Point  *vertices,
    SurfaceFunction *fur_surface)
{
    m_numtris   = num_tris;
    m_numverts  = num_verts;
    
    // get height (default 0.2)
    float height = fur_surface->InitializeFloat(LRT_FUR_HEIGHT, 0.2f);

    // get noise (default 0)
    float height_noise = fur_surface->InitializeFloat(LRT_FUR_HEIGHT_NOISE, 0.0f);

    // get normal bias (none)
    Vector normal_bias = fur_surface->InitializeVector(LRT_FUR_NORMAL_BIAS, Vector(0.0f, 0.0f, 0.0f));
    Vector normal_bias_noise = fur_surface->InitializeVector(LRT_FUR_NORMAL_NOISE_BIAS, Vector(0.0f, 0.0f, 0.0f));
        
    // allocate temp space for face normals
    Vector *face_normals = new Vector[m_numtris];
    Vector *vertex_normals = new Vector[m_numverts];
    
    // compute face normals
    int i;
    int *vptr = indices;
    for (i = 0; i < m_numtris; i++)
    {
        int vertexIndex0 = *vptr++;
        int vertexIndex1 = *vptr++;
        int vertexIndex2 = *vptr++;
        
    	Vector edge1 = vertices[vertexIndex1] - vertices[vertexIndex0];
    	Vector edge2 = vertices[vertexIndex2] - vertices[vertexIndex0];
                
        face_normals[i] = Cross(edge2, edge1);
        face_normals[i].Normalize();
    }
    
    m_vertices  = new FurryMeshVertex[m_numverts];
    
    // compute averaged vertex normals
    for (i = 0; i < m_numverts; i++)
    {
        Vector sum(0.0f, 0.0f, 0.0f);
        int count = 0;
        int index = 0;
        
        // copy over base vertex
        m_vertices[i].p_base = vertices[i];
        
        // walk tri face list
        for (int j = 0; j < m_numtris; j++)
        {
            if ((i == indices[index]) || (i == indices[index + 1]) || (i == indices[index + 2]))
            {
                count++;
                sum += face_normals[j];
            }
            
            index += 3;
        }
        
        vertex_normals[i] = (sum / (float) count);   
        vertex_normals[i].Normalize();

        // normal bias        
        float bias_x = normal_bias.x + RandomFloat(-normal_bias_noise.x, normal_bias_noise.x);
        float bias_y = normal_bias.y + RandomFloat(-normal_bias_noise.y, normal_bias_noise.y);
        float bias_z = normal_bias.z + RandomFloat(-normal_bias_noise.z, normal_bias_noise.z);
        
        vertex_normals[i].x += bias_x;
        vertex_normals[i].y += bias_y;
        vertex_normals[i].z += bias_z;
        vertex_normals[i].Normalize();
                
        // compute tip point from base (use height noise)
        float curr_height = height + RandomFloat(0.0f, height_noise);
        m_vertices[i].p_tip = m_vertices[i].p_base + curr_height * vertex_normals[i];
    }
    
    delete vertex_normals;
    delete face_normals;
}

//////////////////////////////////////////////////////////////////////
//  FurryTriangle code
//////////////////////////////////////////////////////////////////////

FurryTriangle::FurryTriangle(const TriangleMesh *m, int n, FurryMesh *in_furmesh,
    PrimitiveAttributes *a, SurfaceFunction *v)
    : Triangle(m, n)
{
    attributes = a;
    surfaceFunction = v;
    
    furmesh = in_furmesh;
    
    int index0  = mesh->vertexIndex[triNum * 3];
    int index1  = mesh->vertexIndex[triNum * 3 + 1];
    int index2  = mesh->vertexIndex[triNum * 3 + 2];
    
    p0_base     = &(furmesh->m_vertices[index0].p_base);
    p0_tip      = &(furmesh->m_vertices[index0].p_tip);
    
    p1_base     = &(furmesh->m_vertices[index1].p_base);
    p1_tip      = &(furmesh->m_vertices[index1].p_tip);
    
    p2_base     = &(furmesh->m_vertices[index2].p_base);
    p2_tip      = &(furmesh->m_vertices[index2].p_tip);
}

BBox
FurryTriangle::BoundObjectSpace() const
{
    BBox bounds(*p0_base, *p0_tip);
    
    bounds = Union(bounds, *p1_base);
    bounds = Union(bounds, *p1_tip);
    
    bounds = Union(bounds, *p2_base);
    bounds = Union(bounds, *p2_tip);
    
    return bounds;
}

BBox
FurryTriangle::BoundWorldSpace() const
{   
    Transform o2w = mesh->attributes->ObjectToWorld;
    
    Point p0_b  = o2w(*p0_base);
    Point p0_t  = o2w(*p0_tip);
    
    Point p1_b  = o2w(*p1_base);
    Point p1_t  = o2w(*p1_tip);
    
    Point p2_b  = o2w(*p2_base);
    Point p2_t  = o2w(*p2_tip);
    
    BBox bounds(p0_b, p0_t);
        
    bounds = Union(bounds, p1_b);
    bounds = Union(bounds, p1_t);
    
    bounds = Union(bounds, p2_b);
    bounds = Union(bounds, p2_t);
    
    return bounds;
}

float
FurryTriangle::IntersectSingle(const Ray &ray, Point &v0, Point &v1, Point &v2) const
{
    // moller-trumbone ray-tri intersection algorithm
    
	Vector E1 = v1 - v0;
	Vector E2 = v2 - v0;
	Vector S_1 = Cross(ray.D, E2);
	Float divisor = Dot(S_1, E1);
	if (divisor == 0.)
		return false;
	Float invDivisor = 1. / divisor;

	Vector T = ray.O - v0;
	Float u = Dot(T, S_1) * invDivisor;
	if (u < 0. || u > 1.0)
		return -1.0f;

	Vector S_2 = Cross(T, E1);
	Float v = Dot(ray.D, S_2) * invDivisor;
	if (v < 0 || u + v > 1.0)
		return -1.0f;

	Float t = Dot(E2, S_2) * invDivisor;
		
    return t;
}

float
FurryTriangle::IntersectOutside(const Ray &ray)
{
    // ray origin assumed to be inside the triangle volume
    
    float t;
    
    if ((t = IntersectSingle(ray, *p0_base, *p1_base, *p2_base)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p0_base, *p1_base, *p0_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p0_tip,  *p1_base, *p1_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p0_tip,  *p1_tip,  *p2_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p1_base, *p2_base, *p1_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p2_base, *p1_tip,  *p2_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p0_base, *p2_base, *p0_tip)) > 0.0f)
    {
        return t;
    }
    
    if ((t = IntersectSingle(ray, *p2_base, *p0_tip,  *p2_tip)) > 0.0f)
    {
        return t;
    }
    
    return -1.0f;
}

bool
FurryTriangle::IntersectClosest(
    const Ray  &ray,
    Float       mint,
    Float      *maxt,
	HitInfo    *hit) const
{       
    // find entry-exit values of t
    float tvalues[8] = { -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f, -1.0f };

    tvalues[0] = IntersectSingle(ray, *p0_base, *p1_base, *p2_base);
    tvalues[1] = IntersectSingle(ray, *p0_base, *p1_base, *p0_tip);
    tvalues[2] = IntersectSingle(ray, *p0_tip,  *p1_base, *p1_tip);
    tvalues[3] = IntersectSingle(ray, *p0_tip,  *p1_tip,  *p2_tip);
    tvalues[4] = IntersectSingle(ray, *p1_base, *p2_base, *p1_tip);
    tvalues[5] = IntersectSingle(ray, *p2_base, *p1_tip,  *p2_tip);
    tvalues[6] = IntersectSingle(ray, *p0_base, *p2_base, *p0_tip);
    tvalues[7] = IntersectSingle(ray, *p2_base, *p0_tip,  *p2_tip);
    
    // compute entry tmin
    bool intersects = false;
    float tmin = RI_INFINITY;
    float tmax = 0.0f;
    for (int i = 0; i < 8; i++)
    {
        if (tvalues[i] > 0.0f && tvalues[i] < tmin)
        {
            intersects = true;
            tmin = tvalues[i];
        }
        
        if (tvalues[i] > tmax)
        {
            tmax = tvalues[i];
        }
    }
    
    if (!intersects)
    {
        return false;
    }
    
    if (tmin > *maxt)
    {
        return false;
    }
    
    *maxt = tmin;
    
    float flagged_tmax = tmax;
    
    if (tmax == tvalues[0])
    {
        // leaving through base!
        flagged_tmax -= HIT_BOTTOM_FLAG;
    }
    
    if (tmin == tmax)
    {        
        hit->RecordHit(ray(0.0f), Normal(0,1,0), 0.0f, flagged_tmax, this);
        return true;
    }
    
    if (hit)
    {
        hit->RecordHit(ray(tmin), Normal(0,1,0), tmin, flagged_tmax, this);
    }
        
    return true;
}

Point
FurryTriangle::InverseMapTexel(const Point &p)
{
    // given point p = (x,y,z), compute p' = (u,s,v)
    // where 0 <= u,v,s <= 1
        
    // find s first by approximation
    float min_s = 0.0f;
    float max_s = 1.0f;
    float s;
    
    // approximate [s] by bisection -- slow :(
    for (int k = 0; k < 7; k++)
    {
        s = (min_s + max_s) * 0.5f;
        
        // find points on triangle with this value of [s]
        Point p0_s = (*p0_base) + s * (*p0_tip - *p0_base);
        Point p1_s = (*p1_base) + s * (*p1_tip - *p1_base);
        Point p2_s = (*p2_base) + s * (*p2_tip - *p2_base);
        
        // find normal of this triangle
        Vector edge2_s  = p2_s - p0_s;
        Vector edge1_s  = p1_s - p0_s;
        Vector norm_s   = Cross(edge2_s, edge1_s).Hat();
        
        // find distance from incoming point [p] to plane
        Vector diff_s   = (p - p0_s).Hat();
        float dist_s    = Dot(norm_s, diff_s);
        
        if (fabs(dist_s) < 0.001f)
        {
            break;
        }
        
        if (dist_s > 0.0f)
        {
            min_s = s;
        }
        else
        {
            max_s = s;
        }
    }
        
    s = MAX(s, 0.0f);
    s = MIN(s, 1.0f);
    
    Point p0_s = (*p0_base) + s * (*p0_tip - *p0_base);
    Point p1_s = (*p1_base) + s * (*p1_tip - *p1_base);
    Point p2_s = (*p2_base) + s * (*p2_tip - *p2_base);    
    
    // we have suitable value for [s]
    // now solve for [u, v]
    
    // form matrix M
    //
    //      a b c
    //      d e f
    //      g h i
    //
    // such that (x,y,z) = M(u,v,1)
    
    float a = p0_s.x - p2_s.x;
    float b = p1_s.x - p2_s.x;
    float c = p2_s.x;
    
    float d = p0_s.y - p2_s.y;
    float e = p1_s.y - p2_s.y;
    float f = p2_s.y;
    
    float g = p0_s.z - p2_s.z;
    float h = p1_s.z - p2_s.z;
    float i = p2_s.z;
    
    // form inverse matrix
    //
    //      aa bb cc
    //      dd ee ff
    //      gg hh ii
    
    float determ = -c*e*g + b*f*g + c*d*h - a*f*h - b*d*i + a*e*i;
    
    if (fabs(determ) < RI_EPSILON)
    {
        printf("uh oh determ = 0!!\n");
        exit(1);
    }
    
    // compute matrix inverse
    float inv_determ = 1.0f / determ;
    float aa = (-f*h + e*i) * inv_determ;
    float bb = (c*h - b*i) * inv_determ;
    float cc = (-c*e + b*f) * inv_determ;
    float dd = (f*g - d*i) * inv_determ;
    float ee = (-c*g + a*i) * inv_determ;
    float ff = (c*d - a*f) * inv_determ;
    
    // compute u and v
    // by solving (u,v,1) = Inverse(M) * (x,y,z)
    float u = (aa * p.x) + (bb * p.y) + (cc * p.z);
    float v = (dd * p.x) + (ee * p.y) + (ff * p.z);
    
    // clamp u and v
    u = MAX(u, 0.0f);
    u = MIN(u, 1.0f);
    
    v = MAX(v, 0.0f);
    v = MIN(v, 1.0f);
    
    return Point(u, s, v);
}
