
#include "shading.h"
#include "color.h"
#include "reflection.h"
#include "texture.h"
#include "light.h"
#include "trimesh.h"
#include "scene.h"


ShadeContext::ShadeContext(const HitInfo * hi, const Vector & w)
:wo(w.Hat())
{
	hitInfo = hi;

	Ng = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->NgObj);
	Ng.Normalize();
	if (Dot(Ng, wo) < 0.) {
		Ng = -Ng;
		entering = false;
	} else {
		entering = true;
	}

	const Normal *Ni = hitInfo->GetNormal(RI_N);
	if (Ni != NULL) {
		Ns = hitInfo->hitPrim->attributes->ObjectToWorld(*Ni).Hat();
		if (Dot(Ns, wo) < 0.)
			Ns = -Ns;
	} else {
		Ns = Ng;
	}

}

Float ShadeContext::InitializeFloat(RtToken token, Float def) const
{
	const Float *ret = hitInfo->GetFloat(token);
	if (ret)
		return *ret;
	if (token == RI_S)
		return hitInfo->u;
	else if (token == RI_T)
		return hitInfo->v;
	return def;
}

Spectrum ShadeContext::InitializeColor(RtToken token, const Spectrum & def) const
{
	const Spectrum *ret = hitInfo->GetColor(token);
	if (ret)
		return *ret;
	if (token == RI_CS)
		return hitInfo->hitPrim->attributes->Color;
	else if (token == RI_OS)
		return hitInfo->hitPrim->attributes->Opacity;
	return def;
}

Point ShadeContext::InitializePoint(RtToken token, const Point & def) const
{
	const Point *ret = hitInfo->GetPoint(token);
	if (ret)
		return *ret;
	else
		return def;
}

const char *ShadeContext::InitializeString(RtToken token, const char *def) const
{
	const char *ret = hitInfo->GetString(token);
	if (ret)
		return ret;
	else
		return def;
}

Vector ShadeContext::InitializeVector(RtToken token, const Vector & def) const
{
	const Vector *ret = hitInfo->GetVector(token);
	if (ret)
		return *ret;
	else
		return def;
}

Normal ShadeContext::InitializeNormal(RtToken token, const Normal & def) const
{
	const Normal *ret = hitInfo->GetNormal(token);
	if (ret)
		return *ret;
	else
		return def;
}

StringHashTable ShadeContext::textures;


TextureMap *ShadeContext::GetTexture(const char *filename)
{
	TextureMap *ret = (TextureMap *) textures.Search(filename);
	if (!ret) {
		ret = new TextureMap(filename);
		textures.Add(filename, ret);
	}
	return ret;
}

Material::~Material()
{
	delete surfaceFunction;
}

BRDF *MatteSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	Float surfaceKd = sc.InitializeFloat(RI_KD, Kd);
	return CreateLambertian(surfaceKd * Cs, sc.Ns);
}

BRDF *PlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	return PlasticShade(sc, Cs);
}

BRDF *PlasticSurface::PlasticShade(const ShadeContext & sc,
								   const Spectrum & Cs) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Float surfKs = sc.InitializeFloat(RI_KS, Ks);
	Float surfRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum surfSpecColor =
		sc.InitializeColor(RI_SPECULARCOLOR, SpecularColor);

	ScatteringMixture *mix = CreateScatteringMixture();
	mix->AddFunction(CreateLambertian(surfKd * Cs, sc.Ns));
	mix->AddFunction(CreateBlinnGlossy(surfKs * surfSpecColor,
									   Clamp(surfRoughness, 0., 1.), sc.Ns,
									   sc.wo));
	return mix;

}

PaintedPlasticSurface::PaintedPlasticSurface(SurfaceFunction *data)
		: PlasticSurface(data) {
	texName = surfaceFunction->InitializeString(RI_TEXTURENAME,
								NULL);
}

BRDF *PaintedPlasticSurface::Shade(const ShadeContext & sc) const
{
	Spectrum surfColor = sc.InitializeColor(RI_CS, Spectrum(1.));
	const char *surfTexName = sc.InitializeString(RI_TEXTURENAME, texName);
	TextureMap *tmap = sc.GetTexture(surfTexName);

	if (tmap) {
		Float s = sc.InitializeFloat(RI_S, 0.);
		Float t = sc.InitializeFloat(RI_T, 0.);
		surfColor *= tmap->Lookup(s, t);
	}
	return PlasticShade(sc, surfColor);
}

BRDF *CheckeredSurface::Shade(const ShadeContext & sc) const
{

	Float surfKd = sc.InitializeFloat(RI_KD, Kd);
	Spectrum surfCColor1 = sc.InitializeColor(LRT_CHECKCOLOR1,
											  CheckColor1);
	Spectrum surfCColor2 = sc.InitializeColor(LRT_CHECKCOLOR2,
											  CheckColor2);
	Float freq = sc.InitializeFloat(LRT_CHECKFREQUENCY,
									CheckFrequency);

	Point Pshade = sc.InitializePoint(RI_P, Point(0, 0, 0));
	Point Pcheck(freq * Pshade.x, freq * Pshade.y, freq * Pshade.z);
	if (Mod(Round(Pcheck.x) + Round(Pcheck.y) + Round(Pcheck.z), 2) == 0) {
		return CreateLambertian(surfKd * surfCColor1, sc.Ns);
	} else {
		return CreateLambertian(surfKd * surfCColor2, sc.Ns);
	}

}

BRDF *GlassSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeKt = sc.InitializeFloat(LRT_KT, Kt);
	Float shadeIndex = sc.InitializeFloat(LRT_INDEX, Index);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(0.));

	if (shadeKr == 0. && shadeKt == 0.)
		return NULL;

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(shadeKr * Cs,
												  sc.Ns, sc.wo));
	}
	if (shadeKt > 0.) {

		if (sc.entering) {
			mix->AddFunction(CreateSpecularTransmission(Kt * Cs,
														sc.Ns, sc.wo,
														1.0, shadeIndex));
		} else {
			mix->AddFunction(CreateSpecularTransmission(Kt * Cs,
														sc.Ns, sc.wo,
														shadeIndex, 1.0));
		}

	}
	return mix;
}

BRDF *ShinyMetalSurface::Shade(const ShadeContext & sc) const
{

	Float shadeKs = sc.InitializeFloat(RI_KS, Ks);
	Float shadeKr = sc.InitializeFloat(RI_KR, Kr);
	Float shadeRoughness = sc.InitializeFloat(RI_ROUGHNESS, Roughness);
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));

	ScatteringMixture *mix = CreateScatteringMixture();
	if (shadeKs > 0.) {
		mix->AddFunction(CreateBlinnGlossy(shadeKs * Cs,
										   shadeRoughness, sc.Ns, sc.wo));
	}
	if (shadeKr > 0.) {
		mix->AddFunction(CreateSpecularReflection(shadeKr * Cs,
												  sc.Ns, sc.wo));
	}
	return mix;
}

BRDF *ViewSTSurface::Shade(const ShadeContext & sc) const
{
	Float sp = sc.InitializeFloat(RI_S, 0.);
	Float tp = sc.InitializeFloat(RI_T, 0.);
	return CreateLambertian(Spectrum(sp, tp, 0.), sc.Ns);
}

Material *MaterialCreate(const char *name, SurfaceFunction * bindings)
{
	if (strcmp(name, RI_MATTE) == 0)
		return new MatteSurface(bindings);
	else if (strcmp(name, RI_PLASTIC) == 0)
		return new PlasticSurface(bindings);
	else if (strcmp(name, RI_PAINTEDPLASTIC) == 0)
		return new PaintedPlasticSurface(bindings);
	else if (strcmp(name, RI_SHINYMETAL) == 0)
		return new ShinyMetalSurface(bindings);
	else if (strcmp(name, LRT_VIEWST) == 0)
		return new ViewSTSurface(bindings);
	else if (strcmp(name, LRT_CHECKERED) == 0)
		return new CheckeredSurface(bindings);
	else if (strcmp(name, LRT_GLASS) == 0)
		return new GlassSurface(bindings);
    else if (strcmp(name, LRT_FUR) == 0) // ec -- fur surface type
        return new FurSurface(bindings);
	else {
		Warning("Unknown surface shader: %s," "returning \"matte\"", name);
		return new MatteSurface(bindings);
	}
}

SurfaceFunction::~SurfaceFunction()
{

	for (u_int i = 0; i < vertexPoints.size(); ++i)
		delete[]vertexPoints[i].second;

	for (u_int i = 0; i < uniformStrings.size(); ++i)
		delete uniformStrings[i].second;

	for (u_int i = 0; i < vertexFloats.size(); ++i)
		delete[]vertexFloats[i].second;
	for (u_int i = 0; i < vertexHPoints.size(); ++i)
		delete[]vertexHPoints[i].second;
	for (u_int i = 0; i < vertexVectors.size(); ++i)
		delete[]vertexVectors[i].second;
	for (u_int i = 0; i < vertexNormals.size(); ++i)
		delete[]vertexNormals[i].second;
	for (u_int i = 0; i < vertexColors.size(); ++i)
		delete[]vertexColors[i].second;

}

SurfaceFunction::SurfaceFunction(const SurfaceFunction & sf)
{
	nVertex = sf.nVertex;

	vector<pair<RtToken, Point> >::const_iterator uniformPointIter;
	for (uniformPointIter = sf.uniformPoints.begin();
		 uniformPointIter != sf.uniformPoints.end(); uniformPointIter++) {
		uniformPoints.push_back(make_pair((*uniformPointIter).first,
										  (*uniformPointIter).second));
	}

	vector<pair<RtToken, Point *> >::const_iterator vertexPointIter;
	for (vertexPointIter = sf.vertexPoints.begin();
		 vertexPointIter != sf.vertexPoints.end(); vertexPointIter++) {
		Point *values = new Point[nVertex];
		memcpy(values,
			   (*vertexPointIter).second, nVertex * sizeof(*values));
		vertexPoints.
			push_back(make_pair((*vertexPointIter).first, values));
	}

	vector<pair<RtToken, Float> >::const_iterator uniformFloatIter;
	for (uniformFloatIter = sf.uniformFloats.begin();
		 uniformFloatIter != sf.uniformFloats.end(); uniformFloatIter++) {
		uniformFloats.push_back(make_pair((*uniformFloatIter).first,
										  (*uniformFloatIter).second));
	}
	vector<pair<RtToken, Float> >::const_iterator uniformHPointIter;
	for (uniformHPointIter = sf.uniformHPoints.begin();
		 uniformHPointIter != sf.uniformHPoints.end(); uniformHPointIter++) {
		uniformHPoints.push_back(make_pair((*uniformHPointIter).first,
										   (*uniformHPointIter).second));
	}
	vector<pair<RtToken, Vector> >::const_iterator uniformVectorIter;
	for (uniformVectorIter = sf.uniformVectors.begin();
		 uniformVectorIter != sf.uniformVectors.end(); uniformVectorIter++) {
		uniformVectors.push_back(make_pair((*uniformVectorIter).first,
										   (*uniformVectorIter).second));
	}
	vector<pair<RtToken, Normal> >::const_iterator uniformNormalIter;
	for (uniformNormalIter = sf.uniformNormals.begin();
		 uniformNormalIter != sf.uniformNormals.end(); uniformNormalIter++) {
		uniformNormals.push_back(make_pair((*uniformNormalIter).first,
										   (*uniformNormalIter).second));
	}
	vector<pair<RtToken, Spectrum> >::const_iterator uniformColorIter;
	for (uniformColorIter = sf.uniformColors.begin();
		 uniformColorIter != sf.uniformColors.end(); uniformColorIter++) {
		uniformColors.push_back(make_pair((*uniformColorIter).first,
										  (*uniformColorIter).second));
	}
	vector<pair<RtToken, char *> >::const_iterator uniformStringIter;
	for (uniformStringIter = sf.uniformStrings.begin();
		 uniformStringIter != sf.uniformStrings.end(); uniformStringIter++) {
		uniformStrings.push_back(make_pair((*uniformStringIter).first,
										   Strdup((*uniformStringIter).
												  second)));
	}

	vector<pair<RtToken, Float *> >::const_iterator vertexFloatIter;
	for (vertexFloatIter = sf.vertexFloats.begin();
		 vertexFloatIter != sf.vertexFloats.end(); vertexFloatIter++) {
		Float *values = new Float[nVertex];
		memcpy(values, (*vertexFloatIter).second,
			   nVertex * sizeof(*values));
		vertexFloats.
			push_back(make_pair((*vertexFloatIter).first, values));
	}
	vector<pair<RtToken, Float *> >::const_iterator vertexHPointIter;
	for (vertexHPointIter = sf.vertexHPoints.begin();
		 vertexHPointIter != sf.vertexHPoints.end(); vertexHPointIter++) {
		Float *values = new Float[4 * nVertex];
		memcpy(values, (*vertexHPointIter).second,
			   4 * nVertex * sizeof(*values));
		vertexHPoints.
			push_back(make_pair((*vertexHPointIter).first, values));
	}
	vector<pair<RtToken, Vector *> >::const_iterator vertexVectorIter;
	for (vertexVectorIter = sf.vertexVectors.begin();
		 vertexVectorIter != sf.vertexVectors.end(); vertexVectorIter++) {
		Vector *values = new Vector[nVertex];
		memcpy(values, (*vertexVectorIter).second,
			   nVertex * sizeof(*values));
		vertexVectors.
			push_back(make_pair((*vertexVectorIter).first, values));
	}
	vector<pair<RtToken, Normal *> >::const_iterator vertexNormalIter;
	for (vertexNormalIter = sf.vertexNormals.begin();
		 vertexNormalIter != sf.vertexNormals.end(); vertexNormalIter++) {
		Normal *values = new Normal[nVertex];
		memcpy(values, (*vertexNormalIter).second,
			   nVertex * sizeof(*values));
		vertexNormals.
			push_back(make_pair((*vertexNormalIter).first, values));
	}
	vector<pair<RtToken, Spectrum *> >::const_iterator vertexColorIter;
	for (vertexColorIter = sf.vertexColors.begin();
		 vertexColorIter != sf.vertexColors.end(); vertexColorIter++) {
		Spectrum *values = new Spectrum[nVertex];
		memcpy(values, (*vertexColorIter).second,
			   nVertex * sizeof(*values));
		vertexColors.
			push_back(make_pair((*vertexColorIter).first, values));
	}

}

void SurfaceFunction::AddUniformPoint(RtToken name, const Float * data)
{
	uniformPoints.
		push_back(make_pair(name, Point(data[0], data[1], data[2])));
}

void SurfaceFunction::AddVertexPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Point *values = new Point[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexPoints.push_back(make_pair(name, values));
}

void SurfaceFunction::AddUniformFloat(RtToken name, const Float * data)
{
	uniformFloats.push_back(make_pair(name, *data));
}
void SurfaceFunction::AddUniformHPoint(RtToken name, const Float * data)
{
	uniformHPoints.push_back(make_pair(name, data[0]));
	uniformHPoints.push_back(make_pair(name, data[1]));
	uniformHPoints.push_back(make_pair(name, data[2]));
	uniformHPoints.push_back(make_pair(name, data[3]));
}
void SurfaceFunction::AddUniformVector(RtToken name, const Float * data)
{
	uniformVectors.
		push_back(make_pair(name, Vector(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformNormal(RtToken name, const Float * data)
{
	uniformNormals.
		push_back(make_pair(name, Normal(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformColor(RtToken name, const Float * data)
{
	uniformColors.
		push_back(make_pair(name, Spectrum(data[0], data[1], data[2])));
}
void SurfaceFunction::AddUniformString(RtToken name, const char *data)
{
	uniformStrings.push_back(make_pair(name, Strdup(data)));
}

void SurfaceFunction::AddVertexFloat(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[nVertex];
	memcpy(values, data, nVertex * sizeof(Float));
	vertexFloats.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexHPoint(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Float *values = new Float[4 * nVertex];
	memcpy(values, data, 4 * nVertex * sizeof(Float));
	vertexHPoints.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexVector(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Vector *values = new Vector[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexVectors.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexNormal(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Normal *values = new Normal[nVertex];
	for (u_int i = 0; i < nVertex; ++i) {
		values[i].x = data[3 * i];
		values[i].y = data[3 * i + 1];
		values[i].z = data[3 * i + 2];
	}
	vertexNormals.push_back(make_pair(name, values));
}
void SurfaceFunction::AddVertexColor(RtToken name, const Float * data)
{
	Assert(nVertex > 0);
	Spectrum *values = new Spectrum[nVertex];
	for (u_int i = 0; i < nVertex; ++i)
		values[i] =
			Spectrum(data[3 * i], data[3 * i + 1], data[3 * i + 2]);
	vertexColors.push_back(make_pair(name, values));
}

const Point *SurfaceFunction::GetUniformPoint(RtToken name) const
{
	for (u_int i = 0; i < uniformPoints.size(); ++i) {
		if (name == uniformPoints[i].first)
			return &uniformPoints[i].second;
	}
	return NULL;
}

const Point *SurfaceFunction::GetVertexPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		if (name == vertexPoints[i].first)
			return vertexPoints[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetUniformFloat(RtToken name) const
{
	for (u_int i = 0; i < uniformFloats.size(); ++i) {
		if (name == uniformFloats[i].first)
			return &uniformFloats[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetUniformVector(RtToken name) const
{
	for (u_int i = 0; i < uniformVectors.size(); ++i) {
		if (name == uniformVectors[i].first)
			return &uniformVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetUniformNormal(RtToken name) const
{
	for (u_int i = 0; i < uniformNormals.size(); ++i) {
		if (name == uniformNormals[i].first)
			return &uniformNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetUniformColor(RtToken name) const
{
	for (u_int i = 0; i < uniformColors.size(); ++i) {
		if (name == uniformColors[i].first)
			return &uniformColors[i].second;
	}
	return NULL;
}
const char *SurfaceFunction::GetUniformString(RtToken name) const
{
	for (u_int i = 0; i < uniformStrings.size(); ++i) {
		if (name == uniformStrings[i].first)
			return uniformStrings[i].second;
	}
	return NULL;
}

const Float *SurfaceFunction::GetVertexFloat(RtToken name) const
{
	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		if (name == vertexFloats[i].first)
			return vertexFloats[i].second;
	}
	return NULL;
}
const Float *SurfaceFunction::GetVertexHPoint(RtToken name) const
{
	for (u_int i = 0; i < vertexHPoints.size(); ++i) {
		if (name == vertexHPoints[i].first)
			return vertexHPoints[i].second;
	}
	return NULL;
}
const Vector *SurfaceFunction::GetVertexVector(RtToken name) const
{
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		if (name == vertexVectors[i].first)
			return vertexVectors[i].second;
	}
	return NULL;
}
const Normal *SurfaceFunction::GetVertexNormal(RtToken name) const
{
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		if (name == vertexNormals[i].first)
			return vertexNormals[i].second;
	}
	return NULL;
}
const Spectrum *SurfaceFunction::GetVertexColor(RtToken name) const
{
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		if (name == vertexColors[i].first)
			return vertexColors[i].second;
	}
	return NULL;
}

Point SurfaceFunction::InitializePoint(RtToken name, const Point & def) const
{
	const Point *ptr = GetUniformPoint(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Float SurfaceFunction::InitializeFloat(RtToken name, Float def) const
{
	const Float *ptr = GetUniformFloat(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Spectrum SurfaceFunction::InitializeColor(RtToken name,
										  const Spectrum & def) const
{
	const Spectrum *ptr = GetUniformColor(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Vector SurfaceFunction::InitializeVector(RtToken name, const Vector & def) const
{
	const Vector *ptr = GetUniformVector(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}

Normal SurfaceFunction::InitializeNormal(RtToken name, const Normal & def) const
{
	const Normal *ptr = GetUniformNormal(name);
	if (ptr != NULL)
		return *ptr;
	else
		return def;
}
const char *SurfaceFunction::InitializeString(RtToken name,
											  const char *def) const
{
	const char *ptr = GetUniformString(name);
	if (ptr != NULL)
		return ptr;
	else
		return def;
}

InterpolatedPrimData::InterpolatedPrimData(const SurfaceFunction * pd)
{
	primitiveData = pd;

	if (primitiveData->vertexPoints.size() > 0) {
		int nvp = primitiveData->vertexPoints.size();
		interpVertexPoints = new Point[nvp];
	} else
		interpVertexPoints = NULL;

	if (primitiveData->vertexFloats.size() > 0) {
		interpVertexFloats = new Float[primitiveData->vertexFloats.size()];
	} else {
		interpVertexFloats = NULL;
	}
	if (primitiveData->vertexVectors.size() > 0) {
		int nv = primitiveData->vertexVectors.size();
		interpVertexVectors = new Vector[nv];
	} else {
		interpVertexVectors = NULL;
	}
	if (primitiveData->vertexNormals.size() > 0) {
		int vn = primitiveData->vertexNormals.size();
		interpVertexNormals = new Normal[vn];
	} else {
		interpVertexNormals = NULL;
	}
	if (primitiveData->vertexColors.size() > 0) {
		int nc = primitiveData->vertexColors.size();
		interpVertexColors = new Spectrum[nc];
	} else {
		interpVertexColors = NULL;
	}

}

InterpolatedPrimData::~InterpolatedPrimData()
{

	delete[]interpVertexPoints;

	delete[]interpVertexFloats;
	delete[]interpVertexVectors;
	delete[]interpVertexNormals;
	delete[]interpVertexColors;

}

InterpolatedPrimData *SurfaceFunction::Interpolate(int n,
												   const int *offsets,
												   const Float * weights) const
{
	InterpolatedPrimData *ret = new InterpolatedPrimData(this);

	for (u_int i = 0; i < vertexPoints.size(); ++i) {
		Point value(0, 0, 0);
		Point *pts = vertexPoints[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value.x += weights[j] * pts[o].x;
			value.y += weights[j] * pts[o].y;
			value.z += weights[j] * pts[o].z;
		}
		ret->interpVertexPoints[i] = value;
	}

	for (u_int i = 0; i < vertexFloats.size(); ++i) {
		Float value = 0.;
		Float *f = vertexFloats[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * f[o];
		}
		ret->interpVertexFloats[i] = value;
	}
	for (u_int i = 0; i < vertexVectors.size(); ++i) {
		Vector value(0, 0, 0);
		Vector *vec = vertexVectors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * vec[o];
		}
		ret->interpVertexVectors[i] = value;
	}
	for (u_int i = 0; i < vertexNormals.size(); ++i) {
		Normal value(0, 0, 0);
		Normal *norm = vertexNormals[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * norm[o];
		}
		ret->interpVertexNormals[i] = value;
	}
	for (u_int i = 0; i < vertexColors.size(); ++i) {
		Spectrum value(0.);
		Spectrum *cp = vertexColors[i].second;
		for (int j = 0; j < n; ++j) {
			int o = offsets[j];
			value += weights[j] * cp[o];
		}
		ret->interpVertexColors[i] = value;
	}

	return ret;
}

const Point *InterpolatedPrimData::GetPoint(RtToken token) const
{
	const Point *ret = primitiveData->GetUniformPoint(token);
	if (ret)
		return ret;

	const vector<pair<RtToken, Point *> > &vpp =
	primitiveData->vertexPoints;
	for (u_int i = 0; i < vpp.size(); ++i) {
		if (token == vpp[i].first)
			return &interpVertexPoints[i];
	}
	return NULL;

}

const Float *InterpolatedPrimData::GetFloat(RtToken token) const
{
	const vector<pair<RtToken, Float *> > &vfp = primitiveData->vertexFloats;
	for (u_int i = 0; i < vfp.size(); ++i) {
		if (token == vfp[i].first)
			return &interpVertexFloats[i];
	}
	return NULL;
}
const Vector *InterpolatedPrimData::GetVector(RtToken token) const
{
	const vector<pair<RtToken, Vector *> > &vvp = primitiveData->vertexVectors;
	for (u_int i = 0; i < vvp.size(); ++i) {
		if (token == vvp[i].first)
			return &interpVertexVectors[i];
	}
	return NULL;
}
const Normal *InterpolatedPrimData::GetNormal(RtToken token) const
{
	const vector<pair<RtToken, Normal *> > &vnp = primitiveData->vertexNormals;
	for (u_int i = 0; i < vnp.size(); ++i) {
		if (token == vnp[i].first)
			return &interpVertexNormals[i];
	}
	return NULL;
}
const Spectrum *InterpolatedPrimData::GetColor(RtToken token) const
{
	const vector<pair<RtToken, Spectrum *> > &vcp = primitiveData->vertexColors;
	for (u_int i = 0; i < vcp.size(); ++i) {
		if (token == vcp[i].first)
			return &interpVertexColors[i];
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////////
//  FurSurface code
//////////////////////////////////////////////////////////////////////

#define MIN_TRANS 0.01

FurSurface::FurSurface(SurfaceFunction *data)
    : Material(data)
{
    // get list of shader names (space-delimited)
	const char *shader_list_src = surfaceFunction->InitializeString(LRT_TEXEL_SHADER, NULL);
	char *shader_list;
	
	// init
	m_numshaders = 0;
	m_shaders = NULL;
	m_seglen  = surfaceFunction->InitializeFloat(LRT_FUR_INCREMENT, 0.1f);
	
	// must have at least 1 shader
	if (!shader_list_src)
	{
	    Error("<*> attempted to create FurSurface without texel shaders! aborting");
	    exit(1);
	}
	
	// get ready to parse shader list
	shader_list = new char[strlen(shader_list_src) + 1];
	strcpy(shader_list, shader_list_src);
	
	// first count number of shaders
	char *curr_shader_name = shader_list;
	for (;;)
	{
    	char *space_loc = strchr(curr_shader_name, 0x20);
   	
    	m_numshaders++;
        curr_shader_name = space_loc + 1;
    	
    	if (!space_loc)
    	{
    	    break;
    	} 
	}
	
	printf("<*> creating FurSurface: num texel shaders = %d\n", m_numshaders);
	m_shaders = new (Texel *)[m_numshaders];
	
	// then get their names
	curr_shader_name = shader_list;
	for (int i = 0; i < m_numshaders; i++)
	{
    	char *space_loc = strchr(curr_shader_name, 0x20);
    	
    	if (space_loc)
    	{
    	    *space_loc = '\0';
    	}
    		    
    	printf("<*> building shader <%s>\n", curr_shader_name);
    	
    	m_shaders[i] = new Texel(curr_shader_name);
    	
    	curr_shader_name = space_loc + 1;
	}
	
	delete [] shader_list;
}

BRDF *
FurSurface::Shade(const ShadeContext &sc) const
{
    // matte for now?
    
	Spectrum Cs = sc.InitializeColor(RI_CS, Spectrum(1.0));
	// Float surfaceKd = sc.InitializeFloat(RI_KD, Kd);
	return CreateLambertian(Cs, sc.Ns);    
}

float
FurSurface::Transparency(const Ray &ray, FurryTriangle *tri, int shader_index, float in_tmin, float in_tmax)
{
    float t = in_tmin;
    float t_max = in_tmax;
    
    if (t_max < 0.0f)
    {
        t_max = tri->IntersectOutside(ray);
    }
    
    float density_sum = 0.0f;
    while (t < t_max)
    {
        float t_next    = MIN(t + m_seglen, t_max);
        float t_random  = RandomFloat(t, t_next);
        
        Point p_object  = ray(t_random);
        Point p_texel   = tri->InverseMapTexel(p_object);
        
        density_sum += m_shaders[shader_index]->GetDensity(p_texel) * (t_next - t);
        t += m_seglen;
    }
    
    return exp(-m_shaders[shader_index]->GetOpticalDensity() * density_sum);
    
}

Spectrum
FurSurface::ApplyShaders(const Ray &ray, HitInfo *hit, float *ray_trans)
{   
    float trans_to_bg = 1.0f;
    
    Spectrum L(0.0f);
    
    float t_min = hit->u;
    float t_max = hit->v;
    
    // check hit bottom flag
    if (t_max < 0.0f)
    {
        t_max += HIT_BOTTOM_FLAG;
    }
    
    FurryTriangle *tri = (FurryTriangle *) (hit->hitPrim);
    
    for (int i = 0; i < m_numshaders; i++)
    {
        // teddy-bear paper algorithm is here
        trans_to_bg *= Transparency(ray, tri, i, t_min, t_max);    
        
        float t = t_min;
        
        float trans = 1.0f;
        
        while (t < t_max && trans > MIN_TRANS)
        {
            float t_next = MIN(t + m_seglen, t_max);
            float t_sample = RandomFloat(t, t_next);
            
            Point point_object = ray(t_sample);
            Point point_texel  = tri->InverseMapTexel(point_object);
            
            // use first shader?
            float density = m_shaders[i]->GetDensity(point_texel);
            if (density > 0.0f)
            {
                Vector tangent = m_shaders[i]->GetTangent(point_texel);
                Spectrum dL(0.0f);
                
                Point point_world = hit->hitPrim->attributes->ObjectToWorld(point_object);
                
                // for each light ...
                const list <Light *> &lights = hit->hitPrim->attributes->Lights;
        		list <Light *>::const_iterator iter = lights.begin();
        		while (iter != lights.end())
        		{
        			const Light *curr_light = *iter;
        			
        			Point light_world;
        			
        			Spectrum dE = curr_light->dE(point_world, &light_world);
        			
        			Point light_object = hit->hitPrim->attributes->WorldToObject(light_world);
        			
        			// construct light ray
        			Vector light_dir = (light_object - point_object).Hat();
        			Ray light_ray(point_object, light_dir);
        			
        			// compute transparency along light ray
        			float light_trans = Transparency(light_ray, tri, i);
        			
        			// compute light diffuse contribution
        			float cosine = Dot(tangent, light_dir);
        			float diffuse = (float) sqrt(1.0f - cosine * cosine);
        			
        			dL +=  diffuse * light_trans * dE * m_shaders[i]->GetColor(true);
        			
        			iter++;
        		}
        		
        		L += m_shaders[i]->GetOpticalDensity() * trans * dL * density * (t_next - t);
        		trans *= exp(-m_shaders[i]->GetOpticalDensity() * density * (t_next - t));
            }
                    
            t += m_seglen;
        }
    }
    
    // return total transparency
    *ray_trans = trans_to_bg;
    
    return L;
}

Spectrum
FurSurface::ShadeBase(const Ray &ray, HitInfo *hitInfo)
{
    Spectrum L(0.0f);
    
    
    FurryTriangle *tri = (FurryTriangle *) (hitInfo->hitPrim);
    
    // compute normal at base
	Vector E2 = tri->p2_base[0] - tri->p0_base[0];
	Vector E1 = tri->p1_base[0] - tri->p0_base[0];
	hitInfo->RecordHit(hitInfo->Pobj, Normal(Cross(E2, E1)), hitInfo->u, hitInfo->v, tri);
	
    ShadeContext shadeContext(hitInfo, -ray.D);
    
	BRDF *brdf = hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
	
	// compute intersection point in world space
	Point Pw = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
	
	// for each light ...
	const list < Light * >&lights = hitInfo->hitPrim->attributes->Lights;
	list < Light * >::const_iterator iter = lights.begin();
	while (iter != lights.end())
	{
		const Light *lt = *iter;
		Point Plight;
		
		// get contribution from light at this point
		Spectrum dE = lt->dE(Pw, &Plight);
		
		if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight))
		{
		    // standard lighting equation
			Vector wi = (Plight - Pw).Hat();
			const Normal & Nw = shadeContext.Ns;
			
			L += dE * brdf->fr(wi) * Dot(wi, Nw);
		}

		++iter;
	}
	
	delete brdf;
	
	return L;
}

//////////////////////////////////////////////////////////////////////
//  Texel code
//////////////////////////////////////////////////////////////////////

Texel::Texel(const char *in_filename)
{    
    m_xsize = 1;
    m_ysize = 1;
    m_zsize = 1;
    
    // not used -- grrr
    m_seglen = 0.1f;
    
    // read in from file
    FILE *in_fp = fopen(in_filename, "rb");
    
    if (!in_fp)
    {
        printf("<*> Error in Texel creation: shader <%s> not found!\n", in_filename);
        exit(1);
    }
    
    // read dim
    fread(&m_xsize, sizeof(int), 1, in_fp);
    fread(&m_ysize, sizeof(int), 1, in_fp);
    fread(&m_zsize, sizeof(int), 1, in_fp);
    printf("<*> - size: %d x %d x %d\n", m_xsize, m_ysize, m_zsize);
    
    // read color
    fread(m_color, sizeof(float), 3, in_fp);
    printf("<*> - color: (%.2f, %.2f, %.2f)\n", m_color[0], m_color[1], m_color[2]);
    
    // read color noise
    fread(m_colornoise, sizeof(float), 3, in_fp);
    printf("<*> - color noise: (%.3f, %.3f, %.3f)\n",
        m_colornoise[0], m_colornoise[1], m_colornoise[2]);
    
    // read optical density
    fread(&m_opt_density, sizeof(float), 1, in_fp);
    printf("<*> - optical density: %.3f\n", m_opt_density);
    
    // allocate texel storage
    m_cells = new (TexelCell **)[m_xsize];    
    for (int i = 0; i < m_xsize; i++)
    {
        m_cells[i] = new (TexelCell *)[m_ysize];        
        for (int j = 0; j < m_ysize; j++)
        {
            m_cells[i][j] = new TexelCell[m_zsize];
            
            for (int k = 0; k < m_zsize; k++)
            {
                // read in texel content
                float values[4];    // values[0]-values[2] = tangent, values[3] = density
                fread(values, sizeof(float), 4, in_fp);
                
                m_cells[i][j][k].tangent = Vector(values[0], values[1], values[2]);
                m_cells[i][j][k].density = values[3];
            }
        }
    }
    
    fclose(in_fp);
}

Texel::~Texel()
{
    for (int i = 0; i < m_xsize; i++)
    {
        for (int j = 0; j < m_ysize; j++)
        {
            delete m_cells[i][j];
        }
        delete m_cells[i];
    }
    delete m_cells;
}

Spectrum
Texel::GetColor(bool noise)
{
    if (!noise)
    {
        return Spectrum(m_color[0], m_color[1], m_color[2]);
    }
    else
    {
        float r = m_color[0] + RandomFloat(0.0f, m_colornoise[0]);
        float g = m_color[1] + RandomFloat(0.0f, m_colornoise[1]);
        float b = m_color[2] + RandomFloat(0.0f, m_colornoise[2]);

        r = MIN(r, 1.0f);
        r = MAX(r, 0.0f);
        
        g = MIN(g, 1.0f);
        g = MAX(g, 0.0f);
        
        b = MIN(b, 1.0f);
        b = MAX(b, 0.0f);

        return Spectrum(r, g, b);
    }
}

Spectrum
Texel::Shade(Ray ray, HitInfo *hit, float *alpha)
{
    // texel rendering algorithm
    // see Kajiya SIGGRAPH 89 p.274
    
    ray.D.Normalize();
    
    Spectrum L(0.0f);
    
    float t_min, t_max;
    
    // prob. don't need this later because we already
    // did scene->intersect test -- sanity check for now
    if (!intersects(ray, &t_min, &t_max))
    {
        *alpha = 0.0f;
        return Spectrum(0.0f);
    }
    
    
    
    *alpha = 1.0f - transparency(ray);
 
    float t = t_min;
    float trans = 1.0f;
        
    while (t < t_max && trans > MIN_TRANS)
    {
        float t_next = MIN(t + m_seglen, t_max);
        float t_sample = RandomRange(t, t_next);
        
        Point point_texel = ray(t_sample);
        Point point_world = hit->hitPrim->attributes->ObjectToWorld(point_texel);
        
        float density = GetDensity(point_texel);
        if (density > 0.0f)
        {
            Vector tangent = GetTangent(point_texel);
            Spectrum dL(0.0f);
            
            // for each light ...
            const list <Light *> &lights = hit->hitPrim->attributes->Lights;
    		list <Light *>::const_iterator iter = lights.begin();
    		while (iter != lights.end())
    		{
    			const Light *curr_light = *iter;
    			
    			Point light_world;
    			
    			Spectrum dE = curr_light->dE(point_world, &light_world);
    			
    			Point light_texel = hit->hitPrim->attributes->WorldToObject(light_world);
    			
    			Vector light_dir = (light_texel - point_texel).Hat();
    			
    			Ray light_ray(point_texel, light_dir);
    			
    			float light_trans = transparency(light_ray);
    			
    			float cosine = Dot(tangent, light_dir);
    			float diffuse = (float) sqrt(1.0f - cosine * cosine);
    			
    			// eventually add texel color here???
    			dL +=  diffuse * light_trans * dE;
    			
    			iter++;
    		}
    		
    		L += m_opt_density * trans * dL * density * (t_next - t);
    		trans *= exp(-m_opt_density * density * (t_next - t));
        }
                
        t += m_seglen;
    }
    
    return L;
}

float
Texel::transparency(const Ray &ray)
{
    // see Kajiya SIGGRAPH 89 p.273
    
    // find entry and exit t values
    float t_min, t_max;
    if (!intersects(ray, &t_min, &t_max))
    {
        return 1.0f;
    }
    
    // sample attenuation integral along ray
    float t = t_min;
    float density_sum = 0.0f;
    while (t < t_max)
    {
        float t_next    = MIN(t + m_seglen, t_max);
        float t_random  = RandomRange(t, t_next);
        
        Point p = ray(t_random);
        
        density_sum += GetDensity(p) * (t_next - t);
        t += m_seglen;
    }
    
    return exp(-m_opt_density * density_sum);
}

//
//  intersects
//
//  If ray origin is outside box, then if the ray intersects
//  the box, returns true, and [t_min] and [t_max] contain the
//  entry and exit values of t. Otherwise returns false.
//
//  If ray origin is inside box, then returns true, [t_min]
//  is 0, and [t_max] is the exit value of t.
//

bool
Texel::intersects(const Ray &ray, float *t_min, float *t_max)
{    
    float min_y = 0.0f;
	float max_y = 1.0f;
	
	float min_z = 0.0f;
	float max_z = 1.0f;
	
	float min_x = 0.0f;
	float max_x = 1.0f;
	
	float tx = -1.0f;
	float ty = -1.0f;
	float tz = -1.0f;
	float far_tx = RI_INFINITY;
	float far_ty = RI_INFINITY;
	float far_tz = RI_INFINITY;
	
    // check x
	if (ray.D.x > RI_EPSILON)
	{
	    tx = (min_x - ray.O.x) / ray.D.x;
	    far_tx = (max_x - ray.O.x) / ray.D.x;
	}
	else if (ray.D.x < -RI_EPSILON)
	{
	    tx = (max_x - ray.O.x) / ray.D.x;
	    far_tx = (min_x - ray.O.x) / ray.D.x;
	}
    
    // check y
	if (ray.D.y > RI_EPSILON)
	{
	    ty = (min_y - ray.O.y) / ray.D.y;
	    far_ty = (max_y - ray.O.y) / ray.D.y;
	}
	else if (ray.D.y < -RI_EPSILON)
	{
	    ty = (max_y - ray.O.y) / ray.D.y;
	    far_ty = (min_y - ray.O.y) / ray.D.y;
	}
	
    // check z
	if (ray.D.z > RI_EPSILON)
	{
	    tz = (min_z - ray.O.z) / ray.D.z;
	    far_tz = (max_z - ray.O.z) / ray.D.z;
	}
	else if (ray.D.z < -RI_EPSILON)
	{
	    tz = (max_z - ray.O.z) / ray.D.z;
	    far_tz = (min_z - ray.O.z) / ray.D.z;
	}	
	
	// check ray origin point
	if (ray.O.x < min_x || ray.O.x > max_x
	    || ray.O.y < min_y || ray.O.y > max_y
	    || ray.O.z < min_z || ray.O.z > max_z)
    {
        // ray origin is outside box
        // find entry point t_min
    		
    	if (tx > ty && tx > tz)
    	{
    	    Point p = ray(tx);
    	    
    	    if (tx < 0.0f || p.y < min_y || p.y > max_y || p.z < min_z || p.z > max_z)
    	    {
    	        return false;
    	    }
    	    
    	    *t_min = tx;
    	}
    	else if (ty > tz)
    	{
    	    Point p = ray(ty);
    	    
    	    if (ty < 0.0f || p.x < min_x || p.x > max_x || p.z < min_z || p.z > max_z)
    	    {
    	        return false;
    	    }
    	    
    	    *t_min = ty;
    	}
    	else
    	{
    	    Point p = ray(tz);
    	    
    	    if (tz < 0.0f || p.y < min_y || p.y > max_y || p.x < min_x || p.x > max_x)
    	    {
    	        return false;
    	    }
    	    
    	    *t_min = tz;
    	}
    }
    else
    {
        // ray origin is inside box
        
        *t_min = 0.0f;
    }
    
	//
	//  find exit point
	//
	
	if (far_tx < far_ty && far_tx < far_tz)
	{
	    *t_max = far_tx;
	}
	else if (far_ty < far_tz)
	{
	    *t_max = far_ty;
	}
	else
	{
	    *t_max = far_tz;
	}
	
	return true;
}

void
Texel::SetSegmentLength(float seg_length)
{
    m_seglen = seg_length;
}

void
Texel::SetDensity(int i, int j, int k, float density)
{
    i = ClampInt(i, 0, m_xsize - 1);
    j = ClampInt(j, 0, m_ysize - 1);
    k = ClampInt(k, 0, m_zsize - 1);
    
    m_cells[i][j][k].density = density;
}

float
Texel::GetDensity(Point point)
{    
    float x         = point.x * (m_xsize - 0.01f);
    float y         = point.y * (m_ysize - 0.01f);
    float z         = point.z * (m_zsize - 0.01f);
    int int_x       = (int) x;
    int int_y       = (int) y;
    int int_z       = (int) z;
    float frac_x    = x - int_x;
    float frac_y    = y - int_y;
    float frac_z    = z - int_z;
    
    float d;
    
    int next_x      = MIN(int_x + 1, m_xsize - 1);
    int next_y      = MIN(int_y + 1, m_ysize - 1);
    int next_z      = MIN(int_z + 1, m_zsize - 1);
    
    float d000 = m_cells[int_x][int_y][int_z].density;
    float d100 = m_cells[next_x][int_y][int_z].density;
    
    float d001 = m_cells[int_x][int_y][next_z].density;
    float d101 = m_cells[next_x][int_y][next_z].density;
    
    float d010 = m_cells[int_x][next_y][int_z].density;
    float d110 = m_cells[next_x][next_y][int_z].density;
    
    float d011 = m_cells[int_x][next_y][next_z].density;
    float d111 = m_cells[next_x][next_y][next_z].density;
    
    // lerp 4 times in x
    d = d000;
    float d00 = d + frac_x * (d100 - d);
    
    d = d001;
    float d01 = d + frac_x * (d101 - d);
    
    d = d010;
    float d10 = d + frac_x * (d110 - d);
    
    d = d011;
    float d11 = d + frac_x * (d111 - d);      
    
    // lerp 2 times in y
    float d0 = d00 + frac_y * (d10 - d00);
    float d1 = d01 + frac_y * (d11 - d01);
    
    // final lerp in z
    return (d0 + frac_z * (d1 - d0));
}

float
Texel::GetDensityDiscrete(int i, int j, int k)
{    
    return m_cells[i][j][k].density;
}

void
Texel::SetTangent(int i, int j, int k, Vector tangent)
{
    i = ClampInt(i, 0, m_xsize - 1);
    j = ClampInt(j, 0, m_ysize - 1);
    k = ClampInt(k, 0, m_zsize - 1);
    
    m_cells[i][j][k].tangent = tangent;
}

Vector
Texel::GetTangent(Point point)
{
    float x         = point.x * (m_xsize - 0.01f);
    float y         = point.y * (m_ysize - 0.01f);
    float z         = point.z * (m_zsize - 0.01f);
    int int_x       = (int) x;
    int int_y       = (int) y;
    int int_z       = (int) z;
    float frac_x    = x - int_x;
    float frac_y    = y - int_y;
    float frac_z    = z - int_z;
    
    Vector t;
    
    int next_x      = MIN(int_x + 1, m_xsize - 1);
    int next_y      = MIN(int_y + 1, m_ysize - 1);
    int next_z      = MIN(int_z + 1, m_zsize - 1);
    
    // lerp 4 times in x
    t = GetTangentDiscrete(int_x, int_y, int_z);
    Vector t00 = t + frac_x * (GetTangentDiscrete(next_x, int_y, int_z) - t);
    
    t = GetTangentDiscrete(int_x, int_y, next_z);
    Vector t01 = t + frac_x * (GetTangentDiscrete(next_x, int_y, next_z) - t);    
  
    t = GetTangentDiscrete(int_x, next_y, int_z);
    Vector t10 = t + frac_x * (GetTangentDiscrete(next_x, next_y, int_z) - t);
    
    t = GetTangentDiscrete(int_x, next_y, next_z);
    Vector t11 = t + frac_x * (GetTangentDiscrete(next_x, next_y, next_z) - t);    
  
    // lerp 2 times in y
    Vector t0 = t00 + frac_y * (t10 - t00);
    Vector t1 = t01 + frac_y * (t11 - t01);
    
    // final lerp in z
    Vector final_t = t0 + frac_z * (t1 - t0);
    final_t.Normalize();
    
    return final_t;
}

Vector
Texel::GetTangentDiscrete(int i, int j, int k)
{
    return m_cells[i][j][k].tangent;
}

void
Texel::SetOpticalDensity(float opt_density)
{
    m_opt_density = opt_density;
}

float
Texel::GetOpticalDensity(void)
{
    return m_opt_density;
}

float
Texel::RandomRange(float lo, float hi)
{
	float x = (float) (rand() & 0x7fff) / (float) 0x7fff;
	return (x * (hi - lo) + lo);
}

float
Texel::ClampFloat(float value, float low, float high)
{
    if (value < low) return low;
    if (value > high) return high;
    return value;
}

int
Texel::ClampInt(int value, int low, int high)
{
    if (value < low) return low;
    if (value > high) return high;
    return value;
}

