
#include "lrt.h"
#include "color.h"
#include "reflection.h"

BRDF::~BRDF()
{
}

Spectrum Lambertian::fr(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return R;
	} else {
		return Spectrum(0.);
	}
}

BlinnGlossy::BlinnGlossy(const Spectrum & reflectance,
						 Float roughness, const Normal & normal,
						 const Vector & w)
{
	R = reflectance;
	roughness /= 8.;			// hack to match BMRT, prman
	invRoughness = 1. / roughness;
	N = normal;
	wo = w;
	costhetao = Dot(wo, N);
}

Spectrum BlinnGlossy::fr(const Vector & wi) const
{
	Vector H = (wi + wo).Hat();
	Float costhetah = Dot(N, H);
	Float costhetai = Dot(N, wi);
	if (costhetah > 0. && costhetai > 0.) {
		return R * pow(costhetah, invRoughness) / (costhetai * costhetao);
	} else {
		return Spectrum(0.);
	}
}

int BRDF::SpecularComponents() const
{
	return 0;
}

Spectrum BRDF::SampleSpecular(int component, Vector * wo) const
{
	Severe("SampleSpecular() shouldn't be called for BRDFs "
		   "without specular components.");
	return Spectrum(0.);
}

SpecularReflection::SpecularReflection(const Spectrum & r,
									   const Normal & N, const Vector & wo)
{
	R = r;
	reflectedDirection = -wo - 2. * Dot(N, -wo) * Vector(N);
}

Spectrum SpecularReflection::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = reflectedDirection;
	return R;
}

SpecularTransmission::SpecularTransmission(const Spectrum & r,
										   const Normal & N,
										   const Vector & wo, Float indexi,
										   Float indext)
{
	R = r;

	Float eta = indexi / indext;
	Float c1 = Dot(wo, N);
	Float c2 = 1 - (eta * eta * (1. - c1));
	if (c2 < 0.) {
		// total internal reflection
		R = 0.;
		DirT = wo;
	} else {
		c2 = sqrt(c2);
		DirT = -eta * wo + (eta * c1 - c2) * Vector(N);
	}

}

Spectrum SpecularTransmission::SampleSpecular(int component, Vector * wi) const
{
	Assert(component == 0);
	*wi = DirT;
	return R;
}

ScatteringMixture::~ScatteringMixture()
{
	vector < BRDF * >::const_iterator iter;
	for (iter = funcs.begin(); iter != funcs.end(); ++iter)
		delete *iter;
}

Spectrum ScatteringMixture::fr(const Vector & wi) const
{
	Spectrum ret(0);
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		ret += (*weightIter) * (*funcIter)->fr(wi);
		funcIter++, weightIter++;
	}
	return ret;
}

Spectrum ScatteringMixture::SampleSpecular(int component, Vector * wi) const
{
	Assert(component < SpecularComponents());
	vector < BRDF * >::const_iterator funcIter = funcs.begin();
	vector < Float >::const_iterator weightIter = weights.begin();

	while (funcIter != funcs.end()) {
		if (component >= (*funcIter)->SpecularComponents())
			component -= (*funcIter)->SpecularComponents();
		else
			return (*weightIter) *
				(*funcIter)->SampleSpecular(component, wi);
		++funcIter, ++weightIter;
	}

	// We shouldn't get to this point.
	Severe("Major logic error in" " ScatteringMixture::SampleSpecular()");
	return Spectrum(0);
}

#if 0
Lafortune::Lafortune(int n, const LafortuneLobe l[], const Vector & w,
					 const Normal & norm, const Vector & dpdu)
{
	nLobes = n;
	lobes = new LafortuneLobe[n];
	for (int i = 0; i < nLobes; ++i)
		lobes[i] = l[i];
	wo = w;
	N = norm.Hat();
	dPdu = dpdu.Hat();
	dPdv = Cross((Vector) N, dPdu);
}
#endif

#if 0
Spectrum Lafortune::fr(const Vector & wi) const
{
	Spectrum value(0.);

	Float xd = Dot(dPdu, wi) * Dot(dPdu, wo);
	Float yd = Dot(dPdv, wi) * Dot(dPdv, wo);
	Float zd = Dot(N, wi) * Dot(N, wo);

	for (int i = 0; i < nLobes; ++i) {

		Spectrum dot =
			xd * lobes[i].Cx + yd * lobes[i].Cy + zd * lobes[i].Cz;
		value += dot.Pow(lobes[i].Exponent);

	}
	return value;
}
#endif

BRDF *CreateLambertian(const Spectrum & reflectance, const Normal & normal)
{
	return new Lambertian(reflectance, normal);
}

BRDF *CreateBlinnGlossy(const Spectrum & reflectance,
						Float roughness,
						const Normal & normal, const Vector & wo)
{
	return new BlinnGlossy(reflectance, roughness, normal, wo);
}

BRDF *CreateSpecularReflection(const Spectrum & r,
							   const Normal & N, const Vector & wo)
{
	return new SpecularReflection(r, N, wo);
}

BRDF *CreateSpecularTransmission(const Spectrum & r,
								 const Normal & N,
								 const Vector & wo,
								 Float indexi, Float indext)
{
	return new SpecularTransmission(r, N, wo, indexi, indext);
}

ScatteringMixture *CreateScatteringMixture()
{
	return new ScatteringMixture;
}

#if 0
BRDF *CreateLafortune(int nLobes, const LafortuneLobe lobes[],
					  const Vector & wi, const Normal & N,
					  const Vector & dPdu)
{
	Assert(1 == 0);
	return NULL;
	// need to implement the sampling methods
//  return new Lafortune(nLobes, lobes, wi, N, dPdu);
}
#endif

Spectrum Lambertian::Sample(Float u[2], Vector * wi, Float * pdf) const
{

	Float u0 = 2 * u[0] - 1.;
	Float u1 = 2 * u[1] - 1.;
	if (u0 * u0 + u1 * u1 > 1.) {
		*pdf = 0.;
		return Spectrum(0.);
	}

	Vector Du, Dv;
	if (N.x == 0. && N.y == 0.) {
		Du = Vector(1, 0, 0);
		Dv = Vector(0, 1, 0);
	} else {
		Du = Cross(Vector(N.y, N.x, 0.), Vector(N)).Hat();
		Dv = Cross(Du, Vector(N)).Hat();
	}

	*wi = u0 * Du + u1 * Dv + sqrt(1. - (u0 * u0 + u1 * u1)) * Vector(N);
	*pdf = Dot(N, *wi) / M_PI;

	return fr(*wi);
}

Spectrum BlinnGlossy::Sample(Float u[2], Vector * wi, Float * pdf) const
{
    // pick theta and phi
	Float theta = acos(pow(u[0], 1.0f / (invRoughness + 2)));
	Float phi = 2.0f * M_PI * u[1];
	
	// compute vectors orthogonal to N to define
	// local coord system
	
	Vector dx, dy;
	
	if (N.x == 0.0f && N.y == 0.0f)
	{
	    dx = Vector(1.0f, 0.0f, 0.0f);
	    dy = Vector(0.0f, 1.0f, 0.0f);
	}
	else
	{
	    dx = Cross(Vector(N.y, N.x, 0.0f), Vector(N)).Hat();
	    dy = Cross(Vector(N), dx).Hat();
	}
	
	if (phi > M_PI * 0.5f && phi < M_PI * 1.5f)
	{
	    dx = -dx;
	}
	
	if (phi > M_PI)
	{
	    dy = -dy;
	}
	
	// construct H
	Float scale_y   = fabs(tan(phi));
	Vector s        = (dx + scale_y * dy).Hat();
	Float k         = 1.0f / tan(theta);
    Vector H        = (s + k * Vector(N)).Hat();
    
    Float costheta = Dot(N, H);
	
	*wi = 2.0f * Dot(wo, H) * H - wo;
	*pdf = (invRoughness + 2) / (2 * M_PI) * pow(costheta, invRoughness + 1);
	
	Float costhetai = Dot(*wi, H);
	
	return fr(*wi) * 4.0f * costhetai * costhetai / *pdf;
}

Spectrum ScatteringMixture::Sample(Float u[2], Vector * wi, Float * pdf) const
{
	int fnum = int (RandomFloat() * funcs.size());
	funcs[fnum]->Sample(u, wi, pdf);
	*pdf /= funcs.size();
	return fr(*wi);
}

Float Lambertian::Pdf(const Vector & wi) const
{
	Float costheta = Dot(wi, N);
	if (costheta > 0.) {
		return 1. / (M_PI * costheta);
	} else {
		return 0.;
	}
}

Float BlinnGlossy::Pdf(const Vector & wi) const
{
	// YOUR CODE HERE
	
	Float costheta = Dot(N, wi);
	
	return (invRoughness + 2) / (2 * M_PI) * pow(costheta, invRoughness + 1);
	
	//return 0.;
}

Float ScatteringMixture::Pdf(const Vector & wi) const
{
	Float pdfSum = 0.;
	for (u_int i = 0; i < funcs.size(); ++i)
		pdfSum += funcs[i]->Pdf(wi);
	return pdfSum / funcs.size();
}
