
#include "nurbs.h"
#include "trimesh.h"

static int KnotOffset(const float *knot, int order, int np, float t)
{
	int firstKnot = order - 1;
	int lastKnot = np;

	int knotOffset = firstKnot;
	while (t > knot[knotOffset + 1])
		++knotOffset;
	assert(knotOffset < lastKnot);
	assert(t >= knot[knotOffset] && t <= knot[knotOffset + 1]);
	return knotOffset;
}

// doesn't handle flat out discontinuities in the curve...

struct Homogeneous3 {
	Homogeneous3() {
		x = y = z = w = 0.;
	} Homogeneous3(Float xx, Float yy, Float zz, Float ww) {
		x = xx;
		y = yy;
		z = zz;
		w = ww;
	}
	Float x, y, z, w;
};

static Homogeneous3
NURBSEvaluate(int order, const float *knot, const Homogeneous3 * cp,
			  int np, int cpStride, float t, Vector * deriv = NULL)
{
//    int nKnots = np + order;
	float alpha;

	int knotOffset = KnotOffset(knot, order, np, t);
	knot += knotOffset;

	int cpOffset = knotOffset - order + 1;
	assert(cpOffset >= 0 && cpOffset < np);

	Homogeneous3 *cpWork =
		(Homogeneous3 *) alloca(order * sizeof(Homogeneous3));
	for (int i = 0; i < order; ++i)
		cpWork[i] = cp[(cpOffset + i) * cpStride];

	for (int i = 0; i < order - 2; ++i)
		for (int j = 0; j < order - 1 - i; ++j) {
			alpha = (knot[1 + j] - t) /
				(knot[1 + j] - knot[j + 2 - order + i]);
			assert(alpha >= 0. && alpha <= 1.);

			cpWork[j].x =
				cpWork[j].x * alpha + cpWork[j + 1].x * (1. - alpha);
			cpWork[j].y =
				cpWork[j].y * alpha + cpWork[j + 1].y * (1. - alpha);
			cpWork[j].z =
				cpWork[j].z * alpha + cpWork[j + 1].z * (1. - alpha);
			cpWork[j].w =
				cpWork[j].w * alpha + cpWork[j + 1].w * (1. - alpha);
		}

	alpha = (knot[1] - t) / (knot[1] - knot[0]);
	assert(alpha >= 0. && alpha <= 1.);

	Homogeneous3 val(cpWork[0].x * alpha + cpWork[1].x * (1. - alpha),
					 cpWork[0].y * alpha + cpWork[1].y * (1. - alpha),
					 cpWork[0].z * alpha + cpWork[1].z * (1. - alpha),
					 cpWork[0].w * alpha + cpWork[1].w * (1. - alpha));

	if (deriv) {
		float factor = (order - 1) / (knot[1] - knot[0]);
		Homogeneous3 delta((cpWork[1].x - cpWork[0].x) * factor,
						   (cpWork[1].y - cpWork[0].y) * factor,
						   (cpWork[1].z - cpWork[0].z) * factor,
						   (cpWork[1].w - cpWork[0].w) * factor);

		deriv->x = delta.x / val.w - (val.x * delta.w / (val.w * val.w));
		deriv->y = delta.y / val.w - (val.y * delta.w / (val.w * val.w));
		deriv->z = delta.z / val.w - (val.z * delta.w / (val.w * val.w));
	}

	return val;
}

static Point
NURBSEvaluateSurface(int uOrder, const float *uKnot, int ucp, float u,
					 int vOrder, const float *vKnot, int vcp, float v,
					 const Homogeneous3 * cp, Vector * dPdu, Vector * dPdv)
{
	Homogeneous3 *iso = (Homogeneous3 *) alloca(max(uOrder, vOrder) *
												sizeof(Homogeneous3));

	int uOffset = KnotOffset(uKnot, uOrder, ucp, u);
	int uFirstCp = uOffset - uOrder + 1;
	assert(uFirstCp >= 0 && uFirstCp + uOrder - 1 < ucp);

	for (int i = 0; i < uOrder; ++i)
		iso[i] = NURBSEvaluate(vOrder, vKnot, &cp[uFirstCp + i], vcp,
							   ucp, v);

	int vOffset = KnotOffset(vKnot, vOrder, vcp, v);
	int vFirstCp = vOffset - vOrder + 1;
	assert(vFirstCp >= 0 && vFirstCp + vOrder - 1 < vcp);

	Homogeneous3 P = NURBSEvaluate(uOrder, uKnot, iso - uFirstCp, ucp,
								   1, u, dPdu);

	if (dPdv) {
		for (int i = 0; i < vOrder; ++i)
			iso[i] =
				NURBSEvaluate(uOrder, uKnot, &cp[(vFirstCp + i) * ucp],
							  ucp, 1, u);
		(void) NURBSEvaluate(vOrder, vKnot, iso - vFirstCp, vcp, 1, v,
							 dPdv);
	}
	return Point(P.x / P.w, P.y / P.w, P.z / P.w);;
}

NURBS::NURBS(int numu, int uo, Float * uk, Float u0, Float u1,
			 int numv, int vo, Float * vk, Float v0, Float v1,
			 Float * p, bool homogeneous, PrimitiveAttributes * attr,
			 SurfaceFunction * surf)
:	Primitive(attr, surf)
{
	nu = numu;
	uorder = uo;
	umin = u0;
	umax = u1;
	nv = numv;
	vorder = vo;
	vmin = v0;
	vmax = v1;
	isHomogeneous = homogeneous;
	if (isHomogeneous) {
		P = new Float[4 * nu * nv];
		memcpy(P, p, 4 * nu * nv * sizeof(Float));
	} else {
		P = new Float[3 * nu * nv];
		memcpy(P, p, 3 * nu * nv * sizeof(Float));
	}
	uknot = new Float[nu + uorder];
	memcpy(uknot, uk, (nu + uorder) * sizeof(Float));
	vknot = new Float[nv + vorder];
	memcpy(vknot, vk, (nv + vorder) * sizeof(Float));
}

NURBS::~NURBS()
{
	delete[]P;
	delete[]uknot;
	delete[]vknot;
}

BBox NURBS::BoundObjectSpace() const
{
	if (!isHomogeneous) {

		Float *pp = P;
		BBox bound = Point(pp[0], pp[1], pp[2]);
		for (int i = 0; i < nu * nv; ++i, pp += 3)
			bound = Union(bound, Point(pp[0], pp[1], pp[2]));
		return bound;

	} else {

		Float *pp = P;
		BBox bound = Point(pp[0] / pp[3], pp[1] / pp[3], pp[2] / pp[3]);
		for (int i = 0; i < nu * nv; ++i, pp += 4)
			bound =
				Union(bound,
					  Point(pp[0] / pp[3], pp[1] / pp[3], pp[2] / pp[3]));
		return bound;

	}
}

BBox NURBS::BoundWorldSpace() const
{
	if (!isHomogeneous) {

		Float *pp = P;
		Point pt = attributes->ObjectToWorld(Point(pp[0], pp[1], pp[2]));
		BBox bound = pt;
		for (int i = 0; i < nu * nv; ++i, pp += 3) {
			pt = attributes->ObjectToWorld(Point(pp[0], pp[1], pp[2]));
			bound = Union(bound, pt);
		}
		return bound;

	} else {

		Float *pp = P;
		Point pt = attributes->ObjectToWorld(Point(pp[0] / pp[3],
												   pp[1] / pp[3],
												   pp[2] / pp[3]));
		BBox bound = pt;
		for (int i = 0; i < nu * nv; ++i, pp += 4) {
			pt = attributes->ObjectToWorld(Point(pp[0] / pp[3],
												 pp[1] / pp[3],
												 pp[2] / pp[3]));
			bound = Union(bound, pt);
		}
		return bound;

	}
}


void NURBS::Refine(vector < Primitive * >*refined) const
{

	int diceu = 30, dicev = 30;
	Float *ueval = new Float[diceu];
	Float *veval = new Float[dicev];
	Point *evalPs = new Point[diceu * dicev];
	Point *evalNs = new Point[diceu * dicev];
	int i;
	for (i = 0; i < diceu; ++i)
		ueval[i] = Lerp((Float) i / (Float) (diceu - 1), umin, umax);
	for (i = 0; i < dicev; ++i)
		veval[i] = Lerp((Float) i / (Float) (dicev - 1), vmin, vmax);

	memset(evalPs, 0, diceu * dicev * sizeof(Point));
	memset(evalNs, 0, diceu * dicev * sizeof(Point));

	int nTris = 2 * (diceu - 1) * (dicev - 1);
	int *vertices = new int[3 * nTris];
	int *vertp = vertices;

	for (int v = 0; v < dicev - 1; ++v) {
		for (int u = 0; u < diceu - 1; ++u) {
#define VN(u,v) ((v)*diceu+(u))
			*vertp++ = VN(u, v);
			*vertp++ = VN(u + 1, v);
			*vertp++ = VN(u + 1, v + 1);

			*vertp++ = VN(u, v);
			*vertp++ = VN(u + 1, v + 1);
			*vertp++ = VN(u, v + 1);
#undef VN
		}
	}

	refined->push_back(new TriangleMesh(nTris, diceu * dicev, vertices,
										evalPs, attributes, NULL));
	// XXX Need to make a new surface function and hold the N values

	delete[]ueval;
	delete[]veval;
	delete[]evalPs;
	delete[]evalNs;

}
