
#include "lrt.h"
#include "primitives.h"
#include "accel.h"
#include "color.h"
#include "light.h"
#include "scene.h"
#include "reflection.h"
#include "sampling.h"
#include "shading.h"
#include "transport.h"
#include "math.h"

Integrator::~Integrator()
{
}

Spectrum ColorIntegrator::Integrate(const Ray & ray,
									HitInfo * hitInfo, Float * hitDist,
									Float * alpha) const
{
	if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
		*alpha = 1.;
		const Spectrum *Csp = hitInfo->GetColor(RI_CS);
		if (Csp) {
			return *Csp;
		} else {
			return hitInfo->hitPrim->attributes->Color;
		}
	} else {
		*alpha = 0.;
		return Spectrum(0, 0, 0);
	}
}

Spectrum RayCastingIntegrator::Integrate(const Ray & ray,
										 HitInfo * hitInfo,
										 Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 0., hitDist, hitInfo)) {
		ShadeContext shadeContext(hitInfo, -ray.D);
		Spectrum L(0.);
		*alpha = 1.;

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

Spectrum WhittedIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
									  Float * hitDist, Float * alpha) const
{
	if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
		Spectrum L(0.);
		*alpha = 1.;

		ShadeContext shadeContext(hitInfo, -ray.D);

		if (hitInfo->hitPrim->attributes->LightShader != NULL) {
			L +=
				hitInfo->hitPrim->attributes->LightShader->
				Le(shadeContext);
		}

		if (!hitInfo->hitPrim->attributes->Surface)
			return L;
		BRDF *brdf =
			hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);

		Point Pw =
			hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
		const list < Light * >&lights =
			hitInfo->hitPrim->attributes->Lights;
		list < Light * >::const_iterator iter = lights.begin();
		while (iter != lights.end()) {

			const Light *lt = *iter;
			Point Plight;
			Spectrum dE = lt->dE(Pw, &Plight);
			if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) {
				Vector wi = (Plight - Pw).Hat();
				const Normal & Nw = shadeContext.Ns;
				L += dE * brdf->fr(wi) * Dot(wi, Nw);
			}

			++iter;
		}

		if (++RayDepth < 5) {
			Ray newRay;
			newRay.O = Pw;
			for (int i = 0; i < brdf->SpecularComponents(); ++i) {

				Spectrum kernel = brdf->SampleSpecular(i, &newRay.D);
				HitInfo hitInfo2;
				Float alpha2, maxt = INFINITY;
				L += kernel * Integrate(newRay, &hitInfo2, &maxt, &alpha2);

			}
		}
		--RayDepth;

		delete brdf;

		return L;
	} else {
		*alpha = 0.;
		return Spectrum(0.);
	}
}

int WhittedIntegrator::RayDepth = 0;

Float MCIntegrator::IntegrateSS(const Ray & ray, Float w, 
				   Point surface_p, Vector norm, 
				   const Ray & outer_ray) const
{
  int debug = 0;

  Float sigma_s = 50.0;               // scattering cross section
  Float sigma_a = 3.8;                // absorption cross section
  Float sigma_t = sigma_s + sigma_a;  // total cross section
  Float g       = 0.3;                // mean cosine of phase function
  Float r       = RandomFloat();      // random float between 0 and 1

  Float thickness = 0.1;

  Float weight = w;

  // 2A Step: estimate distance to next interaction:
  //
  // d = - log r / sigma t

  Float d = - log(r) / sigma_t;

  //
  // compute new position: p = p + ds
  //

  //  Point p = surface_p + d*ray.D;
  //  printf("ray.O=(%f, %f, %f), p1=(%f, %f, %f)\n", ray.O.x, ray.O.y, ray.O.z, p.x, p.y, p.z);
  Point p = ray.O + d*ray.D.Hat();

  if(debug) {
    printf("ray.O=(%f, %f, %f), ray.D=(%f, %f, %f)\n", ray.O.x, ray.O.y, ray.O.z, 
	   ray.D.x, ray.D.y, ray.D.z);
    printf("d=%f, p=(%f, %f, %f)\n", d, p.x, p.y, p.z);
    printf("surface_p=(%f, %f, %f)\n", surface_p.x, surface_p.y, surface_p.z);
  }
  // check to see if we have exited from the subsurface

  // calculate equation of the plane:

  Float A = norm.x;
  Float B = norm.y;
  Float C = norm.z;
  Float D = A*surface_p.x + B*surface_p.y + C*surface_p.z;

  Vector p_to_orig = Vector(p.x-surface_p.x, p.y-surface_p.y, p.z-surface_p.z);
  Float  new_dist  = Dot(norm, p_to_orig);

  /*
  Float distABC = sqrt(A*A + B*B + C*C);
  printf("A=%f B=%f C=%f D=%f dist=%f\n", A, B, C, D, distABC);
  Float distance =  (A*p.x + B*p.y + C*p.z + D)/sqrt(A*A + B*B + C*C);
  printf("distance from plane=%f, new_dist=%f\n", distance, new_dist);
  */
  if(debug)
    printf("new_dist=%f\n", new_dist);


  // if point is above surface:
  //  if ((A*p.x + B*p.y + C*p.z)<0) {
  if (new_dist>0) {

    // find distance to surface along ray
    /*
    Float dist = 
      (A*ray.O.x + B*ray.O.y + C*ray.O.z) /
      (A*(ray.O.x-p.x) + B*(ray.O.y-p.y) + C*(ray.O.z-p.z));

    printf("dist = %f\n", dist);
    */

    if(debug)
      printf("  ray has bounced back out of subsurface, weight=%f\n", weight);

    // calculate refraction angle coming out of subsurface

    Float refr_indx = 1.33;
    
    Float costhetaprime = Dot(ray.D, norm);

    // if angle is greater than the critial angle, then we have
    // total internal reflection

    if (acos(costhetaprime) > asin(1/refr_indx))
      return 0.; //Spectrum(0.);

    else {
      HitInfo hitInfo2;
      Float alpha2, maxt = INFINITY;

      weight = w * sigma_s / (sigma_s + .5*sigma_a);

      return weight; // * Integrate(outer_ray, &hitInfo2, &maxt, &alpha2);
    }
  }

  // else check if point has penetrated through the subsurface
  //  else if( (A*p.x + B*p.y + C*p.z + D)/sqrt(A*A + B*B + C*C) > thickness) {
  else if( -new_dist > thickness) {

    if(debug)
      printf("  ray has penetrated through subsurface (transmission)\n");

    return 0.; //Spectrum(0.); 
    // our modelling of the petals doesn't allow for transmission :(

    /*
    Vector Li = ray.D.Hat();

    Float costhetai = Dot(-Li, norm);
    Float thetai = acos(costhetai);

    Vector v = Cross(-Li, norm);
    Transform thetaprime_rot = Rotate(Degrees(thetai), v);
    Vector vLti = -thetaprime_rot(-norm);

    Ray ray2 = Ray(outer_ray.O - vLti, vLti);  // get old point & subtract off vLti ???
    HitInfo hitInfo2;
    Float alpha2, maxt = INFINITY;

    return weight * Integrate(ray2, &hitInfo2, &maxt, &alpha2);
    */
  }

  //   return Integrate();
  //

  // set particle weight to:
  //
  //   w = w * sigma s / (sigma s + sigma a)
  //

  weight = w * sigma_s / (sigma_s + sigma_a);

  if(debug)
    printf("  weight = %f\n", weight);

  // if weight falls below 5%, return 0;
  if(weight < 0.05)
    return 0; //Spectrum(0.);

  // 2B Scatter: estimate cosine of the scattering angle for the Henyey-Greenstein 
  //             phase function

  Float cos_j = (1./fabs(2.*g)) * (1. + g*g - pow( (1.-g*g)/(1.-g+2.*g*r), 2) );
  Float sin_j = sin(acos(cos_j));

  Float cos_phi   = cos(2 * M_PI * r);
  Float sin_phi   = sin(2 * M_PI * r);
  
  Float cos_theta = ray.D.z;
  Float sin_theta = sqrt(1 - ray.D.z*ray.D.z);

  Ray new_ray;

  if(sin_theta != 0) {
    Vector t = Vector((ray.D.x*cos_phi*cos_theta - ray.D.y*sin_phi) / sin_theta,
		      (ray.D.y*cos_phi*cos_theta - ray.D.x*sin_phi) / sin_theta,
		      sin_theta);
    if(debug) {
      printf("t=(%f, %f, %f)\n", t.x, t.y, t.z);
      printf("cos_j=%f, sin_j=%f\n", cos_j, sin_j);
    }
    new_ray = Ray(p, ray.D*cos_j + t*sin_j);
  }
  else
    new_ray = Ray(p, ray.D);

  return IntegrateSS(new_ray, weight, surface_p, norm, outer_ray);

}
//#endif

Spectrum MCIntegrator::Integrate(const Ray & ray, HitInfo * hitInfo,
				 Float * hitDist, Float * alpha) const
{
  int debug = 0;

  if (scene->Intersect(ray, 1e-4, hitDist, hitInfo)) {
    ShadeContext shadeContext(hitInfo, -ray.D);
    const Normal & Nw = shadeContext.Ns;

    //    printf(" Nw=(%f, %f, %f)\n", Nw.x, Nw.y, Nw.z);

    Spectrum L(0.);
    *alpha = 1.;
    
    if (hitInfo->hitPrim->attributes->LightShader != NULL) {
      L +=
	hitInfo->hitPrim->attributes->LightShader->
	Le(shadeContext);
    }
    
    if (!hitInfo->hitPrim->attributes->Surface)
      return L;
    BRDF *brdf =
      hitInfo->hitPrim->attributes->Surface->Shade(shadeContext);
    
    if (hitInfo->hitPrim->attributes->Sampling ==
	PrimitiveAttributes::SampleSurface) {
      
    }
    else if (hitInfo->hitPrim->attributes->Sampling ==
	     PrimitiveAttributes::SampleLight) {
      
      Point Pw = hitInfo->hitPrim->attributes->ObjectToWorld(hitInfo->Pobj);
      const list < Light * >&lights = hitInfo->hitPrim->attributes->Lights;
      list < Light * >::const_iterator iter = lights.begin();
      while (iter != lights.end()) {
      
	const Light *lt = *iter;
	Point Plight;
	Spectrum dE;
	int samples = lt->NumSamples();
	//	Spectrum lightL(0.0);

	for( int i = 0; i < samples; i++ ) {	
	  dE = lt->dE(Pw, &Plight);
	  if (!lt->CastsShadows() || scene->Unoccluded(Pw, Plight)) { 
	    Vector wi = (Plight - Pw).Hat();
	    //	    const Normal & Nw = shadeContext.Ns;
	    L += dE * brdf->fr(wi) * Dot(wi, Nw) / ((Float) samples);
	  }
	}		
	++iter;
      }
    }
    
    else if (hitInfo->hitPrim->attributes->Sampling ==
	     PrimitiveAttributes::SampleCombination) {
      // YOUR CODE HERE: integrate by sampling a little bit
      // from both and then combining the results intelligently
    }
    
    delete brdf;
    
    // Lr (reflected radiance) = 
    //
    //   Lr,s (reflected radiance due to surface scattering) +
    // 
    //   Lr,v (reflected radiance due to subsurface scattering)
    //
    // fr = R*fr,s + (1-R)*fr,v
    //

    // calculate theta' - the refracted angle between the Lss
    // and the surface normal

    Float R = 0.75;

    Float  refr_indx = 1.33;
    Vector Norm = Vector(Nw.x, Nw.y, Nw.z).Hat();
    Vector Li   = ray.D.Hat();

    Float costhetai = Dot(-Li, Norm);
    Float thetai = acos(costhetai);
    Float thetaprime = asin(sin(thetai)/refr_indx);

    //printf("thetai=%f, thetaprime=%f\n", thetai, thetaprime);

    Vector v = Cross(-Li, Norm);
    Transform thetaprime_rot = Rotate(-Degrees(thetaprime), v);
    Vector vLss = thetaprime_rot(-Norm);

    /*
    printf("Li=(%f, %f, %f) vLss=(%f, %f, %f)\n",
    	   Li.x, Li.y, Li.z, vLss.x, vLss.y, vLss.z);
    */

    Ray rLss = Ray(hitInfo->Pobj, vLss);
    //    HitInfo hitInfo2;
    //    Float alpha2, maxt = INFINITY;
    Float Lrs; // weight of the contribution from subsurface scattering

    if(RayDepth == 0) {
      RayDepth++;

      if(debug)
	printf("calling IntegrateSS...\n");
      Lrs = IntegrateSS(rLss, 1, hitInfo->Pobj, Norm, ray);

      RayDepth--;

      return (R + (1-R)*Lrs) * L;
    }
    else
      return L;

  } else {
    *alpha = 0.;
    return Spectrum(0.);
  }
}

int MCIntegrator::RayDepth = 0;


#include "sl.h"


static void null_shader(RmanShaderParams & params)
{
	params.Ci = SLTriple(0);
	params.Oi = SLTriple(1);
}

RendermanShader::RendermanShader(const char *name)
{
	char path[4096];
	this->name = strdup(name);
	char *shader_dir = getenv("LRT_SHADER_DIR");
	if (!shader_dir)
		shader_dir = "/home/humper/work/litrt/src/shaders";

	sprintf(path, "%s/%s.so", shader_dir, name);
	dso = new DSO(path);

	already_warned = false;

#define RMAN_MUST_HAVE( var, sym ) \
	void *var = dso->GetSym( sym ); \
	if (!var) { \
		Warning( "No %s in %s", sym, path );\
		type = RMAN_BROKEN_SHADER; \
		shader = null_shader; \
		return; \
	}

	RMAN_MUST_HAVE(type_ptr, "shader_type");
	RMAN_MUST_HAVE(shader_ptr, "shader");

#undef RMAN_MUST_HAVE

	type = *((RmanShaderType *) type_ptr);
	shader = (shader_func) shader_ptr;
}

void RendermanShader::Run(RmanShaderParams & params)
{
	switch (type) {
	case RMAN_SURFACE_SHADER:
		params.Cs =
			SLTriple(params.shadeContext.
					 InitializeColor(RI_CS, Spectrum(1.0)));
		params.Os = SLTriple(Spectrum(1, 1, 1));
		params.N = SLTriple(params.shadeContext.Ns);
		params.P =
			SLTriple(params.hitInfo->hitPrim->attributes->
					 ObjectToWorld(params.hitInfo->Pobj));
		params.I = SLTriple(params.shadeContext.wo);
		params.Ci = SLTriple(0, 0, 0);
		params.Os = SLTriple(1, 1, 1);
		shader(params);
		break;
	case RMAN_BROKEN_SHADER:
		if (!already_warned) {
			already_warned = 1;
			Warning("Shader %s is broken.", name);
		}
		params.Ci = SLTriple(RandomFloat(), RandomFloat(), RandomFloat());
		params.Oi = SLTriple(1, 1, 1);
		break;
	default:
		if (!already_warned) {
			already_warned = 1;
			Warning("I don't know how to handle shader %s", name);
		}
		break;
	}
}

Spectrum RendermanIntegrator::Integrate(const Ray & ray,
										HitInfo * hitInfo, Float * hitDist,
										Float * alpha) const
{

	if (scene->Intersect(ray, 0., hitDist, hitInfo) &&
		hitInfo->hitPrim->attributes->Surface != NULL) {
		ShadeContext shadeContext(hitInfo, -ray.D);
		RmanShaderParams params(hitInfo->hitPrim->attributes->Surface->
								surfaceFunction, shadeContext,
								*scene->sampler, hitInfo);

		*alpha = 1;

		RendermanShader *shader = NULL;
		Assert(shader != NULL);
		/* XXX MMP: shader creation and management like this should happen up at
		   the RI level, I think, so that this guy can just grab a useful Shader *
		   from the HitInfo.  I've commented this out for now so that I can clean
		   up stuff in shading.nw for 348 assignment 3. */
#if 0
		RendermanShader *shader =
			(RendermanShader *) shaders.Search(hitInfo->attributes->
											   Surface->Name());
		if (!shader) {
			shader =
				new RendermanShader(hitInfo->attributes->Surface->Name());
			shaders.Add(hitInfo->attributes->Surface->Name(), shader);
		}
#endif
		shader->Run(params);

		return params.Ci.MakeSpectrum();
	} else {
		*alpha = 0;
		return Spectrum(0.);
	}
}
